# -*- coding: utf-8 -*-

import sys
import logging
logger = logging.getLogger(__name__)

from collections import defaultdict, namedtuple
from itertools import chain

import networkx as nx

import friendlysam as fs
from friendlysam.opt import Constraint, VariableCollection, namespace
from friendlysam.compat import ignored



class ConstraintCollection(object):
    """
    Generates constraints from functions.

    This class aggregates functions that generate constraints. 
    It is primarily meant to be used as an attribute of the :class:`Part` class.

    Add functions to it, functions that return constraints. Then call
    the :meth:`make` with an index and receive the set of
    all constraints generated by the contained functions with that index.

    See docs for :attr:`Part.constraints` for details.

    """
    def __init__(self, owner):
        super().__init__()
        self._owner = owner
        self._constraint_funcs = set()

    _origin_tuple = namedtuple('CallTo', ['func', 'index', 'owner'])


    def make(self, index):
        """Create constraints from contained functions.

        Args:
            index: The index to call the constraint functions with.

        Examples:

            c = ConstraintCollection(owner)
            c.add(func1)
            c += func2 # Alternative syntax.
            c(index) # Returns a set with all the constraints from func1 and func2

        """
        constraints = set()

        for func in self._constraint_funcs:
            origin = self._origin_tuple(func=func, index=index, owner=self._owner)
            func_output = func(index)
            try:
                func_output = iter(func_output)
            except TypeError: # not iterable
                func_output = (func_output,)

            for constraint in func_output:
                if isinstance(constraint, fs.Relation):
                    constraint = Constraint(constraint)

                if constraint.origin is None:
                    constraint.origin = origin

                constraints.add(constraint)

        return constraints

    def _add_constraint_func(self, func):
        if not callable(func):
            raise ValueError('constraint funcs must be callable but {} is not'.format(func))
        self._constraint_funcs.add(func)

    def add(self, addition):
        """Add a constraint function, or an iterable of constraint functions.

        Args:
            addition (callable or iterable of callables): Constraint function(s)
                to add.

        Examples:

            c = ConstraintCollection(owner)
            c.add(constraint_func)
            c.add([func1, func2])

            c += constraint_func
            c += [func1, func2]
        """
        try:
            for func in addition:
                self._add_constraint_func(func)
        except TypeError:
            self._add_constraint_func(addition)

    def __iadd__(self, addition):
        self.add(addition)
        return self


class Part(object):
    """A part of a model.

    End users probably primarily want to use the subclasses
    :class:`Node`, :class:`FlowNetwork`, etc.

    The ``Part`` class serves several purposes:

        1. ``Part`` has the attribute :attr:`constraints`.

        2. A ``Part`` can contain other parts. Read more about

            * :meth:`add_part`, :meth:`remove_part`
            * :meth:`parts`
            * :meth:`find`
            * :attr:`children`, :attr:`children_and_self`, :attr:`descendants`, :attr:`descendants_and_self`

        3. ``Part`` implements Friendly Sam's time model. Read more about

            * :meth:`step_time`
            * :meth:`times` and :meth:`iter_times`
            * :meth:`times_between` and :meth:`iter_times_between`

    Args:
        name (str, optional): The :attr:`name` of the part.

    Examples:

        See :class:`Node` for examples.

    """

    _subclass_counters = defaultdict(int)

    def __new__(cls, *args, **kwargs):
        self = super().__new__(cls)
        self._subclass_counters[type(self)] += 1
        self._constraints = ConstraintCollection(self)
        self._parts = set()

        self.name = '{}{:04d}'.format(type(self).__name__, self._subclass_counters[type(self)])
        return self

    def __init__(self, name=None):
        if name is not None:
            self.name = name

    @property
    def name(self):
        """A name for the object.

        The name is for debugging purposes. It has nothing to do
        with the identity of the object, so does not have to be unique
        """
        return self._name
    @name.setter
    def name(self, value):
        self._name = value

    _time_unit = 1
    @property
    def time_unit(self):
        """The time unit used in :meth:`step_time`.

        The default value is 1.

        For more info, see :meth:`step_time`.
        """
        return self._time_unit
    @time_unit.setter
    def time_unit(self, value):
        self._time_unit = value
    
    def step_time(self, index, num_steps):
        """A function for stepping forward or backward in time.

        A :class:`Part` (or subclass) instance may use any logic 
        for stepping in time. To change time stepping, you may have to change
        :attr:`time_step` or override :meth:`step_time`.

        Args:
            index (any object): The index to step from.
            num_steps (int): The number of steps to take.

        Returns:
            the new time, ``num_steps`` away from index.

        Examples:

            If your model is indexed in evenly spaced integers, the default
            implementation is enough. A step is taken as follows::

                def step_time(self, index, num_steps):
                    return index + self.time_unit * num_steps

            The default ``time_unit`` is ``1``, so time stepping is done
            by adding an integer.

                >>> part = Part()
                >>> part.step_time(3, -2)
                1
                >>> part.time_unit = 10
                >>> part.step_time(3, 2)
                23

            Let's assume your model is indexed with ``pandas.Timestamp`` in
            2-hour increments, then it's still sufficient to change the time unit::

                >>> from pandas import Timestamp, Timedelta
                >>> part = Part()
                >>> part.time_unit = Timedelta('2h')
                >>> part.step_time(Timestamp('2015-06-10 16:00'), 1)
                Timestamp('2015-06-10 18:00:00')
                >>> part.step_time(Timestamp('2015-06-10 16:00'), -3)
                Timestamp('2015-06-10 10:00:00')

            If your model is indexed with something more complicated, you may have to change
            the :meth:`step_time` method. For example, assume a model is indexed
            with two-element tuples where the first element is an integer representing time.
            Then override the :meth:`step_time` as follows::

                >>> def my_step_time(index, step):
                ...     t, other = index
                ...     t += step
                ...     return (t, other)
                ...
                >>> part = Part()
                >>> part.step_time = my_step_time
                >>> part.step_time((1, 'ABC'), 2)
                (3, 'ABC')


        """
        return index + self.time_unit * num_steps

    def iter_times(self, start, *range_args):
        """A generator yielding a sequence of times.

        See also: :meth:`times`, :meth:`iter_times_between`, :meth:`times_between`.

        Equivalent to::

            for num_steps in range(*range_args):
                yield self.step_time(start, num_steps)

        Args:
            start (any object): The index to start from.
            *range_args: Args exactly like for the built-in ``range()``.

        Examples:

            >>> part = Part()
            >>> for t in part.iter_times(3, 5):
            ...     print(t)
            ...
            3
            4
            5
            6
            7
            >>> for t in part.iter_times(0, -5, 1, 2):
            ...     print(t)
            ...
            -5
            -3
            -1
        """

        for num_steps in range(*range_args):
            yield self.step_time(start, num_steps)

    def times(self, start, *range_args):
        """Get a sequence of times.

        See also: :meth:`iter_times`, :meth:`iter_times_between`, :meth:`times_between`.

        This works exactly like :meth:`iter_times`, but returns a tuple.

        Args:
            start (any object): The index to start from.
            *range_args: Args exactly like for the built-in ``range()``.

        Examples:

            >>> part = Part()
            >>> part.times(3, 5)
            (3, 4, 5, 6, 7)
            >>> part.times(0, -5, 1, 2)
            (-5, -3, -1)

        """
        return tuple(self.iter_times(start, *range_args))

    def iter_times_between(self, start, end):
        """A generator yielding all times between two points.

        See also: :meth:`times_between`, :meth:`iter_times`, :meth:`times`.

        Takes one time step at a time from ``start`` while ``<= end``.

        Args:
            start (any object): The index to start from.
            end (any object): The index to go to.

        Note:

            This function only works if times are orderable,
            or specifically that the ``<=`` operator is implemented.

        Examples:

            >>> from pandas import Timestamp, Timedelta
            >>> part = Part()
            >>> part.time_unit = Timedelta('7 days')
            >>> start, end = Timestamp('2011-02-14'), Timestamp('2011-02-28')
            >>> between = part.iter_times_between(start, end)
            >>> next(between)
            Timestamp('2011-02-14 00:00:00')
            >>> next(between)
            Timestamp('2011-02-21 00:00:00')
            >>> next(between)
            Timestamp('2011-02-28 00:00:00')

        """
        time = start
        while time <= end:
            yield time
            time = self.step_time(time, 1)

    def times_between(self, start, end):
        """Get a tuple of all times between two points.

        See also: :meth:`times_between`, :meth:`iter_times`, :meth:`times`.

        Takes one time step at a time from ``start`` while ``<= end``.
        This works exactly like :meth:`iter_times_between`, but returns a tuple.

        Args:
            start (any object): The index to start from.
            end (any object): The index to go to.

        Note:

            This function only works if times are orderable,
            or specifically that the ``<=`` operator is implemented.

        Examples:

            >>> from pandas import Timestamp, Timedelta
            >>> part = Part()
            >>> part.time_unit = Timedelta('7 days')
            >>> start, end = Timestamp('2011-02-14'), Timestamp('2011-02-28')
            >>> part.times_between(start, end)
            (Timestamp('2011-02-14 00:00:00'), Timestamp('2011-02-21 00:00:00'), Timestamp('2011-02-28 00:00:00'))

        """
        return tuple(self.iter_times_between(start, end))

    @property
    def constraints(self):
        '''For defining and generating constraints.

        This is a :class:`ConstraintCollection` instance. Add functions or 
        iterables of functions to it. Each function
        should return a constraint or an iterable of constraints.
        (In this context, a constraint is :class:`~friendlysam.opt.Constraint`,
        :class:`~friendlysam.opt.Relation`, :class:`~friendlysam.opt.SOS1` or
        :class:`~friendlysam.opt.SOS2`.)

        If a constraint function returns a :class:`~friendlysam.opt.Relation`,
        it automatically packaged in a :class:`~friendlysam.opt.Constraint` object
        and marked with :attr:`~friendlysam.opt.Constraint.origin` after creation.
        A constraint function may also return an iterable of constraints,
        even a generator.

        All the added constraint functions are called when 
        :meth:`~ConstraintCollection.make` is called.

        Examples:

            There are many ways to formulate constraint functions. Here
            is a wonderfully contrived example:

            >>> from friendlysam.opt import VariableCollection, Constraint, Eq
            >>> class MyNode(Node):
            ...     def __init__(self, k):
            ...         self.k = k
            ...         self.var = VariableCollection('x')
            ...         self.production['foo'] = self.var
            ...         # += and .add() are just alternative syntaxes
            ...         self.constraints.add(lambda t: self.var(t) >= k[t] - 1)
            ...         self.constraints += self.constraint_func_1, self.constraint_func_2
            ...         self.constraints.add(self.constraint_func_3)
            ...
            ...     def constraint_func_1(self, t):
            ...         return Constraint(
            ...             self.var(t) <= self.k[t] * 2,
            ...             desc='Some description')
            ...
            ...     def constraint_func_2(self, t):
            ...         constraints = []
            ...         t_plus_1 = self.step_time(t, 1)
            ...         constraints.append(self.var(t) <= self.k[t] * self.var(t_plus_1))
            ...         constraints.append(self.var(t) >= self.k[t_plus_1] * self.var(t_plus_1))
            ...         return constraints
            ...
            ...     def constraint_func_3(self, t):
            ...         for prev in self.times_between(0, t):
            ...             expr = Eq(self.k[prev] * self.var(prev), self.k[t] * self.var(t))
            ...             desc = 'Why make such a constraint? (k(t)={})'.format(self.k[t])
            ...             yield Constraint(expr, desc=desc)
            ...
            >>> my_k = {i: i ** 2.4 for i in range(100)}
            >>> node = MyNode(my_k)
            >>> constraints = node.constraints.make(20)
            >>> len(constraints)
            26


            Constraints can also be added from "outside":

            >>> node.constraints += lambda index: node.production['foo'](index) >= index
            >>> constraints = node.constraints.make(20)
            >>> len(constraints)
            27

            '''

        return self._constraints

    @constraints.setter
    def constraints(self, value):
        if value is not self._constraints:
            raise AttributeError("Don't replace the constraint collection. "
                "Use constraints.add() to add constraint functions.")

    def __repr__(self):
        if self.name:
            return '<{} at {}: {}>'.format(self.__class__.__name__, hex(id(self)), self)
        else:
            return '<{} at {} (unnamed)>'.format(self.__class__.__name__, hex(id(self)))

    def __str__(self):
        return self.name


    def find(self, name):
        """Try to find a part by name.

        Searches among :attr:`descendants_and_self`, comparing the :attr:`name`.
        If there is exactly one match, it is returned.

        Args:
            name: The name to search for.

        Returns:
            A part named ``name``, if one exists among :attr:`descendants_and_self`.

        Raises:
            ValueError: If there is no match or several matches.
        """
        matches = [part for part in self.descendants_and_self if part.name == name]
        if len(matches) == 1:
            return matches[0]
        elif len(matches) == 0:
            raise ValueError("'{}' has no part '{}'".format(repr(self), name))
        elif len(matches) > 1:
            raise ValueError(
                "'{}' has more than one part '{}'".format(repr(self), name))


    def parts(self, depth='inf', include_self=True):
        """Get contained parts, recursively.

        See also properties :attr:`children`, :attr:`children_and_self`,
        :attr:`descendants`, :attr:`descendants_and_self`.

        Args:
            depth (integer or 'inf', optional): The recursion depth to search with.
                depth=1 searches among the parts directly contained by this part.
                depth=2 among children and their children, etc.
            include_self (boolean, optional): Include this part in the results?

        Returns:
            A set of parts.

        Examples:

            >>> child = Part(name='Baby')
            >>> parent = Part(name='Mommy')
            >>> parent.add_part(child)
            >>> grandparent = Part('Granny')
            >>> grandparent.add_part(parent)
            >>> grandparent.children == {parent}
            True
            >>> grandparent.descendants == {parent, child}
            True
            >>> grandparent.descendants_and_self == {grandparent, parent, child}
            True
            >>> grandparent.parts(depth=1, include_self=False) == grandparent.children
            True

        """
        parts = set()
        depth = float(depth)

        if depth >= 1:
            parts.update(self._parts)

        parts.update(*(subpart.parts(depth=depth - 1, include_self=False) for subpart in parts))

        if include_self:
            parts.add(self)

        return parts

    @property    
    def children(self):
        """Parts in this part, excluding ``self``.

        To add children, use :meth:`add_part`.
        """
        return self.parts(depth=1, include_self=False)

    @property
    def children_and_self(self):
        """Parts in this part, including ``self``.

        To add children, use :meth:`add_part`.
        """
        return self.parts(depth=1, include_self=True)

    @property
    def descendants(self):
        """All :attr:`children`, children of children, etc, excluding ``self``."""
        return self.parts(depth='inf', include_self=False)

    @property
    def descendants_and_self(self):
        """All :attr:`children`, children of children, etc, including ``self``."""
        return self.parts(depth='inf', include_self=True)


    def add_part(self, part):
        """Add a part to this part.

        Args:
            part (:class:`Part` or subclass instance): The part to add.

        Raises:
            InsanityError: If the calling part is a descendant of the part to add.
                (This would generate a cyclic relationship.)
        """
        if self in part.descendants_and_self:
            raise fs.InsanityError(
                ('cannot add {} to {} because it would '
                'generate a cyclic relationship').format(part, self))

        self._parts.add(part)


    def remove_part(self, part):
        """Remove a part from this part.

        Args:
            p (:class:`Part` or subclass instance): The part to remove.

        Raises:
            KeyError: If the part is not there.
        """
        with ignored(KeyError):
            self._parts.remove(part)


    def state_variables(self, index):
        """The state variables of the part.

        Each subclass may define :meth:`state_variables`, returning
        an iterable of the :class:`Varible` instances that define the state
        of the part at the specified index.

        :class:`friendlysam.models.MyopicDispatchModel` is an example
        of how it can be used.

        Args:
            index: The index of the state.

        Examples:

            >>> from friendlysam import VariableCollection, Domain
            >>> class ChocolateFactory(Node):
            ...     def __init__(self):
            ...         self.total = VariableCollection('total production')
            ...         self.mc = VariableCollection('milk chocolate production')
            ...         self.production['dark chocolate'] = lambda t: self.total(t) - self.mc(t)
            ...         self.production['milk chocolate'] = self.mc
            ...         
            ...     def state_variables(self, t):
            ...         return (self.total, self.mc)

        """
        msg = "{} has not defined state_variables".format(repr(self))
        raise AttributeError(msg).with_traceback(sys.exc_info()[2])

class Node(Part):
    """A node with balance constraints.

    Suitable for modeling nodes in a flow network. A :class:`Node`
    instance produces balance constraints for all its :attr:`resources`.
    The dictionaries :attr:`consumption`, :attr:`production`, 
    :attr:`accumulation`, :attr:`inflows`, and :attr:`outflows`, are the
    basis for the balance constraints.

    Args:
        name (str, optional): A name for the node.

    Examples:

        >>> class PowerPlant(Node):
        ...     def __init__(self, efficiency):
        ...         with namespace(self):
        ...             x = VariableCollection('output')
        ...         self.production['power'] = x
        ...         self.consumption['fuel'] = lambda t: x(t) / efficiency
        ...
        >>> power_plant = PowerPlant(0.85)
        >>> constraints = power_plant.constraints.make(42)
        >>> constraints
        {<friendlysam.opt.Constraint at 0x...>, <friendlysam.opt.Constraint at 0x...>}
        >>> {c.desc for c in constraints} == {'Balance constraint (resource=power)',
        ...                                   'Balance constraint (resource=fuel)'}
        True

    """

    def __new__(cls, *args, **kwargs):
        self = super().__new__(cls, *args, **kwargs)
        self._consumption = dict()
        self._production = dict()
        self._accumulation = dict()
        self._inflows = defaultdict(set)
        self._outflows = defaultdict(set)
        self._clusters = dict()

        self.constraints += self.balance_constraints

        return self

    def __init__(self, name=None):
        if name is not None:
            self.name = name

    @property
    def consumption(self):
        """A dictionary of consumption functions.

        Each key in the
        dictionary is a resource, and the value is a function, taking one argument
        ``index``, returning the consumption at that index.
        """
        return self._consumption
    
    @property
    def production(self):
        """A dictionary of production functions.

        See :attr:`consumption`.
        """
        return self._production

    @property
    def accumulation(self):
        """A dictionary of accumulation functions.

        See :attr:`consumption`.
        """
        return self._accumulation

    @property
    def inflows(self):
        """A dictionary of sets of inflow functions.

        Each key in the dictionary
        is a resource, and the corresponding value is a set. Each item in each set
        is a function, taking one argument ``index``, returning the
        an inflow of that resource at that index.
        """
        return self._inflows
    
    @property
    def outflows(self):
        """A dictionary of sets of outflow functions.

        Each key in the dictionary
        is a resource, and the corresponding value is a set. Each item in each set
        is a function, taking one argument ``index``, returning the
        an outflow of that resource at that index.
        """
        return self._outflows


    def set_cluster(self, cluster):
        """Add this node to a :class:`Cluster`.

        You should probably use :meth:`Cluster.add_part` instead.

        Args:
            cluster: The :node:`Cluster` instance to add to.
        """
        res = cluster.resource
        if res in self._clusters:
            if self._clusters[res] is not cluster:
                raise fs.InsanityError('already in another cluster with that resource!')
            else:
                raise fs.InsanityError('this has already been done')
        else:
            self._clusters[res] = cluster

        if not self in cluster.children:
            cluster.add_part(self)


    def unset_cluster(self, cluster):
        """Remove from a :class:`Cluster`.

        You should probably use :meth:`Cluster.remove_part` instead.

        Args:
            cluster: The :node:`Cluster` instance to remove from.
        """
        res = cluster.resource
        if not (res in self._clusters and self._clusters[res] is cluster):
            raise fs.InsanityError('cannot unset Cluster {} because it is not set'.format(cluster))        
        del self._clusters[res]
        if self in cluster.children:
            cluster.remove_part(self)


    def cluster(self, resource):
        """Get a :class:`Cluster` this node is in.

        Args:
            resource: The :attr:`Cluster.resource`.

        Returns:
            cluster: The Cluster if this node is in a
            :class:`Cluster` with :attr:`Cluster.resource``` == resource``,
            None otherwise.
        """
        return self._clusters.get(resource, None)


    def _balance_constraint(self, resource, index):
        inflow = fs.Sum(flow(index) for flow in self.inflows[resource])
        outflow = fs.Sum(flow(index) for flow in self.outflows[resource])

        lhs = inflow
        rhs = outflow

        if resource in self.production:
            lhs += self.production[resource](index)

        if resource in self.consumption:
            rhs += self.consumption[resource](index)

        if resource in self.accumulation:
            rhs += self.accumulation[resource](index)

        return Constraint(fs.Eq(lhs, rhs), desc='Balance constraint (resource={})'.format(resource))


    @property
    def resources(self):
        """The set of resources this node handles.

        This is the set of all keys found in the following dictionaries:

            * :attr:`consumption`
            * :attr:`production`
            * :attr:`accumulation`
            * :attr:`inflows`
            * :attr:`outflows`
        """
        balance_dicts = (
            self.consumption, self.production, self.accumulation, self.inflows, self.outflows)
        return set(chain(*(d.keys() for d in balance_dicts)))

    def balance_constraints(self, index):
        """Balance constraints for all resources.

        Returns one constraint for each resource in :attr:`resources`,
        except for the resources for which this node is in a :class:`Cluster`.

        Args:
            index: The index to get the resources for.

        Returns:
            set: The balance constraints.
        """
        # Enforce balance constraints for all resources, except those resources
        # which this node is in a cluster for. The cluster instead makes an aggregated
        # balance constraint for those.
        resources_to_be_balanced = (r for r in self.resources if r not in self._clusters)
        return set(self._balance_constraint(r, index) for r in resources_to_be_balanced)


class Cluster(Node):
    """A node containing other nodes, fully connected.

    A cluster is used to create a free flow of a resource ``R`` among
    a set of nodes. All :attr:`children` of a cluster get their
    :attr:`balance_constraints` turned off for the resource ``R``,
    and instead the cluster makes an aggregated balance constraint
    for all the nodes. In this way, a :class:`Cluster` is like a
    :class:`FlowNetwork` for resource ``R`` where all the parts are
    connected to one another.

    Args:
        *parts (optional): Zero or more parts to put in the cluster.
        resource: The resource this cluster handles.
        name (optional): A name for the cluster.

    Examples:

        Let's create three nodes:

        >>> producer = Node(name='producer')
        >>> producer.production['R'] = VariableCollection('prod')
        >>> consumer = Node(name='consumer')
        >>> consumer.consumption['R'] = VariableCollection('cons')
        >>> storage = Storage(resource='R', name='storage')
        >>> nodes = [consumer, producer, storage]

        Now they all make a balance constraint at any given index:

        >>> sum(len(n.constraints.make(5)) for n in nodes)
        3

        After clustering, they don't make balance constraints:

        >>> cluster = Cluster(*nodes, resource='R', name='cluster')
        >>> sum(len(n.constraints.make(5)) for n in nodes) # They all make a balance constraint
        0

        But the :class:`Cluster` does:

        >>> for constr in cluster.constraints.make(5):
        ...     print(constr.expr)
        ...
        prod(5) == cons(5) + storage.volume(6) - storage.volume(5)


    """
    
    def __init__(self, *parts, resource=None, name=None):
        super().__init__(name=name)
        if resource is None:
            msg = '{} is not a valid resource'.format(resource)
            raise ValueError(msg).with_traceback(sys.exc_info()[2])
        self._resource = resource
        for p in parts:
            self.add_part(p)

        self.consumption[self._resource] = self._get_aggr_func('consumption')
        self.production[self._resource] = self._get_aggr_func('production')
        self.accumulation[self._resource] = self._get_aggr_func('accumulation')


    def _get_aggr_func(self, attr_name):
        # attr_name is the attribute to aggregate, like "production", "consumption", or "accumulation"
        def aggregation(index):
            terms = []
            for part in self.children:
                func_dict = getattr(part, attr_name)
                if self._resource in func_dict:
                    func = func_dict[self._resource]
                    try:
                        term = func(index)
                        terms.append(term)
                    except TypeError as e:
                        if callable(func):
                            raise
                        else:
                            msg = 'The node {} has a non-callable value of {}[{}]: {}'.format(
                                part,
                                attr_name,
                                self._resource,
                                repr(func))

                            raise TypeError(msg) from e

            return fs.Sum(terms)

        return aggregation


    @property
    def resource(self):
        """The resource this cluster collects. Read only."""
        return self._resource


    def add_part(self, part):
        """Add a part to this cluster.

        Args:
            part (:class:`Part` or subclass instance): The part to add.

        Raises:
            InsanityError: If the calling part is a descendant of the part to add.
                (This would generate a cyclic relationship.)
        """
        super().add_part(part)
        if not part.cluster(self.resource) is self:
            try:
                part.set_cluster(self) # May raise an exception.
            except fs.InsanityError as e:
                super().remove_part(part) # Roll back on exception.
                raise


    def remove_part(self, part):
        """Remove a part from this part.

        Args:
            p (:class:`Part` or subclass instance): The part to remove.

        Raises:
            KeyError: If the part is not there.
        """
        super().remove_part(part)
        if part.cluster(self.resource) is not None:
            part.unset_cluster(self)


    def state_variables(self, index):
        """Cluster does not have state variables."""
        return tuple()


class Storage(Node):
    """Simple storage model.

    The storage has a volume function. It should be thought of as the
    volume at the beginning of a time step, such that
    ``volume(t) + accumulation[resource](t) == volume(t+1)``, or more
    exactly,

        >>> s = Storage('my_resource')
        >>> t = 42
        >>> t_plus_1 = s.step_time(t, 1)
        >>> s.accumulation['my_resource'](t) == s.volume(t_plus_1) - s.volume(t)
        True

    Args:
        resource: The resource to store.
        capacity (float, optional): The maximum amount that can be stored. If
            ``None`` (the default), there is no limit.
        maxchange (float, optional): The :attr:`maxchange` of the storage.
        name (str, optional): The :attr:`name` of the node.

    Examples:

        >>> from pandas import Timestamp, Timedelta
        >>> battery = Storage('power', name='Battery')
        >>> battery.time_unit = Timedelta('3h')
        >>> t = Timestamp('2015-06-10 18:00')
        >>> print(battery.accumulation['power'](t))
        Battery.volume(2015-06-10 21:00:00) - Battery.volume(2015-06-10 18:00:00)

    """
    def __init__(self, resource, capacity=None, maxchange=None, name=None):
        if name is not None:
            self.name = name
        self._resource = resource
        #:The maximum net inflow/outflow the storage
        #:can manage in one time step. In mathematical terms, it requires
        #:``abs(accumulation[resource](index)) <= maxchange`` for each ``index``.
        #:If ``None`` (the default), there is no limit.
        self.maxchange = maxchange

        with namespace(self):
            #:The volume of the storage at any time. Think of it as the
            #:volume at the **start** of a time step. To be concrete,
            #:``accumulation[resource](t) == volume(t+1) - volume(t)`` if
            #:the model is simply indexed in integers representing time.
            #:(And in more general terms,
            #:``accumulation[resource](idx) == volume(step_time(idx, 1)) - volume(idx)``
            #:for any index ``idx``.)
            self.volume = VariableCollection('volume', lb=0., ub=capacity)
        self.accumulation[resource] = self._compute_accumulation

        self.constraints += self._maxchange_constraints

    @property
    def resource(self):
        """The resource this storage stores. Read only."""
        return self._resource
    

    def _compute_accumulation(self, index):
        return self.volume(self.step_time(index, 1)) - self.volume(index)

    def _maxchange_constraints(self, index):
        acc, maxchange = self.accumulation[self.resource](index), self.maxchange
        if maxchange is None:
            return ()
        return (
            RelConstraint(acc <= maxchange, 'Max net inflow'),
            RelConstraint(-maxchange <= acc, 'Max net outflow'))

    def state_variables(self, index):
        """The only state variable is :attr:`volume` ``(index)``."""
        return {self.volume(index)}


class FlowNetwork(Part):
    """Manages flows between nodes.

    :class:`FlowNetwork` creates flow variables and can connect
    :class:`Node` instances by changing their :attr:`~Node.inflows`
    and :attr:`~Node.outflows`.

    Args:
        resource: The resource flowing in the network.
        name (str, optional): The :attr:`name` of the network.

    Examples:

        We create three nodes: A producer, a storage, and a consumer.

        >>> producer = Node(name='producer')
        >>> producer.production['R'] = VariableCollection('prod')
        >>> consumer = Node(name='consumer')
        >>> consumer.consumption['R'] = VariableCollection('cons')
        >>> storage = Storage(resource='R', name='storage')

        Connect the producer to the storage, and the storage to the consumer.

        >>> network = FlowNetwork(resource='R', name='network')
        >>> network.connect(producer, storage)
        >>> network.connect(storage, consumer)
        >>> for part in [producer, consumer, storage]:
        ...     for constr in part.constraints.make(5):
        ...         print(constr.origin.owner)
        ...         print(constr.expr)
        ...         print()
        ...
        producer
        prod(5) == network.flow(producer-->storage)(5)
        <BLANKLINE>
        consumer
        network.flow(storage-->consumer)(5) == cons(5)
        <BLANKLINE>
        storage
        network.flow(producer-->storage)(5) == network.flow(storage-->consumer)(5) + storage.volume(6) - storage.volume(5)
        <BLANKLINE>

    """
    def __init__(self, resource, name=None):
        super().__init__(name=name)
        self._resource = resource
        self._graph = nx.DiGraph()
        self._flows = dict()

    @property
    def graph(self):
        """A graph of all the flows.

        Gets a NetworkX ``DiGraph`` representation of the graph of how nodes
        are connected. See https://networkx.github.io/ for details.

        The graph object is a copy of the internal graph, so changing
        it does not affect the ``FlowNetwork``

        Examples:

            >>> FlowNetwork('resource').graph
            <networkx.classes.digraph.DiGraph object at 0x...>
        """
        return self._graph.copy()
    

    def remove_part(self, part):
        raise NotImplementedError('need to also remove edges then')


    def connect(self, n1, n2, bidirectional=False, capacity=None):
        """Connect two nodes.

        Creates a flow and adds it to
        ``n1.outflows[resource]`` and ``n2.inflows[resource]``,
        if it does not already exist. Calling again makes no difference.

        The flow must be nonnegative. For bidirectional flows, use
        ``bidirectional=True``.

        Args:
            n1: The node the flow goes from.
            n2: The node the flow goes to.
            bidirectional (boolean, optional): Create a two-way flow?
            capacity (float, optional): The maximum amount that can flow
                between the nodes. Creates an upper bound ``ub=capacity``
                on the flow :class:`~friendlysam.opt.Variable` for each index.

        """
        edges = self._graph.edges()
            
        if not (n1, n2) in edges:
            self.add_part(n1)
            self.add_part(n2)
            self._graph.add_edge(n1, n2)
            name = 'flow({}-->{})'.format(n1, n2)
            with namespace(self):
                flow = VariableCollection(name, lb=0, ub=capacity)
            self._flows[(n1, n2)] = flow
            n1.outflows[self._resource].add(flow)
            n2.inflows[self._resource].add(flow)

        if bidirectional and (n2, n1) not in edges:
            self.connect(n2, n1)

    def get_flow(self, n1, n2):
        """Get a flow between two nodes.

        Args:
            n1: The node the flow goes from.
            n2: The node the flow goes to.

        Returns:
            a ``VariableCollection``

        Raises:
            KeyError: If the flow does not exist.
        """

        return self._flows[n1, n2]

    def state_variables(self, index):
        """The state variables are all the flow variables."""
        return tuple(var(index) for var in self._flows.values())
