#!/usr/bin/env python
# -*- coding: utf-8 -*-

import unittest
import os, os.path
import json
from flask import Flask, request, json

import ablog_api
from ablog_api.main import app
from ablog_api.util import LEVEL

DATA_TEST = """Test
====

.. post:: Aug 02, 2016
   :tags: tag1,tag2
   :category: cat1,cat2


It's a test !!!
"""

class MyTest(unittest.TestCase):
    """
        Class for Unitaire Test for ablog_api
    """
    def setUp(self):
        self.app = app
        self.app.testing = True
        self.app.config['ABLOG_LEVEL_LOG']= LEVEL['DEBUG']
        self.app.config['ABLOG_CONF_DIR'] = os.path.join(os.path.dirname(__file__),'blog')
        self.app.config.complete()
        
        import logging
        self.app.logger.setLevel(self.app.config['ABLOG_LEVEL_LOG'])
        stream_handler = logging.StreamHandler()
        stream_handler.setLevel(self.app.config['ABLOG_LEVEL_LOG'])
        self.app.logger.addHandler(stream_handler)
 
        
    def test_version(self):
        with self.app.test_client() as c:
            rv = c.get('/api/version')
            self.assertEqual(rv.status_code , 200)

    def test_unauthorized(self):
        with self.app.test_client() as c:
            rv = c.post('/api/ls')
            self.assertEqual(rv.status_code, 401)

    def test_login(self):
        with self.app.test_client() as c:
            rv = c.post('/api/login', data = json.dumps(app.config['USERS'][0]))
            self.assertEqual(rv.status_code, 200)

    def test_login_ko(self):
        with self.app.test_client() as c:
            rv = c.post('/api/login', data = json.dumps({'username':'noguest', 'password':'nopassword'}))
            self.assertEqual(rv.status_code, 401)

    def test_ls(self):
        with self.app.test_client() as c:
            rv = c.post('/api/ls')
            self.assertEqual(rv.status_code, 401)
            rv = c.post('/api/login', data = json.dumps(app.config['USERS'][0]))
            rv = c.post('/api/ls')
            self.assertEqual(rv.status_code, 200)

    def test_build(self):
        with self.app.test_client() as c:
            rv = c.post('/api/build')
            self.assertEqual(rv.status_code, 401)
            rv = c.post('/api/login', data = json.dumps(app.config['USERS'][0]))
            rv = c.post('/api/build')
            self.assertEqual(rv.status_code, 200)
            rv = c.post('/api/build', data = json.dumps({'all':'true'}))
            self.assertEqual(rv.status_code, 200)

    def test_clean(self):
        with self.app.test_client() as c:
            rv = c.post('/api/clean')
            self.assertEqual(rv.status_code, 401)
            rv = c.post('/api/login', data = json.dumps(app.config['USERS'][0]))
            rv = c.post('/api/clean')
            self.assertEqual(rv.status_code, 200)
            rv = c.post('/api/build')
            rv = c.post('/api/clean', data = json.dumps({'deep':'true'}))
            self.assertEqual(rv.status_code, 200)
            rv = c.post('/api/build')
 
#    def test_edit(self):
#        with self.app.test_client() as c:
#            rv = c.post('/api/edit')
#            self.assertEqual(rv.status_code, 401)
#            rv = c.post('/api/login', data = json.dumps(app.config['USERS'][0]))
#            rv = c.post('/api/edit', data = json.dumps({'docname': 'test', 'data': DATA_TEST}))
#            self.assertEqual(rv.status_code, 200)
#            rv = c.post('/api/edit', data = json.dumps({'data': DATA_TEST}))
#            self.assertEqual(rv.status_code, 418)
#            rv = c.post('/api/edit', data = json.dumps({'docname': 'test'}))
#            self.assertEqual(rv.status_code, 418)
#            rv = c.post('/api/edit', data = json.dumps({'docname': 'test', 'data': DATA_TEST + 'add'}))
#            self.assertEqual(rv.status_code, 200)
 
    def test_rm(self):
        
        with open(os.path.join(self.app.config['ABLOG_CONF_DIR'],'test.rst'),'w') as f:
            f.write('test')
        with self.app.test_client() as c:
            rv = c.post('/api/rm')
            self.assertEqual(rv.status_code, 401)
            rv = c.post('/api/login', data = json.dumps(app.config['USERS'][0]))
            rv = c.post('/api/rm', data = json.dumps({'docname': 'test'}))
            self.assertEqual(rv.status_code, 200)
            rv = c.post('/api/rm')
            self.assertEqual(rv.status_code, 418)
 
    def test_get(self):
        with self.app.test_client() as c:
            rv = c.post('/api/get', data = json.dumps({'docname': 'first-post'}))
            self.assertEqual(rv.status_code, 200)
            rv = c.post('/api/get')
            self.assertEqual(rv.status_code, 418)
            rv = c.post('/api/get', data = json.dumps({'docname': 'unknown'}))
            self.assertEqual(rv.status_code, 418)
 
    def test_conf(self):
        with self.app.test_client() as c:
            rv = c.post('/api/conf')
            self.assertEqual(rv.status_code, 401)
            rv = c.post('/api/login', data = json.dumps(app.config['USERS'][0]))
            rv = c.post('/api/conf', data = json.dumps({'key': 'ABLOG_SRC_DIR'}))
            self.assertEqual(rv.status_code, 200)
            rv = c.post('/api/conf', data = json.dumps({'key': 'source_suffix'}))
            self.assertEqual(rv.status_code, 200)
            rv = c.post('/api/conf')
            self.assertEqual(rv.status_code, 418)
 
    def test_download(self):
        with self.app.test_client() as c:
            rv = c.post('/api/download')
            self.assertEqual(rv.status_code, 401)
            rv = c.post('/api/login', data = json.dumps(app.config['USERS'][0]))
            rv = c.post('/api/download', data = json.dumps({'path': 'about.rst'}))
            self.assertEqual(rv.status_code, 200)
            rv = c.post('/api/download')
            self.assertEqual(rv.status_code, 418)
            rv = c.post('/api/download', data = json.dumps({'path': 'unknow.rst'}))
            self.assertEqual(rv.status_code, 404)

# for test upload
# import requests
# r = requests.post('http://127.0.0.1:5000/api/upload/img/foo.png', data = open('test/foo.png', 'rb'))
# r = requests.post('http://127.0.0.1:5000/api/upload/foo.png', data = open('test/foo.png', 'rb'))
# r = requests.post('http://127.0.0.1:5000/api/upload/foo.exe', data = open('test/foo.png', 'rb'))

    def test_load_config(self):
        self.app.config['ABLOG_HOST']='0.0.0.0'
        self.assertEqual(self.app.config['ABLOG_HOST'], '0.0.0.0')
        self.app.config.from_pyfile(os.path.join(os.path.dirname(__file__),'conf.py'), silent=True)
        self.assertEqual(self.app.config['ABLOG_HOST'], '1.1.1.1')
        os.environ['ABLOG_HOST']='2.2.2.2'
        self.app.config.from_env('ABLOG_')
        self.assertEqual(self.app.config['ABLOG_HOST'], '2.2.2.2')
        


if __name__ == '__main__':
    unittest.main()
