# -*- coding: utf-8 -*-
import logging

from li_common.padroes import servicos
from li_common.comunicacao import requisicao
from pagador import configuracoes

logger = logging.getLogger('PAGADOR')


class SituacaoPedido(object):
    SITUACAO_PEDIDO_PENDENTE = 1
    SITUACAO_AGUARDANDO_PAGTO = 2
    SITUACAO_PAGTO_EM_ANALISE = 3
    SITUACAO_PEDIDO_PAGO = 4
    SITUACAO_PAGTO_EM_DISPUTA = 6
    SITUACAO_PAGTO_DEVOLVIDO = 7
    SITUACAO_PEDIDO_CANCELADO = 8
    SITUACAO_PEDIDO_EFETUADO = 9
    SITUACAO_PEDIDO_ENVIADO = 11
    SITUACAO_PRONTO_RETIRADA = 13
    SITUACAO_PEDIDO_ENTREGUE = 14
    SITUACAO_PEDIDO_EM_SEPARACAO = 15
    SITUACAO_PAGTO_CHARGEBACK = 16

    NOMES_SITUACAO = {
        1: u'Pendente',
        2: u'Aguardando Pagamento',
        3: u'Pagamento em Análise',
        4: u'Pedido Pago',
        6: u'Pagamento em Disputa',
        7: u'Pagamento Devolvido',
        8: u'Pedido Cancelado',
        9: u'Pedido Efetuado',
        11: u'Pedido Enviado ao Cliente',
        13: u'Pedido Pronto para ser Retirado',
        14: u'Pedido já foi Entregue',
        15: u'Pedido em Separação para Entrega',
        16: u'Pagamento em Processo de Devolução'
    }

    @classmethod
    def mensagens_complementares(cls, situacao_id):
        if situacao_id in [2]:
            return u'A forma de pagamento escolhida está aguardando o seu pagamento.'
        if situacao_id in [3]:
            return u'Você precisa verificar com a forma de pagamento escolhida que está analisando o seu pagamento.'
        if situacao_id in [4]:
            return u'Seu pagamento já está pago e estamos processando o envio.'
        if situacao_id in [6, 16]:
            return u'O pagamento está em processo de devolução. Verifique com a forma de pagamento escolhida o andamento do processo'
        if situacao_id in [7]:
            return u'Seu pagamento foi devolvido e o pedido cancelado e não pode ser mais usado. Você precisa fazer um novo pedido na loja.'
        if situacao_id in [8]:
            return u'Seu pedido foi cancelado e não pode ser mais usado. Você precisa fazer um novo pedido na loja.'
        if situacao_id in [9]:
            return u'Seu pedido foi efetuado mas ainda não foi realizado o pagamento. Você pode clicar no botão Efetuar Pagamento para dar andamento no processo.'
        if situacao_id in [11]:
            return u'Esse pedido já foi pago e enviado. Entre em contato com o SAC da loja para saber do andamento.'
        if situacao_id in [13]:
            return u'Esse pedido já foi pago e está pronto para ser retirado. Entre em contato com o SAC da loja para combinar a entrega.'
        if situacao_id in [14]:
            return u'Esse pedido já foi pago e marcado como entregue. Entre em contato com o SAC da loja caso não tenha recebido o produto.'
        if situacao_id in [15]:
            return u'Esse pedido já foi pago e está sendo preparado para ser entregue. Entre em contato com o SAC da loja para saber do andamento.'


class ServicoBase(servicos.Servico):
    extensao = None

    @classmethod
    def cria_entidade_pagador(cls, entidade, **kwargs):
        nome_modulo = 'pagador'
        return super(ServicoBase, cls).cria_entidade(nome_modulo, entidade, **kwargs)


class Credenciador(requisicao.Credenciador):
    def __init__(self, tipo, configuracao):
        super(Credenciador, self).__init__(tipo=tipo)
        self.configuracao = configuracao

    def atualiza_credenciais(self):
        pass


class InstalaMeioDePagamento(ServicoBase):
    campos = []

    def __init__(self, loja_id, dados):
        self.loja_id = loja_id
        self.dados = dados

    class InstalacaoNaoFinalizada(Exception):
        def __init__(self, msg, status=500):
            super(InstalaMeioDePagamento.InstalacaoNaoFinalizada, self).__init__(msg)
            self.status = status

    def montar_url_autorizacao(self):
        return ''

    def obter_dados(self):
        return {}

    def desinstalar(self, dados):
        return ''


class EntregaPagamento(ServicoBase):

    class PedidoJaRealizado(Exception):
        pass

    class EnvioNaoRealizado(Exception):
        def __init__(self, msg, loja_id, pedido_numero, dados_envio=None, erros=None, status=500):
            super(EntregaPagamento.EnvioNaoRealizado, self).__init__(msg)
            self.status = status

    class EnvioDePagamentoInvalido(Exception):
        def __init__(self, mensagem, status=500, fatal=True):
            super(EntregaPagamento.EnvioDePagamentoInvalido, self).__init__(mensagem)
            self.status = status
            self.fatal = fatal

    def __init__(self, loja_id, plano_indice=1, dados=None):
        self.loja_id = loja_id
        self.tem_malote = False
        self.malote = None
        self.situacao_pedido = None
        self.dados_pagamento = None
        self.configuracao = None
        self.resultado = {'sucesso': False, 'mensagem': None, 'fatal': True}
        self.pedido = None
        self.reenviar = False
        self.identificacao_pagamento = None
        self.plano_indice = plano_indice
        self.faz_http = False
        self.redirect_para = None
        self.dados = dados
        self.resposta = None
        self.dados_enviados = None
        self.pedido_processado = True

    @property
    def sandbox(self):
        return 'sandbox.' if (configuracoes.ENVIRONMENT == 'local' or configuracoes.ENVIRONMENT == 'development') else ''

    def define_pedido_e_configuracao(self, pedido_numero):
        self.pedido = self.cria_entidade_pagador('Pedido', numero=pedido_numero, loja_id=self.loja_id)
        if self.pedido.codigo_meio_pagamento != self.extensao:
            raise self.EnvioDePagamentoInvalido(u'O pedido {} enviado da loja {} não usa a forma de pagamento {}.'.format(self.pedido.numero, self.loja_id, self.extensao))
        self.configuracao = self.cria_entidade_extensao('ConfiguracaoMeioPagamento', loja_id=self.loja_id, codigo_pagamento=self.extensao)
        if not self.configuracao:
            raise self.EnvioDePagamentoInvalido(u'A forma de pagamento escolhida não está mais ativa na loja.', status=400, fatal=True)
        self.configuracao.plano_indice_alvo = self.plano_indice
        if not self.configuracao.em_uso_na_loja:
            raise self.EnvioDePagamentoInvalido(u'A forma de pagamento escolhida não está mais em uso na loja.', status=400, fatal=True)

    def montar_malote(self):
        self.malote = self.cria_entidade_extensao('Malote', configuracao=self.configuracao)
        parametros = self.cria_entidade_pagador('ParametrosDeContrato', loja_id=self.loja_id).obter_para(self.extensao)
        self.malote.monta_conteudo(pedido=self.pedido, parametros_contrato=parametros, dados=self.dados)

    def define_credenciais(self):
        pass

    def envia_pagamento(self, tentativa=1):
        pass

    def processa_dados_pagamento(self):
        pass


class SituacoesDePagamento(object):
    DE_PARA = {}

    @classmethod
    def do_tipo(cls, tipo):
        return cls.DE_PARA.get(tipo, None)


class RegistraResultado(ServicoBase):

    class RegistroDePagamentoInvalido(Exception):
        def __init__(self, mensagem, status=500, fatal=True):
            super(RegistraResultado.RegistroDePagamentoInvalido, self).__init__(mensagem)
            self.status = status
            self.fatal = fatal

    def __init__(self, loja_id, dados=None):
        self.loja_id = loja_id
        self.dados = dados
        self.configuracao = None
        self.pedido_numero = None
        self.dados_pagamento = {}
        self.situacao_pedido = None
        self.redirect_para = None
        self.faz_http = False
        self.resultado = {'sucesso': False, 'mensagem': None, 'fatal': True}
        self.resposta = None
        self.dados_enviados = None
        self.reenviar = False
        self.identificacao_pagamento = None
        self.pedido_processado = True

    def define_configuracao(self):
        self.configuracao = self.cria_entidade_extensao('ConfiguracaoMeioPagamento', loja_id=self.loja_id, codigo_pagamento=self.extensao)
        if not self.configuracao:
            raise self.RegistroDePagamentoInvalido(u'A forma de pagamento {} não está configurada na loja.'.format(self.extensao), status=400, fatal=True)

    @property
    def sandbox(self):
        return 'sandbox.' if (configuracoes.ENVIRONMENT == 'local' or configuracoes.ENVIRONMENT == 'development') else ''

    def define_credenciais(self):
        pass

    def obtem_informacoes_pagamento(self):
        pass

    def monta_dados_pagamento(self):
        pass


class GerenciaPedido(ServicoBase):

    class Credenciador(requisicao.Credenciador):
        def obter_credenciais(self):
            return configuracoes.AUTENTICACAO_API_PEDIDO

    def __init__(self, loja_id, pedido_numero, codigo_pagamento):
        self.pedido_numero = pedido_numero
        self.loja_id = loja_id
        self.pedido_pagamento = None
        if self.pedido_numero:
            self.pedido_pagamento = self.cria_entidade_pagador('PedidoPagamento', pedido_numero=pedido_numero, loja_id=loja_id, codigo_pagamento=codigo_pagamento)
        self.resultado = {'sucesso': False, 'resposta': 'sem resposta do servidor', 'excecao': None, 'alterou_status': False, 'acoes': []}

    def atualiza_resultado(self, acao, sucesso=True, resposta=None, excecao=None, alterou_status=False):
        self.resultado['sucesso'] = sucesso
        resultado_resposta = 'Sem resposta do servidor.'
        if resposta:
            alterou_status = resposta.get('situacao_alterada', False)
            if alterou_status:
                resultado_resposta = u'Situação do pedido alterada de {} para {}.'.format(
                    resposta.get('situacao_anterior', {}).get('situacao_nome', ''),
                    resposta.get('situacao_atual', {}).get('situacao_nome', '')
                )
            else:
                resultado_resposta = u'Situação do pedido não foi alterada.'
        self.resultado['resposta'] = resultado_resposta
        if excecao:
            excecao = u'{}'.format(excecao)
        self.resultado['excecao'] = excecao
        self.resultado['alterou_status'] = alterou_status
        self.resultado['acoes'].append(acao)

    def grava_identificacao_pagamento(self, identificador):
        try:
            self.pedido_pagamento.identificador_id = identificador
            self.pedido_pagamento.grava_identificacao_pagamento()
        except Exception, ex:
            self.atualiza_resultado('grava-identificacao', sucesso=False, excecao=ex)

    def grava_dados_pagamento(self, dados_pagamento):
        try:
            self.pedido_pagamento.preencher_com(dados_pagamento)
            self.pedido_pagamento.grava_dados_pagamento(dados_pagamento.keys())
        except Exception, ex:
            self.atualiza_resultado('grava-dados-pagamento', sucesso=False, excecao=ex)

    def altera_situacao_pedido(self, situacao_pedido):
        try:
            url = '{}loja/{}/pedido/{}/situacao/{}/api_pagador'.format(configuracoes.API_PEDIDO, self.loja_id, self.pedido_numero, situacao_pedido)
            conexao = self.obter_conexao()
            conexao.credenciador = self.Credenciador()
            resposta = conexao.put(url)
            self.atualiza_resultado('altera-situacao', sucesso=resposta.sucesso, resposta=resposta.conteudo)
        except Exception, ex:
            self.atualiza_resultado('altera-situacao', sucesso=False, excecao=ex)


class GravaEvidencia(ServicoBase):

    class Credenciador(requisicao.Credenciador):
        def obter_credenciais(self):
            return configuracoes.EVIDENCIA_AUTENTICACAO

    def __init__(self, meio_pagamento, loja_id, pedido_numero):
        self.meio_pagamento = meio_pagamento
        self.loja_id = loja_id
        self.pedido_numero = pedido_numero
        self.conexao = self.obter_conexao(formato_resposta=requisicao.Formato.texto)
        self.url = configuracoes.EVIDENCIA_URL
        
    def post(self, evidencia):
        try:
            if configuracoes.EVIDENCIA_USA_AUTENTICACAO_HEADER:
                self.conexao.credenciador = self.Credenciador()
            resposta = self.conexao.post(self.url, dados=evidencia)
            logger.info(resposta.conteudo)
        except Exception, ex:
            logger.error(ex)


    def grava_evidencia_retorno(self, registrador, tipo_retorno='resultado', resultado_api_pedido=None):
        if type(registrador.resultado) is dict:
            resultado = registrador.resultado
        else:
            resultado = {'status': registrador.resultado}
        resposta = getattr(registrador.resposta, 'conteudo', {})
        if not type(resposta) is dict:
            resposta = {'mensagem': resposta}
        evidencia = {
            'index': 'pagador',
            'tipo_evidencia': '{}-{}'.format(self.meio_pagamento, tipo_retorno),
            'evidencia': {
                'loja_id': self.loja_id,
                'pedido_numero': self.pedido_numero,
                'enviado': {
                    'dados': registrador.dados_enviados,
                    'url':  getattr(registrador, 'url', None)
                },
                'recebido': {
                    'dados': registrador.dados,
                },
                'resposta': resposta,
                'resultado': resultado,
                'identificacao_pagamento': registrador.identificacao_pagamento,
                'situacao_pedido': registrador.situacao_pedido,
                'resultado_api_pedido': resultado_api_pedido,
                'dados_pagamento': registrador.dados_pagamento,
            }
        }
        self.post(evidencia)

    def grava_evidencia_envio(self, entregador, resultado_api_pedido=None):
        if type(entregador.resultado) is dict:
            resultado = entregador.resultado
        else:
            resultado = {'status': entregador.resultado}
        resposta = getattr(entregador.resposta, 'conteudo', {})
        if not type(resposta) is dict:
            resposta = {'mensagem': resposta}
        evidencia = {
            'index': 'pagador',
            'tipo_evidencia': '{}-envio'.format(self.meio_pagamento),
            'evidencia': {
                'conta_id': self.loja_id,
                'pedido_numero': self.pedido_numero,
                'enviado': {
                    'dados': entregador.dados_enviados,
                    'url':  getattr(entregador, 'url', None)
                },
                'recebido': {
                    'dados': entregador.dados,
                },
                'resposta': resposta,
                'resultado': resultado,
                'identificacao_pagamento': entregador.identificacao_pagamento,
                'situacao_pedido': entregador.situacao_pedido,
                'resultado_api_pedido': resultado_api_pedido,
                'dados_pagamento': entregador.dados_pagamento,
            }
        }
        self.post(evidencia)


class AtualizaTransacoes(ServicoBase):
    class TransacaoDePagamentoInvalida(Exception):
        pass

    def __init__(self, loja_id, dados):
        self.loja_id = loja_id
        self.dados = dados
        self.configuracao = None
        self.resposta = None
        self.erros = None
        self.dados_enviados = {}
        self.dados_pedido = {}

    @property
    def sandbox(self):
        return 'sandbox.' if (configuracoes.ENVIRONMENT == 'local' or configuracoes.ENVIRONMENT == 'development') else ''

    def define_credenciais(self):
        pass

    def consulta_transacoes(self):
        pass

    def analisa_resultado_transacoes(self):
        pass

    def define_configuracao(self):
        self.configuracao = self.cria_entidade_extensao('ConfiguracaoMeioPagamento', loja_id=self.loja_id, codigo_pagamento=self.extensao)
        if not self.configuracao:
            raise self.TransacaoDePagamentoInvalida(u'A forma de pagamento {} não está configurada na loja.'.format(self.extensao))
