# -*- coding: utf-8 -*-
import json
from repositories.configuracao import models
from repositories.plataforma import models as plataforma_models
from repositories.pedido import models as pedido_models
from li_common.padroes import repositorios


class PedidoPagamentoRepositorio(repositorios.Repositorio):
    classe_model = pedido_models.PedidoVendaFormaPagamento
    manager = pedido_models.PedidoVendaFormaPagamento.objects

    def obter_do_banco(self, loja_id, numero, codigo_pagamento):
        if self.do_banco:
            if self.do_banco.conta_id != loja_id or self.do_banco.pedido.numero != numero or self.do_banco.pagamento.codigo != codigo_pagamento:
                self.do_banco = None
        if not self.do_banco:
            self.do_banco = self._obter(conta_id=loja_id, pedido__numero=numero, pagamento__codigo=codigo_pagamento)
        return {
            'identificador_id': self.do_banco.identificador_id,
            'transacao_id': self.do_banco.transacao_id,
            'valor_pago': self.do_banco.valor_pago,
            'conteudo_json': self.do_banco.conteudo_json
        }

    def gravar_identificacao_pagamento(self, loja_id, pedido_numero, codigo_pagamento, identificador_id):
        self.obter_do_banco(loja_id, pedido_numero, codigo_pagamento)
        self.do_banco.identificador_id = identificador_id
        self.do_banco.save()

    def gravar_dados_pagamento(self, loja_id, pedido_numero, codigo_pagamento, dados):
        self.obter_do_banco(loja_id, pedido_numero, codigo_pagamento)
        for dado in dados:
            setattr(self.do_banco, dado, dados[dado])
        self.do_banco.save()


class PedidoRepositorio(repositorios.Repositorio):
    classe_model = pedido_models.PedidoVenda
    manager = pedido_models.PedidoVenda.objects

    def _monta_endereco(self, endereco):
        return {
            'tipo': endereco.tipo,
            'nome': endereco.nome,
            'razao_social': endereco.razao_social,
            'cpf': endereco.cpf,
            'cnpj': endereco.cnpj,
            'rg': endereco.rg,
            'endereco': endereco.endereco,
            'numero': endereco.numero,
            'complemento': endereco.complemento,
            'bairro': endereco.bairro,
            'cidade': endereco.cidade,
            'estado': endereco.estado,
            'cep': endereco.cep,
        }

    def obter_com_numero(self, loja_id, numero):
        self.do_banco = self._obter(conta_id=loja_id, numero=numero)
        return {
            'codigo_meio_pagamento': self.do_banco.pagamento.codigo,
            'cliente': {
                'id': self.do_banco.cliente.id,
                'nome': self.do_banco.cliente.nome,
                'email': self.do_banco.cliente.email,
                'sexo': self.do_banco.cliente.sexo,
                'data_criacao': self.do_banco.cliente.data_criacao,
                'data_nascimento': self.do_banco.cliente.data_nascimento,
                'telefone_principal': self.do_banco.cliente.telefone_principal,
                'telefone_comercial': self.do_banco.cliente.telefone_comercial,
                'telefone_celular': self.do_banco.cliente.telefone_celular
            },
            'email_contato_loja': self.do_banco.conta.email_contato,
            'data_criacao': self.do_banco.data_criacao,
            'prazo_entrega': self.do_banco.prazo_entrega,
            '_valor_envio': self.do_banco.valor_envio,
            '_valor_desconto': self.do_banco.valor_desconto,
            '_valor_subtotal': self.do_banco.valor_subtotal,
            'valor_total': self.do_banco.valor_total,
            'forma_envio': self.do_banco.pedido_envio.envio.nome,
            'forma_envio_tipo': self.do_banco.pedido_envio.envio.tipo,
            'forma_envio_codigo': self.do_banco.pedido_envio.envio.codigo,
            'endereco_cliente': self._monta_endereco(self.do_banco.cliente.endereco),
            'endereco_entrega': self._monta_endereco(self.do_banco.endereco_entrega),
            'endereco_pagamento': self._monta_endereco(self.do_banco.endereco_pagamento),
            'telefone_principal': self.do_banco.telefone_principal,
            'telefone_celular': self.do_banco.telefone_celular,
            'conteudo_json': self.do_banco.conteudo_json,
            'situacao_id': self.do_banco.situacao_id,
            'itens_no_pedido': [{'sku': item.sku, 'nome': item.nome, 'quantidade': item.quantidade, 'preco_venda': item.preco_venda, 'altura': item.altura, 'largura': item.largura, 'profundidade': item.profundidade, 'peso': item.peso} for item in self.do_banco.itens.all()]
        }

    def cliente_tem_pedidos_na_loja(self, loja_id, cliente_id):
        return self.manager.filter(conta_id=loja_id, cliente_id=cliente_id).exists()


class ParametrosDeContratoNaoEncontrado(Exception):
    pass


class ParametrosDeContratoRepositorio(repositorios.Repositorio):
    classe_model = plataforma_models.Conta
    manager = plataforma_models.Conta.objects

    def contrato_loja(self, loja_id):
        try:
            resultado = self.manager.filter(id=loja_id).prefetch_related('contrato')[0]
            return resultado.contrato.id
        except IndexError:
            raise ParametrosDeContratoNaoEncontrado(u'Não foi encontrado parâmetros de contrato para a loja_id {}'.format(loja_id))

    def parametros_loja(self, loja_id):
        try:
            resultado = self.manager.filter(id=loja_id).prefetch_related('contrato')[0]
            return resultado.contrato.parametros
        except IndexError:
            raise ParametrosDeContratoNaoEncontrado(u'Não foi encontrado parâmetros de contrato para a loja_id {}'.format(loja_id))

    def parametros_base(self):
        return self.parametros_loja(1)


class CarteiraParaBoletoRepositorio(repositorios.Repositorio):
    classe_model = models.BoletoCarteira
    manager = models.BoletoCarteira.objects

    def _retorno(self, resultado):
        return {
            'id': resultado.id,
            'ativo': resultado.ativo,
            'nome': resultado.nome,
            'numero': resultado.numero,
            'convenio': resultado.convenio,
            'banco_id': resultado.banco_id,
            'banco_nome': resultado.banco.nome,
        }

    def listar_ativas(self):
        resultados = self.manager.filter(ativo=True).prefetch_related('banco')
        return [
            self._retorno(resultado)
            for resultado in resultados
        ]


class BancoRepositorio(repositorios.Repositorio):
    classe_model = models.Banco
    manager = models.Banco.objects

    def _retorno(self, resultado):
        return {
            'id': resultado.id,
            'codigo': resultado.codigo,
            'nome': resultado.nome,
            'imagem': resultado.imagem,
        }

    def _obter(self, **filtros):
        resultado = super(BancoRepositorio, self)._obter(**filtros)
        return self._retorno(resultado)

    def obter_com_codigo(self, codigo):
        return self._obter(codigo=codigo)

    def obter_com_id(self, banco_id):
        return self._obter(id=banco_id)

    def listar(self):
        resultados = self._listar()
        return [
            self._retorno(resultado)
            for resultado in resultados
        ]


class MeioDePagamentoRepositorio(repositorios.Repositorio):
    classe_model = models.FormaPagamento
    manager = models.FormaPagamento.objects

    def listar_ativos(self):
        return self.listar(ativo=True)

    def listar(self, **filtros):
        resultados = self._listar(**filtros)
        return [
            {
                'id': resultado.id,
                'codigo': resultado.codigo,
                'nome': resultado.nome,
                'ativo_no_sistema': resultado.ativo,
                'plano_indice': resultado.plano_indice,
                'valor_minimo_parcela': resultado.valor_minimo_parcela,
                'valor_minimo_parcelamento': resultado.valor_minimo_parcelamento
            }
            for resultado in resultados
        ]


class ConfiguracaoNaoEncontrada(Exception):
    pass


class ConfiguracaoMeioPagamentoRepositorio(repositorios.Repositorio):
    classe_model = models.FormaPagamentoConfiguracao
    manager = models.FormaPagamentoConfiguracao.objects
    manager_lista = models.FormaPagamento.objects
    _select = [
        'pg.pagamento_id',
        'pg.pagamento_nome',
        'pg.pagamento_codigo',
        'pg.pagamento_ativado',
        'pg.pagamento_plano_indice',
        'pg.pagamento_parcela_valor_minimo_parcela',
        'pg.pagamento_parcela_valor_minimo',
        'pgc.pagamento_configuracao_id',
        'pgc.pagamento_configuracao_usuario',
        'pgc.pagamento_configuracao_senha',
        'pgc.pagamento_configuracao_token',
        'pgc.pagamento_configuracao_assinatura',
        'pgc.pagamento_configuracao_codigo_autorizacao',
        'pgc.pagamento_configuracao_ativo',
        'pgc.pagamento_configuracao_json',
        'pgc.pagamento_configuracao_email_comprovante',
        'pgc.pagamento_configuracao_informacao_complementar',
        'pgc.pagamento_configuracao_desconto',
        'pgc.pagamento_configuracao_desconto_tipo',
        'pgc.pagamento_configuracao_desconto_valor',
        'pgc.pagamento_configuracao_desconto_aplicar_no_total',
        'pgc.pagamento_configuracao_quantidade_parcela_maxima',
        'pgc.pagamento_configuracao_quantidade_parcela_sem_juros',
        'pgc.pagamento_configuracao_juros_valor',
        'pgc.pagamento_configuracao_valor_minimo_parcela',
        'pgc.pagamento_configuracao_valor_minimo_aceitado',
        'pgc.pagamento_coonfiguracao_mostrar_parcelamento',
        'pgc.pagamento_configuracao_aplicacao_id',
        'pgc.pagamento_configuracao_usar_antifraude',
        'pgc.pagamento_configuracao_eh_padrao',
        'pgc.pagamento_configuracao_ordem',
        'pgc.conta_id',
        'pgc.contrato_id',
        '''
        (SELECT array_to_json(array_agg(t)) FROM (
          SELECT pp.pagamento_parcela_fator, pp.pagamento_parcela_numero_parcelas FROM configuracao.tb_pagamento_parcela pp WHERE pp.pagamento_id = pg.pagamento_id ORDER BY pp.pagamento_parcela_numero_parcelas
        ) as t) as pagamento_parcelas
        '''
    ]
    _campos = {
        'id': 'forma_pagamento_id',
        'nome': 'pagamento_nome',
        'codigo': 'pagamento_codigo',
        'ativo': 'pagamento_ativado',
        'plano_indice': 'pagamento_plano_indice',
        'pagamento_parcelas': 'pagamento_parcelas',
        'valor_minimo_parcela': 'pagamento_parcela_valor_minimo_parcela',
        'valor_minimo_parcelamento': 'pagamento_parcela_valor_minimo',
        'pagamento_configuracao_id': 'id',
        'pagamento_configuracao_usuario': 'usuario',
        'pagamento_configuracao_senha': 'senha',
        'pagamento_configuracao_token': 'token',
        'pagamento_configuracao_assinatura': 'assinatura',
        'pagamento_configuracao_codigo_autorizacao': 'codigo_autorizacao',
        'pagamento_configuracao_ativo': 'ativo',
        'pagamento_configuracao_json': 'json',
        'pagamento_configuracao_email_comprovante': 'email_comprovante',
        'pagamento_configuracao_informacao_complementar': 'informacao_complementar',
        'pagamento_configuracao_desconto': 'desconto',
        'pagamento_configuracao_desconto_tipo': 'desconto_tipo',
        'pagamento_configuracao_desconto_valor': 'desconto_valor',
        'pagamento_configuracao_desconto_aplicar_no_total': 'aplicar_no_total',
        'pagamento_configuracao_quantidade_parcela_maxima': 'maximo_parcelas',
        'pagamento_configuracao_quantidade_parcela_sem_juros': 'parcelas_sem_juros',
        'pagamento_configuracao_juros_valor': 'juros_valor',
        'pagamento_configuracao_valor_minimo_parcela': 'valor_minimo_parcela',
        'pagamento_configuracao_valor_minimo_aceitado': 'valor_minimo_aceitado',
        'pagamento_coonfiguracao_mostrar_parcelamento': 'mostrar_parcelamento',
        'pagamento_configuracao_aplicacao_id': 'aplicacao',
        'pagamento_configuracao_usar_antifraude': 'usar_antifraude',
        'pagamento_configuracao_eh_padrao': 'eh_padrao',
        'pagamento_configuracao_ordem': 'ordem',
        'conta_id': 'conta_id',
        'contrato_id': 'contrato_id',
    }

    def __init__(self, loja_id):
        super(ConfiguracaoMeioPagamentoRepositorio, self).__init__()
        self.loja_id = loja_id

    def _trata_atributo_do_banco(self, atributo, campo):
        if atributo and campo == 'pagamento_configuracao_json':
            try:
                return json.loads(atributo)
            except ValueError:
                return {}
        return atributo

    def listar_ativas(self, pagamento_id=None):
        parametros = [self.loja_id, pagamento_id]
        resultados = self.manager_lista.raw(self._montar_query_para_ativas(filtro_pagamento=(pagamento_id is not None)), [parametro for parametro in parametros if parametro])
        retorno = []
        for resultado in resultados:
            try:
                retorno.append({self._campos[campo]: self._trata_atributo_do_banco(getattr(resultado, campo, None), campo) for campo in self._campos})
            except self.classe_model.DoesNotExist:
                self.manager.create(conta_id=self.loja_id, forma_pagamento_id=resultado.forma_pagamento_id)
                criado = self.manager.raw(self._montar_query_para_ativas(filtro_pagamento=True), [self.loja_id, resultado.forma_pagamento_id])[0]
                retorno.append({self._campos[campo]: getattr(criado, campo, None) for campo in self._campos})
        return retorno

    def _montar_query_para_ativas(self, filtro_pagamento=False):
        from_ = ['configuracao.tb_pagamento as pg']
        left_join = 'LEFT OUTER JOIN configuracao.tb_pagamento_configuracao as pgc on (pgc.pagamento_id = pg.pagamento_id AND pgc.conta_id = %s)'
        where = ['pg.pagamento_ativado = true']
        if filtro_pagamento:
            where.append('pg.pagamento_id = %s')
        return 'SELECT {} FROM {} {} WHERE {} ORDER BY pgc.pagamento_configuracao_ordem, pg.pagamento_posicao;'.format(
            ', '.join(self._select),
            ', '.join(from_),
            left_join,
            ' AND '.join(where)
        )

    def obter_gateway(self, pagamento_id):
        try:
            return self.listar_ativas(pagamento_id=pagamento_id)[0]
        except IndexError:
            raise ConfiguracaoNaoEncontrada(u'Não foi encontrada uma configuração de pagamento para a loja_id {} e pagamento_id {}'.format(self.loja_id, pagamento_id))

    def instanciar_banco(self, pagamento_id):
        self.do_banco, criado = self.manager.get_or_create(conta_id=self.loja_id, forma_pagamento_id=pagamento_id)
        return self.do_banco

    def salvar_instancia(self):
        self.do_banco.save()

    def definir_valores(self, dados):
        for atributo in dados:
            if hasattr(self.do_banco, atributo):
                setattr(self.do_banco, atributo, dados[atributo])

    def remover_pagamento_padrao(self):
        padrao_atual = self.manager.filter(conta_id=self.loja_id, eh_padrao=True)
        for pagamento in padrao_atual:
            pagamento.eh_padrao = False
            pagamento.save()

    def remover_ordenacao(self):
        com_ordem = self.manager.filter(conta_id=self.loja_id, ordem__gt=0)
        for pagamento in com_ordem:
            pagamento.ordem = 0
            pagamento.save()

    def obter_forma_pagamento_id(self, codigos_pagamentos):
        return {forma_pagamento.codigo: forma_pagamento.id for forma_pagamento in self.manager_lista.filter(codigo__in=codigos_pagamentos)}
