# -*- coding: utf-8 -*-
import math
import collections
from datetime import timedelta, datetime
from decimal import Decimal, InvalidOperation

from li_common.padroes import entidades, extensibilidade
from li_common.comunicacao import requisicao

from pagador import configuracoes, repositorios, servicos

extensibilidade.SETTINGS = configuracoes

FormatoDeEntrega = collections.namedtuple('FormatoDeEntrega', ['querystring', 'json', 'form_urlencode', 'xml'])('querystring', 'json', 'form_urlencode', 'xml')


class ItemDePedido(entidades.Entidade):
    _nao_serializar = ['url_produto']

    def __init__(self, sku=None, nome=None, quantidade=None, preco_venda=None,
                 altura=None, largura=None, profundidade=None, peso=None):
        self.sku = sku
        self.nome = nome
        self.quantidade = quantidade
        self.preco_venda = preco_venda
        self.altura = altura
        self.largura = largura
        self.profundidade = profundidade
        self.peso = peso

    @property
    def url_produto(self):
        return ''


class PedidoPagamento(entidades.Entidade):
    def __init__(self, loja_id, pedido_numero, codigo_pagamento):
        self._repositorio = repositorios.PedidoPagamentoRepositorio()
        self.loja_id = loja_id
        self.pedido_numero = pedido_numero
        self.codigo_pagamento = codigo_pagamento
        self.identificador_id = None
        self.transacao_id = None
        self.valor_pago = None
        self.conteudo_json = None

    def preencher_do_banco(self):
        dados = self._repositorio.obter_do_banco(self.loja_id, self.pedido_numero, self.codigo_pagamento)
        self.preencher_com(dados)

    def grava_identificacao_pagamento(self):
        self._repositorio.gravar_identificacao_pagamento(self.loja_id, self.pedido_numero, self.codigo_pagamento, self.identificador_id)

    def grava_dados_pagamento(self, campos=None):
        dados = {'transacao_id': self.transacao_id, 'valor_pago': self.valor_pago, 'conteudo_json': self.conteudo_json}
        if campos:
            dados = {campo: getattr(self, campo, None) for campo in campos}
        self._repositorio.gravar_dados_pagamento(
            self.loja_id,
            self.pedido_numero,
            self.codigo_pagamento,
            dados
        )


class Pedido(entidades.Entidade):
    _VALOR_ENVIO = 1
    _VALOR_DESCONTO = 2
    _nao_serializar = ['eh_primeira_compra_cliente']

    def _monta_itens(self, itens):
        self.itens = [
            ItemDePedido(sku=item['sku'], nome=item['nome'], quantidade=item['quantidade'], preco_venda=item['preco_venda'],
                         altura=item['altura'], largura=item['largura'], profundidade=item['profundidade'], peso=item['peso'])
            for item in itens
            ]

    def __init__(self, numero, loja_id):
        self.numero = numero
        self.loja_id = loja_id
        self.cliente = None
        self.email_contato_loja = None
        self._valor_envio = Decimal('0.00')
        self._valor_desconto = Decimal('0.00')
        self._valor_subtotal = Decimal('0.00')
        self.valor_total = Decimal('0.00')
        self.forma_envio = None
        self.forma_envio_tipo = None
        self.forma_envio_codigo = None
        self.endereco_cliente = None
        self.endereco_entrega = None
        self.endereco_pagamento = None
        self.telefone_principal = None
        self.telefone_celular = None
        self.itens = None
        self.data_criacao = None
        self.prazo_entrega = None
        self.codigo_meio_pagamento = None
        self.conteudo_json = {}
        self.situacao_id = None
        if numero == 'TESTE-BOLETO':
            self.codigo_meio_pagamento = 'boleto'
            self.endereco_entrega = {'nome': 'CLIENTE TESTE', 'tipo': 'PF'}
            self.endereco_cliente = {'tipo': 'PF', 'cpf': '12345678901'}
            self.endereco_pagamento = {
                'endereco': 'Rua Teste',
                'numero': '0',
                'complemento': 'compl',
                'bairro': 'Bairro',
                'cidade': 'Cidade',
                'estado': 'XX',
                'cep': '00000-000',
            }
            self.data_criacao = datetime.now()
            self.valor_total = Decimal('1.01')
            self.numero = 236
            self.prazo_entrega = 10
        else:
            self._repositorio = repositorios.PedidoRepositorio()
            dados = self._repositorio.obter_com_numero(loja_id=loja_id, numero=numero)
            self.preencher_com(dados, soh_atributo=True)

    @property
    def provavel_data_entrega(self):
        return self.data_criacao + timedelta(days=self.prazo_entrega)

    @property
    def eh_primeira_compra_cliente(self):
        return not self._repositorio.cliente_tem_pedidos_na_loja(self.loja_id, self.cliente['id'])

    def preencher_com(self, dados, campos=None, soh_atributo=False):
        self._monta_itens(dados['itens_no_pedido'])
        super(Pedido, self).preencher_com(dados, campos, soh_atributo)

    @property
    def cliente_nome_ascii(self):
        if not self.cliente:
            return ''
        nome = self.cliente['nome']
        nome = self.formatador.trata_nome(nome)
        return self.formatador.trata_unicode_com_limite(nome, limite=50, ascii=True, trata_espaco_duplo=True)

    @property
    def cliente_primeiro_nome(self):
        if not self.cliente:
            return ''
        return self.cliente['nome'].split(' ')[0]

    @property
    def cliente_ultimo_nome(self):
        if not self.cliente:
            return ''
        nome = self.cliente['nome'].split(' ')[-1]
        if nome == self.cliente['nome']:
            return ''
        return nome

    @property
    def cliente_documento(self):
        if self.endereco_cliente['tipo'] == "PF":
            return self.endereco_cliente['cpf']
        return self.endereco_cliente['cnpj']

    @property
    def cliente_telefone(self):
        if not self.cliente:
            return '', ''
        telefone = None
        if self.cliente['telefone_principal']:
            telefone = self.cliente['telefone_principal']
        elif self.cliente['telefone_comercial']:
            telefone = self.cliente['telefone_comercial']
        elif self.cliente['telefone_celular']:
            telefone = self.cliente['telefone_celular']
        if telefone:
            return self.formatador.converte_tel_em_tupla_com_ddd(telefone)
        return '', ''

    @property
    def nome_endereco_entrega(self):
        if self.endereco_entrega['tipo'] == "PF":
            return self.formatador.trata_nome(self.endereco_entrega['nome'])
        else:
            return self.formatador.trata_nome(self.endereco_entrega['razao_social'])

    def _valores_envio_e_deconto(self, valor_para_retorno):
        valor_envio = self.formatador.converte_para_decimal(self._valor_envio)
        valor_desconto = self.formatador.converte_para_decimal(self._valor_desconto)
        if valor_envio == valor_desconto:
            return self.formatador.converte_para_decimal('0.00')
        if valor_para_retorno == self._VALOR_DESCONTO:
            return valor_desconto
        if valor_para_retorno == self._VALOR_ENVIO:
            return valor_envio

    @property
    def valor_envio(self):
        return self._valores_envio_e_deconto(self._VALOR_ENVIO)

    @property
    def valor_desconto(self):
        return self._valores_envio_e_deconto(self._VALOR_DESCONTO)

    @property
    def valor_subtotal(self):
        valor_subtotal = self.formatador.converte_para_decimal(self._valor_subtotal)
        return valor_subtotal - self.valor_desconto

    # def _dimensoes_a_partir_da_cubagem(self, cubagem):
    #     # Dimensões mínimas para os Correios.
    #     dimensoes_minimas = [2, 16, 11]
    #
    #     # Para saber o valor de cada lado do pacote é necessário
    #     # tirar a raiz cúbica.
    #     dimensao = cubagem ** (Decimal(1) / 3)
    #
    #     if cubagem > 4096:
    #         # A cubagem é maior do que a maior aresta das dimensões mínimas
    #         # dos Correios elevada ao cubo, isso quer dizer que podemos
    #         # aplicar cada dimensão por igual.
    #         altura, largura, profundidade = dimensao
    #     elif cubagem <= 352:
    #         # A cubagem calculada é menor do que a cubagem das dimensões
    #         # mínimas do Correios. Aplicamos as dimensões mínimas.
    #         altura, largura, profundidade = dimensoes_minimas
    #     else:
    #         # Aplicando proporcionalmente as porcentagens relativas ao tamanho
    #         # mínimo possível para enviar pelos Correios, dessa forma os
    #         # pacotes pequenos não são cobrados a mais do que deveriam.
    #
    #         duas_dimensoes = dimensoes_minimas[1] * dimensoes_minimas[2]
    #
    #         somatorio = sum(dimensoes_minimas)
    #         altura = dimensao * ((Decimal('100') * dimensoes_minimas[0]) / somatorio)
    #         largura = dimensao * ((Decimal('100') * dimensoes_minimas[1]) / somatorio)
    #         profundidade = dimensao * ((Decimal('100') * dimensoes_minimas[2]) / somatorio)
    #
    #         # TODO: Terminar o cálculo intermediário.
    #         abc = (max(int(altura), dimensoes_minimas[0]),
    #             max(int(largura), dimensoes_minimas[1]),
    #             max(int(profundidade), dimensoes_minimas[2]))
    #
    #     return altura, largura, profundidade
    #
    # def dimensoes(self):
    #     """Retorna uma lista com as três dimensões do pacote para o pedido
    #     na seguinte ordem: altura, largura, comprimento e peso.
    #     """
    #     cubagem = Decimal('0')
    #     peso = Decimal('0.000')
    #
    #     for item in self.itens:
    #         cubagem += ((item.altura or 1) * (item.largura or 1) *
    #                     (item.profundidade or 1)) * item.quantidade
    #         peso += item.peso or Decimal('0.000')
    #
    #     altura, largura, profundidade = self._dimensoes_a_partir_da_cubagem(cubagem)
    #
    #     return [
    #         altura,
    #         largura,
    #         profundidade,
    #         max(int(peso * 1000), 0.05)
    #     ]

    def dimensoes(self):
        medidas = [0, 0, 0, 0]
        medidas_temporarias = []
        cubagem_total = 0

        #
        # Checa Dimensão Máxima dos Itens
        #
        for item in self.itens:
            for i in range(int(item.quantidade)):
                altura = item.altura or 0
                largura = item.largura or 0
                profundidade = item.profundidade or 0

                if altura > 105 or largura > 105 \
                    or profundidade > 105 or (
                        altura + largura + profundidade) > 200:
                    # Se o tamanho total passar nós limitamos ao tamanho
                    # máximo possível. Essa verificação é redundante pois já
                    # existe uma verificação igual na primeira consulta de
                    # preço.
                    altura = largura = profundidade = 66.66

                medidas_temporarias.append([altura, largura, profundidade])
                medidas[3] += item.peso or 0.0
                cubagem_total += (altura or 1) * \
                    (largura or 1) * (profundidade or 1)

        #
        # Checa Cubagem total dos Itens
        #
        if cubagem_total > (66 * 66 * 66):
            # Se a cubagem total passa do limite, usamos o limite.
            cubagem_total = 200

        alturas, larguras, comprimentos = zip(*medidas_temporarias)
        maiores_medidas = [max(alturas), max(larguras), max(comprimentos)]

        raiz = 0
        if cubagem_total > 0:
            try:
                divisao = float(cubagem_total) / max(maiores_medidas)
                raiz = round(math.sqrt(divisao), 1)
            except ZeroDivisionError:
                raiz = 0

        CORREIOS_LIMITES = {
            'tamanho_min': {
                'comprimento': 16,
                'largura': 11,
                'altura': 2
            },
            'tamanho_max': {
                'comprimento': 105,
                'largura': 105,
                'altura': 105,
                'somado': 200
            },
            'peso_max': 30
        }

        maior = max(maiores_medidas)
        medidas[0] = int(max(
            raiz,
            CORREIOS_LIMITES['tamanho_min']['altura']))
        medidas[1] = int(max(
            raiz,
            CORREIOS_LIMITES['tamanho_min']['largura']))
        medidas[2] = int(max(
            maior,
            CORREIOS_LIMITES['tamanho_min']['comprimento']))
        medidas[3] = max(int(medidas[3] * 1000), 500)
        return medidas


class BaseParaPropriedade(entidades.Entidade):
    _atributos = []

    def __init__(self, **dados):
        for chave in self._chaves_alternativas_para_serializacao:
            setattr(self, chave, dados.get(chave, None))
        for atributo in self._atributos:
            setattr(self, atributo, dados.get(atributo, None))


class Malote(entidades.Entidade):
    class DadosInvalidos(Exception):
        pass

    def __init__(self, configuracao):
        self._nao_serializar = ['configuracao']
        self.configuracao = configuracao

    def monta_conteudo(self, pedido, parametros_contrato=None, dados=None):
        pass


class TipoRetorno(object):
    resultado = 'resultado'
    notificacao = 'notificacao'


class Transacoes(entidades.EntidadeExtensivel):
    _nao_serializar = ['gerenciador_pedidos']

    def __init__(self, loja_id, codigo_pagamento):
        self.loja_id = loja_id
        self.extensao = codigo_pagamento
        self.gerenciador_pedidos = servicos.GerenciaPedido(loja_id=self.loja_id, pedido_numero=None, codigo_pagamento=self.extensao)
        self.atualizados = 0

    def atualizar_dados_pedido(self, dados_pedido):
        if type(dados_pedido) is dict:
            self._atualizando_situacao(dados_pedido)
        else:
            for dado in dados_pedido:
                self._atualizando_situacao(dado)

    def _atualizando_situacao(self, dados):
        self.gerenciador_pedidos.pedido_numero = dados['pedido_numero']
        self.gerenciador_pedidos.altera_situacao_pedido(dados['situacao_pedido'])
        if self.gerenciador_pedidos.resultado['sucesso']:
            self.atualizados += 1

    def atualizar(self, dados):
        try:
            atualizador = self.cria_servico_extensao('AtualizaTransacoes', loja_id=self.loja_id, dados=dados)
        except extensibilidade.ExtensaoNaoDefinido, ex:
            return {'status_code': 404, 'mensagem': u'{}'.format(ex)}
        atualizador.define_configuracao()
        atualizador.define_credenciais()
        atualizador.consulta_transacoes()
        atualizador.analisa_resultado_transacoes()
        if atualizador.dados_pedido:
            self.atualizar_dados_pedido(atualizador.dados_pedido)
            return {'mensagem': u'{} pedidos enviados para atualização'.format(self.atualizados), 'pedidos': atualizador.dados_pedido}
        if atualizador.erros:
            return {'erros': atualizador.erros, 'status_code': 500}
        return {'mensagem': u'Nenhum dado de pedido encontrado para atualização.'}


class Pagamento(entidades.EntidadeExtensivel):
    _nao_serializar = ['gerenciador_pedidos']

    def __init__(self, loja_id, codigo_pagamento, pedido_numero=None):
        self.loja_id = loja_id
        self.extensao = codigo_pagamento
        self.pedido_numero = pedido_numero
        self._gerenciador_pedidos = None

    @property
    def gerenciador_pedidos(self):
        if not self._gerenciador_pedidos:
            self._gerenciador_pedidos = servicos.GerenciaPedido(loja_id=self.loja_id, pedido_numero=self.pedido_numero, codigo_pagamento=self.extensao)
        return self._gerenciador_pedidos

    def _grava_evidencia_envio(self, entregador, ex=None):
        if configuracoes.GRAVA_EVIDENCIA and entregador.faz_http:
            if ex:
                if not entregador.resultado:
                    entregador.resultado = {'sucesso': False, 'mensagem': None, 'fatal': True}
                try:
                    entregador.resultado['mensagem'] = unicode(ex)
                except Exception, ex:
                    entregador.resultado['mensagem'] = u'Erro na geração da mensagem de erro do entregador: {}'.format(ex)
            gravador_evidencia = servicos.GravaEvidencia(self.extensao, self.loja_id, self.pedido_numero)
            gravador_evidencia.grava_evidencia_envio(entregador, self.gerenciador_pedidos.resultado)

    def _finaliza_envio(self, entregador, ex=None):
        if entregador.identificacao_pagamento:
            self.gerenciador_pedidos.grava_identificacao_pagamento(entregador.identificacao_pagamento)
        if entregador.dados_pagamento:
            self.gerenciador_pedidos.grava_dados_pagamento(entregador.dados_pagamento)
        if entregador.situacao_pedido:
            self.gerenciador_pedidos.altera_situacao_pedido(entregador.situacao_pedido)
            entregador.pedido_processado = self.gerenciador_pedidos.resultado['sucesso']
        self._grava_evidencia_envio(entregador, ex=ex)

    def enviar(self, dados, plano_indice):
        try:
            entregador = self.cria_servico_extensao('EntregaPagamento', loja_id=self.loja_id, plano_indice=plano_indice, dados=dados)
        except extensibilidade.ExtensaoNaoDefinido, ex:
            return {'status_code': 404, 'mensagem': u'{}'.format(ex), 'fatal': True}
        if dados.get('tipo', '') == 'BoletoTeste':
            self.pedido_numero = 'TESTE-BOLETO'
        try:
            entregador.define_pedido_e_configuracao(self.pedido_numero)
        except entregador.EnvioDePagamentoInvalido, ex:
            self._finaliza_envio(entregador, ex=ex)
            return {'status_code': 400, 'mensagem': u'{}'.format(ex), 'fatal': True}
        if entregador.tem_malote:
            try:
                entregador.montar_malote()
            except entregador.malote.DadosInvalidos, ex:
                self._finaliza_envio(entregador, ex=ex)
                return {'status_code': 400, 'mensagem': u'{}'.format(ex), 'fatal': True}
        if entregador.faz_http:
            entregador.define_credenciais()
            try:
                entregador.envia_pagamento()
            except entregador.PedidoJaRealizado, ex:
                self._finaliza_envio(entregador, ex=ex)
                return self.gerar_resposta(entregador)
            except requisicao.TempoExcedido, ex:
                self._finaliza_envio(entregador, ex=ex)
                return {'status_code': 408, 'mensagem': u'{}'.format(ex), 'fatal': False}
            except entregador.EnvioNaoRealizado, ex1:
                if entregador.reenviar:
                    try:
                        entregador.envia_pagamento(tentativa=2)
                    except entregador.EnvioNaoRealizado, ex:
                        self._finaliza_envio(entregador, ex=ex)
                        return {'status_code': 400, 'mensagem': u'{}'.format(ex), 'fatal': True}
                else:
                    self._finaliza_envio(entregador, ex=ex1)
                    return {'status_code': ex1.status, 'mensagem': u'{}'.format(ex1), 'fatal': True}
        try:
            entregador.processa_dados_pagamento()
        except entregador.EnvioNaoRealizado, ex:
            self._finaliza_envio(entregador, ex=ex)
            fatal = getattr(ex, 'fatal', True)
            return {'status_code': ex.status, 'mensagem': u'{}'.format(ex), 'fatal': fatal, 'pago': False}
        self._finaliza_envio(entregador)
        return self.gerar_resposta(entregador)

    def _grava_evidencia_retorno(self, registrador, tipo_retorno, ex=None):
        if configuracoes.GRAVA_EVIDENCIA:
            if ex:
                if not registrador.resultado:
                    registrador.resultado = {'sucesso': False, 'mensagem': None, 'fatal': True}
                try:
                    registrador.resultado['mensagem'] = unicode(ex)
                except Exception, ex:
                    registrador.resultado['mensagem'] = u'Erro na geração da mensagem de erro do registrador: {}'.format(ex)
            gravador_evidencia = servicos.GravaEvidencia(self.extensao, self.loja_id, self.pedido_numero)
            gravador_evidencia.grava_evidencia_retorno(registrador, tipo_retorno=tipo_retorno, resultado_api_pedido=self.gerenciador_pedidos.resultado)

    def registrar(self, dados, tipo_retorno):
        try:
            if tipo_retorno == TipoRetorno.resultado:
                registrador = self.cria_servico_extensao('RegistraResultado', loja_id=self.loja_id, dados=dados)
            else:
                registrador = self.cria_servico_extensao('RegistraNotificacao', loja_id=self.loja_id, dados=dados)
        except Exception, ex:
            return {'status_code': 404, 'mensagem': u'{}'.format(ex), 'fatal': True}
        try:
            registrador.define_configuracao()
        except registrador.RegistroDePagamentoInvalido, ex:
            self._grava_evidencia_retorno(registrador, tipo_retorno)
            return {'status_code': 400, 'mensagem': u'{}'.format(ex), 'fatal': True}
        if registrador.faz_http:
            registrador.define_credenciais()
        try:
            registrador.obtem_informacoes_pagamento()
            registrador.monta_dados_pagamento()
        except requisicao.TempoExcedido, ex:
            self._grava_evidencia_retorno(registrador, tipo_retorno)
            return {'status_code': 408, 'mensagem': u'{}'.format(ex), 'fatal': True}
        except Exception, ex:
            self._grava_evidencia_retorno(registrador, tipo_retorno)
            return {'status_code': 500, 'mensagem': u'{}'.format(ex), 'fatal': True}
        self.pedido_numero = registrador.pedido_numero
        if registrador.dados_pagamento:
            self.gerenciador_pedidos.grava_dados_pagamento(registrador.dados_pagamento)
        if registrador.situacao_pedido:
            self.gerenciador_pedidos.altera_situacao_pedido(registrador.situacao_pedido)
            registrador.pedido_processado = self.gerenciador_pedidos.resultado['sucesso']
        self._grava_evidencia_retorno(registrador, tipo_retorno)
        return self.gerar_resposta(registrador)

    @classmethod
    def gerar_resposta(cls, servico):
        if isinstance(servico.resultado, dict):
            if 'status_code' not in servico.resultado:
                servico.resultado['status_code'] = 200 if servico.pedido_processado else 500
        if servico.redirect_para:
            retorno = {'redirect': servico.redirect_para}
            retorno.update(servico.resultado)
            return retorno
        return servico.resultado


class ParametrosDeContrato(entidades.Entidade):
    def __init__(self, loja_id):
        self.loja_id = loja_id
        self._repositorio = repositorios.ParametrosDeContratoRepositorio()

    def obter_para(self, codigo_pagamento):
        try:
            return self._repositorio.parametros_loja(self.loja_id)[codigo_pagamento]
        except (repositorios.ParametrosDeContratoNaoEncontrado, KeyError):
            try:
                return self._repositorio.parametros_base()[codigo_pagamento]
            except KeyError:
                return None


class ParcelaDePagamento(entidades.Entidade):
    def __init__(self):
        self.fator = None
        self.numero = None


class CarteiraParaBoleto(entidades.Entidade):
    def __init__(self):
        self._repositorio = repositorios.CarteiraParaBoletoRepositorio()
        self.id = None
        self.ativo = False
        self.nome = None
        self.numero = None
        self.convenio = False
        self.banco_id = None
        self.banco_nome = None

    def listar_ativas(self):
        resultado = []
        listagem = self._repositorio.listar_ativas()
        for dados in listagem:
            resultado.append(CarteiraParaBoleto.criar_apartir_de(dados))
        return resultado


class Banco(entidades.Entidade):
    def __init__(self, banco_id=None, codigo=None):
        self._repositorio = repositorios.BancoRepositorio()
        self.id = banco_id
        self.codigo = codigo
        self.nome = None
        self.imagem = None
        if banco_id:
            self.preencher_com(self._repositorio.obter_com_id(banco_id))
        if not banco_id and codigo:
            self.preencher_com(self._repositorio.obter_com_codigo(codigo))

    def listar_todos(self):
        resultado = []
        listagem = self._repositorio.listar()
        for dados in listagem:
            resultado.append(Banco.criar_apartir_de(dados))
        return resultado


class MeioDePagamento(entidades.Entidade):
    def __init__(self):
        self._repositorio = repositorios.MeioDePagamentoRepositorio()
        self.id = None
        self.codigo = None
        self.nome = None
        self.ativo_no_sistema = False
        self.plano_indice = None
        self.parcelas = []
        self.valor_minimo_parcela = Decimal(0.00)
        self.valor_minimo_parcelamento = Decimal(0.00)

    def listar(self, como_dicionario=False):
        resultado = []
        listagem = self._repositorio.listar_ativos()
        for dados in listagem:
            if como_dicionario:
                resultado.append(MeioDePagamento.criar_apartir_de(dados).to_dict())
            else:
                resultado.append(MeioDePagamento.criar_apartir_de(dados))
        return resultado

    def define_parcelas(self, dados):
        pagamento_parcelas = dados.get('pagamento_parcelas', None) or []
        for parcela in pagamento_parcelas:
            self.parcelas.append(ParcelaDePagamento.criar_apartir_de({'numero': parcela['pagamento_parcela_numero_parcelas'], 'fator': parcela['pagamento_parcela_fator']}))


class ConfiguracaoMeioPagamento(entidades.EntidadeExtensivel):
    _nao_serializar = ['eh_listagem', 'eh_gateway', 'campos', 'codigo_gateway']
    codigo_gateway = None
    eh_gateway = False
    campos = []
    modos_pagamento_aceitos = {}

    def __init__(self, loja_id=None, codigo_pagamento=None, plano_indice=1, eh_listagem=False):
        self.extensao = codigo_pagamento
        self.loja_id = loja_id
        self._repositorio = repositorios.ConfiguracaoMeioPagamentoRepositorio(loja_id)
        self.meio_pagamento = None
        self.token = None
        self.id = None
        self.usuario = None
        self.senha = None
        self.assinatura = None
        self.codigo_autorizacao = None
        self.ativo = False
        self.json = None
        self.email_comprovante = None
        self.informacao_complementar = None
        self.desconto = None
        self.desconto_tipo = 'porcentagem'
        self.desconto_valor = Decimal('0.00')
        self.aplicar_no_total = False
        self.mostrar_parcelamento = False
        self.maximo_parcelas = -2
        self.parcelas_sem_juros = 0
        self.juros_valor = Decimal('0.00')
        self.valor_minimo_parcela = Decimal('0.00')
        self.valor_minimo_aceitado = Decimal('0.00')
        self.aplicacao = None
        self.usar_antifraude = False
        self.eh_padrao = False
        self.ordem = 0
        self.plano_indice_alvo = plano_indice
        self.eh_aplicacao = False
        self.exige_https = False
        self.formulario = None
        self.parcelas_disponiveis = []
        self.eh_listagem = eh_listagem
        self._parcelas = None
        if not self.eh_listagem and self.eh_gateway:
            self.preencher_gateway(self.codigo_gateway, self.campos)

    def atualiza_meios_pagamento(self):
        pass

    def preencher_gateway(self, codigo, campos):
        dados = self._repositorio.obter_gateway(codigo)
        self.preencher_com(dados, campos)
        self.define_meio_pagamento(dados)
        self.atualiza_meios_pagamento()

    @property
    def instalado(self):
        return self.configurado

    @property
    def configurado(self):
        return (
            self.usuario is not None or
            self.senha is not None or
            self.assinatura is not None or
            self.token is not None or
            self.codigo_autorizacao is not None or
            self.json is not None
        )

    @property
    def em_uso_na_loja(self):
        if not self.meio_pagamento:
            return False
        return (
            self.configurado and
            self.ativo and
            self.meio_pagamento.ativo_no_sistema and
            self.meio_pagamento.plano_indice <= self.plano_indice_alvo
        )

    def define_meio_pagamento(self, dados):
        self.meio_pagamento = MeioDePagamento.criar_apartir_de({
            'id': dados['forma_pagamento_id'],
            'codigo': dados['pagamento_codigo'],
            'nome': dados['pagamento_nome'],
            'ativo_no_sistema': dados['pagamento_ativado'],
            'plano_indice': dados['pagamento_plano_indice'],
            'valor_minimo_parcela': dados['pagamento_parcela_valor_minimo_parcela'],
            'valor_minimo_parcelamento': dados['pagamento_parcela_valor_minimo'],
        })
        self.meio_pagamento.define_parcelas(dados)

    def aceita_pagamento_no_valor(self, valor_pagamento):
        try:
            valor_pagamento = Decimal(valor_pagamento)
        except (ValueError, TypeError, InvalidOperation):
            return True
        if not self.valor_minimo_aceitado:
            return True
        return valor_pagamento >= self.valor_minimo_aceitado

    def simula_parcelamento(self, maximo_parcelas, parcelas_sem_juros, valor_pagamento):
        self.maximo_parcelas = maximo_parcelas
        self.parcelas_sem_juros = parcelas_sem_juros
        return self.adiciona_parcelas_disponiveis(valor_pagamento)

    def _calcula_valor_parcela(self, parcela, valor_pagamento):
        if parcela.fator:
            return valor_pagamento * Decimal(parcela.fator), False
        else:
            juros_valor = self.juros_valor / Decimal(100)
            if not juros_valor:
                return valor_pagamento / Decimal(parcela.numero), True
            juros_potencia = pow((1 + juros_valor), parcela.numero)
            return valor_pagamento * juros_valor * juros_potencia / (juros_potencia - Decimal(1)), False

    def _parcelas_possiveis(self):
        if self._parcelas:
            if self.maximo_parcelas > 0:
                indice = self._parcelas.index(self.maximo_parcelas)
                return self.meio_pagamento.parcelas[:(indice + 1)]
            if self.parcelas_sem_juros:
                indice = self._parcelas.index(self.parcelas_sem_juros)
                return self.meio_pagamento.parcelas[:(indice + 1)]
        if self.maximo_parcelas > 0:
            return self.meio_pagamento.parcelas[:self.maximo_parcelas]
        if self.parcelas_sem_juros:
            return self.meio_pagamento.parcelas[:self.parcelas_sem_juros]
        return self.meio_pagamento.parcelas

    def adiciona_parcelas_disponiveis(self, valor_pagamento):
        try:
            valor_pagamento = Decimal(valor_pagamento)
        except (ValueError, TypeError, InvalidOperation):
            return self
        if self.desconto:
            desconto = 1 - (self.desconto_valor / 100)

            valor_pagamento = valor_pagamento * desconto

        for parcela in self._parcelas_possiveis():
            valor_parcelado, sem_juros = self._calcula_valor_parcela(parcela, valor_pagamento)
            if self.parcelas_sem_juros and self.parcelas_sem_juros >= parcela.numero:
                sem_juros = True
                valor_parcelado = valor_pagamento / Decimal(parcela.numero)
            if parcela.numero == 1 or valor_parcelado >= self.valor_minimo_parcela:
                self.parcelas_disponiveis.append({
                    'valor_parcelado': self.formatador.formata_decimal(valor_parcelado, como_float=True),
                    'numero': parcela.numero,
                    'sem_juros': sem_juros
                })

        return self

    def pode_ser_usado_na_loja(self, valor_pagamento, loja_usa_https):
        pode_usar = self.em_uso_na_loja and self.aceita_pagamento_no_valor(valor_pagamento)
        if self.exige_https:
            return pode_usar and loja_usa_https
        return pode_usar

    def listar_em_uso(self, como_dicionario=False, valor_pagamento=None, usa_https=False):
        listagem = self.listar()
        resultado = [item.adiciona_parcelas_disponiveis(valor_pagamento) for item in listagem if item.pode_ser_usado_na_loja(valor_pagamento, usa_https)]
        if como_dicionario:
            return [item.to_dict() for item in resultado]
        return resultado

    def listar(self, como_dicionario=False, codigo_pagamento=None):
        resultado = []
        listagem = self._repositorio.listar_ativas()
        for dados in listagem:
            if codigo_pagamento and dados['pagamento_codigo'] != codigo_pagamento:
                continue
            configuracao = self.do_gateway(dados['pagamento_codigo'], eh_listagem=True)
            if configuracao:
                configuracao.preencher_com(dados)
                configuracao.define_meio_pagamento(dados)
                configuracao.atualiza_meios_pagamento()
                configuracao.loja_id = self.loja_id
                configuracao.plano_indice_alvo = self.plano_indice_alvo
                if como_dicionario:
                    resultado.append(configuracao.to_dict())
                else:
                    resultado.append(configuracao)
        return resultado

    def do_gateway(self, codigo, eh_listagem=False):
        self.extensao = codigo
        try:
            return self.cria_entidade_extensao(self.__class__.__name__, loja_id=self.loja_id, codigo_pagamento=codigo, eh_listagem=eh_listagem)
        except extensibilidade.ExtensaoNaoDefinido:
            return None

    def _normaliza_dados_de_parcelamento(self, dados):
        maximo_parcelas = dados.get('maximo_parcelas')
        parcelas_sem_juros = dados.get('parcelas_sem_juros')

        if not maximo_parcelas or not parcelas_sem_juros:
            return

        maximo_parcelas = int(maximo_parcelas)
        parcelas_sem_juros = int(parcelas_sem_juros)

        if maximo_parcelas != 0 and parcelas_sem_juros > maximo_parcelas:
            parcelas_sem_juros = maximo_parcelas

        dados['maximo_parcelas'] = maximo_parcelas
        dados['parcelas_sem_juros'] = parcelas_sem_juros

    def salvar_formulario(self, dados):
        if 'eh_padrao' in dados:
            self.definir_padrao(dados)
        else:
            self.formulario.validar_valores(dados)
            self._normaliza_dados_de_parcelamento(dados)
            self.preencher_com(dados, self.formulario.nomes_campos)
            self._repositorio.instanciar_banco(self.meio_pagamento.id)
            self.formulario.definir_valores(self._repositorio.do_banco, dados)
            self._repositorio.salvar_instancia()

    def salvar(self, dados):
        self._normaliza_dados_de_parcelamento(dados)
        self.preencher_com(dados)
        self._gravar_dados(dados)

    def _gravar_dados(self, dados):
        self._repositorio.instanciar_banco(self.meio_pagamento.id)
        self._repositorio.definir_valores(dados)
        self._repositorio.salvar_instancia()

    def definir_padrao(self, dados):
        self._repositorio.remover_pagamento_padrao()
        if dados['eh_padrao']:
            self.salvar({'eh_padrao': True})

    def atualizar(self, dados=None):
        if dados and 'ordenacao' in dados:
            return self.ordenar_formas_pagamento(dados['ordenacao'])
        self._gravar_dados(self.to_dict())
        return {'resultado', 'OK'}

    def ordenar_formas_pagamento(self, ordenacao):
        self._repositorio.remover_ordenacao()
        forma_pagamento_ids = self._repositorio.obter_forma_pagamento_id(ordenacao)
        for ordem, codigo_pagamento in enumerate(ordenacao):
            forma_pagamento = ConfiguracaoMeioPagamento(loja_id=self.loja_id)
            forma_pagamento.define_meio_pagamento({'forma_pagamento_id': forma_pagamento_ids[codigo_pagamento], 'pagamento_codigo': None, 'pagamento_nome': None, 'pagamento_ativado': None, 'pagamento_plano_indice': None, 'pagamento_parcela_valor_minimo_parcela': None, 'pagamento_parcela_valor_minimo': None})
            forma_pagamento.salvar({'ordem': ordem + 1})
        return {'resultado': 'OK'}

    def instalar(self, dados):
        if not self.extensao:
            return None
        instalador = self.cria_servico_extensao('InstalaMeioDePagamento', loja_id=self.loja_id, dados=dados)
        fase_atual = dados.get('fase_atual', '')
        if '?' in fase_atual:
            fase_atual = fase_atual.split('?')
            resto_query = fase_atual[1].split('&')
            for resto in resto_query:
                par = resto.split('=')
                try:
                    dados[par[0]] = par[1]
                except IndexError:
                    dados[par[0]] = None
            fase_atual = fase_atual[0]
        if fase_atual == FaseDeInstalacao.montando_url_autorizacao:
            try:
                return instalador.montar_url_autorizacao()
            except instalador.InstalacaoNaoFinalizada, ex:
                return {'mensagem': u'Ocorreu um erro durante a instalação: {}'.format(ex).encode('utf-8'), 'status': 500}
        if fase_atual == FaseDeInstalacao.obtendo_dados_gateway:
            erro = None
            try:
                dados_instalacao = instalador.obter_dados()
                self.salvar(dados_instalacao)
            except instalador.InstalacaoNaoFinalizada, ex:
                erro = u'Ocorreu um erro durante a instalação: {}'.format(ex).encode('utf-8')
            if 'next_url' in dados:
                if erro:
                    return {'redirect': dados['next_url'], 'erro': erro}
                return {'redirect': dados['next_url']}
            return 'Sucesso'

    def desinstalar(self, dados):
        if not self.extensao:
            return None
        dados_desinstalacao = {}
        instalador = self.cria_servico_extensao('InstalaMeioDePagamento', loja_id=self.loja_id, dados=dados)
        for campo in instalador.campos:
            dados[campo] = getattr(self, campo, None)
            dados_desinstalacao[campo] = None
        self.salvar(dados_desinstalacao)
        try:
            resultado = instalador.desinstalar(dados)
        except instalador.InstalacaoNaoFinalizada, ex:
            return {'mensagem': u'Ocorreu um erro durante a desinstalação: {}'.format(ex).encode('utf-8'), 'status': getattr(ex, 'status', 500)}
        return resultado


class FaseDeInstalacao(object):
    montando_url_autorizacao = '1'
    obtendo_dados_gateway = '2'
