# -*- coding: utf-8 -*-
"""
/***************************************************************************
 * 
 * Copyright (c) 2014 Baidu.com, Inc. All Rights Reserved
 * $Id$ 
 * 
 **************************************************************************/
 
 
 
/**
 * @file httpmcpackmgr.py
 * @author daiwenkai(daiwenkai@baidu.com)
 * @date 2014/12/29 23:07:23
 * @version $Revision$ 
 * @brief 
 *  
 **/
"""

import json
import httplib
import logging
import time
import datetime
import sys
import time

import os
os.sys.path.append(os.path.dirname(os.path.abspath(__file__))+"/../")

import globals.globals as globals
import connections.connectionmgr as connectionmgr
import tools.log as log
import tools.utils as utils

import libs.mcpack as mcpack

class HttpMcpackMgr(connectionmgr.ConnectionMgr):
    """
    HttpMcpackMgr 
    """

    def __init__(self):
        """
        init HttpMcpackMgr
        """
        super(HttpMcpackMgr, self).__init__()

    def gen_id(self):
        """
        gen_id 利用时间戳生成请求adcore的id
        """
        # *1000: 毫秒级的时间戳
        id = int(time.time() * 1000)

        return id

    def print_log(self, service, method, part, type, content):
        """
        print log 将method header/body content打印出来
        type: req/res
        part: body/header
        """
        #ret = "\n" + "\033[4;32m" + "+"*10
        ret = ""
        ret += " [SERVICE] : " + service.upper() 
        ret += " [METHOD] : " + method.upper() 
        if type == "req":
            ret += " [REQUEST] " + part.upper() 
        elif type == "res":
            ret += " [RESPONSE] " + part.upper() 
        #ret += "+" * 10 + "\033[0m\n"
        ret += content
        logging.info(ret) 

    def create_conn(self, url, port, timeout):
        """
        create_conn 建立与服务器的连接
        """
        try:
            conn = httplib.HTTPConnection(url, port, timeout=timeout)
            connectionmgr.ConnectionMgr.set_connection(self, conn)
        except Exception as e:
            utils.print_exception(e, __file__)
            conn = None

    def make_body(self, method, params):
        """
        make body
        mcpack的rpc请求的body必须具备的格式
        """
        body = {
                "jsonrpc": "2.0", 
                "method": method,
                "id": self.gen_id(),
                "params": params
                }
        return body 

    def process(self, service, method, params):
        """
        process
        总控函数：
        构造请求包->进行连接->分析返回包
        """
        start_time = time.time() * 1000 #ms
        # 构造请求包
        headers = {'Content-type': 'application/baidu.mcpack-rpc;charset=GBK', 'Connection': "close"}
        
        body = self.make_body(method, params)

        # 填充header和body
        connectionmgr.ConnectionMgr.prepare(self, headers, body)

        type = "req"
        self.print_log(service, method, "header", type, str(json.dumps(headers)))
        self.print_log(service, method, "body", type, str(json.dumps(body)))
       
        try: 
            # mcpack.dumps()只接受字典类型，如果body不是字典，会抛异常
            mybody = mcpack.dumps(body)
        except Exception as e:
            utils.print_exception(e, __file__)
            mybody = {}
            res = self.analyse_response(mybody)
            return res

        start_conn_time = time.time() * 1000 #ms
        # 连接server
        response = self.connect(service, method, headers, mybody)
        end_conn_time = time.time() * 1000 #ms
        conn_proc_time = end_conn_time - start_conn_time
        logging.info("[method:%s][conn proc_time:%fms]" % (method, conn_proc_time))
       
        # 分析返回包
        res = self.analyse_response(response)
        end_time = time.time() * 1000 # ms
        proc_time = end_time - start_time
        logging.info("[method:%s][total proc_time:%fms]" % (method, proc_time))

        return res

    def connect(self, service, method, headers, body):
        """
        connect
        发起rpc请求，并记录最原始的返回包
        """
        res_dict = ""

        try:
            self.conn.request("POST", service, body, headers)

            response = self.conn.getresponse()
            res = response.read()
            if "<body>" in res or str(res) == "":
                # 返回包不是mcpack形式
                # 分以下两种情况：
                # 1.由于是http请求，所以返回的是一个404之类的html页面,返回结果中有<body>
                #   出现此情况的原因一般是service(如/WinfoModAPIProxy)不存在
                # 2.如果是服务有问题等情况，则返回结果会是空串
                logging.debug("[res: %s]" % (str(res)))
                res_dict = res
            else:
                res_dict = mcpack.loads(res)
            
            type = "res"
            res_header = response.getheaders()

            self.print_log(service, method, "header", type, str(res_header))
            self.print_log(service, method, "body", type, str(res_dict))

        except Exception as e:
            utils.print_exception(e, __file__)
            res_dict = {}

        return res_dict

    def analyse_response(self, res):
        """
        analyse response
        依据返回结果对res_info, res_err_detail以及res_data进行填充
        """
        if "<body>" in res or str(res) == "":
            # 返回包不是mcpack形式
            # 分以下两种情况：
            # 1.由于是http请求，所以返回的是一个404之类的html页面,返回结果中有<body>
            #   出现此情况的原因一般是service(如/WinfoModAPIProxy)不存在
            # 2.如果是服务有问题等情况，则返回结果会是空串
            logging.debug("res not mcpack")
            res_status = globals.STATUS_EXTERNAL_ERR_HTML
            res_err_detail = {}
            res_err_detail["error"] = res
            res_data = {}

        elif "result" in res and res["id"] is not None:
            # 正常返回，对response['result']['status']字段分类处理
            logging.debug("result exists")
            result = res["result"]
            
            res_status = int(result["status"])
            logging.debug("res_status is " + str(res_status))
            
            if globals.STATUS_SUCCESS == res_status:
                # 全部成功
                logging.info("status: success!!")
                if "errorDetail" in result.keys() and result["errorDetail"] is not None:
                    res_err_detail = result["errorDetail"]
                else:
                    res_err_detail = {}
                res_data = result["data"]

            elif globals.STATUS_PARTIAL_SUCCESS == res_status:
                # 部分成功
                logging.info("status: partial success!!")
                if "errorDetail" in result.keys() and result["errorDetail"] is not None:
                    res_err_detail = result["errorDetail"]
                else:
                    res_err_detail = {}
                res_data = result["data"]

            elif globals.STATUS_FAIL == res_status:
                # 如果返回400,
                # 业务级的错误会填充在errorDetail字段中，
                # 系统级的错误会填充在error字段中。
                logging.warning("status: fail!!")
                if "errorDetail" in result.keys() and result["errorDetail"] is not None:
                    # 业务级的错误
                    logging.debug("errorDetail exists!!")
                    res_err_detail = result["errorDetail"]
                elif "error" in result.keys() and result["error"] is not None:
                    # 系统级的错误,如请求包缺少必填字段或字段为空
                    logging.debug("errorDetail not exist!! error exists!!")
                    res_err_detail = {}
                    res_err_detail['error'] = result["error"]
                else:
                    res_err_detail = {}
                
                if "data" in result.keys() and result["data"] is not None:
                    res_data = result["data"]
                else:
                    res_data = {}

        elif "error" in res and res['error'] is not None:
            # mcpack-rpc的错误信息
            # 请求包格式出错等情况
            logging.debug("req format error")
            msg = "请求包格式出错等情况" + json.dumps(res['error'])
            logging.warning(msg)
            res_status = globals.STATUS_EXTERNAL_ERR
            res_err_detail = res['error']
            res_data = {} 
        else:
            # 连接出错等情况
            logging.debug("server conn error")
            res_status = globals.STATUS_SERVER_CONN_ERR
            res_data = {}
            res_err_detail = {}
       
        res_info = {}
        res_info["status"] = res_status
        res_info["succ_count"] = len(res_data)
        res_info["fail_count"] = len(res_err_detail.keys())
        
        return (res_info, res_err_detail, res_data)

    def get_response(self, service, method, params):
        """
        get response调用process函数
        """
        res = self.process(service, method, params)

        return res

"""
/* vim: set ts=4 sw=4 sts=4 tw=100 */
"""
