"""
Numdifftools implementation

"""


# Author:      Per A. Brodtkorb
#
# Created:     01.08.2008
# Copyright:   (c) pab 2008
# Licence:     New BSD
#
# Based on matlab functions derivest.m gradest.m hessdiag.m, hessian.m
# and jacobianest.m by:
#
# Author: John D'Errico
# e-mail: woodchips@rochester.rr.com
# Release: 1.0
# Release date: 12/27/2006

# !/usr/bin/env python

from __future__ import division, print_function
import numpy as np
import scipy.linalg as linalg
import scipy.misc as misc
# import scipy.interpolate as si
import warnings
import matplotlib.pyplot as plt

__all__ = [
    'dea3', 'Derivative', 'Jacobian', 'Gradient', 'Hessian', 'Hessdiag'
]

_TINY = np.finfo(float).tiny
_EPS = np.finfo(float).eps


# def _extrapolate(h1, der_romb, errest):
#     i0 = np.argmin(errest)-1
#     x = h1[:i0]
#     try:
#         y = der_romb[:i0] - der_romb[i0]
#         # val, abserr = dea3(y[:-2], y[1:-1], y[2:])
#         # return val +der_romb[i0], abserr
#         sy = np.sign(y[-1])
#         k = np.flatnonzero(sy == np.sign(y))
#         logy = np.log(sy * y[k])[::-1]
#         logx = np.log(x[k])[::-1]
#         logxi = np.log(h1[i0:])[::-1]
#         tck = si.splrep(logx, logy, k=1, s=0, xb=logxi[0])
#         der_romb = (sy * np.exp(si.splev(np.log(h1[::-1]), tck))[::-1] +
#                     der_romb[i0])
#         # return der_romb, errest
#         val, abserr = dea3(der_romb[2:], der_romb[1:-1], der_romb[2:],
#                            symmetric=True)
#         return val[:i0], abserr[:i0]
#     except:
#         return der_romb, errest


class Dea(object):
    '''
    LIMEXP  is the maximum number of elements the
    epsilon table data can contain. The epsilon table
    is stored in the first (LIMEXP+2) entries of EPSTAB.


    LIST OF MAJOR VARIABLES
    -----------------------
    E0,E1,E2,E3 - DOUBLE PRECISION
                  The 4 elements on which the computation of
                  a new element in the epsilon table is based.
    NRES   - INTEGER
             Number of extrapolation results actually
             generated by the epsilon algorithm in prior
             calls to the routine.
    NEWELM - INTEGER
             Number of elements to be computed in the
             new diagonal of the epsilon table. The
             condensed epsilon table is computed. Only
             those elements needed for the computation of
             the next diagonal are preserved.
    RES    - DOUBLE PREISION
             New element in the new diagonal of the
             epsilon table.
    ERROR  - DOUBLE PRECISION
             An estimate of the absolute error of RES.
             Routine decides whether RESULT=RES or
             RESULT=SVALUE by comparing ERROR with
             ABSERR from the previous call.
    RES3LA - DOUBLE PREISION
             Vector of DIMENSION 3 containing at most
             the last 3 results.
    '''
    def __init__(self, limexp=3):
        self.limexp = 2 * (limexp // 2) + 1
        self.epstab = np.zeros(limexp+5)
        self.ABSERR = 10.
        self._n = 0
        self._nres = 0
        if (limexp < 3):
            raise ValueError('LIMEXP IS LESS THAN 3')

    def _compute_error(self, RES3LA, NRES, RES):
        fact = [6.0, 2.0, 1.0][min(NRES-1, 2)]
        error = fact * np.abs(RES - RES3LA[:NRES]).sum()
        return error

    def _shift_table(self, EPSTAB, N, NEWELM, NUM):
        i_0 = 1 if ((NUM // 2) * 2 == NUM - 1) else 0
        i_n = 2 * NEWELM + 2
        EPSTAB[i_0:i_n:2] = EPSTAB[i_0 + 2:i_n + 2:2]

        if (NUM != N):
            i_n = NUM - N
            EPSTAB[:N + 1] = EPSTAB[i_n:i_n + N + 1]
        return EPSTAB

    def _update_RES3LA(self, RES3LA, RESULT, NRES):
        if NRES > 2:
            RES3LA[:2] = RES3LA[1:]
            RES3LA[2] = RESULT
        else:
            RES3LA[NRES] = RESULT

    def __call__(self, SVALUE):

        EPSTAB = self.epstab
        RES3LA = EPSTAB[-3:]
        RESULT = SVALUE
        N = self._n
        NRES = self._nres
        EPSTAB[N] = SVALUE
        if (N == 0):
            ABSERR = abs(RESULT)
        elif (N == 1):
            ABSERR = 6.0 * abs(RESULT - EPSTAB[0])
        else:
            ABSERR = self.ABSERR
            EPSTAB[N + 2] = EPSTAB[N]
            NEWELM = N // 2
            NUM = N
            K1 = N
            for I in range(NEWELM):
                E0 = EPSTAB[K1 - 2]
                E1 = EPSTAB[K1 - 1]
                E2 = RES = EPSTAB[K1 + 2]
                DELTA2, DELTA3 = E2 - E1, E1 - E0
                ERR2, ERR3 = abs(DELTA2), abs(DELTA3)
                TOL2 = max(abs(E2), abs(E1)) * _EPS
                TOL3 = max(abs(E1), abs(E0)) * _EPS
                converged = (ERR2 <= TOL2 and ERR3 <= TOL3)
                if converged:
                    ABSERR = ERR2 + ERR3
                    RESULT = RES
                    break
                if (I != 0):
                    E3 = EPSTAB[K1]
                    DELTA1 = E1 - E3
                    ERR1 = abs(DELTA1)
                    TOL1 = max(abs(E1), abs(E3)) * _EPS
                    converged = (ERR1 <= TOL1 or ERR2 <= TOL2 or
                                 ERR3 <= TOL3)
                    if not converged:
                        SS = 1.0 / DELTA1 + 1.0 / DELTA2 - 1.0 / DELTA3
                else:
                    converged = (ERR2 <= TOL2 or ERR3 <= TOL3)
                    if not converged:
                        SS = 1.0 / DELTA2 - 1.0 / DELTA3
                EPSTAB[K1] = E1
                if (converged or abs(SS * E1) <= 1e-04):
                    N = 2 * I
                    if (NRES == 0):
                        ABSERR = ERR2 + ERR3
                        RESULT = RES
                    else:
                        RESULT = RES3LA[min(NRES-1, 2)]
                    break
                RES = E1 + 1.0 / SS
                EPSTAB[K1] = RES
                K1 = K1 - 2
                if (NRES == 0):
                    ABSERR = ERR2 + abs(RES - E2) + ERR3
                    RESULT = RES
                    continue
                ERROR = self._compute_error(RES3LA, NRES, RES)

                if (ERROR > 10.0 * ABSERR):
                    continue
                ABSERR = ERROR
                RESULT = RES
            else:
                ERROR = self._compute_error(RES3LA, NRES, RES)

            # 50
            if (N == self.limexp - 1):
                N = 2 * (self.limexp // 2) - 1
            EPSTAB = self._shift_table(EPSTAB, N, NEWELM, NUM)
            self._update_RES3LA(RES3LA, RESULT, NRES)

            ABSERR = max(ABSERR, 10.0*_EPS * abs(RESULT))
            NRES = NRES + 1

        N += 1
        self._n = N
        self._nres = NRES
        # EPSTAB[-3:] = RES3LA
        self.ABSERR = ABSERR
        return RESULT, ABSERR


def test_dea():
    linfun = lambda i: np.linspace(0, np.pi/2., 2**i+1)
    dea = Dea(limexp=11)
    print('NO. PANELS      TRAP. APPROX          APPROX W/EA           ABSERR')
    for k in np.arange(10):
        x = linfun(k)
        val = np.trapz(np.sin(x), x)
        vale, err = dea(val)
        print('%5d %20.8f  %20.8f  %20.8f' % (len(x)-1, val, vale, err))


def dea3(v0, v1, v2, symmetric=False):
    """
    Extrapolate a slowly convergent sequence

    Parameters
    ----------
    v0, v1, v2 : array-like
        3 values of a convergent sequence to extrapolate

    Returns
    -------
    result : array-like
        extrapolated value
    abserr : array-like
        absolute error estimate

    Description
    -----------
    DEA3 attempts to extrapolate nonlinearly to a better estimate
    of the sequence's limiting value, thus improving the rate of
    convergence. The routine is based on the epsilon algorithm of
    P. Wynn, see [1]_.

     Example
     -------
     # integrate sin(x) from 0 to pi/2

     >>> import numpy as np
     >>> import numdifftools as nd
     >>> Ei= np.zeros(3)
     >>> linfun = lambda i : np.linspace(0, np.pi/2., 2**(i+5)+1)
     >>> for k in np.arange(3):
     ...    x = linfun(k)
     ...    Ei[k] = np.trapz(np.sin(x),x)
     >>> [En, err] = nd.dea3(Ei[0], Ei[1], Ei[2])
     >>> truErr = Ei-1.
     >>> (truErr, err, En)
     (array([ -2.00805680e-04,  -5.01999079e-05,  -1.25498825e-05]),
     array([ 0.00020081]), array([ 1.]))

     See also
     --------
     dea

     Reference
     ---------
     .. [1] C. Brezinski (1977)
            "Acceleration de la convergence en analyse numerique",
            "Lecture Notes in Math.", vol. 584,
            Springer-Verlag, New York, 1977.
    """
    E0, E1, E2 = np.atleast_1d(v0, v1, v2)
    abs, max = np.abs, np.maximum  # @ReservedAssignment
    delta2, delta1 = E2 - E1, E1 - E0
    err2, err1 = abs(delta2), abs(delta1)
    tol2, tol1 = max(abs(E2), abs(E1)) * _EPS, max(abs(E1), abs(E0)) * _EPS

    with warnings.catch_warnings():
        warnings.simplefilter("ignore")  # ignore division by zero and overflow
        ss = 1.0 / delta2 - 1.0 / delta1
        smalle2 = (abs(ss * E1) <= 1.0e-3).ravel()
    converged = (err1 <= tol1) & (err2 <= tol2).ravel() | smalle2
    result = np.where(converged, E2 * 1.0, E1 + 1.0 / ss)
    abserr = err1 + err2 + np.where(converged, tol2 * 10, abs(result-E2))
    if symmetric and len(result) > 1:
        return result[:-1], abserr[1:]
    return result, abserr


def vec2mat(vec, n, m):
    ''' forms the matrix M, such that M[i,j] = vec[i+j]
    '''
    [i, j] = np.ogrid[0:n, 0:m]
    return np.matrix(vec[i + j])


class _Derivative(object):

    ''' Object holding common variables and methods for the numdifftools

    Parameters
    ----------
    fun : callable
        function to differentiate.
    n : Integer from 1 to 4             (Default 1)
        defining derivative order.
    order : Integer from 1 to 4        (Default 2)
        defining order of basic method used.
        For 'central' methods, it must be from the set [2,4].
    method : Method of estimation.  Valid options are:
        'central', 'forward' or 'backward'.          (Default 'central')
    romberg_terms : integer from 0 to 3  (Default 2)
        Number of Romberg terms used in the extrapolation.
        Note: 0 disables the Romberg step completely.
    step_nom : vector   default maximum(log1p(abs(x0)), 0.02)
        Nominal step. (The steps: h_i = step_nom[i] * delta)
    step_max : real scalar  (Default 2.0)
        Maximum allowed excursion from step_nom as a multiple of it.
    step_ratio: real scalar  (Default 2.0)
        Ratio used between sequential steps in the estimation of the derivative
    step_num : integer  (Default 26)
        The minimum step_num for making romberg extrapolation work is
            7 + np.ceil(self.n/2.) + self.order + self.romberg_terms
    delta : vector default step_max*step_ratio**(-arange(steps_num))
        Defines the steps sizes used in derivation: h_i = step_nom[i] * delta
    vectorized : Bool
        True  - if your function is vectorized.
        False - loop over the successive function calls (default).

    Uses a semi-adaptive scheme to provide the best estimate of the
    derivative by its automatic choice of a differencing interval. It uses
    finite difference approximations of various orders, coupled with a
    generalized (multiple term) Romberg extrapolation. This also yields the
    error estimate provided. See the document DERIVEST.pdf for more explanation
    of the algorithms behind the parameters.

     Note on order: higher order methods will generally be more accurate,
             but may also suffer more from numerical problems. First order
             methods would usually not be recommended.
     Note on method: Central difference methods are usually the most accurate,
            but sometimes one can only allow evaluation in forward or backward
            direction.
    '''

    def __init__(self, fun, n=1, order=2, method='central', romberg_terms=2,
                 step_max=2.0, step_nom=None, step_ratio=2.0, step_num=26,
                 delta=None, vectorized=False, verbose=False,
                 use_dea=True):
        self.fun = fun
        self.n = n
        self.order = order
        self.method = method
        self.romberg_terms = romberg_terms
        self.step_max = step_max
        self.step_ratio = step_ratio
        self.step_nom = step_nom
        self.step_num = step_num
        self.delta = delta
        self.vectorized = vectorized
        self.verbose = verbose
        self.use_dea = use_dea

        self._check_params()

        self.error_estimate = None
        self.final_delta = None

        # The remaining member variables are set by _initialize
        self._fd_rule = None
        self._rmat = None
        self._qromb = None
        self._rromb = None
        self._diff_fun = None

    def _set_delta(self, delta=None):
        ''' Set the steps to use in derivation.

        Member variables used:
        step_ratio, step_num, step_max
        '''
        if delta is None:
            step_ratio = self._make_exact(self.step_ratio)
            if self.step_num is None:
                num_steps = self._get_min_num_steps()
            else:
                num_steps = max(self.step_num, 1)
            step1 = self._make_exact(self.step_max)
            self._delta = step1 * step_ratio ** (-np.arange(num_steps))
        else:
            self._delta = np.atleast_1d(delta)

    delta = property(lambda cls: cls._delta, fset=_set_delta)
    finaldelta = property(lambda cls: cls.final_delta)

    def _check_params(self):
        ''' check the parameters for acceptability
        '''
        atleast_1d = np.atleast_1d
        kwds = self.__dict__
        for name in ['n', 'order']:
            val = np.atleast_1d(kwds[name])
            if ((len(val) != 1) or (val not in (1, 2, 3, 4))):
                raise ValueError('%s must be scalar, one of [1 2 3 4].' % name)
        name = 'romberg_terms'
        val = atleast_1d(kwds[name])
        if not ((len(val) == 1) and (val in (0, 1, 2, 3))):
            raise ValueError('%s must be scalar, one of [0 1 2 3].' % name)

        for name in ('step_max', 'step_num'):
            val = kwds[name]
            if val is not None and (len(atleast_1d(val)) > 1 or val <= 0):
                raise ValueError('%s must be None or a scalar, >0.' % name)

        valid_methods = dict(c='central', f='forward', b='backward')
        method = valid_methods.get(kwds['method'][0])
        if method is None:
            t = 'Invalid method: Must start with one of c, f, b characters!'
            raise ValueError(t)
        if method[0] == 'c' and kwds['method'] in (1, 3):
            t = 'order 1 or 3 is not possible for central difference methods'
            raise ValueError(t)

    def _initialize(self):
        '''Set derivative parameters:
            differention rule and romberg extrapolation matrices
        '''
        self._set_fd_rule()
        self._set_romb_qr()
        self._set_difference_function()

    def _fder(self, fun, f_x0i, x0i, h):
        '''
        Return derivative estimates of f at x0 for a sequence of stepsizes h

        Member variables used
        ---------------------
        n
        _fd_rule
        romberg_terms
        '''
        fd_rule = self._fd_rule
        n_fdr = fd_rule.size
        n_h = h.size

        f_del = self._diff_fun(fun, f_x0i, x0i, h)

        if f_del.size != n_h:
            raise ValueError('fun did not return data of correct size ' +
                             '(it must be vectorized)')

        ne = max(n_h + 1 - n_fdr - self.romberg_terms, 1)
        der_init = np.asarray(vec2mat(f_del, ne, n_fdr) * fd_rule.T).ravel()
        der_init = der_init / (h[:ne]) ** self.n

        return der_init, h[:ne]

#     def _trim_estimates(self, der_romb, errors, h):
#         '''
#         trim off the estimates at each end of the scale
#         '''
#         trimdelta = h.copy()
#         der_romb = np.atleast_1d(der_romb)
#         num_vals = len(der_romb)
#         nr_rem_min = int((num_vals - 1) / 2)
#         nr_rem = min(2 * max((self.n - 1), 1), nr_rem_min)
#         if nr_rem > 0:
#             tags = der_romb.argsort()
#             tags = tags[nr_rem:-nr_rem]
#             der_romb = der_romb[tags]
#             errors = errors[tags]
#             trimdelta = trimdelta[tags]
#         return der_romb, errors, trimdelta

    def _plot_errors(self, h2, errors, step_nom_i, der_romb):
        i = np.argsort(h2)
        ii = np.arange(len(h2))
        ii[i] = np.arange(len(h2))
        print('  Stepsize             Value             Errors')
        fmt = ''.join(('%10.2g    %20.15g    %10.2g\n',)*len(h2))
        tmp = (np.vstack((h2[i], der_romb[i], errors[i])).T).ravel().tolist()
        print(fmt % tuple(tmp))
        # ind = self._get_arg_min(errors)
        # plt.ioff()
        try:
            # diff_df = np.diff(np.hstack(((der_romb[i], 0))))[ii]
            diff_df = np.diff(np.hstack(((0, der_romb[i], 0))), n=1)  # [ii]
            diff_df = np.diff(np.log(abs(diff_df)))[ii]
            ip = diff_df >= 0
            plt.subplot(2, 1, 1)
            plt.semilogx(h2[i], np.abs(diff_df[i]) + _EPS, 'k',
                         h2[ip], diff_df[ip] + _EPS, 'r.',
                         h2[~ip], -diff_df[~ip] + _EPS, 'b.')
            small = 2 * np.sqrt(_EPS) ** (1. / np.sqrt(self.n))
            plt.vlines(small, 1e-15, 1)
            plt.title('abs(df_i(x)-df_best(x)) as function of stepsize. ' +
                      '(nom=%g)' % step_nom_i)
            plt.subplot(2, 1, 2)
            plt.loglog(h2[i], errors[i], 'r--', h2, errors, 'r.')
            plt.vlines(small, 1e-15, 1)
            plt.title('Estimated error as function of stepsize. (nom=%g)' %
                      step_nom_i)

            # plt.show()
        except:
            pass

    def _get_arg_min(self, errest):
        arg_mins = np.flatnonzero(errest == np.min(errest))
        n = arg_mins.size
        return arg_mins[n // 2]

    def _get_step_nom(self, step_nom, x0):
        if step_nom is None:
            step_nom = np.maximum(np.log1p(np.abs(x0)), 0.02)
        return self._make_exact(np.atleast_1d(step_nom)) * np.ones(x0.shape)

    def _eval_first(self, fun, x0):
        f_x0 = np.zeros(x0.shape)
        # will we need fun(x0)?
        even_order = np.remainder(self.n, 2) == 0
        if even_order or not self.method[0] == 'c':
            if self.vectorized:
                f_x0 = fun(x0)
            else:
                f_x0 = np.asfarray([fun(x0j) for x0j in x0])
        return f_x0

    def _remove_non_positive(self, h):
        threshold = (self.n > 1) * 10.0 ** (-15 + self.n)
        if (h <= threshold).any():
            warnings.warn('Some of the steps are too small, either because ' +
                          'step_max*step_nom is too small or ' +
                          'step_ratio**step_num too large!')
            return h[h > threshold]
        return h

    def _make_exact(self, h):
        '''Make sure h is an exact representable number
        This is important when calculating numerical derivatives and is
        accomplished by adding 1 and then subtracting 1..
        '''
        return (h + 1.0) - 1.0

    def _get_steps(self, step_nom):
        h = self._make_exact((1.0 * step_nom) * self._delta)
        return self._remove_non_positive(h)

    def _best_der(self, der_romb, errors, h2):
        # der_romb, errors, h2 = self._trim_estimates(der_romb, errors, h2)
        i = self._get_arg_min(errors)
        return der_romb[i], errors[i], h2[i]

    def _derivative(self, fun, x00, step_nom=None):
        x0 = np.atleast_1d(x00)
        step_nom = self._get_step_nom(step_nom, x0)

        f_x0 = self._eval_first(fun, x0)
        n, nx0 = x0.size, x0.shape
        der, err, delta = np.zeros(nx0), np.zeros(nx0), np.zeros(nx0)
        for i in range(n):
            x0i, f_x0i = float(x0[i]), float(f_x0[i])
            h = self._get_steps(step_nom[i])
            der_init, h1 = self._fder(fun, f_x0i, x0i, h)
            der_romb, errors, h2 = self._romb_extrap(der_init, h1)
            if self.verbose:
                self._plot_errors(h2, errors, step_nom[i], der_romb)
            der[i], err[i], delta[i] = self._best_der(der_romb, errors, h2)
        return der, err, delta

    def _fd_mat(self, parity, nterms):
        ''' Return matrix for finite difference derivation.

        Parameters
        ----------
        parity : scalar, integer
            0 (one sided, all terms included but zeroth order)
            1 (only odd terms included)
            2 (only even terms included)
        nterms : scalar, integer
            number of terms

        Member variables used
        ---------------------
        step_ratio
        '''
        srinv = 1.0 / self.step_ratio
        factorial = misc.factorial
        arange = np.arange
        [i, j] = np.ogrid[0:nterms, 0:nterms]
        if parity == 0:
            c = 1.0 / factorial(arange(1, nterms + 1))
            mat = c[j] * srinv ** (i * (j + 1))
        elif parity == 1 or parity == 2:
            c = 1.0 / factorial(arange(parity, 2 * nterms + 1, 2))
            mat = c[j] * srinv ** (i * (2 * j + parity))
        else:
            raise ValueError('Parity must be 0, 1 or 2! (%d)' % parity)
        return np.matrix(mat)

    def _set_fd_rule(self):
        '''
        Generate finite differencing rule in advance.

        The rule is for a nominal unit step size, and will
        be scaled later to reflect the local step size.

        Member methods used
        -------------------
        _fd_mat

        Member variables used
        ---------------------
        n
        order
        method
        '''
        der_order = self.n
        met_order = self.order
        method = self.method[0]

        matrix = np.matrix
        zeros = np.zeros
        fd_rule = matrix(der_order)

        pinv = linalg.pinv
        if method == 'c':  # 'central'
            if met_order == 2:
                if der_order == 3:
                    fd_rule = matrix([0, 1]) * pinv(self._fd_mat(1, 2))
                elif der_order == 4:
                    fd_rule = matrix([0, 1]) * pinv(self._fd_mat(2, 2))
            elif der_order == 1:
                fd_rule = matrix([1, 0]) * pinv(self._fd_mat(1, 2))
            elif der_order == 2:
                fd_rule = matrix([1, 0]) * pinv(self._fd_mat(2, 2))
            elif der_order == 3:
                fd_rule = matrix([0, 1, 0]) * pinv(self._fd_mat(1, 3))
            elif der_order == 4:
                fd_rule = matrix([0, 1, 0]) * pinv(self._fd_mat(2, 3))
        else:
            if met_order == 1:
                if der_order != 1:
                    v = zeros(der_order)
                    v[der_order - 1] = 1
                    fd_rule = matrix(v) * pinv(self._fd_mat(0, der_order))
            else:
                v = zeros(der_order + met_order - 1)
                v[der_order - 1] = 1
                dpm = der_order + met_order - 1
                fd_rule = matrix(v) * pinv(self._fd_mat(0, dpm))
            if method == 'b':  # 'backward' rule
                fd_rule = -fd_rule
        self._fd_rule = fd_rule.ravel()

    def _get_min_num_steps(self):
        n0 = 5 if self.method[0] == 'c' else 7
        return int(n0 + np.ceil(self.n / 2.) + self.order + self.romberg_terms)

    def _set_romb_qr(self):
        '''
        Member variables used
        order
        method
        romberg_terms
        '''
        num_terms = self.romberg_terms
        add1 = self.method[0] == 'c'
        rombexpon = (1 + add1) * np.arange(num_terms) + self.order

        srinv = self._make_exact(1.0 / self.step_ratio)
        rmat = np.ones((num_terms + 2, num_terms + 1))
        if num_terms > 0:
            for n in range(1, num_terms + 2):
                rmat[n, 1:] = srinv ** (n * rombexpon)
        rmat = np.matrix(rmat)
        self._qromb, self._rromb = linalg.qr(rmat)
        self._rmat = rmat

    def _set_difference_function(self):
        ''' Set _diff_fun function according to method
        '''
        get_diff_fun = dict(c=self._central, b=self._backward,
                            f=self._forward)[self.method[0]]
        self._diff_fun = get_diff_fun()

    def _central(self):
        ''' Return central difference function

        Member variables used
            n
            fun
            vectorized
        '''
        even_order = (np.remainder(self.n, 2) == 0)

        if self.vectorized:
            if even_order:
                f_del = lambda fun, f_x0i, x0i, h: (
                    fun(x0i + h) + fun(x0i - h)).ravel() / 2.0 - f_x0i
            else:
                f_del = lambda fun, f_x0i, x0i, h: (
                    fun(x0i + h) - fun(x0i - h)).ravel() / 2.0
        else:
            if even_order:
                f_del = lambda fun, f_x0i, x0i, h: np.asfarray(
                    [fun(x0i + h_j) + fun(x0i - h_j)
                     for h_j in h]).ravel() / 2.0 - f_x0i
            else:
                f_del = lambda fun, f_x0i, x0i, h: np.asfarray(
                    [fun(x0i + h_j) - fun(x0i - h_j)
                     for h_j in h]).ravel() / 2.0
        return f_del

    def _forward(self):
        ''' Return forward difference function

        Member variables used
            fun
            vectorized
        '''
        if self.vectorized:
            f_del = lambda fun, f_x0i, x0i, h: (fun(x0i + h) - f_x0i).ravel()
        else:
            f_del = lambda fun, f_x0i, x0i, h: np.asfarray(
                [fun(x0i + h_j) - f_x0i for h_j in h]).ravel()
        return f_del

    def _backward(self):
        ''' Return backward difference function

        Member variables used
        ---------------------
        fun
        vectorized

        '''
        if self.vectorized:
            f_del = lambda fun, f_x0i, x0i, h: (fun(x0i - h) - f_x0i).ravel()
        else:
            f_del = lambda fun, f_x0i, x0i, h: np.asfarray(
                [fun(x0i - h_j) - f_x0i for h_j in h]).ravel()
        return f_del

    def _remove_non_finite(self, der_init, h1):
        isnonfinite = 1 - np.isfinite(der_init)
        i_nonfinite, = isnonfinite.ravel().nonzero()
        if i_nonfinite.size > 0:
            i = np.max(i_nonfinite)
            allfinite_start = i + 1
            warnings.warn('The stepsize (%g) is possibly too large!' % h1[i])
            der_init = der_init[allfinite_start:]
            h1 = h1[allfinite_start:]
        return der_init, h1

    def _predict_uncertainty(self, rombcoefs):
        '''uncertainty estimate of derivative prediction'''
        coefs = rombcoefs[0][0]
        s = np.sqrt(rombcoefs[1])
        rinv = np.asarray(linalg.pinv(self._rromb))
        cov1 = np.sum(rinv ** 2, axis=1)  # 1 spare dof
        errest = np.maximum(s * 12.7062047361747 * np.sqrt(cov1[0]),
                            s * _EPS * 10.)

        tmp_err = np.abs(np.diff(np.hstack((0, coefs, 0)))) * (len(coefs) > 1)
        abserr = tmp_err[:-1] + tmp_err[1:] + np.abs(coefs) * _EPS * 10.0
        return np.maximum(errest, abserr)

    def _romb_extrap(self, der_init, h1):
        ''' Return Romberg extrapolated derivatives and error estimates
            based on the initial derivative estimates

        Parameter
        ---------
        der_init - initial derivative estimates
        h1 - stepsizes used in the derivative estimates

        Returns
        -------
        der_romb - derivative estimates returned
        errest - error estimates
        hout - stepsizes returned

        Member variables used
        ---------------------
        step_ratio - Ratio decrease in step
        romberg_terms - higher order terms to cancel using the romberg step
        '''
        # amp = np.linalg.cond(self._rromb)
        # amp - noise amplification factor due to the romberg step
        # the noise amplification is further amplified by the Romberg step
        der_romb, hout = self._remove_non_finite(der_init, h1)
        # this does the extrapolation to a zero step size.
        num_terms = self.romberg_terms
        ne = der_romb.size

        if ne < num_terms + 2:
            errest = np.ones(der_init.shape) * hout
        else:
            rhs = vec2mat(der_romb, num_terms + 2, max(1, ne - num_terms - 2))

            rombcoefs = linalg.lstsq(self._rromb, (self._qromb.T * rhs))
            der_romb = rombcoefs[0][0, :]
            hout = hout[:der_romb.size]
            errest = self._predict_uncertainty(rombcoefs)

        if self.use_dea and der_romb.size > 2:
            der_romb, errest = dea3(der_romb[0:-2], der_romb[1:-1],
                                    der_romb[2:], symmetric=True)
            hout = hout[2:-1]
            # der_romb, errest = _extrapolate(hout, der_romb, errest)

        return der_romb, errest + _EPS, hout


class _PartialDerivative(_Derivative):

    def _partial_der(self, x00):
        ''' Return partial derivatives
        '''
        x0 = np.atleast_1d(x00)
        nx = len(x0)
        df, err, delta = np.zeros(nx), np.zeros(nx), np.zeros(nx)

        step_nom = [None, ] * nx if self.step_nom is None else self.step_nom

        fun = self._fun
        self._x = np.asarray(x0, dtype=float)
        for i in range(nx):
            self._ix = i
            df[i], err[i], delta[i] = self._derivative(fun, x0[i], step_nom[i])
            # err[i] = self.error_estimate
            # delta[i] = self.final_delta
        # self.error_estimate = err
        # self.final_delta = delta
        return df, err, delta

    def _fun(self, xi):
        x = self._x.copy()
        x[self._ix] = xi
        return self.fun(x)


class Derivative(_Derivative):
    __doc__ = ('''Estimate n'th derivative of fun at x0, with error estimate

    %s

    Examples
    --------
     >>> import numpy as np
     >>> import numdifftools as nd

     # 1'st and 2'nd derivative of exp(x), at x == 1
     >>> fd = nd.Derivative(np.exp)       # 1'st derivative
     >>> fdd = nd.Derivative(np.exp,n=2)  # 2'nd derivative
     >>> fd(1)
     array([ 2.71828183])

     >>> d2 = fdd([1, 2])
     >>> d2
     array([ 2.71828183,  7.3890561 ])

     >>> np.abs(d2-np.exp([1,2]))< fdd.error_estimate # Check error estimate
     array([ True,  True], dtype=bool)

     # 3'rd derivative of x.^3+x.^4, at x = [0,1]
     >>> fun = lambda x: x**3 + x**4
     >>> dfun = lambda x: 6 + 4*3*2*np.asarray(x)
     >>> fd3 = nd.Derivative(fun,n=3)
     >>> fd3([0,1])          #  True derivatives: [6,30]
     array([  6.,  30.])

     >>> np.abs(fd3([0,1])-dfun([0,1])) <= fd3.error_estimate
     array([ True,  True], dtype=bool)

     See also
     --------
     Gradient,
     Hessdiag,
     Hessian,
     Jacobian
    ''' % _Derivative.__doc__.partition('\n')[2])

    def __call__(self, x):
        return self.derivative(x)

    def derivative(self, x):
        ''' Return estimate of n'th derivative of fun at x
            using romberg extrapolation
        '''
        self._initialize()
        x0 = np.atleast_1d(x)
        shape = x0.shape
        der, err, delta = self._derivative(self.fun, x0.ravel(), self.step_nom)
        self.error_estimate = err.reshape(shape)
        self.final_delta = delta.reshape(shape)
        return der.reshape(shape)


class Jacobian(_Derivative):
    __doc__ = ('''Estimate Jacobian matrix, with error estimate
    %s

    The Jacobian matrix is the matrix of all first-order partial derivatives
    of a vector-valued function.

    Assumptions
    -----------
    fun : (vector valued)
        analytical function to differentiate.
        fun must be a function of the vector or array x0.

    x0 : vector location at which to differentiate fun
        If x0 is an N x M array, then fun is assumed to be
        a function of N*M variables.

    Examples
    --------
    >>> import numpy as np
    >>> import numdifftools as nd

    #(nonlinear least squares)
    >>> xdata = np.reshape(np.arange(0,1,0.1),(-1,1))
    >>> ydata = 1+2*np.exp(0.75*xdata)
    >>> fun = lambda c: (c[0]+c[1]*np.exp(c[2]*xdata) - ydata)**2
    >>> Jfun = nd.Jacobian(fun)
    >>> h = Jfun([1., 2., 0.75]) # should be numerically zero
    >>> np.abs(h) < 1e-14
    array([[ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True]], dtype=bool)

    >>> np.abs(h) <= 100 * Jfun.error_estimate
    array([[ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True],
           [ True,  True,  True]], dtype=bool)

    See also
    --------
    Gradient,
    Derivative,
    Hessdiag,
    Hessian
    ''' % _Derivative.__doc__.partition('\n')[2].replace(
        'Integer from 1 to 4             (Default 1)', '1').replace(
        'defining derivative order.',
        'Derivative order is always 1.'))

    def __call__(self, x):
        return self.jacobian(x)

    def jacobian(self, x):
        '''
        Return Jacobian matrix of a vector valued function of n variables


        Parameter
        ---------
        x : vector
            location at which to differentiate fun.
            If x is an nxm array, then fun is assumed to be
            a function of n*m variables.

        Member variable used
        --------------------
        fun : (vector valued) analytical function to differentiate.
                fun must be a function of the vector or array x0.

        Returns
        -------
        jac : array-like
           first partial derivatives of fun. Assuming that x0
           is a vector of length p and fun returns a vector
           of length n, then jac will be an array of size (n,p)

        err - vector
            of error estimates corresponding to each partial
            derivative in jac.

        See also
        --------
        Derivative,
        Gradient,
        Hessian,
        Hessdiag
        '''
        self.n = 1
        fun = self.fun
        self._initialize()

        zeros = np.zeros
        newaxis = np.newaxis
        x0 = np.atleast_1d(x)
        nx = x0.size

        f0 = fun(x0)
        f0 = f0.ravel()
        n = f0.size

        jac = zeros((n, nx))
        if n == 0:
            self.error_estimate = jac
            return jac

        step_nom = self._get_step_nom(self.step_nom, x0)

        err, delta = jac.copy(), jac.copy()
        for i in range(nx):
            x0_i = x0[i]
            h = self._get_steps(step_nom[i])
            nsteps = h.size
            # evaluate at each step, centered around x0_i
            # difference to give a second order estimate
            fdel = zeros((n, nsteps))
            xp, xm = x0.copy(), x0.copy()
            for j in range(nsteps):
                xp[i], xm[i] = x0_i + h[j], x0_i - h[j]
                fdif = fun(xp) - fun(xm)
                fdel[:, j] = 0.5 * fdif.ravel()
            derest = fdel / h[newaxis, :]

            for j in range(n):
                der_romb, errors, h1 = self._romb_extrap(derest[j, :], h)
                jac[j, i], err[j, i], delta[j, i] = self._best_der(der_romb,
                                                                   errors, h1)

        self.final_delta = delta
        self.error_estimate = err
        return jac


class Gradient(_PartialDerivative):
    __doc__ = ('''Estimate gradient of fun at x, with error estimate
    %s

    Assumptions
    -----------
    fun : SCALAR analytical function to differentiate.
        fun must be a function of the vector or array x,
        but it needs not to be vectorized.

    x : vector location at which to differentiate fun
        If x is an N x M array, then fun is assumed to be
        a function of N*M variables.


    Examples
    --------
    >>> import numpy as np
    >>> import numdifftools as nd
    >>> fun = lambda x: np.sum(x**2)
    >>> dfun = nd.Gradient(fun)
    >>> dfun([1,2,3])
    array([ 2.,  4.,  6.])

    # At [x,y] = [1,1], compute the numerical gradient
    # of the function sin(x-y) + y*exp(x)

    >>> sin = np.sin; exp = np.exp
    >>> z = lambda xy: sin(xy[0]-xy[1]) + xy[1]*exp(xy[0])
    >>> dz = nd.Gradient(z)
    >>> grad2 = dz([1, 1])
    >>> grad2
    array([ 3.71828183,  1.71828183])

    # At the global minimizer (1,1) of the Rosenbrock function,
    # compute the gradient. It should be essentially zero.

    >>> rosen = lambda x : (1-x[0])**2 + 105.*(x[1]-x[0]**2)**2
    >>> rd = nd.Gradient(rosen)
    >>> grad3 = rd([1,1])
    >>> np.abs(grad3 - 0)<=rd.error_estimate
    array([ True,  True], dtype=bool)


    See also
    --------
    Derivative, Hessdiag, Hessian, Jacobian
    ''' % _Derivative.__doc__.partition('\n')[2].replace(
        'Integer from 1 to 4             (Default 1)', '1').replace(
        'defining derivative order.',
        'Derivative order is always 1.'))

    def __call__(self, x):
        return self.gradient(x)

    def gradient(self, x):
        '''Returns gradient

         See also derivative, hessian, jacobian
        '''
        self.n = 1
        self.vectorized = False
        self._initialize()
        pder, self.error_estimate, self.final_delta = self._partial_der(x)
        return pder


class Hessdiag(_PartialDerivative):
    __doc__ = ('''
    Estimate diagonal elements of Hessian of fun at x, with error estimate
    %s

    HESSDIAG return a vector of second order partial derivatives of fun.
    These are the diagonal elements of the Hessian matrix, evaluated
    at x0.  When all that you want are the diagonal elements of the hessian
    matrix, it will be more efficient to call HESSDIAG than HESSIAN.
    HESSDIAG uses DERIVATIVE to provide both second derivative estimates
    and error estimates.

    Assumptions
    ------------
    fun : SCALAR analytical function to differentiate.
        fun must be a function of the vector or array x0,
        but it needs not to be vectorized.

    x : vector location at which to differentiate fun
        If x is an N x M array, then fun is assumed to be
        a function of N*M variables.

    Examples
    --------
    >>> import numpy as np
    >>> import numdifftools as nd
    >>> fun = lambda x : x[0] + x[1]**2 + x[2]**3
    >>> ddfun = lambda x : np.asarray((0, 2, 6*x[2]))
    >>> Hfun = nd.Hessdiag(fun)
    >>> hd = Hfun([1,2,3]) # HD = [ 0,2,18]
    >>> hd
    array([  0.,   2.,  18.])
    >>> np.abs(ddfun([1,2,3])-hd) <= Hfun.error_estimate
    array([ True,  True,  True], dtype=bool)


    See also
    --------
    Gradient, Derivative, Hessian, Jacobian
    ''' % _Derivative.__doc__.partition('\n')[2].replace(
        'Integer from 1 to 4             (Default 1)', '2').replace(
        'defining derivative order.',
        'Derivative order is always 2.'))

    def __call__(self, x):
        return self.hessdiag(x)

    def hessdiag(self, x):
        ''' Diagonal elements of Hessian matrix

         See also derivative, gradient, hessian, jacobian
        '''
        self.n = 2
        self.vectorized = False
        self._initialize()
        dder, self.error_estimate, self.final_delta = self._partial_der(x)
        return dder


class Hessian(Hessdiag):
    __doc__ = (''' Estimate Hessian matrix, with error estimate
    %s

    HESSIAN estimate the matrix of 2nd order partial derivatives of a real
    valued function FUN evaluated at X. HESSIAN is NOT a tool for frequent
    use on an expensive to evaluate objective function, especially in a large
    number of dimensions. Its computation will use roughly  O(6*n^2) function
    evaluations for n parameters.

    Assumptions
    -----------
    fun : SCALAR analytical function
        to differentiate. fun must be a function of the vector or array x0,
        but it needs not to be vectorized.

    x : vector location
        at which to differentiate fun
        If x is an N x M array, then fun is assumed to be a function
        of N*M variables.

    Examples
    --------
    >>> import numpy as np
    >>> import numdifftools as nd

    # Rosenbrock function, minimized at [1,1]
    >>> rosen = lambda x : (1.-x[0])**2 + 105*(x[1]-x[0]**2)**2
    >>> Hfun = nd.Hessian(rosen)
    >>> h = Hfun([1, 1])
    >>> h
    array([[ 842., -420.],
           [-420.,  210.]])
    >>> Hfun.error_estimate < 5.e-10
    array([[ True,  True],
           [ True,  True]], dtype=bool)

    # cos(x-y), at (0,0)
    >>> cos = np.cos
    >>> fun = lambda xy : cos(xy[0]-xy[1])
    >>> Hfun2 = nd.Hessian(fun)
    >>> h2 = Hfun2([0, 0])
    >>> h2
    array([[-1.,  1.],
           [ 1., -1.]])
    >>> np.abs(h2-np.array([[-1,  1],[ 1, -1]])) < 10*Hfun2.error_estimate
    array([[ True,  True],
           [ True,  True]], dtype=bool)

    >>> Hfun2.romberg_terms = 3
    >>> h3 = Hfun2([0,0])
    >>> h3
    array([[-1.,  1.],
           [ 1., -1.]])
    >>> np.abs(h3-np.array([[-1.,  1.],[ 1., -1.]])) < 10*Hfun2.error_estimate
    array([[ True,  True],
           [ True,  True]], dtype=bool)


    See also
    --------
    Gradient,
    Derivative,
    Hessdiag,
    Jacobian
    ''' % _Derivative.__doc__.partition('\n')[2].replace(
        'Integer from 1 to 4             (Default 1)', '2').replace(
        'defining derivative order.',
        'Derivative order is always 2.'))

    def __call__(self, x):
        return self.hessian(x)

    def _get_step_max(self):
        # Decide on intelligent step sizes for the mixed partials
        best_step_size = self.final_delta
        num_steps = self._get_min_num_steps()
        if self.step_num is not None:
            num_steps = min(num_steps, self.step_num)
        deltas = (1.0 * self.step_ratio) ** (-np.arange(num_steps))
        deltas = self._make_exact(deltas)
        stepmax = best_step_size / deltas[num_steps // 2]
        return stepmax, deltas

    def hessian(self, x):
        '''Hessian matrix i.e., array of 2nd order partial derivatives

         See also derivative, gradient, hessdiag, jacobian
        '''
        x0 = np.atleast_1d(x)
        nx = len(x0)
        self.method = 'central'

        hess = self.hessdiag(x0)
        err = self.error_estimate

        hess, err = np.diag(hess), np.diag(err)
        if nx < 2:
            return hess  # the hessian matrix is 1x1. all done

        stepmax, dfac = self._get_step_max()
        ndel = dfac.size
        fun = self.fun
        zeros = np.zeros
        for i in range(1, nx):
            for j in range(i):
                dij, step = zeros(ndel), zeros(nx)
                step[[i, j]] = stepmax[[i, j]]
                for k in range(ndel):
                    x1 = x0 + step * dfac[k]
                    x2 = x0 - step * dfac[k]
                    step[j] = -step[j]
                    x3 = x0 + step * dfac[k]
                    step = -step
                    x4 = x0 + step * dfac[k]
                    step[i] = -step[i]
                    dij[k] = fun(x1) + fun(x2) - fun(x3) - fun(x4)
                h2 = stepmax[[i, j]].prod() * (dfac ** 2)
                dij = dij / (4 * h2)

                hess_romb, errors, h = self._romb_extrap(dij, np.sqrt(h2))
                hess[i, j], err[i, j], h = self._best_der(hess_romb, errors, h)
                hess[j, i], err[j, i] = hess[i, j], err[i, j]

        self.error_estimate = err
        return hess


def _example(x=0.0001, fun_name='inv', n=1, method='central', step_max=100,
             step_ratio=2, step_num=30, romberg_terms=2, use_dea=True):
    '''
    '''
    sinh, cosh, tanh = np.sinh, np.cosh, np.tanh
    f_dic = dict(cos=(np.cos,
                      lambda x: -np.sin(x),
                      lambda x: -np.cos(x),
                      lambda x: np.sin(x),
                      lambda x: np.cos(x),),
                 tanh=(tanh,
                       lambda x: 1. / cosh(x) ** 2,
                       lambda x: -2 * sinh(x) / cosh(x) ** 3,
                       lambda x: 2.*(3 * tanh(x)**2 - 1) / cosh(x)**2,
                       lambda x: (6 + 4*sinh(x)*(cosh(x) -
                                                 3*tanh(x))) / cosh(x)**4),
                 log=(np.log,
                      lambda x: 1. / x,
                      lambda x: -1. / x ** 2,
                      lambda x: 2. / x ** 3,
                      lambda x: -6. / x ** 4),
                 inv=(lambda x: 1. / x,
                      lambda x: -1. / x ** 2,
                      lambda x: 2. / x ** 3,
                      lambda x: -6. / x ** 4,
                      lambda x: 24. / x ** 5))
    funs = f_dic.get(fun_name)
    fun0 = funs[0]
    dfun = funs[n]

    h = 1e-4
    fd = Derivative(fun0, n=n, method=method, step_max=step_max,
                    step_ratio=step_ratio,
                    step_num=step_num, verbose=True, vectorized=True,
                    romberg_terms=romberg_terms, use_dea=use_dea)

    t = fd(x)
    txt = ''
    if n == 1:
        txt = (' (f(x+h)-f(x))/h = %g\n' %
               ((fun0(x + h) - fun0(x)) / h))
    print((txt +
           '      true df(x) = %20.15g\n' +
           ' estimated df(x) = %20.15g\n' +
           ' true err = %g\n err estimate = %g\n relative err = %g\n' +
           ' final_delta = %g\n') % (dfun(x), t, dfun(x) - t,
                                     fd.error_estimate,
                                     fd.error_estimate / t,
                                     fd.final_delta))
    plt.show('hold')


def _test_rosen():
    rosen = lambda x: (1.-x[0])**2 + 105*(x[1]-x[0]**2)**2
    Hfun = Hessian(rosen, delta=[1e-6], step_num=1)
    print(Hfun([1, 1]))
    print(Hfun.final_delta)
    print(Hfun.error_estimate)


def test_docstrings():
    import doctest
    doctest.testmod(optionflags=doctest.NORMALIZE_WHITESPACE)


if __name__ == '__main__':
    # _test_rosen()
    # test_dea()
    # _example(x=0.001, fun_name='tanh', n=1, method='c', step_max=20.,
    #         step_ratio=3., step_num=15, romberg_terms=3, use_dea=True)

    test_docstrings()
