from collections import namedtuple
from copy import deepcopy
import cPickle as pickle
import datetime as dt
import logging
import os
import re

from microdrop.experiment_log import ExperimentLog
from microdrop.protocol import Protocol
from path_helpers import path
import dstat_interface as di
import dstat_interface.analysis
import numpy as np
import pandas as pd
import si_prefix as si

logger = logging.getLogger(__name__)


ExperimentLogDir = namedtuple('ExperimentLogDir', ['log_dir', 'instrument_id'])


def combine_data_from_microdrop_logs(exp_log_paths):
    '''
    Scrape microdrop experiment log directories to generate a
    `pandas.DataFrame` with the combined experimental data from a
    set of experiments.

    Args
    ----

        exp_log_paths (list of ExperimentLogDir named tuples) : Each tuple in
        the list describes the path to a Microdrop experiment log directory and
        and instrument_id (optional). If the instrument_id is None, this
        function will try to extract an instrument_id from the experiment log
        metadata.

    Returns
    -------

        (pandas.DataFrame) : Combined experimental data.
    '''
    combined_data_df = pd.DataFrame()

    for log_dir, instrument_id in exp_log_paths:
        error = False
        exp_id = str(log_dir.name)
        output_path = log_dir / path('wheelerlab.dropbot_dx_accessories')

        # Check for existing csv file which provides a cached version
        # of the data generated by this script.
        cached_file_exists = False

        if output_path.isdir():
            for file_path in output_path.files('*.csv'):
                # skip the calibrator fil
                if file_path.name == 'calibrator.csv':
                    continue

                df = pd.DataFrame().from_csv(path=file_path, index_col=None)
                if 'experiment_uuid' in df.columns and 'utc_timestamp' in df.columns:
                    df = df.set_index(['experiment_uuid', 'utc_timestamp'])
                    combined_data_df = combined_data_df.append(df)
                    cached_file_exists = True

        # if we have a cached file
        if cached_file_exists:
            continue

        # if there is no dstat data for this experiment, continue
        if not len(log_dir.files('*Measure*.txt')):
            continue

        log_file = log_dir / 'data'

        try:
            # load the experiment log
            print 'load experiment log %s' % log_file
            log = ExperimentLog.load(log_file)
        except Exception, e:
            print "Couldn't load exp %s" % exp_id
            continue

        step_numbers = log.get('step')
        protocol = Protocol.load(log_dir / 'protocol')

        step_labels = []
        for step in protocol.steps:
            step_labels.append(pickle.loads(step.plugin_data['wheelerlab.step_label_plugin'])['label'])

        relative_humidity = []
        temperature_celsius = []

        for line in log.get('environment state', 'wheelerlab.dropbot_dx_accessories'):
            if line:
                temperature_celsius.append(line['temperature_celsius'])
                relative_humidity.append(line['relative_humidity'])
            else:
                temperature_celsius.append(None)
                relative_humidity.append(None)

        dstat_enabled = []
        magnet_engaged = []

        for step in protocol.steps:
            dx_data = pickle.loads(step.plugin_data['wheelerlab.dropbot_dx_accessories'])
            dstat_enabled.append(dx_data['dstat_enabled'])
            magnet_engaged.append(dx_data['magnet_engaged'])

        experiment_df = pd.DataFrame()
        for file_path in log_dir.files('*Measure*.txt'):
            if error:
                continue
            df = di.analysis.dstat_to_frame(file_path)
            df.rename(columns={'name': 'step_label'}, inplace=True)
            df['experiment_uuid'] = log.uuid
            df['experiment_id'] = log.experiment_id
            df = df.reset_index().set_index(['utc_timestamp', 'experiment_uuid'])

            print file_path.name
            match = re.match('(?P<label>.*)(?P<attempt>\d+)\-data.txt', file_path.name)
            attempt = 0
            if match:
                attempt = int(match.group('attempt'))
            else:
                match = re.match(r'(?P<label>.*)-data.txt', file_path.name)

            label = match.group('label')
            try:
                step_number = step_labels.index(label)
            except ValueError:
                print 'Error scanning log %s. Label "%s" not found in protocol.' % (log_dir, label)
                error = True
                continue

            # get the uuid of the calibrator
            calibrator_file = output_path / 'calibrator.csv'
            calibrator_uuid = ''
            if calibrator_file.isfile():
                try:
                    calibrator_df = pd.read_csv(calibrator_file)
                    calibrator_uuid = calibrator_df[calibrator_df['step_label'] == label] \
                        ['experiment_uuid'].values[0]
                except:
                    print "Couldn't get calibrator uuid."

            df['calibrator_uuid'] = calibrator_uuid

            index = 0
            for i in range(attempt + 1):
                index = step_numbers.index(step_number, index + 1)

            df['step_number'] = step_number
            df['attempt_number'] = attempt
            df['temperature_celsius'] = temperature_celsius[index]
            df['relative_humidity'] = relative_humidity[index]

            experiment_df = experiment_df.append(df)

        if error:
            continue

        try:
            metadata = deepcopy(log.metadata['wheelerlab.metadata_plugin'])
            device_id = metadata.get('device_id', '')
            sample_id = metadata.get('sample_id', '')

            cre_device_id = re.compile(r'#(?P<batch_id>[a-fA-F0-9]+)'
                                       r'%(?P<device_id>[a-fA-F0-9]+)$')

            # If `device_id` is in the form '#<batch-id>%<device-id>', extract batch and
            # device identifiers separately.
            match = cre_device_id.match(device_id)
            if match:
                metadata['device_id'] = unicode(match.group('device_id'))
                metadata[u'batch_id'] = unicode(match.group('batch_id'))
            else:
                metadata['device_id'] = ''
                metadata[u'batch_id'] = ''

            experiment_df['device_id'] = metadata['device_id']
            experiment_df['batch_id'] = metadata['batch_id']
            experiment_df['sample_id'] = metadata['sample_id']
        except:
            experiment_df['device_id'] = ''
            experiment_df['batch_id'] = ''
            experiment_df['sample_id'] = ''

        # TODO: get instrument_id from experiment log metadata
        experiment_df['instrument_id'] = instrument_id
        start_time = log.get('start time')[0]
        experiment_df['experiment_start'] = dt.datetime.fromtimestamp(start_time).isoformat()
        experiment_df['experiment_length_min'] = log.get('time')[-1] / 60

        if not output_path.isdir():
            output_path.mkdir()

        experiment_df.to_csv(output_path / \
             path('e[%s]-d[%s]-s[%s].csv' % (str(log.uuid)[:8], experiment_df['device_id'].values[0],
             re.sub(r'[:/\\><\?]', '_', experiment_df['sample_id'].values[0]))))

        combined_data_df = combined_data_df.append(experiment_df)
    return combined_data_df


def reduce_microdrop_dstat_data(df_md_dstat, settling_period_s=2.,
                                bandwidth=1., extra_summary_fields=None,
                                verbose=False):
    '''
    Reduce measurements for each Microdrop DStat acquisition step in
    `df_md_dstat` to a single row with an aggregate signal value.

    For continuous detection, the aggregate signal column corresponds to the
    mean `current_amps`.

    For synchronous detection experiments (i.e., where `target_hz` is greater
    than 0), the aggregate signal corresponds to the integrated amplitude of
    the `current_amps` FFT within the bandwidth around target frequency.

    See `dstat_interface.analysis.reduce_microdrop_dstat_data` for more
    details.

    Args
    ----

        df_md_dstat (pandas.DataFrame) : Microdrop DStat measurements in a
            table with at least the columns `experiment_uuid`, `step_number`,
            `attempt_number`, `target_hz`, `sample_frequency_hz`, `current_amps`,
            and `time_s`.
        settling_period_s (float) : Measurement settling period in seconds.
            Measurements taken before start time will not be included in
            calculations.
        bandwidth (float) : Bandwidth (centered at synchronous detection
            frequency) to integrate within.

    Returns
    -------

        (pd.DataFrame) : Table containing the columns `experiment_start`,
            `experiment_length_min`, `sample_id`, `experiment_uuid`,
            `step_label`, `instrument_id`, `relative_humidity`,
            `temperature_celsius`, `sample_frequency_hz`, `target_hz`,
            'calibrator_uuid', and `signal` (i.e., the aggregate signal value).
    '''
    summary_fields = ['utc_timestamp', 'experiment_start',
                      'experiment_length_min', 'sample_id', 'instrument_id',
                      'relative_humidity', 'temperature_celsius',
                      'sample_frequency_hz', 'target_hz', 'calibrator_uuid']
    if extra_summary_fields is None:
        extra_summary_fields = []
    groupby = ['experiment_uuid', 'step_label', 'step_number', 'attempt_number']

    experiment_uuids = df_md_dstat.experiment_uuid.unique()
    frames = []
    for i, (experiment_uuid_i, df_i) in enumerate(df_md_dstat
                                                  .groupby('experiment_uuid')):
        if verbose:
            instrument_id_i = df_i.instrument_id.iloc[0]
            experiment_id_i = df_i.experiment_id.iloc[0]
            print ('\rExperiment: {}/{} ({}-{}: {})        '
                   .format(i + 1, experiment_uuids.shape[0],
                           instrument_id_i, experiment_id_i,
                           str(experiment_uuid_i)[:8])),
        try:
            extra_summary_fields_i = [f for f in extra_summary_fields
                                      if f in df_i]
            df_dstat_reduced_i = \
                di.analysis.reduce_dstat_data(df_i, groupby=groupby,
                                              summary_fields=summary_fields +
                                              extra_summary_fields_i)
        except Exception, e:
            logger.warning('\n[warning] experiment %s %s: %s', i + 1,
                           experiment_uuid_i[:8], e)
        else:
            frames.append(df_dstat_reduced_i)
    df_dstat_reduced = pd.concat(frames).reset_index(drop=True)
    df_dstat_reduced.sort_values(['experiment_uuid', 'utc_timestamp'],
                                 inplace=True)

    return df_dstat_reduced


def microdrop_dstat_summary_table(df_md_reduced, calibrator_csv_path=None,
                                  numeric=False, unit=None, **kwargs):
    '''
    Args
    ----

        df_md_reduced (pandas.DataFrame) : Reduced Microdrop DStat measurements
            (see `reduce_microdrop_dstat_data`) in a table with at least the
            columns `step_number`, `attempt_number`, `target_hz`, and `signal`.
        calibrator_csv_path (str) : Path to calibrator CSV file.
        numeric (bool) : If `True`, signal columns will have floating point
            type.  If `False`, signal columns will be string type using SI
            units.
        unit (str) : Express signal values in terms of the specified SI unit
            prefix.  Must be one of 'y', 'z', 'a', 'f', 'p', 'n', 'u', 'm',
            'k', 'M', 'G', 'T', 'P', 'E', 'Z', or 'Y'.

    For the remaining keyword arguments, see the `reduce_microdrop_dstat_data`
    function.

    Returns
    -------

        (pandas.DataFrame) : Summary dataframe matching each step label to the
            corresponding reduced signal value.  If `calibrator_csv_path` is
            specified, normalize signal to respective calibrator signal.

    Examples
    ========

    Without calibrator:

                    step_number  i signal   fft
        step_label
        background            1  1  47.2p  True
        background            1  2  62.4p  True
        test 1                2  1  48.7p  True
        test 1                2  2  72.7p  True

    With calibrator:

                    i signal calibrator   fft normalized
        step_label
        background  1  47.2p      23.6p  True     200.0%
        background  2  62.4p      23.6p  True     264.3%
        test 1      1  48.7p      24.3p  True     200.4%
        test 1      2  72.7p      24.3p  True     299.2%
    '''
    if unit is not None and unit not in si.SI_PREFIX_UNITS:
        raise KeyError('Unrecognized unit "{}".  The following SI units are '
                       'supported: {}'.format(unit, si.SI_PREFIX_UNITS))

    df_md_reduced = df_md_reduced.copy()
    df_md_reduced['fft'] = (df_md_reduced.target_hz > 0
                            if 'target_hz' in df_md_reduced
                            else False)
    df_display = (df_md_reduced[['step_label', 'step_number',
                                 'attempt_number', 'signal', 'fft']]
                  .set_index('step_label'))

    if df_display.groupby([df_display.index.get_level_values('step_label'),
                           'step_number'])['step_number'].count().max() < 2:
         del df_display['step_number']

    signal_columns = ['signal']
    df_display.rename(columns={'attempt_number': 'i'}, inplace=True)

    if calibrator_csv_path and os.path.isfile(calibrator_csv_path):
        with open(calibrator_csv_path, 'r') as csv_file:
            df_calibrator = pd.read_csv(csv_file, na_values=['nan'])
        df_calibrator.drop_duplicates(subset=['step_label', 'attempt_number'],
                                      inplace=True)
        df_calibrator['fft'] = (df_calibrator.target_hz > 0
                                if 'target_hz' in df_calibrator
                                else False)

        # Join experiment signal data with calibrator signal data.
        df_display = (df_display
                      .join(df_calibrator.set_index('step_label'),
                            rsuffix='_calibrator', how='left')
                      [['i', 'attempt_number', 'signal',
                        'signal_calibrator', 'fft',
                        'fft_calibrator']]).copy()
        assert((df_display.fft == df_display.fft_calibrator).all())
        del df_display['fft_calibrator']
        df_display.rename(columns={'attempt_number': 'calib_i',
                                   'signal_calibrator': 'calibrator'},
                          inplace=True)
        if df_display.calib_i.max() < 2:
            del df_display['calib_i']
        signal_columns += ['calibrator']
        df_display['normalized'] = df_display.signal / df_display.calibrator
        df_display.loc[:, 'normalized'] = (df_display.normalized
                                           .map(lambda v:
                                                '{:.1f}%'.format(100 * v)))
    if df_display.i.max() < 2:
        del df_display['i']

    if not numeric:
        if unit is None:
            # Format signals using SI units.
            df_display.loc[:, signal_columns] = (df_display[signal_columns]
                                                 .applymap(si.si_format))
        else:
            # Convert unit to float scalar.
            unit_scale = si.si_parse('1{}'.format(unit))
            df_display.loc[:, signal_columns] = \
                df_display[signal_columns].applymap(lambda v: '{:.01f}{}'
                                                    .format(v / unit_scale,
                                                            unit))
    return df_display


def subtract_background_signal(df_dstat_summary, inplace=False):
    '''
    Subtract background from each signal.

    Each row starting with `'background'` is considered a background.

    Each background is matched to other rows based on the suffix of the
    background index label.  For example, `'background-a'` will be matched to
    all rows with an index label ending with `'-a'`.  Rows with no matching
    suffix will be matched to the background row `'background'` if it exists,
    or will remained unchanged if the background row `'background'` does not
    exist.

    Args
    ----

        df_dstat_summary (pandas.DataFrame) : Table with at least the column
            `signal`.

    Returns
    -------

        (pandas.DataFrame) : Table with same structure as input with respective
            background subtracted from each corresponding signal.
    '''
    if not inplace:
        df_dstat_summary = df_dstat_summary.copy()
    if not df_dstat_summary.shape[0]:
        return df_dstat_summary

    index = df_dstat_summary.index.str.lower()

    # Find mask of all rows with index value starting with `'background'`.
    background_mask = (index.str.startswith('background'))
    if background_mask.max() < 1:
        # No background steps.
        return df_dstat_summary

    # Find list of all distinct background index value suffixes.
    background_labels = index[background_mask].drop_duplicates()
    background_prefixes = background_labels.str.slice(0, len('background'))
    background_suffixes = background_labels.str.slice(len('background'))

    # Find the mask all non-background rows that share a non-empty suffix with
    # a background row.
    non_empty_suffix_masks = np.array([index.str.endswith(s.lower())
                                       for s in background_suffixes if s])
    if non_empty_suffix_masks.shape[0]:
        # Use column-wise maximum of boolean non-empty suffix masks to `OR`
        # together
        non_empty_suffix = non_empty_suffix_masks.max(axis=0)
    else:
        non_empty_suffix = np.zeros(index.shape[0], dtype=bool)

    if '' in background_suffixes:
        # Find mask for for rows corresponding to base `'background'` signal.
        row_mask = ~non_empty_suffix & ~background_mask

        i = index.tolist().index('background')
        # Subtract base background from corresponding rows.
        df_dstat_summary.loc[row_mask, 'signal'] -= df_dstat_summary.ix[i].signal

    for prefix_i, suffix_i in zip(background_prefixes, background_suffixes):
        if suffix_i:
            # Find mask for for rows corresponding to `'background'` signal
            # with suffix `s`.
            row_mask = index.str.endswith(suffix_i.lower()) & ~background_mask

            i = index.tolist().index(prefix_i + suffix_i)
            # Subtract background with `s` suffix from corresponding rows.
            df_dstat_summary.loc[row_mask, 'signal'] -= df_dstat_summary.ix[i].signal
    return df_dstat_summary


# Define list of columns to copy from background experiments.
background_columns = (u'step_label', u'step_number', u'attempt_number',
                      u'utc_timestamp', u'sample_id', u'instrument_id',
                      u'relative_humidity', u'temperature_celsius', u'sample_frequency_hz',
                      u'target_hz', u'calibrator_uuid', u'Measure 1', u'Measure 2',
                      u'Measure 3', u'Measure 4', u'sample_first_id', u'sample_second_id',
                      u'sample_third_id', u'sample_fourth_id', u'metadata_schema',
                      u'signal', u'label')


def merge_experiments(df_dstat_reduced, target_uuid, source_uuids):
    '''
    Args
    ----

        df_dstat_reduced (pandas.DataFrame) :  As returned by
            `ea.reduce_microdrop_dstat_data`, one row per DStat acquisition.
        target_uuid (str) : UUID of target experiment to merge source data into.
        source_uuids (iterable) : UUIDs of source experiments to merge signal data from.

    Returns
    -------

        (pandas.DataFrame) : Table containing new rows, gathered from source experiments.
            Columns related to experiment metadata (e.g., UUID) contain data from *target*
            experiment, not *source* experiments.
    '''
    # Get rows to merge from source experiments.
    df_merge_rows = df_dstat_reduced.loc[df_dstat_reduced.experiment_uuid
                                         .isin(list(source_uuids))].copy()

    # Overwrite columns in source merge rows with data from target experiment.
    experiment_i = df_dstat_reduced.loc[df_dstat_reduced.experiment_uuid ==
                                        target_uuid].iloc[0]
    overwrite_columns = [k for k in df_dstat_reduced.columns
                         if k not in background_columns]
    df_merge_rows.loc[:, overwrite_columns] = experiment_i[overwrite_columns].values
    return df_merge_rows


def merge_instrument_experiments(df_dstat_reduced, instrument_id, target_id, source_ids):
    '''
    Merge experiments based on instrument id and experiment ids instead of experiment
    UUIDs.

    Args
    ----

        df_dstat_reduced (pandas.DataFrame) :  As returned by
            `ea.reduce_microdrop_dstat_data`, one row per DStat acquisition.
        instrument_id (str) : Instrument identifier (e.g., `'MR-BOX1'`).
        target_uuid (str) : ID of target experiment to merge source data into.
        source_ids (iterable) : ID of source experiments to merge signal data from.

    Returns
    -------

        (pandas.DataFrame) : Table containing new rows, gathered from source experiments.
            Columns related to experiment metadata (e.g., UUID) contain data from *target*
            experiment, not *source* experiments.
    '''
    df_instrument = df_dstat_reduced.loc[df_dstat_reduced.instrument_id == instrument_id]
    target_uuid = df_instrument.loc[df_dstat_reduced.experiment_id == target_id,
                                'experiment_uuid'].iloc[0]
    source_uuids = df_instrument.loc[df_dstat_reduced.experiment_id.isin(source_ids),
                                 'experiment_uuid'].unique()
    return merge_experiments(df_dstat_reduced, target_uuid, source_uuids)
