'''
This module defines the `Group` object, a mix-in class for everything that
saves state variables, e.g. `NeuronGroup` or `StateMonitor`.
'''
import collections
from collections import OrderedDict
import weakref

import numpy as np

from brian2.core.base import BrianObject
from brian2.core.preferences import prefs
from brian2.core.variables import (Variables, Constant, Variable,
                                   ArrayVariable, DynamicArrayVariable,
                                   Subexpression)
from brian2.core.functions import Function
from brian2.core.namespace import (get_local_namespace,
                                   DEFAULT_FUNCTIONS,
                                   DEFAULT_UNITS,
                                   DEFAULT_CONSTANTS)
from brian2.codegen.codeobject import create_runner_codeobj, check_code_units
from brian2.equations.equations import BOOLEAN, INTEGER, FLOAT
from brian2.units.fundamentalunits import (fail_for_dimension_mismatch, Unit,
                                           get_unit)
from brian2.units.allunits import second
from brian2.utils.logger import get_logger
from brian2.utils.stringtools import get_identifiers, SpellChecker

__all__ = ['Group', 'CodeRunner']

logger = get_logger(__name__)


def _conflict_warning(message, resolutions):
    '''
    A little helper functions to generate warnings for logging. Specific
    to the `Namespace.resolve` method and should only be used by it.

    Parameters
    ----------
    message : str
        The first part of the warning message.
    resolutions : list of str
        A list of (namespace, object) tuples.
    '''
    if len(resolutions) == 0:
        # nothing to warn about
        return
    elif len(resolutions) == 1:
        second_part = ('but also refers to a variable in the %s namespace:'
                       ' %r') % (resolutions[0][0], resolutions[0][1])
    else:
        second_part = ('but also refers to a variable in the following '
                       'namespaces: %s') % (', '.join([r[0] for r in resolutions]))

    logger.warn(message + ' ' + second_part,
                'Group.resolve.resolution_conflict', once=True)


def get_dtype(equation, dtype=None):
    '''
    Helper function to interpret the `dtype` keyword argument in `NeuronGroup`
    etc.

    Parameters
    ----------
    equation : `SingleEquation`
        The equation for which a dtype should be returned
    dtype : `dtype` or dict, optional
        Either the `dtype` to be used as a default dtype for all float variables
        (instead of the `core.default_float_dtype` preference) or a
        dictionary stating the `dtype` for some variables; all other variables
        will use the preference default

    Returns
    -------
    d : `dtype`
        The dtype for the variable defined in `equation`
    '''
    # Check explicitly provided dtype for compatibility with the variable type
    if isinstance(dtype, collections.Mapping):
        if equation.varname in dtype:
            BASIC_TYPES = {BOOLEAN: 'b',
                           INTEGER: 'iu',
                           FLOAT: 'f'}
            provided_dtype = np.dtype(dtype[equation.varname])
            if not provided_dtype.kind in BASIC_TYPES[equation.var_type]:
                raise TypeError(('Error determining dtype for variable %s: %s '
                                 'is not a correct type for %s variables') % (equation.varname,
                                                              provided_dtype.name,
                                                              equation.var_type))
            else:
                return dtype[equation.varname]
        else:  # continue as if no dtype had been specified at all
            dtype = None

    # Use default dtypes (or a provided standard dtype for floats)
    if equation.var_type == BOOLEAN:
        return np.bool
    elif equation.var_type == INTEGER:
        return prefs['core.default_integer_dtype']
    elif equation.var_type == FLOAT:
        if dtype is not None:
            dtype = np.dtype(dtype)
            if not dtype.kind == 'f':
                raise TypeError(('%s is not a valid floating point '
                                 'dtype') % dtype)
            return dtype
        else:
            return prefs['core.default_float_dtype']
    else:
        raise ValueError(('Do not know how to determine a dtype for '
                          'variable %s of type %s' ) % (equation.varname,
                                                        equation.var_type))


def _same_value(obj1, obj2):
    '''
    Helper function used during namespace resolution.
    '''
    if obj1 is obj2:
        return True
    try:
        obj1 = obj1.get_value()
    except (AttributeError, TypeError):
        pass

    try:
        obj2 = obj2.get_value()
    except (AttributeError, TypeError):
        pass

    return obj1 is obj2


def _same_function(func1, func2):
    '''
    Helper function, used during namespace resolution for comparing whether to
    functions are the same. This takes care of treating a function and a
    `Function` variables whose `Function.pyfunc` attribute matches as the
    same. This prevents the user from getting spurious warnings when having
    for example a numpy function such as :np:func:`~random.randn` in the local
    namespace, while the ``randn`` symbol in the numpy namespace used for the
    code objects refers to a `RandnFunction` specifier.
    '''
    # use the function itself if it doesn't have a pyfunc attribute
    func1 = getattr(func1, 'pyfunc', func1)
    func2 = getattr(func2, 'pyfunc', func2)

    return func1 is func2


class Indexing(object):
    '''
    Object responsible for calculating flat index arrays from arbitrary group-
    specific indices. Stores strong references to the necessary variables so
    that basic indexing (i.e. slicing, integer arrays/values, ...) works even
    when the respective `Group` no longer exists. Note that this object does
    not handle string indexing.
    '''
    def __init__(self, group, default_index='_idx'):
        self.group = weakref.proxy(group)
        self.N = group.variables['N']
        self.default_index = default_index

    def __call__(self, item=None, index_var=None):
        '''
        Return flat indices to index into state variables from arbitrary
        group specific indices. In the default implementation, raises an error
        for multidimensional indices and transforms slices into arrays.

        Parameters
        ----------
        item : slice, array, int
            The indices to translate.

        Returns
        -------
        indices : `numpy.ndarray`
            The flat indices corresponding to the indices given in `item`.

        See Also
        --------
        SynapticIndexing
        '''
        if index_var is None:
            index_var = self.default_index

        if hasattr(item, '_indices'):
            item = item._indices()

        if isinstance(item, tuple):
            raise IndexError(('Can only interpret 1-d indices, '
                              'got %d dimensions.') % len(item))
        else:
            if item is None or item == 'True':
                item = slice(None)
            if isinstance(item, slice):
                if index_var == '0':
                    return 0
                if index_var == '_idx':
                    start, stop, step = item.indices(self.N.get_value())
                else:
                    start, stop, step = item.indices(index_var.size)
                index_array = np.arange(start, stop, step, dtype=np.int32)
            else:
                index_array = np.asarray(item)
                if index_array.dtype == np.bool:
                    index_array = np.nonzero(index_array)[0]
                elif not np.issubdtype(index_array.dtype, np.int):
                    raise TypeError(('Indexing is only supported for integer '
                                     'and boolean arrays, not for type '
                                     '%s' % index_array.dtype))

            if index_var != '_idx':
                try:
                    return index_var.get_value()[index_array]
                except IndexError as ex:
                    # We try to emulate numpy's indexing semantics here:
                    # slices never lead to IndexErrors, instead they return an
                    # empty array if they don't match anything
                    if isinstance(item, slice):
                        return np.array([], dtype=np.int32)
                    else:
                        raise ex
            else:
                return index_array


class IndexWrapper(object):
    '''
    Convenience class to allow access to the indices via indexing syntax. This
    allows for example to get all indices for synapses originating from neuron
    10 by writing `synapses.indices[10, :]` instead of
    `synapses._indices.((10, slice(None))`.
    '''
    def __init__(self, group):
        self.group = weakref.proxy(group)
        self.indices = group._indices

    def __getitem__(self, item):
        if isinstance(item, basestring):
            variables = Variables(None)
            variables.add_auxiliary_variable('_indices', unit=Unit(1),
                                             dtype=np.int32)
            variables.add_auxiliary_variable('_cond', unit=Unit(1),
                                             dtype=np.bool)

            abstract_code = '_cond = ' + item
            check_code_units(abstract_code, self.group,
                             additional_variables=variables,
                             level=1)
            from brian2.devices.device import get_default_codeobject_class
            codeobj = create_runner_codeobj(self.group,
                                            abstract_code,
                                            'group_get_indices',
                                            additional_variables=variables,
                                            level=1,
                                            codeobj_class=get_default_codeobject_class('codegen.string_expression_target')
                                            )
            return codeobj()
        else:
            return self.indices(item)


class Group(BrianObject):
    '''
    Mix-in class for accessing arrays by attribute.
    
    # TODO: Overwrite the __dir__ method to return the state variables
    # (should make autocompletion work)
    '''

    def _enable_group_attributes(self):
        if not hasattr(self, 'variables'):
            raise ValueError('Classes derived from Group need variables attribute.')
        if not hasattr(self, 'codeobj_class'):
            self.codeobj_class = None
        if not hasattr(self, '_indices'):
            self._indices = Indexing(self)
        if not hasattr(self, 'indices'):
            self.indices = IndexWrapper(self)
        if not hasattr(self, '_stored_states'):
            self._stored_states = {}
        if not hasattr(self, '_stored_clocks'):
            self._stored_clocks = {}
        self._group_attribute_access_active = True

    def state(self, name, use_units=True, level=0):
        '''
        Return the state variable in a way that properly supports indexing in
        the context of this group

        Parameters
        ----------
        name : str
            The name of the state variable
        use_units : bool, optional
            Whether to use the state variable's unit.
        level : int, optional
            How much farther to go down in the stack to find the namespace.
        Returns
        -------
        var : `VariableView` or scalar value
            The state variable's value that can be indexed (for non-scalar
            values).
        '''
        try:
            var = self.variables[name]
        except KeyError:
            raise KeyError("State variable "+name+" not found.")

        if use_units:
            return var.get_addressable_value_with_unit(name=name, group=self)
        else:
            return var.get_addressable_value(name=name, group=self)

    def __getattr__(self, name):
        # We do this because __setattr__ and __getattr__ are not active until
        # _group_attribute_access_active attribute is set, and if it is set,
        # then __getattr__ will not be called. Therefore, if getattr is called
        # with this name, it is because it hasn't been set yet and so this
        # method should raise an AttributeError to agree that it hasn't been
        # called yet.
        if name=='_group_attribute_access_active':
            raise AttributeError
        if not '_group_attribute_access_active' in self.__dict__:
            raise AttributeError
        
        # We want to make sure that accessing variables without units is fast
        # because this is what is used during simulations
        # We do not specifically check for len(name) here, we simply assume
        # that __getattr__ is not called with an empty string (which wouldn't
        # be possible using the normal dot syntax, anyway)
        try:
            if name[-1] == '_':
                name = name[:-1]
                use_units = False
            else:
                use_units = True
            return self.state(name, use_units)

        except KeyError:
            raise AttributeError('No attribute with name ' + name)

    def __setattr__(self, name, val, level=0):
        # attribute access is switched off until this attribute is created by
        # _enable_group_attributes
        if not hasattr(self, '_group_attribute_access_active') or name in self.__dict__:
            object.__setattr__(self, name, val)
        elif name in self.variables:
            var = self.variables[name]
            if not isinstance(val, basestring):
                fail_for_dimension_mismatch(val, var.unit,
                                            'Incorrect units for setting %s' % name)
            if var.read_only:
                raise TypeError('Variable %s is read-only.' % name)
            # Make the call X.var = ... equivalent to X.var[:] = ...
            var.get_addressable_value_with_unit(name, self).set_item(slice(None),
                                                                     val,
                                                                     level=level+1)
        elif len(name) and name[-1]=='_' and name[:-1] in self.variables:
            # no unit checking
            var = self.variables[name[:-1]]
            if var.read_only:
                raise TypeError('Variable %s is read-only.' % name[:-1])
            # Make the call X.var = ... equivalent to X.var[:] = ...
            var.get_addressable_value(name[:-1], self).set_item(slice(None),
                                                                val,
                                                                level=level+1)
        elif hasattr(self, name) or name.startswith('_'):
            object.__setattr__(self, name, val)
        else:
            # Try to suggest the correct name in case of a typo
            checker = SpellChecker([varname for varname, var in self.variables.iteritems()
                                    if not (varname.startswith('_') or var.read_only)])
            if name.endswith('_'):
                suffix = '_'
                name = name[:-1]
            else:
                suffix = ''
            error_msg = 'Could not find a state variable with name "%s".' % name
            suggestions = checker.suggest(name)
            if len(suggestions) == 1:
                suggestion, = suggestions
                error_msg += ' Did you mean to write "%s%s"?' % (suggestion,
                                                                 suffix)
            elif len(suggestions) > 1:
                error_msg += (' Did you mean to write any of the following: %s ?' %
                              (', '.join(['"%s%s"' % (suggestion, suffix)
                                          for suggestion in suggestions])))
            error_msg += (' Use the add_attribute method if you intend to add '
                          'a new attribute to the object.')
            raise AttributeError(error_msg)

    def add_attribute(self, name):
        '''
        Add a new attribute to this group. Using this method instead of simply
        assigning to the new attribute name is necessary because Brian will
        raise an error in that case, to avoid bugs passing unnoticed
        (misspelled state variable name, un-declared state variable, ...).

        Parameters
        ----------
        name : str
            The name of the new attribute

        Raises
        ------
        AttributeError
            If the name already exists as an attribute or a state variable.
        '''
        if name in self.variables:
            raise AttributeError('Cannot add an attribute "%s", it is already '
                                 'a state variable of this group.' % name)
        if hasattr(self, name):
            raise AttributeError('Cannot add an attribute "%s", it is already '
                                 'an attribute of this group.' % name)
        object.__setattr__(self, name, None)

    def get_states(self, vars=None, units=True, format='dict',
                   subexpressions=False, level=0):
        '''
        Return a copy of the current state variable values. The returned arrays
        are copies of the actual arrays that store the state variable values,
        therefore changing the values in the returned dictionary will not affect
        the state variables.

        Parameters
        ----------
        vars : list of str, optional
            The names of the variables to extract. If not specified, extract
            all state variables (except for internal variables, i.e. names that
            start with ``'_'``). If the ``subexpressions`` argument is ``True``,
            the current values of all subexpressions are returned as well.
        units : bool, optional
            Whether to include the physical units in the return value. Defaults
            to ``True``.
        format : str, optional
            The output format. Defaults to ``'dict'``.
        subexpressions: bool, optional
            Whether to return subexpressions when no list of variable names
            is given. Defaults to ``False``. This argument is ignored if an
            explicit list of variable names is given in ``vars``.
        level : int, optional
            How much higher to go up the stack to resolve external variables.
            Only relevant if extracting subexpressions that refer to external
            variables.

        Returns
        -------
        values
            The variables specified in ``vars``, in the specified ``format``.

        '''
        # For the moment, 'dict' is the only supported format -- later this will
        # be made into an extensible system, see github issue #306
        if format != 'dict':
            raise NotImplementedError("Format '%s' is not supported" % format)
        if vars is None:
            vars = [name for name, var in self.variables.iteritems()
                    if not name.startswith('_') and
                    (subexpressions or not isinstance(var, Subexpression))]
        data = {}
        for var in vars:
            data[var] = np.array(self.state(var, use_units=units,
                                            level=level+1),
                                 copy=True, subok=True)
        return data

    def set_states(self, values, units=True, format='dict', level=0):
        '''
        Set the state variables.

        Parameters
        ----------
        values : depends on ``format``
            The values according to ``format``.
        units : bool, optional
            Whether the ``values`` include physical units. Defaults to ``True``.
        format : str, optional
            The format of ``values``. Defaults to ``'dict'``
        level : int, optional
            How much higher to go up the stack to resolve external variables.
            Only relevant when using string expressions to set values.
        '''
        # For the moment, 'dict' is the only supported format -- later this will
        # be made into an extensible system, see github issue #306
        if format != 'dict':
            raise NotImplementedError("Format '%s' is not supported" % format)
        for key, value in values.iteritems():
            self.state(key, use_units=units, level=level+1)[:] = value

    def _store(self, name='default'):
        logger.debug('Storing state at for object %s' % self.name)
        state = {}
        for var in self.variables.itervalues():
            if isinstance(var, ArrayVariable):
                state[var] = (var.get_value().copy(), var.size)
        self._stored_states[name] = state
        self._stored_clocks[name] = (self.clock.t_, self.clock.dt_)
        for obj in self._contained_objects:
            if hasattr(obj, '_store'):
                obj._store(name)

    def _restore(self, name='default'):
        logger.debug('Restoring state at for object %s' % self.name)
        if not name in self._stored_states:
            raise ValueError(('No state with name "%s" to restore -- '
                              'did you call store()?') % name)
        for var, (values, size) in self._stored_states[name].iteritems():
            if isinstance(var, DynamicArrayVariable):
                var.resize(size)
            var.set_value(values)
        t, dt = self._stored_clocks[name]
        self.clock.dt_ = dt
        self.clock._set_t_update_dt(t=t*second)
        for obj in self._contained_objects:
            if hasattr(obj, '_restore'):
                obj._restore(name)

    def _check_expression_scalar(self, expr, varname, level=0,
                                 run_namespace=None):
        '''
        Helper function to check that an expression only refers to scalar
        variables, used when setting a scalar variable with a string expression.

        Parameters
        ----------
        expr : str
            The expression to check.
        varname : str
            The variable that is being set (only used for the error message)
        level : int, optional
            How far to go up in the stack to find the local namespace (if
            `run_namespace` is not set).
        run_namespace : dict-like, optional
            A specific namespace provided for this expression.

        Raises
        ------
        ValueError
            If the expression refers to a non-scalar variable.
        '''
        identifiers = get_identifiers(expr)
        referred_variables = self.resolve_all(identifiers,
                                              run_namespace=run_namespace,
                                              level=level+1)
        for ref_varname, ref_var in referred_variables.iteritems():
            if not getattr(ref_var, 'scalar', False):
                raise ValueError(('String expression for setting scalar '
                                  'variable %s refers to %s which is not '
                                  'scalar.') % (varname, ref_varname))

    def resolve(self, identifier, user_identifier=True,
                additional_variables=None, run_namespace=None, level=0):
        '''
        Resolve an identifier (i.e. variable, constant or function name) in the
        context of this group. This function will first lookup the name in the
        state variables, then look for a standard function or unit of that
        name and finally look in `Group.namespace` and in `run_namespace`. If
        the latter is not given, it will try to find the variable in the local
        namespace where the original function call took place. See
        :ref:`external-variables`.

        Parameters
        ----------
        identifiers : str
            The name to look up.
        user_identifier : bool, optional
            Whether this is an identifier that was used by the user (and not
            something automatically generated that the user might not even
            know about). Will be used to determine whether to display a
            warning in the case of namespace clashes. Defaults to ``True``.
        additional_variables : dict-like, optional
            An additional mapping of names to `Variable` objects that will be
            checked before `Group.variables`.
        run_namespace : dict-like, optional
            An additional namespace, provided as an argument to the
            `Network.run` method.
        level : int, optional
            How far to go up in the stack to find the original call frame.

        Returns
        -------
        obj : `Variable` or `Function`
            Returns a `Variable` object describing the variable or a `Function`
            object for a function. External variables are represented as
            `Constant` objects

        Raises
        ------
        KeyError
            If the `identifier` could not be resolved
        '''
        resolved_internal = None

        if identifier in (additional_variables or {}):
            resolved_internal = additional_variables[identifier]
        elif identifier in getattr(self, 'variables', {}):
            resolved_internal = self.variables[identifier]

        if resolved_internal is not None:
            if not user_identifier:
                return resolved_internal  # no need to go further
            # We already found the identifier, but we try to resolve it in the
            # external namespace nevertheless, to report a warning if it is
            # present there as well.
            try:
                resolved_external = self._resolve_external(identifier,
                                                           run_namespace=run_namespace,
                                                           level=level+1)
                # If we arrive here without a KeyError then the name is present
                # in the external namespace as well

                # Do not raise a warning if both correspond to the same constant
                # value, a typical case being an externally defined "N" with the
                # the number of neurons and a later use of "N" in an expression
                # (which refers to the internal variable storing the number of
                # neurons in the group)
                if not (isinstance(resolved_internal, Constant) and
                            isinstance(resolved_external, Constant) and
                                resolved_internal.get_value() == resolved_external.get_value()):

                    message = ('Variable {var} is present in the namespace but is '
                               'also an internal variable of {name}, the internal '
                               'variable will be used.'.format(var=identifier,
                                                               name=self.name))
                    logger.warn(message, 'Group.resolve.resolution_conflict',
                                once=True)
            except KeyError:
                pass  # Nothing to warn about

            return resolved_internal

        # We did not find the name internally, try to resolve it in the external
        # namespace
        return self._resolve_external(identifier, run_namespace=run_namespace,
                                      level=level+1)

    def resolve_all(self, identifiers, user_identifiers=None,
                    additional_variables=None, run_namespace=None, level=0):
        '''
        Resolve a list of identifiers. Calls `Group.resolve` for each
        identifier.

        Parameters
        ----------
        identifiers : iterable of str
            The names to look up.
        user_identifiers : iterable of str, optional
            The names in ``identifiers`` that were provided by the user (i.e.
            are part of user-specified equations, abstract code, etc.). Will
            be used to determine when to issue namespace conflict warnings. If
            not specified, will be assumed to be identical to ``identifiers``.
        additional_variables : dict-like, optional
            An additional mapping of names to `Variable` objects that will be
            checked before `Group.variables`.
        run_namespace : dict-like, optional
            An additional namespace, provided as an argument to the
            `Network.run` method.
        level : int, optional
            How far to go up in the stack to find the original call frame.
        do_warn : bool, optional
            Whether to warn about names that are defined both as an internal
            variable (i.e. in `Group.variables`) and in some other namespace.
            Defaults to ``True`` but can be switched off for internal variables
            used in templates that the user might not even know about.

        Returns
        -------
        variables : dict of `Variable` or `Function`
            A mapping from name to `Variable`/`Function` object for each of the
            names given in `identifiers`

        Raises
        ------
        KeyError
            If one of the names in `identifier` cannot be resolved
        '''
        if user_identifiers is None:
            user_identifiers = identifiers
        resolved = {}
        for identifier in identifiers:
            resolved[identifier] = self.resolve(identifier,
                                                identifier in user_identifiers,
                                                additional_variables=additional_variables,
                                                run_namespace=run_namespace,
                                                level=level+1)
        return resolved

    def _resolve_external(self, identifier, user_identifier=True,
                          run_namespace=None, level=0):
        '''
        Resolve an external identifier in the context of a `Group`. If the `Group`
        declares an explicit namespace, this namespace is used in addition to the
        standard namespace for units and functions. Additionally, the namespace in
        the `run_namespace` argument (i.e. the namespace provided to `Network.run`)
        or, if this argument is unspecified, the implicit namespace of
        surrounding variables in the stack frame where the original call was made
        is used (to determine this stack frame, the `level` argument has to be set
        correctly).

        Parameters
        ----------
        identifier : str
            The name to resolve.
        user_identifier : bool, optional
            Whether this is an identifier that was used by the user (and not
            something automatically generated that the user might not even
            know about). Will be used to determine whether to display a
            warning in the case of namespace clashes. Defaults to ``True``.
        group : `Group`
            The group that potentially defines an explicit namespace for looking up
            external names.
        run_namespace : dict, optional
            A namespace (mapping from strings to objects), as provided as an
            argument to the `Network.run` function.
        level : int, optional
            How far to go up in the stack to find the calling frame.
        '''
        # We save tuples of (namespace description, referred object) to
        # give meaningful warnings in case of duplicate definitions
        matches = []

        namespaces = OrderedDict()
        # Default namespaces (units and functions)
        namespaces['constants'] = DEFAULT_CONSTANTS
        namespaces['units'] = DEFAULT_UNITS
        namespaces['functions'] = DEFAULT_FUNCTIONS
        if getattr(self, 'namespace', None) is not None:
            namespaces['group-specific'] = self.namespace

        # explicit or implicit run namespace
        if run_namespace is not None:
            namespaces['run'] = run_namespace
        else:
            namespaces['implicit'] = get_local_namespace(level+1)

        for description, namespace in namespaces.iteritems():
            if identifier in namespace:
                matches.append((description, namespace[identifier]))

        if len(matches) == 0:
            # No match at all
            raise KeyError(('The identifier "%s" could not be resolved.') %
                           (identifier))
        elif len(matches) > 1:
            # Possibly, all matches refer to the same object
            first_obj = matches[0][1]
            found_mismatch = False
            for m in matches:
                if _same_value(m[1], first_obj):
                    continue
                if _same_function(m[1], first_obj):
                    continue
                try:
                    proxy = weakref.proxy(first_obj)
                    if m[1] is proxy:
                        continue
                except TypeError:
                    pass

                # Found a mismatch
                found_mismatch = True
                break

            if found_mismatch and user_identifier:
                _conflict_warning(('The name "%s" refers to different objects '
                                   'in different namespaces used for resolving '
                                   'names in the context of group "%s". '
                                   'Will use the object from the %s namespace '
                                   'with the value %r') %
                                  (identifier, getattr(self, 'name',
                                                       '<unknown>'),
                                   matches[0][0],
                                   first_obj), matches[1:])

        # use the first match (according to resolution order)
        resolved = matches[0][1]

        # Replace pure Python functions by a Functions object
        if callable(resolved) and not isinstance(resolved, Function):
            resolved = Function(resolved, stateless=False)

        if not isinstance(resolved, (Function, Variable)):
            # Wrap the value in a Constant object
            unit = get_unit(resolved)
            value = np.asarray(resolved)
            if value.shape != ():
                raise KeyError('Variable %s was found in the namespace, but is'
                               ' not a scalar value' % identifier)
            resolved = Constant(identifier, unit=unit, value=value)

        return resolved

    def runner(self, *args, **kwds):
        raise AttributeError("The 'runner' method has been renamed to "
                             "'run_regularly'.")

    def custom_operation(self, *args, **kwds):
        raise AttributeError("The 'custom_operation' method has been renamed "
                             "to 'run_regularly'.")

    def run_regularly(self, code, dt=None, clock=None, when='start',
                      order=0, name=None, codeobj_class=None):
        '''
        Run abstract code in the group's namespace. The created `CodeRunner`
        object will be automatically added to the group, it therefore does not
        need to be added to the network manually. However, a reference to the
        object will be returned, which can be used to later remove it from the
        group or to set it to inactive.

        Parameters
        ----------
        code : str
            The abstract code to run.
        dt : `Quantity`, optional
            The time step to use for this custom operation. Cannot be combined
            with the `clock` argument.
        clock : `Clock`, optional
            The update clock to use for this operation. If neither a clock nor
            the `dt` argument is specified, defaults to the clock of the group.
        when : str, optional
            When to run within a time step, defaults to the ``'start'`` slot.
        name : str, optional
            A unique name, if non is given the name of the group appended with
            'run_regularly', 'run_regularly_1', etc. will be used. If a
            name is given explicitly, it will be used as given (i.e. the group
            name will not be prepended automatically).
        codeobj_class : class, optional
            The `CodeObject` class to run code with. If not specified, defaults
            to the `group`'s ``codeobj_class`` attribute.

        Returns
        -------
        obj : `CodeRunner`
            A reference to the object that will be run.
        '''
        if name is None:
            name = self.name + '_run_regularly*'

        if dt is None and clock is None:
            clock = self._clock

        runner = CodeRunner(self, 'stateupdate', code=code, name=name,
                            dt=dt, clock=clock, when=when, order=order,
                            codeobj_class=codeobj_class)
        self.contained_objects.append(runner)
        return runner


class CodeRunner(BrianObject):
    '''
    A "code runner" that runs a `CodeObject` every timestep and keeps a
    reference to the `Group`. Used in `NeuronGroup` for `Thresholder`,
    `Resetter` and `StateUpdater`.
    
    On creation, we try to run the before_run method with an empty additional
    namespace (see `Network.before_run`). If the namespace is already complete
    this might catch unit mismatches.
    
    Parameters
    ----------
    group : `Group`
        The group to which this object belongs.
    template : `Template`
        The template that should be used for code generation
    code : str, optional
        The abstract code that should be executed every time step. The
        `update_abstract_code` method might generate this code dynamically
        before every run instead.
    dt : `Quantity`, optional
        The time step to be used for the simulation. Cannot be combined with
        the `clock` argument.
    user_code : str, optional
        The abstract code as specified by the user, i.e. without any additions
        of internal code that the user not necessarily knows about. This will
        be used for warnings and error messages.
    clock : `Clock`, optional
        The update clock to be used. If neither a clock, nor the `dt` argument
        is specified, the `defaultclock` will be used.
    when : str, optional
        In which scheduling slot to execute the operation during a time step.
        Defaults to ``'start'``.
    order : int, optional
        The priority of this operation for operations occurring at the same time
        step and in the same scheduling slot. Defaults to 0.
    name : str, optional 
        The name for this object.
    check_units : bool, optional
        Whether the units should be checked for consistency before a run. Is
        activated (``True``) by default but should be switched off for state
        updaters (units are already checked for the equations and the generated
        abstract code might have already replaced variables with their unit-less
        values)
    template_kwds : dict, optional
        A dictionary of additional information that is passed to the template.
    needed_variables: list of str, optional
        A list of variables that are neither present in the abstract code, nor
        in the ``USES_VARIABLES`` statement in the template. This is only
        rarely necessary, an example being a `StateMonitor` where the
        names of the variables are neither known to the template nor included
        in the abstract code statements.
    override_conditional_write: list of str, optional
        A list of variable names which are used as conditions (e.g. for
        refractoriness) which should be ignored.
    codeobj_class : class, optional
        The `CodeObject` class to run code with. If not specified, defaults to
        the `group`'s ``codeobj_class`` attribute.
    '''
    add_to_magic_network = True
    invalidates_magic_network = True
    def __init__(self, group, template, code='', user_code=None,
                 dt=None, clock=None, when='start',
                 order=0, name='coderunner*', check_units=True,
                 template_kwds=None, needed_variables=None,
                 override_conditional_write=None,
                 codeobj_class=None,
                 ):
        BrianObject.__init__(self, clock=clock, dt=dt, when=when, order=order,
                             name=name)
        self.group = weakref.proxy(group)
        self.template = template
        self.user_code = user_code
        self.abstract_code = code
        self.check_units = check_units
        if needed_variables is None:
            needed_variables = []
        self.needed_variables = needed_variables
        self.template_kwds = template_kwds
        self.override_conditional_write = override_conditional_write
        if codeobj_class is None:
            codeobj_class = group.codeobj_class
        self.codeobj_class = codeobj_class
        self.codeobj = None

    def update_abstract_code(self, run_namespace=None, level=0):
        '''
        Update the abstract code for the code object. Will be called in
        `before_run` and should update the `CodeRunner.abstract_code`
        attribute.
        
        Does nothing by default.
        '''
        pass

    def before_run(self, run_namespace=None, level=0):
        self.update_abstract_code(run_namespace=run_namespace, level=level+1)
        # If the CodeRunner has variables, add them
        if hasattr(self, 'variables'):
            additional_variables = self.variables
        else:
            additional_variables = None
        self.codeobj = create_runner_codeobj(group=self.group,
                                             code=self.abstract_code,
                                             user_code=self.user_code,
                                             template_name=self.template,
                                             name=self.name+'_codeobject*',
                                             check_units=self.check_units,
                                             additional_variables=additional_variables,
                                             needed_variables=self.needed_variables,
                                             run_namespace=run_namespace,
                                             level=level+1,
                                             template_kwds=self.template_kwds,
                                             override_conditional_write=self.override_conditional_write,
                                             codeobj_class=self.codeobj_class
                                             )
        self.code_objects[:] = [weakref.proxy(self.codeobj)]
