from betahaus.openmember.tests.base import TestCase
import unittest
from Products.PloneTestCase import PloneTestCase as ptc
from Products.CMFCore.utils import getToolByName
from Products.Archetypes.event import ObjectInitializedEvent
from zope.event import notify
from DateTime.DateTime import DateTime
from betahaus.openmember.interfaces import IMemberData
from Products.CMFCore.utils import _checkPermission as checkPerm
from betahaus.openmember.permissions import OPENMEMBER_SEARCH_DATABASE,\
    OPENMEMBER_MANAGE_DATABASE


ptc.setupPloneSite()

class TestDatabase(TestCase):
    """ """
    def afterSetUp(self):
        """ """
        self.om_catalog = getToolByName(self.portal , 'openmember_catalog')
        self.membership = getToolByName(self.portal, 'portal_membership')

        self.membership.addMember('portal_member', 'pm', ['Member'], [], {})
        self.membership.addMember('manager', 'test_manager', ['Manager'], [], {})
        self.membership.addMember('editor', 'test_editor', ['Editor'], [], {})

        portal_setup = getToolByName(self.portal , 'portal_setup')
        portal_setup.runAllImportStepsFromProfile('profile-betahaus.openmember:betahaus.openmember-personcontent', purge_old=False)
        
        
    def test_database_and_member_creation(self):
        """ Tests that there is no db at beginning and that one is created and that we can add content """
        self.login('manager')
        self.failIf(hasattr(self.portal , 'openmember_database'))
        
        # create the Member that should go into the database.
        memberid = 'member_1'
        id = self.portal.invokeFactory('MemberPerson', id = memberid)
        member = self.portal.get(id, False)
        self.failUnless(member)
        notify(ObjectInitializedEvent(member))
        
        # the database is created on first IOpenMember add.
        self.failUnless(hasattr(self.portal , 'openmember_database'))
        openmember_database = getToolByName(self.portal , 'openmember_database')
        self.assertEquals(len(openmember_database), 1)
        self.failUnless(member.UID() in openmember_database.keys())
        
        # make sure it is the right type
        database_object = openmember_database.get(member.UID())
        self.failUnless(IMemberData.providedBy(database_object))
        self.assertEquals(database_object.getId(), member.UID())

    def test_attribute_mirroring_to_database(self):
        """ Tests that there is no db at beginning and that one is created and that we can add content """
        self.login('manager')
        
        memberid = 'member_1'
        id = self.portal.invokeFactory('MemberPerson', id = memberid)
        member = self.portal.get(id, False)
        self.failUnless(member)
        
        member.setFirst_name('Bob')
        notify(ObjectInitializedEvent(member))
        
        openmember_database = getToolByName(self.portal , 'openmember_database')
        UID = member.UID()
        db_obj = openmember_database.get(UID, False)
        self.failUnless(db_obj)
        self.assertEquals(db_obj['first_name'], 'Bob')
    
    def test_mirrored_attribute_update(self):
        self.login('manager')
        
        memberid = 'member_1'
        id = self.portal.invokeFactory('MemberPerson', id = memberid)
        member = self.portal.get(id, False)
        self.failUnless(member)
        
        member.setFirst_name('Bob')
        notify(ObjectInitializedEvent(member))
        
        openmember_database = getToolByName(self.portal , 'openmember_database')
        UID = member.UID()
        db_obj = openmember_database.get(UID, False)
        self.failUnless(db_obj)
        self.assertEquals(db_obj['first_name'], 'Bob')
        
        member.setFirst_name('Bobby')
        notify(ObjectInitializedEvent(member))

        self.assertEquals(db_obj['first_name'], 'Bobby')
        
       
    def test_attribute_versioning(self):
        """ Tests the attribute versioning system"""
        
        self.login('manager')
        memberid = 'member_1'
        member = self.portal[self.portal.invokeFactory('MemberPerson', id = memberid)]
        
        before_edit_date = DateTime()
        member.setFirst_name('Bob')
        member.setLast_name('Dobalina')
        member.modified = DateTime()
        notify(ObjectInitializedEvent(member))
        after_edit_date = DateTime()
        
        openmember_database = getToolByName(self.portal , 'openmember_database')
        db_obj = openmember_database.get(member.UID(), False)
        
        # make sure that the history is available for both first and last name
        first_name_history = getattr(db_obj, '_first_name_history', False)
        self.failUnless(first_name_history)
        self.assertEquals(len(first_name_history), 1)
        
        last_name_history = getattr(db_obj, '_last_name_history', False)
        self.failUnless(last_name_history)
        self.assertEquals(len(last_name_history), 1)
 
        # check that the data is correct for the first name entry
        history_entry = first_name_history[-1]

        self.assertEquals(history_entry.keys(), ['data', 'modified', 'user'])
        self.assertEquals(history_entry['data'], 'Bob')
        self.failUnless(before_edit_date <= history_entry['modified'] and history_entry['modified'] <= after_edit_date)
        self.assertEquals(history_entry['user'], 'manager')
        
        # login as another user to make sure that the userid is carried thru
        self.login('editor')
        
        # change the first name to something else
        before_edit_date = DateTime()
        member.setFirst_name('Bobby')
        member.modified = DateTime()
        notify(ObjectInitializedEvent(member))
        after_edit_date = DateTime()
        
        # make sure that only the first name history is updated and not the last name
        first_name_history = getattr(db_obj, '_first_name_history', False)
        self.failUnless(first_name_history)
        self.assertEquals(len(first_name_history), 2)

        last_name_history = getattr(db_obj, '_last_name_history', False)
        self.failUnless(last_name_history)
        self.assertEquals(len(last_name_history), 1)

        history_entry = first_name_history[-1]
        self.assertEquals(history_entry.keys(), ['data', 'modified', 'user'])
        self.assertEquals(history_entry['data'], 'Bobby')
        self.failUnless(before_edit_date <= history_entry['modified'] and history_entry['modified'] <= after_edit_date)        
        self.assertEquals(history_entry['user'], 'editor')
        
    def test_attribute_mirroring_with_custom_accessor(self):
        """A Field can set the attribute om_accessor to use instead of getRaw"""
        self.login('manager')
        
        memberid = 'member_1'
        id = self.portal.invokeFactory('MemberPerson', id = memberid)
        member = self.portal.get(id, False)
        self.failUnless(member)
        member.getOm_FirstName = lambda: 'test'
        member.getField('first_name').om_accessor = 'getOm_FirstName'
                
        member.setFirst_name('Bob')
        notify(ObjectInitializedEvent(member))
        
        openmember_database = getToolByName(self.portal , 'openmember_database')
        UID = member.UID()
        db_obj = openmember_database.get(UID, False)
        self.failUnless(db_obj)
        self.assertEquals(db_obj['first_name'], 'test')
        # clean up
        del member.getOm_FirstName 
        del member.getField('first_name').om_accessor

    def test_om_roles_installed(self):
        """Both ManageOMDatabase and SearchOMDatabase should be available after install"""
        self.failUnless('ManageOMDatabase' in self.portal.getGlobalPortalRoles())
        self.failUnless('SearchOMDatabase' in self.portal.getGlobalPortalRoles())

    def test_om_search_role(self):
        """Getting the SearchOMDatabase Role should give you the search permission but not the manage permission"""
        self.failIf(checkPerm(OPENMEMBER_SEARCH_DATABASE, self.portal))
        self.failIf(checkPerm(OPENMEMBER_MANAGE_DATABASE, self.portal))
        self.setRoles(['SearchOMDatabase'])
        self.failUnless(checkPerm(OPENMEMBER_SEARCH_DATABASE, self.portal))
        self.failIf(checkPerm(OPENMEMBER_MANAGE_DATABASE, self.portal))
        self.setRoles(['Member'])
            
    def test_om_manage_role(self):
        """Getting the ManageOMDatabase Role should give you the both the search permission and the manage permission"""
        self.failIf(checkPerm(OPENMEMBER_SEARCH_DATABASE, self.portal))
        self.failIf(checkPerm(OPENMEMBER_MANAGE_DATABASE, self.portal))
        self.setRoles(['ManageOMDatabase'])
        self.failUnless(checkPerm(OPENMEMBER_SEARCH_DATABASE, self.portal))
        self.failUnless(checkPerm(OPENMEMBER_MANAGE_DATABASE, self.portal))
        self.setRoles(['Member'])
        
def test_suite():
    """ """
    suite = unittest.TestSuite()
    suite.addTest(unittest.makeSuite(TestDatabase))
    return suite
