from zope.interface import implements

from Products.Archetypes.utils import shasattr
from Products.CMFCore.utils import getToolByName
#from Products.BTreeFolder2.BTreeFolder2 import manage_addBTreeFolder


from betahaus.openmember.database import manage_addOpenMemberDatabase

from betahaus.openmember import PROJECTNAME, CATALOG_ID, DATABASE_ID, logger
from betahaus.openmember.interfaces import IOMDataHandler
from zope.event import notify
from betahaus.openmember.events import OMemberAddedEvent,\
    OMemberWillBeModifiedEvent, OMemberModifiedEvent, OMemberDeletedEvent
from DateTime.DateTime import DateTime
from betahaus.openmember.content.memberdata import MemberData
from zExceptions import Forbidden

def onlyWhenInstalled(func):
    """A decorator that prevents ``func`` to be executed if the product isn't installed.
    """
    def wrapper(context, *args, **kwargs):
        qi = getToolByName(context, 'portal_quickinstaller')
        if qi and qi.isProductInstalled(PROJECTNAME):
            return func(context, *args, **kwargs)
        else:
            return None
    wrapper.__name__ = func.__name__
    wrapper.__dict__.update(func.__dict__)
    wrapper.__doc__ = func.__doc__
    wrapper.__module__ = func.__module__
    return wrapper

def not_member(context):
    """Objects with not_member = True should not be mirrored at all."""
    return getattr(context, 'not_member', False)
    
@onlyWhenInstalled
def added(context, event):
    """Adds member data when a content type that should be indexed is added. 
    """
    db = IOMDataHandler(context)
    db.addMember(context)
    logger.debug('IOpenMember %s was added' % context.Title())

@onlyWhenInstalled    
def modified(context, event):
    """ Modifies member data
    """
    db = IOMDataHandler(context)
    db.updateMember(context)
    logger.debug('IOpenMember %s was edited' % context.Title())
    
@onlyWhenInstalled    
def deleted(context, event):
    """ Marks member data as orphaned, by removing UID link
    """
    db = IOMDataHandler(context)
    db.markDeletedMember(context)
    logger.debug('IOpenMember %s was orphaned' % context.Title())



def getValueFor(obj, field_name):
    if hasattr(obj.getField(field_name), 'om_accessor'):
        value = getattr(obj, obj.getField(field_name).om_accessor)
    else:
        value = getattr(obj, 'getRaw'+field_name.capitalize(), None)
    #value might be a function (like context.Title()) rather than a property
    if callable(value):
        value = value()
    return value
   
class OMDataHandler(object):
    """Handles communication with the OpenMember database
    """
    implements(IOMDataHandler)
    
    def __init__(self, context, id = DATABASE_ID):
        self.site = getToolByName(context,'portal_url').getPortalObject()
        database = getattr(self.site,id,None)
        if database is None:
            logger.info('No database found\nAdding new database to portal root with id %s.' % id)
            manage_addOpenMemberDatabase(self.site, id = id, title = 'OpenMember Database')
            database = getattr(self.site,id)
        
        if database.meta_type != 'BTreeFolder2':
            return None #FIXME: Throw an exception
        
        #Store for internal use.
        self.database = database
        self.catalog = getToolByName(self.site,CATALOG_ID)
        self.portal_membership = getToolByName(self.site, 'portal_membership')
        
            
    def _setAttributes(self, obj):
        """ Takes a specific object (obj) that is a regular context
            and creates/sets attributes on a MemberData object. Also takes care of versioning of data.
        """
        
        #Get the MemberData object
        memberdata = getattr(self.database, obj.UID())

        #Loop through all indexes to see if something has changed
        #FIXME This should be a property on the database rather than the catalog
        for index in self.catalog.mirrorableIndexes():
            
            # The value of the field that is to be indexed, if it exists
            # use the standard archetypes get methods
            value = getValueFor(obj, index)
            
            #Only set attribute if something has changed
            if value == getattr(memberdata, index, ''):
                continue
            
            # Check if we need to write history for this property
            # Maybe this should be a property same as address_fields
            disable_history = getattr(obj, 'disable_history', [])
            if index not in disable_history:
                # Store the change with history
                memberdata[index] = value
            else:
                #Set main value on MemberData, without history
                setattr(memberdata, index, value)
                
            # There was a change so set the modified date to now.
            memberdata.modified = DateTime()


    def addMember(self, obj):
        """obj is the member that needs to be mapped"""
    
        #Add the actual content type
        obj_uid = obj.UID()
        have_memberdata = shasattr(self.database, obj_uid, acquire = False)
        if not have_memberdata:
            new_member = MemberData(obj_uid)
            self.database._setObject(obj_uid, new_member)
        else:
            self.updateMember(obj)
            return False 
        
        
        self._setAttributes(obj)
        memberdata = getattr(self.database, obj_uid)
        self.catalog.indexObject(memberdata)
        
        notify(OMemberAddedEvent(memberdata, obj))
        return memberdata
                    
    def updateMember(self, obj):
        """Updates member data for obj"""
        have_memberdata = shasattr(self.database, obj.UID(), acquire = False)
        if not have_memberdata:
            return self.addMember(obj)
        else:
            notify(OMemberWillBeModifiedEvent(getattr(self.database, obj.UID()), obj))
            self._setAttributes(obj)
            memberdata = getattr(self.database, obj.UID())
            self.catalog.reindexObject(memberdata)
            
            notify(OMemberModifiedEvent(memberdata, obj))
            return memberdata
            
    def markDeletedMember(self, obj):
        """Markes member data as orphaned by removing the mirrored_object_uid reference"""
        have_memberdata = shasattr(self.database, obj.UID(), acquire = False)
        if have_memberdata:
            memberdata = getattr(self.database, obj.UID())
            self.catalog.reindexObject(memberdata)
            
            notify(OMemberDeletedEvent(memberdata, obj))
            return memberdata

    def removeDeletedMember(self, id):
        """ Delete an object that is orphaned permanently. """
        #FIXME: This is planned and not integrated yet
        have_memberdata = shasattr(self.database, id, acquire = False)
        if have_memberdata:
            memberdata = getattr(self.database, id)
            if memberdata.getMappedObjectUID() != 'orphaned':
                raise Forbidden("Removing memberdata that isn't marked as an orphan isn't allowed."
                                "Tried to remove: %s" % "/".join(memberdata.getPhysicalPath()))
            #This will raise relevant notifications
            self._delObject(id)
        else:
            raise AttributeError("Database doesn't have anything with the id %s" % id)