"""We use our own catalog for member data"""

from zope.interface import implements
from zope.app.schema.vocabulary import IVocabularyFactory

from zope.event import notify
from Products.Archetypes.event import ObjectInitializedEvent

from zope.schema.vocabulary import SimpleVocabulary
from zope.schema.vocabulary import SimpleTerm

from Acquisition import aq_inner
from AccessControl import ClassSecurityInfo, Unauthorized, getSecurityManager

from AccessControl.Permissions import manage_zcatalog_entries as ManageZCatalogEntries
from AccessControl.Permissions import search_zcatalog as SearchZCatalog

from Products.CMFCore.utils import getToolByName
from Products.Archetypes.utils import shasattr
from Products.CMFPlone.CatalogTool import CatalogTool
from Products.CMFPlone.utils import base_hasattr
from Products.CMFPlone.utils import safe_callable
from Products.ZCatalog.Catalog import CatalogError

from betahaus.openmember import CATALOG_ID, OPENMEMBER_SEARCH_DATABASE
from betahaus.openmember.interfaces import IOMDataHandler, IOpenMemberCatalog,\
    IOpenMember
from zope.app.component.hooks import getSite


class OpenMemberCatalogException(Exception):
    def __init(self, value):
        self.value = value
    def __str__(self):
        return repr(self.value)

class _extra:
    pass

class OpenMemberCatalog(CatalogTool):
    id = CATALOG_ID
    security = ClassSecurityInfo()
    
    implements(IOpenMemberCatalog)


    @property
    def openmember_properties(self):
        portal_properties = getToolByName(self, 'portal_properties')
        return getattr(portal_properties, 'openmember_properties', None)
        
    def revisit_mirrored_objects(self):
        """Finds all objects with interface 'betahaus.openmember.interfaces.IOpenMember' and 
        makes sure they are pulled into the database."""
        
        portal_catalog = getToolByName(self, 'portal_catalog')
        members = portal_catalog(object_provides = IOpenMember.__identifier__)
        for member in members:
            notify(ObjectInitializedEvent(member.getObject()))
        
        

    security.declareProtected(ManageZCatalogEntries, 'clearFindAndRebuild')
    def clearFindAndRebuild(self):
        """Empties catalog, then finds all contentish objects (i.e. objects
           with an omcIndexObject method), and reindexes them.
           This may take a long time.
           
           This method is overloaded to use the custom indexer method omcIndexObject instead of indexObject.
           It is also restricted to the object in site root with ID from betahaus.openmember.DATABASE_ID
        """
        def omcIndexObject(obj, path):
            if (base_hasattr(obj, 'omcIndexObject') and
                safe_callable(obj.omcIndexObject)):
                try:
                    obj.omcIndexObject()
                except TypeError:
                    # Catalogs have 'indexObject' as well, but they
                    # take different args, and will fail
                    pass
        self.manage_catalogClear()
        database = IOMDataHandler(aq_inner(self)).database
        self.ZopeFindAndApply(database, search_sub=True, apply_func=omcIndexObject)
        
    security.declareProtected(SearchZCatalog, 'searchResults')
    def searchResults(self, REQUEST=None, **kw):
        """This is a wrapper so that we can set a custom permission for searching in the catalog.
        """
        sm = getSecurityManager()
        # the getattr dance is for when you in ZMI view the content of the catalog there is no context so use self instead
        if sm.checkPermission(OPENMEMBER_SEARCH_DATABASE, getattr(self, 'context', self)):
            #FIXME: This is a temporary patch to remove orphans. It will slow searches down.
            results = super(OpenMemberCatalog, self).searchResults(REQUEST,**kw)
            return [x for x in results if x.getMappedObjectUID != 'orphaned']
        raise Unauthorized('The current user does not have the '
                               'required "%s" permission' % OPENMEMBER_SEARCH_DATABASE)
        
    __call__ = searchResults


    def isKeywordIndex(self, index):
        """Checks if a index name is a KeywordIndex or not in the catalog
        
        @param: index, the name of the index to check
        """
        item = self.openmember_catalog.Indexes.get(index, None)
        
        if item != None and item.meta_type == 'KeywordIndex':
            return True
        return False    

    def isUIDLikeIndex(self, index):
        """Checks if a index name contain UIDs. This is done by checking the first value if it is a UID.
        This is not a guarantee but a decent guess. 
        
        @param: index, the name of the index to check
        """
        item = self.openmember_catalog.Indexes.get(index)
        #Reference UIDs are only stored in KeywordIndexes
        if not self.isKeywordIndex(index):
            return False
        #KeywordIndexes have items attr
        if not shasattr(item, 'items'):
            return False
        
        result = [x[0] for x in item.items()]
        if len(result):
            for val in result:
                return self.isUIDLike(val)
        
        return 'unknown'
    
    def isUIDLike(self, text):
        """ Check if something seems to be a UID.
            If it is part of a list, we can assume that something is a reference
        """
        if not text:
            return False
        if len(text) != 32:
            return False
        import re
        if re.match("[0-9a-f]{32}", text):
            return True
    
    def addIndex(self, name, index_type, extra = None, label = None, internal = False):
        """Adds a index to our catalog.
           also adds a label attribute for display and a 'sortable_' for sorting
           if internal, also adds the name to property internal_indexes in property sheet openmember_properties
                        this property define what indexes that should not show up as a selectable search index.
        """
        if index_type == 'ZCTextIndex' and extra == None:
            if not shasattr(self, 'plone_lexicon', acquire = False):
                # Make sure the lexicon is there
                pl = getattr(getToolByName(self, 'portal_catalog'), 'plone_lexicon')
                setattr(self, 'plone_lexicon', pl)

            # this is a wierd way to add the index but it is the only way I found that works.
            extra = _extra()
            extra.lexicon_id = 'plone_lexicon'
            extra.index_type = 'Okapi BM25 Rank'
            
        super(OpenMemberCatalog, self).addIndex(name, index_type, extra)
        super(OpenMemberCatalog, self).addIndex('sortable_'+name, 'KeywordIndex', extra = {'indexed_attrs': name})
        
        # get the new index, and set the label as an attribute.
        new_index = self.Indexes.get(name)
        if label:    
            setattr(new_index, 'label', label)
        else:
            setattr(new_index, 'label', name)
            
        if internal:
            idxs = set()
            if self.openmember_properties.hasProperty('internal_indexes'):
                [idxs.add(x) for x in self.openmember_properties.getProperty('internal_indexes')]
            idxs.add(name)
            self.openmember_properties.manage_changeProperties(internal_indexes=tuple(idxs))
            
        # yay we are done.

    def delIndex(self, name):
        """removes a index with the correspoding 'sortable_' + name
           if index is defined as internal index it is removed from there as well.
        """
        super(OpenMemberCatalog, self).delIndex(name)
        try:
            super(OpenMemberCatalog, self).delIndex('sortable_' +name)
        except CatalogError:
            pass
        idxs = set()
        if self.openmember_properties.hasProperty('internal_indexes'):
            [idxs.add(x) for x in self.openmember_properties.getProperty('internal_indexes')]
        if name in idxs:
            idxs.remove(name)
            self.openmember_properties.manage_changeProperties(internal_indexes=tuple(idxs))
        

    security.declareProtected(SearchZCatalog, 'indexes')
    def indexes(self):
        return [x for x in self._catalog.indexes.keys() if not x.startswith('sortable')]
    
    security.declareProtected(SearchZCatalog, 'mirrorableIndexes')
    def mirrorableIndexes(self):
        return [x for x in self.indexes() if x not in self.internal_indexes()]
        
    security.declareProtected(SearchZCatalog, 'sortindexes')
    def sortindexes(self):
        return [x for x in self._catalog.indexes.keys() if x.startswith('sortable')]
    
    security.declareProtected(SearchZCatalog, 'allindexes')
    def allindexes(self):
        return self._catalog.indexes.keys()
    
    security.declareProtected(SearchZCatalog, 'internal_indexes')
    def internal_indexes(self):
        if self.openmember_properties.hasProperty('internal_indexes'):
            return self.openmember_properties.getProperty('internal_indexes')
        return ()
    
    
    def setMirrorFields(self, indexes):
        """Takes a list of ('field_name', 'IndexType')
           and adds them to the catalog as indexes and columns.
           
           Removes indexes and columns from the catalog for field_names that are not in the list.
        """
        
        omindexes = self.indexes()
        omcolumns = self.schema()

        # add indexes and columns that are not in the catalog yet.
        # prune the omindex, omcolumns so we know if there are any unneeded indexes, columns that should be removed.
        for name, meta_type, label in indexes:
            if name not in omindexes:
                self.addIndex(name, meta_type, label = label)
            else:
                this_index = self.Indexes.get(name)
                if getattr(this_index, 'label' , '') != label:
                    # Catch if the label has changed
                    this_index.label = label
                    
                if this_index.meta_type != meta_type:
                    # Catch if type is changed
                    self.delIndex(name)
                    self.addIndex(name, meta_type, label = label)
                    
                omindexes.remove(name)
                
                
            if name not in omcolumns:
                self.addColumn(name)
            else:
                omcolumns.remove(name)
                
        # remove any old indexes and columns that were not specified.
        for index in omindexes:
            self.delIndex(index)
        for column in omcolumns:
            self.delColumn(column)
    

class IndexesVocabulary(object):
    """Vocabulary factory for Selectable indexes.

      >>> from zope.component import queryUtility
      >>> from betahaus.openmember.tests.base import DummyContext
      >>> from betahaus.openmember.tests.base import DummyCatalogTool

      >>> name = 'betahaus.openmember.SelectableIndexes'
      >>> util = queryUtility(IVocabularyFactory, name)
      >>> context1 = DummyContext()
      >>> context2 = DummyContext()
      >>> context1.context = context2

      >>> util(context1) is None
      True

      >>> context2.opemember_catalog = DummyCatalogTool()

      >>> indexes = util(context1)
      >>> indexes
      <zope.schema.vocabulary.SimpleVocabulary object at ...>

      >>> len(indexes.by_token)
      2

      >>> mem = types.by_token['membership_references']
      >>> mem.title, mem.token, mem.value
      ('Memberships', 'membership_references', 'membership_references')
    """
    implements(IVocabularyFactory)

    def __call__(self, context):
        context = getattr(context, 'context', context)
        om_catalog = getToolByName(context, CATALOG_ID, None)
        if om_catalog is None:
            return None
        items = []
        for index_name in om_catalog.mirrorableIndexes():
            index = om_catalog.Indexes.get(index_name)
            label = getattr(index, 'label', index.id)
            items.append((label,index.id))

        items.sort()
        items = [SimpleTerm(i[1], i[1], i[0]) for i in items]
        return SimpleVocabulary(items)

IndexesVocabularyFactory = IndexesVocabulary()

class MetaTypesVocabulary(object):
    """Vocabulary factory for Selectable meta types.

      >>> from zope.component import queryUtility
      >>> from betahaus.openmember.tests.base import DummyContext
      >>> from betahaus.openmember.tests.base import DummyCatalogTool

      >>> name = 'betahaus.openmember.SelectableIndexes'
      >>> util = queryUtility(IVocabularyFactory, name)
      >>> context1 = DummyContext()
      >>> context2 = DummyContext()
      >>> context1.context = context2

      >>> util(context1) is None
      True

      >>> context2.opemember_catalog = DummyCatalogTool()

      >>> indexes = util(context1)
      >>> indexes
      <zope.schema.vocabulary.SimpleVocabulary object at ...>

      >>> len(indexes.by_token)
      2

      >>> mem = types.by_token['membership_references']
      >>> mem.title, mem.token, mem.value
      ('Memberships', 'membership_references', 'membership_references')
    """
    implements(IVocabularyFactory)

    def __call__(self, context):
        context = getSite()
        om_catalog = getToolByName(context, CATALOG_ID, None)
        if om_catalog is None:
            return None
        items_set = set()
        [items_set.add(meta_type['name']) for meta_type in om_catalog.Indexes.filtered_meta_types()]

        items = list(items_set)
        items.sort()
        return SimpleVocabulary([SimpleTerm(i, i, i) for i in items])

MetaTypesVocabularyFactory = MetaTypesVocabulary()