

from zope.interface import implements,  Interface
from zope.component import adapts, getMultiAdapter
from zope.formlib import form
from zope.event import notify

from AccessControl import getSecurityManager
from Acquisition import aq_inner

from Products.CMFDefault.formlib.schema import SchemaAdapterBase
from Products.CMFPlone.interfaces import IPloneSiteRoot

from Products.CMFCore.utils import getToolByName
from Products.statusmessages.interfaces import IStatusMessage

from plone.app.form.validators import null_validator

from plone.fieldsets.fieldsets import FormFieldsets

from plone.app.controlpanel.form import ControlPanelForm
from plone.app.controlpanel.events import ConfigurationChangedEvent

from plone.protect import CheckAuthenticator

from betahaus.openmember import OpenMemberMessageFactory as _
from betahaus.openmember import CATALOG_ID, OPENMEMBER_MANAGE_DATABASE, PROJECTNAME
from betahaus.openmember.interfaces import IOMControlPanelDatabaseSchema, IOMControlPanelForm
from Products.Archetypes.debug import deprecated
from betahaus.openmember.browser.widget import OMIndexPair
from docutils.nodes import row

_controlpanel_forms = {}

def getKey(interface, package):
    return "%s.%s" % (package, interface.__name__)

def registerForm(interface, package):
    """Registers an interface as a sub controlpanel form in the OpenMember control panel.
    The interface needs to inherit from the betahaus.openmember.interfaces.IOMControlPanelForm 
    in order to be displayed as a sub control panel."""

    data = {'interface': interface,
            'package': package}
    
    _controlpanel_forms[getKey(interface, package)] = data

from zope.app.form import CustomWidgetFactory
from zope.app.form.browser import ObjectWidget
from zope.app.form.browser import ListSequenceWidget


index_pair_widget = CustomWidgetFactory(ObjectWidget, OMIndexPair)
combination_widget = CustomWidgetFactory(ListSequenceWidget,
                                         subwidget=index_pair_widget)
        
class OMControlPanelAdapter(SchemaAdapterBase):
    """Control panel adapter for the standard database settings"""
    implements(IOMControlPanelDatabaseSchema)
    adapts(IPloneSiteRoot)
    
    custom_widgets = {'fields_to_index':combination_widget}
    
    def getId(self):
        return 'database'
    
    def getLabel(self):
        return  _('Database')
    
    def __init__(self, context):
        super(OMControlPanelAdapter, self).__init__(context)
        self.omcatalog = getToolByName(context, CATALOG_ID)
        self.pprop = getToolByName(context, 'portal_properties')
        self.default_charset = self.pprop.site_properties.default_charset


    def get_fields_to_index(self):
        result = []
        for index in self.omcatalog.mirrorableIndexes():
            item = self.omcatalog.Indexes.get(index)
            if item.id.startswith('sortable'):
                continue
            label = getattr(item, 'label', None)
            if not label:
                label = item.id
                
            result.append(OMIndexPair(item.id, item.meta_type, label.decode(self.default_charset)) )

        return result

    def set_fields_to_index(self, value):

        # add indexes and columns that are not in the catalog yet.
        # prune the omindex, omcolumns so we know if there are any unneeded indexes, columns
        indexes_to_add = []
        indexes_to_remove = self.fields_to_index
        
        
        for row in value:
            if not row.name:
                raise ValueError(_(u'Must have a name'))
            if not row.meta_type:
                raise ValueError(_(u'Must have an index type'))
            if not row.label:
                row.label = row.name
            
            name = row.name.encode(self.default_charset)
            meta_type = row.meta_type.encode(self.default_charset)
            label = row.label.encode(self.default_charset)
            
            if row not in indexes_to_remove:
                indexes_to_add.append(row)
            else:
                indexes_to_remove.remove(row)
                    
        removed = [self.omcatalog.delIndex(x.name) for x in indexes_to_remove]
        added = [self.omcatalog.addIndex(x.name.encode(self.default_charset),
                                         x.meta_type.encode(self.default_charset),
                                         label = x.label.encode(self.default_charset)) for x in indexes_to_add]
        
        if removed or added:
            self.omcatalog.clearFindAndRebuild()

    fields_to_index = property(get_fields_to_index,
                               set_fields_to_index)
    
    def get_title_fields(self):
        """The address fields are stored in the properties tool, on the openmember_properties sheet lines field called address_fields
        """
        portal_properties = getToolByName(self.context, 'portal_properties', None)
        openmember_properties = getattr(portal_properties, 'openmember_properties', None)
        if openmember_properties.hasProperty('title_fields'):
            return list(openmember_properties.getProperty('title_fields'))
        else:
            return []
    
    def set_title_fields(self, value):
        """The address fields are stored in the properties tool, on the openmember_properties sheet lines field called address_fields.
        """
        portal_properties = getToolByName(self.context, 'portal_properties', None)
        openmember_properties = getattr(portal_properties, 'openmember_properties', None)
        openmember_properties.manage_changeProperties(title_fields=tuple(value))
        
    title_fields = property(get_title_fields,
                            set_title_fields)
    
registerForm(IOMControlPanelDatabaseSchema, PROJECTNAME)


class OMControlPanel(ControlPanelForm):
    """Base class for the control panel.
    When rendering it uses all the sub control panels registered using the 
    betahaus.openmember.browser.controlpanel.registerForm method as sub panels.
    """
        
    def __init__(self, context, request):
        super(OMControlPanel, self).__init__(context, request)
        qi = getToolByName(self.context, 'portal_quickinstaller')
        forms = []
        keys = _controlpanel_forms.keys()
        
        if getKey(IOMControlPanelDatabaseSchema, PROJECTNAME) in keys:
            """ this makes sure that the database s always at the front"""
            key = getKey(IOMControlPanelDatabaseSchema, PROJECTNAME)
            keys.remove(key)
            keys.insert(0, key)
            
        for key in keys:
            data = _controlpanel_forms[key]
            
            package = data['package']
            interface = data['interface']

            if qi.isProductInstalled(package) or not qi.isProductInstallable(package):
                if Interface.providedBy(interface) and IOMControlPanelForm in interface.getBases():
                    adapter = interface(self.context)
                    form = FormFieldsets(interface)
                    if hasattr(adapter, 'custom_widgets'):
                        for field in adapter.custom_widgets.keys():
                            form[field].custom_widget = adapter.custom_widgets[field]
                    form.id = adapter.getId()
                    form.label = adapter.getLabel()
                    forms.append(form)
        
        self.form_fields = FormFieldsets(*forms)
        
        
    label = _(u"OpenMember control panel")
    description = _(u"Tools and settings for OpenMember database.")

    def available(self):
        root = aq_inner(self.context).getPhysicalRoot()
        sm = getSecurityManager()
        return sm.checkPermission(OPENMEMBER_MANAGE_DATABASE, root)

    @form.action(_(u'label_save', default=u'Save'), name=u'save')
    def handle_edit_action(self, action, data):
        CheckAuthenticator(self.request)
        if form.applyChanges(self.context, self.form_fields, data,
                             self.adapters):
            self.status = _("Changes saved.")
            notify(ConfigurationChangedEvent(self, data))
        else:
            self.status = _("No changes made.") #Does this work? No Idea?

    @form.action(_(u'label_cancel', default=u'Cancel'),
                 validator=null_validator,
                 name=u'cancel')
    def handle_cancel_action(self, action, data):
        IStatusMessage(self.request).addStatusMessage(_("Changes canceled."),
                                                      type="info")
        url = getMultiAdapter((self.context, self.request),
                              name='absolute_url')()
        self.request.response.redirect(url + '/plone_control_panel')
        return ''
        
    def _on_save(self, data=None):
        pass

    @form.action(_(u'Clear and rebuild catalog'), name=u'clear_and_rebuild')
    def clear_and_rebuild(self, action, data):
        CheckAuthenticator(self.request)
        if not self.available():
            self.status = _(u'text_not_allowed_manage_server',
                            default=u'You are not allowed to manage the Zope server.')        
            return
        catalog = getToolByName(self.context, 'openmember_catalog')
        catalog.clearFindAndRebuild()
        self.status = _(u'Clear and rebuild complete')
        
    @form.action(_(u'Reindex members'), name=u'revisit_members')
    def revisit_members(self, action, data):
        CheckAuthenticator(self.request)
        if not self.available():
            self.status = _(u'text_not_allowed_manage_server',
                            default=u'You are not allowed to manage the Zope server.')        
            return
        catalog = getToolByName(self.context, 'openmember_catalog')
        catalog.revisit_mirrored_objects()
        self.status = _(u'OpenMember catalog updated')
    
        
