#
# -*- coding: utf-8 -*-

"""
Module user_order_index_patch
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
The Sphinx extension to fix generating genindex.html for sortorder.

:copyright: © 2011-2015 by Suzumizaki-Kimitaka(about Additional works)
:license: 2-clause BSD.

This is the extension for `Sphinx <http://sphinx-doc.org/>`_.

This extension hacks :meth:`sphinx.environment.BuildEnvironment.create_index`.
No roles, directives, domains or etc. will be added.
"""

import sphinx.environment
import re
import string
from itertools import groupby
from sphinx.util import split_into

# ========================================================
# Hacking helpers
# ========================================================

# Store 'SortOrderBase' or its subclass defined 'sort_order.py'.
_sort_order_obj = None

# The Dictionary from the entry-name to sort-key(yomi-gana)
_name_to_yomi = None

def yomi_or_given(s):
    """Return sort-key or given string if not found one
    
    :param str_or_unicode s: the string refer to one of entry
    :rtype: str(Python 3) or unicode(Python 2)
    :return: the sort-key refer to the entry or just s if sort-key not found
    """
    s = s.lower()
    if s in _name_to_yomi:
        return _name_to_yomi[s]
    return s

def keyfunc(entry):
    """Sort the index entries

    :param composited entry: the element to sort. entry[0] is the name of it.
    :rtype: str(Python 3) or unicode(Python 2)
    :return: the comparable string to sort.

    Called by :func:`my_env_buildenv_create_index`

    - entry[0] is the name of the entry
    - entry[1] is a tuple
    
      - entry[1][0] is a list of the urls include filename, extension and anchor to the entry.
      - entry[1][1] is the dict of subwords.

    the entries are generated by
    :meth:`sphinx.environment.BuildEnvironment.add_entry` defined inside
    :meth:`sphinx.environment.BuildEnvironment.create_index`.
    (also means inside :meth:`my_env_buildenv_create_index`)
    """
    global _string_of_sort_order_to_yomi
    y = yomi_or_given(entry[0])
    r = _sort_order_obj.get_string_to_sort(y)
    return r

def keyfunc2(item):
    """Group the entries by letter.

    :param sortkey_content_pair item: the tuple (sortkey, content)
    :rtype: str(Python 3) or unicode(Python 2)
    :return: the group name in which (k, v) entry belongs.
    :seealso: :func:`keyfunc`

    Called by :func:`my_env_buildenv_create_index`.
    
    See note written in the document of :func:`keyfunc`.
    """
    k, v = item
    # hack: mutating the subitems dicts to a list in the keyfunc
    v[1] = sorted((si, se) for (si, (se, void)) in v[1].items())
    return _sort_order_obj.get_group_name(yomi_or_given(k))

# ========================================================
# Hacking function
# ========================================================

def my_env_buildenv_create_index(self, builder, group_entries=True,
                                 _fixre=re.compile(r'(.*) ([(][^()]*[)])')):
    """Hack function to alter sphinx.environment.BuildEnvironment.create_index

    In real, this function is almost same as
    :meth:`sphinx.environment.BuildEnvironment.create_index` except:
    
    - keyfunc and keyfunc2 are not inner function
    - setup _name_to_yomi dictionary at first
    - both python 2.7.6 and python 3.3 compatible
    """
    global _name_to_yomi
    _name_to_yomi = {}
    for k, v in self.domaindata['std']['objects'].items():
        if isinstance(v[1], tuple):
            _name_to_yomi[k[1]] = v[1][1]
    new = {}
    
    def add_entry(word, subword, link=True, dic=new):
        entry = dic.get(word)
        if not entry:
            dic[word] = entry = [[], {}]
        if subword:
            add_entry(subword, '', link=link, dic=entry[1])
        elif link:
            try:
                uri = builder.get_relative_uri('genindex', fn) + '#' + tid
            except NoUri:
                pass
            else:
                entry[0].append((main, uri))

    for fn, entries in self.indexentries.items():
        # new entry types must be listed in directives/other.py!
        for type, value, tid, main in entries:
            try:
                if type == 'single':
                    try:
                        entry, subentry = split_into(2, 'single', value)
                    except ValueError:
                        entry, = split_into(1, 'single', value)
                        subentry = ''
                    add_entry(entry, subentry)
                elif type == 'pair':
                    first, second = split_into(2, 'pair', value)
                    add_entry(first, second)
                    add_entry(second, first)
                elif type == 'triple':
                    first, second, third = split_into(3, 'triple', value)
                    add_entry(first, second+' '+third)
                    add_entry(second, third+', '+first)
                    add_entry(third, first+' '+second)
                elif type == 'see':
                    first, second = split_into(2, 'see', value)
                    add_entry(first, _('see %s') % second, link=False)
                elif type == 'seealso':
                    first, second = split_into(2, 'see', value)
                    add_entry(first, _('see also %s') % second, link=False)
                else:
                    self.warn(fn, 'unknown index entry type %r' % type)
            except ValueError as err:
                self.warn(fn, '%r' % err)

    newlist = list(new.items())
    newlist.sort(key=keyfunc)

    if group_entries:
        # fixup entries: transform
        #   func() (in module foo)
        #   func() (in module bar)
        # into
        #   func()
        #     (in module foo)
        #     (in module bar)
        oldkey = ''
        oldsubitems = None
        i = 0
        while i < len(newlist):
            key, (targets, subitems) = newlist[i]
            # cannot move if it has subitems; structure gets too complex
            if not subitems:
                m = _fixre.match(key)
                if m:
                    if oldkey == m.group(1):
                        # prefixes match: add entry as subitem of the
                        # previous entry
                        oldsubitems.setdefault(m.group(2), [[], {}])[0].\
                                    extend(targets)
                        del newlist[i]
                        continue
                    oldkey = m.group(1)
                else:
                    oldkey = key
            oldsubitems = subitems
            i += 1

    return [(key, list(group))
            for (key, group) in groupby(newlist, keyfunc2)]

# ========================================================
# Initializing functions
# ========================================================

def determine_sort_order(app):
    """Determine sort order used in this module

    :param sphinx.config.Config cfg:
       you can give from Sphinx with :code:`app.config`,
       :code:`builder.config` etc.
    :rtype: None
    :return: None
    """
    global _sort_order_obj
    import sortorder
    _sort_order_obj = sortorder.get_sort_order(app.config)
    if _sort_order_obj.__class__ == sortorder.SortOrderLegacy:
        print ('user_ordered_index_patch.py: Using SortOrderLegacy.')

def setup(app):
    """Extend the Sphinx as we want, called from the Sphinx

    :param sphinx.application.Sphinx app: the object to add builder or something.
    """
    app.setup_extension('yogosyu')
    sphinx.environment.BuildEnvironment.create_index = my_env_buildenv_create_index
    app.connect('builder-inited', determine_sort_order)
    return {'version': '2.0.5', 'parallel_read_safe': False}
