# Copyright (c) 2015, The MITRE Corporation. All rights reserved.
# See LICENSE.txt for complete terms.

# stdlib
from StringIO import StringIO
from distutils.version import LooseVersion

# external
from lxml import etree

# internal
import stix
import stix.utils as utils
import stix.utils.parser as parser
import stix.ttp.malware_instance
from stix.ttp.malware_instance import MalwareInstance
import stix.bindings.extensions.malware.maec_4_1 as ext_binding


_MIN_PYTHON_MAEC_VERSION = '4.1.0.12'


class UnsupportedVersion(Exception):
    def __init__(self, message, expected, found):
        super(UnsupportedVersion, self).__init__(message)
        self.expected = expected
        self.found = found


def _check_maec_version():
    """Checks that the installed python-maec has a version greater than or
    equal to the minimum supported version.

    Note:
        We do this rather than having a python-maec dependency requirement
        listed in setup.py because MAEC is used as an extension to STIX and
        not a core component to STIX (like CybOX).

    Raises:
        ImportError: If python-maec is not installed.
        UnsupportedVersion: If the python-maec installation does not satisfy
            the version requirements.

    """
    import maec

    found = maec.__version__
    expected = _MIN_PYTHON_MAEC_VERSION

    if LooseVersion(found) >= LooseVersion(expected):
        return

    fmt = ("Unsupported python-maec version installed: '%s'. Minimum version "
           "is '%s'.")
    error = fmt % (found, expected)
    raise UnsupportedVersion(error, expected=expected, found=found)


try:
    # Check that the correct version of python-maec is installed.
    _check_maec_version()

    # Import maecPackage into global space
    from maec.package.package import Package as maecPackage

    _MAEC_INSTALLED = True
except ImportError:
    maecPackage, Package = None, None
    _MAEC_INSTALLED = False


def is_maec(obj):
    """Checks if the input object is python-maec object.

    Returns:
        True if python-maec is ins

    """
    if not _MAEC_INSTALLED:
        return False

    return isinstance(obj, maecPackage)


class MAECInstance(MalwareInstance):
    _binding = ext_binding
    _binding_class = _binding.MAEC4_1InstanceType
    _namespace = 'http://stix.mitre.org/extensions/Malware#MAEC4.1-1'
    _xml_ns_prefix = "stix-maec"
    _XSI_TYPE = "stix-maec:MAEC4.1InstanceType"
    _TAG_MAEC = "{%s}MAEC" % _namespace

    def __init__(self, maec=None):
        super(MAECInstance, self).__init__()
        self.__input_namespaces__ = {}
        self.__input_schemalocations__ = {}
        self.maec = maec
        
    @property
    def maec(self):
        return self._maec
    
    @maec.setter
    def maec(self, value):
        if value is None:
            self._maec = None
        elif _MAEC_INSTALLED and is_maec(value):
            self._maec = value
        elif utils.is_element(value) or utils.is_etree(value):
            tree = parser.get_etree(value)
            root = parser.get_etree_root(tree)
            self._parse_etree(root)
            self._maec = root
        else:
            error = (
                "Cannot set maec to '{0}'. Expected 'lxml.etree._Element' or "
                "'maec.package.package.Package'."
            )
            error = error.format(type(value))
            raise ValueError(error)

    def _parse_etree(self, root):
        node_tag = root.tag

        if node_tag != self._TAG_MAEC:
            self._cast_maec(root)

        self._collect_namespaces(root)
        self._collect_schemalocs(root)

    def _cast_maec(self, node):
        ns_maec = "http://maec.mitre.org/XMLSchema/maec-package-2"
        node_ns = etree.QName(node).namespace

        if node_ns == ns_maec:
            etree.register_namespace(self._xml_ns_prefix, self._namespace)
            node.tag = self._TAG_MAEC
        else:
            error = "Cannot set maec. Expected tag '{0}' found '{1}'."
            error = error.format(self._TAG_MAEC, node.tag)
            raise ValueError(error)

    def _collect_schemalocs(self, node):
        try:
            schemaloc = parser.get_schemaloc_pairs(node)
            self.__input_schemalocations__ = dict(schemaloc)
        except KeyError:
            self.__input_schemalocations__ = {}

    def _collect_namespaces(self, node):
        self.__input_namespaces__ = dict(node.nsmap.iteritems())

    @classmethod
    def from_obj(cls, obj, return_obj=None):
        if not obj:
            return None
        if not return_obj:
            return_obj = cls()
        
        super(MAECInstance, cls).from_obj(obj, return_obj)

        if _MAEC_INSTALLED:
            return_obj.maec = maecPackage.from_obj(obj.MAEC)
        else:
            return_obj.maec = obj.MAEC

        return return_obj
    
    def to_obj(self, return_obj=None, ns_info=None):
        if not return_obj:
            return_obj = self._binding_class()
            
        super(MAECInstance, self).to_obj(return_obj=return_obj, ns_info=ns_info)

        if _MAEC_INSTALLED and isinstance(self.maec, maecPackage):
            return_obj.MAEC = self.maec.to_obj(ns_info=ns_info)
        else:
            return_obj.MAEC = self.maec

        return return_obj

    @classmethod
    def _maec_from_dict(cls, d):
        if _MAEC_INSTALLED:
            return maecPackage.from_dict(d)

        raise ValueError(
            "Unable to parse 'maec' value in dictionary. Please "
            "install python-maec to parse dictionary value."
        )

    @classmethod
    def from_dict(cls, d, return_obj=None):
        if not d:
            return None

        if not return_obj:
            return_obj = cls()
            
        super(MAECInstance, cls).from_dict(d, return_obj)

        with utils.ignored(KeyError):
            maec = d['maec']
            if isinstance(maec, dict):
                return_obj.maec = cls._maec_from_dict(maec)
            else:
                parser = stix.utils.parser.get_xml_parser()
                return_obj.maec = etree.parse(StringIO(maec), parser=parser)
        
        return return_obj

    def to_dict(self):
        d = super(MAECInstance, self).to_dict()

        if self.maec is not None:
            if _MAEC_INSTALLED and isinstance(self.maec, maecPackage):
                d['maec'] = self.maec.to_dict()
            else:
                d['maec'] = etree.tostring(self.maec)

        return d

# Register the extension
stix.ttp.malware_instance.add_extension(MAECInstance)
