from unittest import TestCase

from collections import namedtuple
from rx import Observable
from cyclotron.rx_runner import run, Driver


class RunnerTestCase(TestCase):

    def test_run_1snk(self):
        ''' Creates a cycle with one sink driver.
        '''
        MainDrivers = namedtuple('MainDrivers', ['drv1'])
        MainSink = namedtuple('MainSink', ['drv1'])
        test_values = []

        def drv1(sink):
            sink.values.subscribe(lambda i: test_values.append(i))
            return None
        Drv1Sink = namedtuple('Drv1Sink', ['values'])
        Drv1Driver = Driver(call=drv1, sink=Drv1Sink)

        def main(sources):
            val = Observable.from_([1, 2, 3])
            return MainSink(drv1=Drv1Sink(values=val))

        drivers = MainDrivers(drv1=Drv1Driver)
        run(main, drivers)

        self.assertEqual(3, len(test_values))
        self.assertEqual(1, test_values[0])
        self.assertEqual(2, test_values[1])
        self.assertEqual(3, test_values[2])

    def test_run_1srcsnk(self):
        ''' Creates a cycle with one sink/source driver.
        '''
        MainDrivers = namedtuple('MainDrivers', ['drv1'])
        MainSource = namedtuple('MainSource', ['drv1'])
        MainSink = namedtuple('MainSink', ['drv1'])
        test_values = []

        Drv1Source = namedtuple('Drv1Source', ['counter'])
        Drv1Sink = namedtuple('Drv1Sink', ['values'])

        def drv1(sink):
            sink.values.subscribe(lambda i: test_values.append(i))
            counter_stream = Observable.from_([1, 2, 3])
            return Drv1Source(counter=counter_stream)
        Drv1Driver = Driver(call=drv1, sink=Drv1Sink)

        def main(sources):
            val = sources.drv1.counter
            return MainSink(drv1=Drv1Sink(values=val))

        drivers = MainDrivers(drv1=Drv1Driver)
        run(main, drivers, MainSource)

        self.assertEqual(3, len(test_values))
        self.assertEqual(1, test_values[0])
        self.assertEqual(2, test_values[1])
        self.assertEqual(3, test_values[2])

    def test_run_1src_1snk(self):
        ''' Creates a cycle with one sink driver and one source driver.
        '''
        test_values = []
        Drv1Sink = namedtuple('Drv1Sink', ['values'])

        def drv1(sink):
            sink.values.subscribe(lambda i: test_values.append(i))
            return None
        Drv1Driver = Driver(call=drv1, sink=Drv1Sink)

        Drv2Sink = namedtuple('Drv1Sink', [])
        Drv2Source = namedtuple('Drv2Source', ['counter'])

        def drv2(sinks):
            counter_stream = Observable.from_([1, 2, 3])
            return Drv2Source(counter=counter_stream)
        Drv2Driver = Driver(call=drv2, sink=Drv2Sink)

        MainDrivers = namedtuple('MainDrivers', ['drv1', 'drv2'])
        MainSource = namedtuple('MainSource', ['drv1', 'drv2'])
        MainSink = namedtuple('MainSink', ['drv1'])

        def main(sources):
            val = sources.drv2.counter
            return MainSink(drv1=Drv1Sink(values=val))

        drivers = MainDrivers(drv1=Drv1Driver, drv2=Drv2Driver)
        run(main, drivers, MainSource)

        self.assertEqual(3, len(test_values))
        self.assertEqual(1, test_values[0])
        self.assertEqual(2, test_values[1])
        self.assertEqual(3, test_values[2])
