from .comment import IComment
from .commentanswer import ICommentAnswer
from .conclusion import IConclusion
from AccessControl import getSecurityManager
from Acquisition import aq_inner
from Acquisition import aq_parent
from esdrt.content import MessageFactory as _
from esdrt.content.subscriptions.interfaces import INotificationSubscriptions
from esdrt.content.subscriptions.interfaces import INotificationUnsubscriptions
from five import grok
from plone import api
from plone.app.contentlisting.interfaces import IContentListing
from plone.app.dexterity.behaviors.discussion import IAllowDiscussion
from plone.app.textfield import RichText
from plone.app.textfield.value import RichTextValue
from plone.directives import dexterity
from plone.directives import form
from plone.directives.form import default_value
from plone.namedfile.interfaces import IImageScaleTraversable
from plone.z3cform.interfaces import IWrappedForm
from Products.CMFCore.utils import getToolByName
from Products.CMFEditions import CMFEditionsMessageFactory as _CMFE
from Products.CMFPlone.utils import safe_unicode
from Products.statusmessages.interfaces import IStatusMessage
from time import time
from z3c.form import button
from z3c.form import field
from z3c.form import interfaces
from z3c.form.browser.checkbox import CheckBoxFieldWidget
from z3c.form.browser.radio import RadioFieldWidget
from z3c.form.form import Form
from z3c.form.interfaces import ActionExecutionError
from zope import schema
from zope.browsermenu.menu import getMenu
from zope.component import getUtility
from zope.container.interfaces import IObjectAddedEvent
from zope.i18n import translate
from zope.interface import alsoProvides
from zope.interface import Invalid
from zope.lifecycleevent.interfaces import IObjectModifiedEvent
from zope.schema.interfaces import IVocabularyFactory

import datetime

HIDDEN_ACTIONS = [
    '/content_status_history',
    '/placeful_workflow_configuration',
]


def hidden(menuitem):
    for action in HIDDEN_ACTIONS:
        if menuitem.get('action').endswith(action):
            return True
    return False


# Interface class; used to define content-type schema.
class IObservation(form.Schema, IImageScaleTraversable):
    """
    New review observation
    """

    text = RichText(
        title=_(u'Short description'),
        description=_(u''),
        required=True,
        )

    country = schema.Choice(
        title=_(u"Country"),
        vocabulary='esdrt.content.eea_member_states',
        required=True,
    )

    year = schema.TextLine(
        title=_(u'Inventory year'),
        required=True
    )

    form.widget(gas=CheckBoxFieldWidget)
    gas = schema.List(
        title=_(u"Gas"),
        value_type=schema.Choice(
            vocabulary='esdrt.content.gas',
            ),
        required=True,
    )

    review_year = schema.Int(
        title=_(u'Review year'),
        required=True,
    )

    fuel = schema.Choice(
        title=_(u"Fuel"),
        vocabulary='esdrt.content.fuel',
        required=False,
    )

    # ghg_source_category = schema.Choice(
    #     title=_(u"CRF category group"),
    #     vocabulary='esdrt.content.ghg_source_category',
    #     required=True,
    # )

    # ghg_source_sectors = schema.Choice(
    #     title=_(u"CRF Sector"),
    #     vocabulary='esdrt.content.ghg_source_sectors',
    #     required=True,
    # )

    ms_key_catagory = schema.Bool(
        title=_(u"MS key category"),
    )

    eu_key_catagory = schema.Bool(
        title=_(u"EU key category"),
    )

    crf_code = schema.Choice(
        title=_(u"CRF category codes"),
        vocabulary='esdrt.content.crf_code',
        required=True,
    )

    form.widget(highlight=CheckBoxFieldWidget)
    highlight = schema.List(
        title=_(u"Highlight"),
        value_type=schema.Choice(
            vocabulary='esdrt.content.highlight',
            ),
        required=False,
    )

    form.widget(parameter=RadioFieldWidget)
    parameter = schema.Choice(
        title=_(u"Parameter"),
        vocabulary='esdrt.content.parameter',
        required=True,
    )

    form.write_permission(closing_comments='cmf.ManagePortal')
    closing_comments = RichText(
        title=_(u'Finish request comments'),
        required=False,
    )

    form.write_permission(closing_deny_comments='cmf.ManagePortal')
    closing_deny_comments = RichText(
        title=_(u'Finish deny comments'),
        required=False,
    )

    form.write_permission(closing_comments_phase2='cmf.ManagePortal')
    closing_comments_phase2 = RichText(
        title=_(u'Finish request comments for phase 2'),
        required=False,
    )

    form.write_permission(closing_deny_comments_phase2='cmf.ManagePortal')
    closing_deny_comments_phase2 = RichText(
        title=_(u'Finish deny comments for phase 2'),
        required=False,
    )


@form.validator(field=IObservation['gas'])
def check_gas(value):
    if len(value) == 0:
        raise Invalid(u'You need to select at least one gas')


@form.validator(field=IObservation['crf_code'])
def check_crf_code(value):
    """ Check if the user is in one of the group of users
        allowed to add this category CRF Code observations
    """
    category = get_category_from_crf_code(value)
    user = api.user.get_current()
    groups = user.getGroups()
    valid = False
    for group in groups:
        if group.startswith('extranet-esd-ghginv-sr-%s-' % category):
            valid = True
        if group.startswith('extranet-esd-esdreview-reviewexp-%s-' % category):
            valid = True

    if not valid:
        raise Invalid(u'You are not allowed to add observations for this sector category')


@form.validator(field=IObservation['country'])
def check_country(value):
    user = api.user.get_current()
    groups = user.getGroups()
    valid = False
    for group in groups:
        if group.startswith('extranet-esd-ghginv-sr-') and \
            group.endswith('-%s' % value):
            valid = True
        if group.startswith('extranet-esd-esdreview-reviewexp-') and \
            group.endswith('-%s' % value):
            valid = True

    if not valid:
        raise Invalid(u'You are not allowed to add observations for this country')


@default_value(field=IObservation['review_year'])
def default_year(data):
    return datetime.datetime.now().year


@grok.subscribe(IObservation, IObjectAddedEvent)
@grok.subscribe(IObservation, IObjectModifiedEvent)
def add_observation(object, event):
    sector = safe_unicode(object.ghg_source_category_value())
    gas = safe_unicode(object.gas_value())
    inventory_year = safe_unicode(str(object.year))
    parameter = safe_unicode(object.parameter_value())
    object.title = u' '.join([sector, gas, inventory_year, parameter])


class Observation(dexterity.Container):
    grok.implements(IObservation)
    # Add your class methods and properties here

    def get_crf_code(self):
        """ stupid method to avoid name-clashes with the existing
        vocabularies when cataloging """
        return self.crf_code

    def get_ghg_source_sectors(self):
        """ stupid method to avoid name-clashes with the existing
        vocabularies when cataloging """
        return self.ghg_source_sectors_value()

    def get_highlight(self):
        """ stupid method to avoid name-clashes with the existing
        vocabularies when cataloging """
        return self.highlight

    def country_value(self):
        return self._vocabulary_value('esdrt.content.eea_member_states',
            self.country
        )

    def crf_code_value(self):
        return self._vocabulary_value('esdrt.content.crf_code',
            self.crf_code
        )

    def ghg_source_category_value(self):
        # XXX Provide a way to extract category from self.crf_code
        return get_category_from_crf_code(self.crf_code)

    def ghg_source_sectors_value(self):
        # XXX Provide a way to extract sector from self.crf_code
        return u'Sector Value'

    def parameter_value(self):
        return self._vocabulary_value('esdrt.content.parameter',
            self.parameter
        )

    def gas_value(self):
        gases = [self._vocabulary_value('esdrt.content.gas',
            g) for g in self.gas]

        return u', '.join(gases)

    def highlight_value(self):

        highlight = [self._vocabulary_value('esdrt.content.highlight',
            h) for h in self.highlight]
        return u', '.join(highlight)

    def finish_reason_value(self):
        return self._vocabulary_value('esdrt.content.finishobservationreasons',
            self.closing_reason
        )

    def finish_deny_reason_value(self):
        return self._vocabulary_value('esdrt.content.finishobservationdenyreasons',
            self.closing_deny_reason
        )

    def _vocabulary_value(self, vocabulary, term):
        vocab_factory = getUtility(IVocabularyFactory, name=vocabulary)
        vocabulary = vocab_factory(self)
        try:
            value = vocabulary.getTerm(term)
            return value.title
        except LookupError:
            return u''

    def get_status(self):
        return api.content.get_state(self)

    def can_draft_conclusions(self):
        questions = [v for v in self.values() if v.portal_type == 'Question']
        if len(questions) > 0:
            q = questions[0]
            return api.content.get_state(q) in [
                'phase1-draft',
                'phase1-drafted',
                'phase1-recalled-lr',
                'phase1-closed',
                'phase2-draft',
                'phase2-drafted',
                'phase2-recalled-lr',
                'phase2-closed',
            ]
        else:
            return True

    def can_close(self):
        if self.get_status() in ['phase1-pending', 'phase2-pending']:
            questions = [v for v in self.values() if v.portal_type == 'Question']
            if len(questions) > 0:
                for q in questions:
                    if api.content.get_state(q) not in ['phase1-closed', 'phase2-closed']:
                        return False
                return True

        return False

    def wf_location(self):
        if self.get_status() == 'phase1-draft':
            return 'Sector expert'
        elif self.get_status() == 'phase2-draft':
            return 'Review expert'
        elif self.get_status() == 'phase1-closed':
            return 'Quality expert'
        elif self.get_status() == 'phase2-closed':
            return 'Lead reviewer'
        elif self.get_status() == 'phase1-conclusions':
            return 'Sector expert'
        elif self.get_status() == 'phase2-conclusions':
            return 'Review expert'
        elif self.get_status() in ['phase1-conclusion-discussion', 'phase2-conclusion-discussion']:
            return 'Counterpart'
        elif self.get_status() == 'phase1-close-requested':
            return 'Quality expert'
        elif self.get_status() == 'phase2-close-requested':
            return 'Lead reviewer'
        else:
            questions = self.values()
            if questions:
                question = questions[0]
                state = api.content.get_state(question)
                if state in ['phase1-draft', 'phase1-closed']:
                    return 'Sector expert'
                if state in ['phase2-draft', 'phase2-closed']:
                    return 'Review expert'
                elif state in ['phase1-counterpart-comments', 'phase2-counterpart-comments']:
                    return 'Counterpart'
                elif state in ['phase1-drafted', 'phase1-recalled-lr']:
                    return 'Quality Expert'
                elif state in ['phase2-drafted', 'phase2-recalled-lr']:
                    return 'Lead reviewer'
                elif state in ['phase1-pending', 'phase1-answered',
                    'phase1-pending-answer-drafting', 'phase1-recalled-msa',
                    'phase2-pending', 'phase2-answered',
                    'phase2-pending-answer-drafting', 'phase2-recalled-msa']:
                    return 'Member state authority'
                elif state in ['phase1-pending-answer', 'phase2-pending-answer']:
                    return 'Member state expert'
            else:
                return 'Sector expert'

    def wf_status(self):
        if self.get_status() in ['phase1-draft', 'phase2-draft']:
            return ['Observation created', "observationIcon"]
        elif self.get_status() in ['phase1-closed', 'phase2-closed']:
            return ['Observation finished', "observationIcon"]
        elif self.get_status() in ['phase1-close-requested', 'phase2-close-requested']:
            return ['Observation finish requested', "observationIcon"]
        elif self.get_status() in ['phase1-conclusions', 'phase2-conclusions']:
            return ["Conclusion ongoing", "conclusionIcon"]
        elif self.get_status() in ['phase1-conclusion-discussion', 'phase2-conclusion-discussion']:
            return ["Counterparts comments requested", "conclusionIcon"]
        else:
            questions = self.values()
            if questions:
                question = questions[-1]
                state = api.content.get_state(question)
                if state in ['phase1-raft', 'phase2-raft']:
                    return ["Question drafted", "questionIcon"]
                elif state in ['phase1-counterpart-comments', 'phase2-counterpart-comments']:
                    return ["Counterpart's comments requested", "questionIcon"]
                elif state in ['phase1-answered', 'phase2-answered']:
                    return ['Pending question', "questionIcon"]
                elif state in ['phase1-pending', 'phase1-pending-answer', 'phase1-pending-answer-validation',
                    'phase1-validate-answer', 'phase1-recalled-msa',
                    'phase2-pending', 'phase2-pending-answer', 'phase2-pending-answer-validation',
                    'phase2-validate-answer', 'phase2-recalled-msa']:
                    return ['Open question', "questionIcon"]
                elif state in ['phase1-draft', 'phase1-ounterpart-comments',
                    'phase1-drafted', 'phase1-recalled-lr',
                    'phase2-draft', 'phase2-ounterpart-comments',
                    'phase2-drafted', 'phase2-recalled-lr']:
                    return ['Draft question', "questionIcon"]
                elif state in ['phase1-closed', 'phase2-closed']:
                    return ['Closed question', "questionIcon"]
            else:
                return ['Observation created', "observationIcon"]

        return ['Unknown', 'observationIcon']

    def observation_status(self):
        status = self.get_status()
        if status in ['phase1-draft', 'phase2-draft']:
            return 'draft'
        elif status in ['phase1-closed', 'phase2-closed']:
            return 'closed'
        elif status in ['phase1-close-requested', 'phase2-close-requested']:
            return 'open'
        elif status in ['phase1-conclusions', 'phase1-conclusion-discussion',
                        'phase2-conclusions', 'phase2-conclusion-discussion']:
            return 'conclusion'
        else:
            return 'open'

    def is_secretariat(self):
        user = api.user.get_current()
        return 'Manager' in user.getRoles()

    def get_author_name(self, userid):
        if userid:
            user = api.user.get(username=userid)
            return user.getProperty('fullname', userid)
        return userid

    def myHistory(self):
        observation_history = self.workflow_history.get('esd-review-workflow', [])
        observation_wf = []
        question_wf = []
        for item in observation_history:
            item['role'] = item['actor']
            item['object'] = 'observationIcon'
            item['author'] = self.get_author_name(item['actor'])
            if item['review_state'] == 'phase1-draft':
                item['state'] = 'Draft observation'
                item['role'] = "Sector expert"
                observation_wf.append(item)
            elif item['review_state'] == 'phase1-pending' and item['action'] == "phase1-approve":
                item['state'] = 'Pending'
                #Do not add
            elif item['review_state'] == 'phase1-pending' and item['action'] == "phase1-reopen":
                item['state'] = 'Observation reopened'
                item['role'] = "Sector expert"
                observation_wf.append(item)
            elif item['review_state'] == 'phase1-closed':
                item['state'] = 'Closed observation'
                item['role'] = "Sector expert"
                observation_wf.append(item)
            elif item['review_state'] == 'phase1-close-requested':
                item['state'] = 'Closure requested'
                item['role'] = "Sector expert"
                observation_wf.append(item)
            elif item['review_state'] == 'phase1-conclusions' and item['action'] == "phase1-deny-closure":
                item['state'] = 'Observation closure denied'
                item['role'] = "Sector expert"
                observation_wf.append(item)
            elif item['review_state'] == 'phase1-conclusion-discussion':
                item['state'] = 'Conclusion comments requested'
                item['role'] = "Sector expert"
                item['object'] = 'conclusionIcon'
                observation_wf.append(item)
            elif item['review_state'] == 'phase1-conclusions' and item['action'] == "phase1-finish-comments":
                item['state'] = 'Conclusion comments closed'
                item['role'] = "Sector expert"
                item['object'] = 'conclusionIcon'
                observation_wf.append(item)
            elif item['review_state'] == 'phase1-conclusions' and item['action'] == "phase1-draft-conclusions":
                item['state'] = 'Conclusion drafting'
                item['role'] = "Sector expert"
                item['object'] = 'conclusionIcon'
                observation_wf.append(item)
            elif item['review_state'] == 'phase2-draft':
                item['state'] = 'Draft observation'
                item['role'] = "Review expert"
                observation_wf.append(item)
            elif item['review_state'] == 'phase2-pending' and item['action'] == "phase2-approve":
                item['state'] = 'Pending'
                #Do not add
            elif item['review_state'] == 'phase2-pending' and item['action'] == "phase2-reopen":
                item['state'] = 'Observation reopened'
                item['role'] = "Review expert"
                observation_wf.append(item)
            elif item['review_state'] == 'phase2-closed':
                item['state'] = 'Closed observation'
                item['role'] = "Review expert"
                observation_wf.append(item)
            elif item['review_state'] == 'phase2-close-requested':
                item['state'] = 'Closure requested'
                item['role'] = "Review expert"
                observation_wf.append(item)
            elif item['review_state'] == 'phase2-conclusions' and item['action'] == "phase2-deny-closure":
                item['state'] = 'Observation closure denied'
                item['role'] = "Review expert"
                observation_wf.append(item)
            elif item['review_state'] == 'phase2-conclusion-discussion':
                item['state'] = 'Conclusion comments requested'
                item['role'] = "Review expert"
                item['object'] = 'conclusionIcon'
                observation_wf.append(item)
            elif item['review_state'] == 'phase2-conclusions' and item['action'] == "phase2-finish-comments":
                item['state'] = 'Conclusion comments closed'
                item['role'] = "Review expert"
                item['object'] = 'conclusionIcon'
                observation_wf.append(item)
            elif item['review_state'] == 'phase2-conclusions' and item['action'] == "phase2-draft-conclusions":
                item['state'] = 'Conclusion drafting'
                item['role'] = "Review expert"
                item['object'] = 'conclusionIcon'
                observation_wf.append(item)
            else:
                item['state'] = '*' + item['review_state'] + '*'
                observation_wf.append(item)

        history = list(observation_wf)
        questions = self.values()

        if questions:
            question = questions[0]

            question_history = question.workflow_history.get('esd-question-review-workflow', [])
            for item in question_history:
                item['role'] = item['actor']
                item['object'] = 'questionIcon'
                item['author'] = self.get_author_name(item['actor'])
                if item['review_state'] == 'phase1-draft' and item['action'] == None:
                    item['state'] = 'Draft question'
                    item['role'] = "Sector expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase1-counterpart-comments':
                    item['state'] = 'Requested counterparts comments'
                    item['role'] = "Sector expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase1-draft' and item['action'] =='phase1-send-comments':
                    item['state'] = 'Counterparts comments closed'
                    item['role'] = "Sector expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase1-drafted':
                    item['state'] = 'Sent to Quality expert'
                    item['role'] = "Sector expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase1-draft' and item['action'] =='phase1-recall-sre':
                    item['state'] = 'Question recalled'
                    item['role'] = "Sector expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase1-draft' and item['action'] =='phase1-redraft':
                    item['state'] = 'Question redrafted'
                    item['role'] = "Quality expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase1-pending' and item['action'] == 'phase1-approve-question':
                    item['state'] = 'Question approved and sent to MSA'
                    item['role'] = "Quality expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase1-recalled-lr':
                    item['state'] = 'Question recalled'
                    item['role'] = "Quality expert"
                elif item['review_state'] == 'phase1-answered':
                    item['state'] = 'Answer sent'
                    item['role'] = "Member state authority"
                    question_wf.append(item)
                elif item['review_state'] == 'phase1-expert-comments':
                    item['state'] = 'Member state expert comments requested'
                    item['role'] = "Member state authority"
                    question_wf.append(item)
                elif item['review_state'] == 'phase1-pending-answer-validation':
                    item['state'] = 'Member state expert comments closed'
                    item['role'] = "Member state authority"
                    question_wf.append(item)
                elif item['review_state'] == 'phase1-recalled-msa':
                    item['state'] = 'Answer recalled'
                    item['role'] = "Member state authority"
                    question_wf.append(item)
                elif item['action'] == 'phase1-validate-answer-msa':
                    item['state'] = 'Sector expert'
                    item['role'] = "Answer acknowledged"
                    question_wf.append(item)
                elif item['review_state'] == 'phase1-draft' and item['action'] == "phase1-reopen":
                    item['state'] = 'Reopened'
                    #Do not add
                elif item['review_state'] == 'phase1-closed':
                    item['state'] = 'Closed'
                    #Do not add
                elif item['review_state'] == 'phase2-draft' and item['action'] == "phase2-reopen":
                    item['state'] = 'Draft question'
                    item['role'] = "Review expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase2-counterpart-comments':
                    item['state'] = 'Requested counterparts comments'
                    item['role'] = "Review expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase2-draft' and item['action'] =='phase2-send-comments':
                    item['state'] = 'Counterparts comments closed'
                    item['role'] = "Review expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase2-drafted':
                    item['state'] = 'Sent to LR'
                    item['role'] = "Review expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase2-draft' and item['action'] =='phase2-recall-sre':
                    item['state'] = 'Question recalled'
                    item['role'] = "Review expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase2-draft' and item['action'] =='phase2-redraft':
                    item['state'] = 'Question redrafted'
                    item['role'] = "Review expert"
                    question_wf.append(item)
                elif item['review_state'] == 'phase2-pending' and item['action'] == 'phase2-approve-question':
                    item['state'] = 'Question approved and sent to MSA'
                    item['role'] = "Lead reviewer"
                    question_wf.append(item)
                elif item['review_state'] == 'phase2-recalled-lr':
                    item['state'] = 'Question recalled'
                    item['role'] = "Lead reviewer"
                elif item['review_state'] == 'phase2-answered':
                    item['state'] = 'Answer sent'
                    item['role'] = "Member state authority"
                    question_wf.append(item)
                elif item['review_state'] == 'phase1-expert-comments':
                    item['state'] = 'Member state expert comments requested'
                    item['role'] = "Member state authority"
                    question_wf.append(item)
                elif item['review_state'] == 'phase2-pending-answer-validation':
                    item['state'] = 'Member state expert comments closed'
                    item['role'] = "Member state authority"
                    question_wf.append(item)
                elif item['review_state'] == 'phase2-recalled-msa':
                    item['state'] = 'Answer recalled'
                    item['role'] = "Member state authority"
                    question_wf.append(item)
                elif item['action'] == 'phase2-validate-answer-msa':
                    item['state'] = 'Review expert'
                    item['role'] = "Answer acknowledged"
                    question_wf.append(item)
                elif item['review_state'] == 'phase2-draft' and item['action'] == "phase2-reopen":
                    item['state'] = 'Reopened'
                    #Do not add
                elif item['review_state'] == 'phase2-closed':
                    item['state'] = 'Closed'
                    #Do not add
                else:
                    item['state'] = '*' + item['review_state'] + '*'
                    item['role'] = item['actor']
                    question_wf.append(item)

            history = list(observation_wf) + list(question_wf)

        history.sort(key=lambda x: x["time"], reverse=False)
        return history

    def can_edit(self):
        sm = getSecurityManager()
        return sm.checkPermission('Modify portal content', self)

    def get_question(self):
        questions = [q for q in self.values() if q.portal_type == 'Question']

        if questions:
            question = questions[0]
            return question

    def observation_question_status(self):
        if self.get_status() != 'pending':
            return self.get_status()
        else:
            questions = self.values()
            if questions:
                question = questions[-1]
                state = api.content.get_state(question)
                return state
            else:
                return ""
# View class
# The view will automatically use a similarly named template in
# templates called observationview.pt .
# Template filenames should be all lower case.
# The view will render when you request a content object with this
# interface with "/@@view" appended unless specified otherwise
# using grok.name below.
# This will make this view the default view for your content-type

grok.templatedir('templates')


class AddForm(dexterity.AddForm):
    grok.name('esdrt.content.observation')
    grok.context(IObservation)
    grok.require('esdrt.content.AddObservation')

    def updateWidgets(self):
        super(AddForm, self).updateWidgets()
        self.fields['IDublinCore.title'].field.required = False
        self.widgets['IDublinCore.title'].mode = interfaces.HIDDEN_MODE
        self.widgets['IDublinCore.description'].mode = interfaces.HIDDEN_MODE
        self.groups = [g for g in self.groups if g.label == 'label_schema_default']

    def updateActions(self):
        super(AddForm, self).updateActions()
        self.actions['save'].title = u'Save Observation'
        self.actions['cancel'].title = u'Delete Observation'

        for k in self.actions.keys():
            self.actions[k].addClass('standardButton')


@grok.subscribe(IObservation, IObjectAddedEvent)
def add_question(context, event):
    """ When adding a question, go directly to
        'open' status on the observation
    """
    observation = context
    review_folder = aq_parent(observation)
    with api.env.adopt_roles(roles=['Manager']):
        if api.content.get_state(obj=review_folder) == 'ongoing-review-phase2':
            api.content.transition(obj=observation, transition='go-to-phase2')
            return
        elif api.content.get_state(obj=observation) == 'phase1-draft':
            api.content.transition(obj=observation, transition='phase1-approve')


class ObservationView(grok.View):
    grok.context(IObservation)
    grok.require('zope2.View')
    grok.name('view')

    def wf_info(self):
        context = aq_inner(self.context)
        wf = getToolByName(context, 'portal_workflow')
        comments = wf.getInfoFor(self.context,
            'comments', wf_id='esd-review-workflow')
        actor = wf.getInfoFor(self.context,
            'actor', wf_id='esd-review-workflow')
        tim = wf.getInfoFor(self.context,
            'time', wf_id='esd-review-workflow')
        return {'comments': comments, 'actor': actor, 'time': tim}

    def isManager(self):
        sm = getSecurityManager()
        context = aq_inner(self.context)
        return sm.checkPermission('Manage portal', context)

    def get_user_name(self, userid):
        # Check users roles
        country = self.context.country_value()
        sector = self.context.ghg_source_sectors_value()
        return ' - '.join([country, sector])

    def get_menu_actions(self):
        context = aq_inner(self.context)
        menu_items = getMenu(
            'plone_contentmenu_workflow',
            context,
            self.request
            )
        return [mitem for mitem in menu_items if not hidden(mitem)]

    def get_questions(self):
        context = aq_inner(self.context)
        items = []
        mtool = api.portal.get_tool('portal_membership')
        for item in context.values():
            if item.portal_type == 'Question' and \
                mtool.checkPermission('View', item):
                items.append(item)

        return IContentListing(items)

    @property
    def repo_tool(self):
        return getToolByName(self.context, "portal_repository")

    def getVersion(self, version):
        question = self.question()
        context = question.getFirstComment()
        if version == "current":
            return context
        else:
            return self.repo_tool.retrieve(context, int(version)).object

    def versionName(self, version):
        """
        Copied from @@history_view
        Translate the version name. This is needed to allow translation
        when `version` is the string 'current'.
        """
        return _CMFE(version)

    def versionTitle(self, version):
        version_name = self.versionName(version)

        return translate(
            _CMFE(u"version ${version}",
              mapping=dict(version=version_name)),
            context=self.request
        )

    def can_delete_observation(self):
        is_draft = api.content.get_state(self.context) == 'phase1-pending'
        questions = len([q for q in self.context.values() if q.portal_type == 'Question'])

        return is_draft and not questions

    def can_add_question(self):
        sm = getSecurityManager()
        questions = len([q for q in self.context.values() if q.portal_type == 'Question'])
        return sm.checkPermission('esdrt.content: Add Question', self) and not questions

    def can_edit(self):
        sm = getSecurityManager()
        return sm.checkPermission('Modify portal content', self.context)

    def get_conclusion(self):
        sm = getSecurityManager()
        conclusions = [c for c in self.context.values() if c.portal_type == 'Conclusion']
        if conclusions and sm.checkPermission('View', conclusions[0]):
            return conclusions[0]

        return None

    def get_conclusion_phase2(self):
        sm = getSecurityManager()
        conclusions = [c for c in self.context.values() if c.portal_type == 'ConclusionsPhase2']
        if conclusions and sm.checkPermission('View', conclusions[0]):
            return conclusions[0]

        return None

    def can_add_conclusion(self):
        sm = getSecurityManager()
        conclusion = self.get_conclusion()
        return sm.checkPermission('esdrt.content: Add Conclusion', self.context) and not conclusion

    def subscription_options(self):
        actions = []
        # actions.append(
        #     dict(
        #         url='/addsubscription',
        #         name=_(u'Add Subscription')
        #     )
        # )
        # actions.append(
        #     dict(
        #         url='/deletesubscription',
        #         name=_(u'Delete Subscription')
        #     )
        # )
        url = self.context.absolute_url()
        actions.append(
            dict(
                action='%s/unsubscribenotifications' % url,
                title=_(u'Unsubscribe from notifications')
            )
        )
        actions.append(
            dict(
                action='%s/deleteunsubscribenotifications' % url,
                title=_(u'Delete unsubscription from notifications')
            )
        )

        return actions

    def show_description(self):
        questions = self.get_questions()
        sm = getSecurityManager()
        if questions:
            question = questions[-1]
            return sm.checkPermission('View', question.getObject())
        else:
            user = api.user.get_current()
            userroles = api.user.get_roles(username=user.getId(),
                obj=self.context)
            if 'MSAuthority' in userroles or 'MSExpert' in userroles:
                return False

            return True

    def add_question_form(self):
        from plone.z3cform.interfaces import IWrappedForm
        form_instance = AddQuestionForm(self.context, self.request)
        alsoProvides(form_instance, IWrappedForm)
        return form_instance()

    #Question view
    def question(self):
        questions = self.get_questions()
        if questions:
            return questions[0].getObject()

    def get_chat(self):
        sm = getSecurityManager()
        question = self.question()
        if question:
            values = [v for v in question.values() if sm.checkPermission('View', v)]
            #return question.values()
            return values

    def actions(self):
        context = aq_inner(self.context)
        question = self.question()
        observation_menu_items = getMenu(
            'plone_contentmenu_workflow',
            context,
            self.request
            )
        menu_items = observation_menu_items
        if question:
            question_menu_items = getMenu(
                'plone_contentmenu_workflow',
                question,
                self.request
                )

            menu_items = question_menu_items + observation_menu_items
        return [mitem for mitem in menu_items if not hidden(mitem)]

    def get_user_name(self, userid, question=None):
        # check users
        if question is not None:
            country = self.context.country_value()
            sector = self.context.ghg_source_sectors_value()
            if question.portal_type == 'Comment':
                return ' - '.join([country, sector])
            elif question.portal_type == 'CommentAnswer':
                return ' - '.join([country, 'Authority'])

        if userid:
            user = api.user.get(username=userid)
            return user.getProperty('fullname', userid)
        return ''

    def can_add_comment(self):
        sm = getSecurityManager()
        question = self.question()
        if question:
            permission = sm.checkPermission('esdrt.content: Add Comment', question)
            questions = [q for q in question.values() if q.portal_type == 'Comment']
            answers = [q for q in question.values() if q.portal_type == 'CommentAnswer']
            obs_state = self.context.get_status()
            return permission and len(questions) == len(answers) and obs_state != 'phase1-closed'
        else:
            return False

    def can_add_answer(self):
        sm = getSecurityManager()
        question = self.question()
        if question:
            permission = sm.checkPermission('esdrt.content: Add CommentAnswer', question)
            questions = [q for q in question.values() if q.portal_type == 'Comment']
            answers = [q for q in question.values() if q.portal_type == 'CommentAnswer']
            return permission and len(questions) > len(answers)
        else:
            return False

    def add_answer_form(self):
        form_instance = AddAnswerForm(self.context, self.request)
        alsoProvides(form_instance, IWrappedForm)
        return form_instance()

    def add_no_answer_form(self):
        form_instance = AddNoAnswerForm(self.context, self.request)
        alsoProvides(form_instance, IWrappedForm)
        return form_instance()

    def can_see_comments(self):
        state = api.content.get_state(self.question())
        #return state in ['draft', 'counterpart-comments', 'drafted']
        return state in ['phase1-counterpart-comments']

    def add_comment_form(self):
        form_instance = AddCommentForm(self.context, self.request)
        alsoProvides(form_instance, IWrappedForm)
        return form_instance()

    def add_conclusion_form(self):
        form_instance = AddConclusionForm(self.context, self.request)
        alsoProvides(form_instance, IWrappedForm)
        return form_instance()

    def update(self):
        question = self.question()
        if question:
            context = question.getFirstComment()
            if context:
                if context.can_edit():
                    try:
                        history_metadata = self.repo_tool.getHistoryMetadata(context)
                    except:
                        history_metadata = None
                    if history_metadata:
                        retrieve = history_metadata.retrieve
                        getId = history_metadata.getVersionId
                        history = self.history = []
                        # Count backwards from most recent to least recent
                        for i in xrange(history_metadata.getLength(countPurged=False)-1, -1, -1):
                            version = retrieve(i, countPurged=False)['metadata'].copy()
                            version['version_id'] = getId(i, countPurged=False)
                            history.append(version)
                        dt = getToolByName(self.context, "portal_diff")

                        version1 = self.request.get("one", None)
                        version2 = self.request.get("two", None)

                        if version1 is None and version2 is None:
                            self.history.sort(lambda x,y: cmp(x.get('version_id', ''), y.get('version_id')), reverse=True)
                            version1 = self.history[-1].get('version_id', 'current')
                            if len(self.history) > 1:
                                version2 = self.history[-2].get('version_id', 'current')
                            else:
                                version2 = 'current'
                        elif version1 is None:
                            version1 = 'current'
                        elif version2 is None:
                            version2 = 'current'

                        self.request.set('one', version1)
                        self.request.set('two', version2)

                        changeset = dt.createChangeSet(
                                self.getVersion(version2),
                                self.getVersion(version1),
                                id1=self.versionTitle(version2),
                                id2=self.versionTitle(version1))
                        self.changes = [change for change in changeset.getDiffs()
                                      if not change.same]
    #def get_chat(self):
    #    question = self.question()
    #    if question:
    #        return question.get_questions()


class AddQuestionForm(Form):

    ignoreContext = True
    fields = field.Fields(IComment).select('text')

    @button.buttonAndHandler(_('Save question'))
    def create_question(self, action):
        context = aq_inner(self.context)
        text = self.request.form.get('form.widgets.text', '')
        transforms = getToolByName(context, 'portal_transforms')
        stream = transforms.convertTo('text/plain', text, mimetype='text/html')
        text = stream.getData().strip()
        if not text:
            raise ActionExecutionError(Invalid(u"Question text is empty"))

        qs = [item for item in self.context.values() if item.portal_type == 'Question']
        if qs:
            question = qs[0]
        else:
            q_id = self.context.invokeFactory(type_name='Question',
                id='question-1',
                title='Question 1',
            )
            question = self.context.get(q_id)

        id = str(int(time()))
        item_id = question.invokeFactory(
                type_name='Comment',
                id=id,
        )
        comment = question.get(item_id)
        comment.text = RichTextValue(text, 'text/html', 'text/html')

        #return self.request.response.redirect(comment.absolute_url())
        return self.request.response.redirect(self.context.absolute_url())

    def updateActions(self):
        super(AddQuestionForm, self).updateActions()
        for k in self.actions.keys():
            self.actions[k].addClass('standardButton')


    # def update(self):
    #     history_metadata = self.repo_tool.getHistoryMetadata(self.context)
    #     retrieve = history_metadata.retrieve
    #     getId = history_metadata.getVersionId
    #     history = self.history = []
    #     # Count backwards from most recent to least recent
    #     for i in xrange(history_metadata.getLength(countPurged=False)-1, -1, -1):
    #         version = retrieve(i, countPurged=False)['metadata'].copy()
    #         version['version_id'] = getId(i, countPurged=False)
    #         history.append(version)
    #     dt = getToolByName(self.context, "portal_diff")

    #     version1 = self.request.get("one", None)
    #     version2 = self.request.get("two", None)

    #     if version1 is None and version2 is None:
    #         self.history.sort(lambda x,y: cmp(x.get('version_id', ''), y.get('version_id')), reverse=True)
    #         version1 = self.history[-1].get('version_id', 'current')
    #         version2 = self.history[-2].get('version_id', 'current')
    #     elif version1 is None:
    #         version1 = 'current'
    #     elif version2 is None:
    #         version2 = 'current'

    #     self.request.set('one', version1)
    #     self.request.set('two', version2)

    #     changeset = dt.createChangeSet(
    #             self.getVersion(version2),
    #             self.getVersion(version1),
    #             id1=self.versionTitle(version2),
    #             id2=self.versionTitle(version1))
    #     self.changes = [change for change in changeset.getDiffs()
    #                   if not change.same]


class AddSubscription(grok.View):
    grok.context(IObservation)
    grok.require('zope2.View')

    def render(self):
        context = self.context
        user = api.user.get_current()
        ok = INotificationSubscriptions(context).add_notifications(user.getId())
        status = IStatusMessage(self.request)
        if ok:
            status.add(_(u'Subscription enabled'), type=u'info')
        else:
            status.add(_(u'Subscription already enabled'), type=u'info')
        return self.request.response.redirect(self.context.absolute_url())


class DeleteSubscription(grok.View):
    grok.context(IObservation)
    grok.require('zope2.View')

    def render(self):
        context = self.context
        user = api.user.get_current()
        ok = INotificationSubscriptions(context).del_notifications(user.getId())
        status = IStatusMessage(self.request)
        if ok:
            status.add(_(u'Correctly unsubscribed'), type=u'info')
        else:
            status.add(_(u'You were not subscribed'), type=u'info')
        return self.request.response.redirect(self.context.absolute_url())


class UnsubscribeNotifications(grok.View):
    grok.context(IObservation)
    grok.require('zope2.View')

    def render(self):
        context = self.context
        user = api.user.get_current()
        ok = INotificationUnsubscriptions(context).unsubscribe(user.getId())
        status = IStatusMessage(self.request)
        if ok:
            status.add(_(u'Correctly unsubscribed'), type=u'info')
        else:
            status.add(_(u'You were already unsubscribed'), type=u'info')
        return self.request.response.redirect(self.context.absolute_url())


class DeleteUnsubscribeNotifications(grok.View):
    grok.context(IObservation)
    grok.require('zope2.View')

    def render(self):
        context = self.context
        user = api.user.get_current()
        ok = INotificationUnsubscriptions(context).delete_unsubscribe(
            user.getId()
        )
        status = IStatusMessage(self.request)
        if ok:
            status.add(_(u'You will receive again notifications'),
                type=u'info')
        else:
            status.add(_(u'You were not in the unsubscription list'),
                type=u'info')
        return self.request.response.redirect(self.context.absolute_url())


class ModificationForm(dexterity.EditForm):
    grok.name('modifications')
    grok.context(IObservation)
    grok.require('cmf.ModifyPortalContent')

    def updateFields(self):
        super(ModificationForm, self).updateFields()

        user = api.user.get_current()
        roles = api.user.get_roles(username=user.getId(), obj=self.context)
        fields = []
        # XXX Needed? Edit rights are controlled by the WF
        if 'ReviewerPhase1' in roles or 'ReviewerPhase2' in roles:
            fields = [f for f in field.Fields(IObservation) if f not in [
                'country',
                'crf_code',
                'review_year',
                'technical_corrections',
                'closing_comments',
                'closing_reason',
                'closing_deny_comments',
                'closing_deny_reason',
                ]]
        elif 'QualityExpert' in roles or 'LeadReviewer' in roles:
            fields = ['text', 'highlight']

        self.fields = field.Fields(IObservation).select(*fields)
        self.groups = [g for g in self.groups if g.label == 'label_schema_default']
        if 'parameter' in fields:
            self.fields['parameter'].widgetFactory = RadioFieldWidget
        if 'highlight' in fields:
            self.fields['highlight'].widgetFactory = CheckBoxFieldWidget
        if 'gas' in fields:
            self.fields['gas'].widgetFactory = CheckBoxFieldWidget

    def updateActions(self):
        super(ModificationForm, self).updateActions()
        for k in self.actions.keys():
            self.actions[k].addClass('standardButton')


class AddAnswerForm(Form):

    ignoreContext = True
    fields = field.Fields(ICommentAnswer).select('text')

    @button.buttonAndHandler(_('Save answer'))
    def add_answer(self, action):
        text = self.request.form.get('form.widgets.text', '')
        transforms = getToolByName(self.context, 'portal_transforms')
        stream = transforms.convertTo('text/plain', text, mimetype='text/html')
        text = stream.getData().strip()
        if not text:
            raise ActionExecutionError(Invalid(u"Answer text is empty"))
        observation = aq_inner(self.context)
        questions = [q for q in observation.values() if q.portal_type == 'Question']
        if questions:
            context = questions[0]
        else:
            raise ActionExecutionError(Invalid(u"Invalid context"))
        id = str(int(time()))
        item_id = context.invokeFactory(
                type_name='CommentAnswer',
                id=id,
        )
        comment = context.get(item_id)
        comment.text = RichTextValue(text, 'text/html', 'text/html')

        return self.request.response.redirect(context.absolute_url())

    def updateActions(self):
        super(AddAnswerForm, self).updateActions()
        for k in self.actions.keys():
            self.actions[k].addClass('standardButton')


class AddAnswerAndRequestComments(grok.View):
    grok.context(IObservation)
    grok.name('add-answer-and-request-comments')
    grok.require('zope2.View')

    def render(self):
        observation = aq_inner(self.context)
        questions = [q for q in observation.values() if q.portal_type == 'Question']
        if questions:
            context = questions[0]
        else:
            raise ActionExecutionError(Invalid(u"Invalid context"))
        text = u'Help me drafting the answer'

        id = str(int(time()))
        item_id = context.invokeFactory(
                type_name='CommentAnswer',
                id=id,
        )
        comment = context.get(item_id)
        comment.text = RichTextValue(text, 'text/html', 'text/html')
        if api.content.get_state(context).startswith('phase1-'):
            action = 'phase1-assign-answerer'
        elif api.content.get_state(context).startswith('phase2-'):
            action = 'phase2-assign-answerer'
        else:
            raise ActionExecutionError(Invalid(u"Invalid context"))
        url = '%s/assign_answerer_form?workflow_action=%s&comment=%s' % (context.absolute_url(), action, item_id)
        return self.request.response.redirect(url)



class AddCommentForm(Form):

    ignoreContext = True
    fields = field.Fields(IComment).select('text')

    @button.buttonAndHandler(_('Add question'))
    def create_question(self, action):
        observation = aq_inner(self.context)
        questions = [q for q in observation.values() if q.portal_type == 'Question']
        if questions:
            context = questions[0]
        else:
            raise

        id = str(int(time()))
        item_id = context.invokeFactory(
                type_name='Comment',
                id=id,
        )
        text = self.request.form.get('form.widgets.text', '')
        comment = context.get(item_id)
        comment.text = RichTextValue(text, 'text/html', 'text/html')

        return self.request.response.redirect(context.absolute_url())

    def updateActions(self):
        super(AddCommentForm, self).updateActions()
        for k in self.actions.keys():
            self.actions[k].addClass('standardButton')


class AddConclusionForm(Form):
    ignoreContext = True
    fields = field.Fields(IConclusion).select('text', 'closing_reason')

    @button.buttonAndHandler(_('Add conclusion'))
    def create_conclusion(self, action):
        context = aq_inner(self.context)
        id = str(int(time()))
        item_id = context.invokeFactory(
                type_name='Conclusion',
                id=id,
        )
        text = self.request.form.get('form.widgets.text', '')
        comment = context.get(item_id)
        comment.text = RichTextValue(text, 'text/html', 'text/html')
        reason = self.request.form.get('form.widgets.closing_reason')
        comment.closing_reason = reason[0]
        adapted = IAllowDiscussion(comment)
        adapted.allow_discussion = True

        return self.request.response.redirect(context.absolute_url())

    def updateActions(self):
        super(AddConclusionForm, self).updateActions()
        for k in self.actions.keys():
            self.actions[k].addClass('standardButton')

def get_category_from_crf_code(value):
    """ get the CRF category this CRF Code matches
        According to the rules previously set. See #21438
    """
    return u'sector1'


class EditConclusionAndCloseComments(grok.View):
    grok.name('edit-conclusions-and-close-comments')
    grok.context(IObservation)
    grok.require('zope2.View')

    def update(self):
        # Some checks:
        waction = self.request.get('workflow_action')
        if waction != 'phase1-finish-comments':
                status = IStatusMessage(self.request)
                msg = _(u'There was an error, try again please')
                status.addStatusMessage(msg, "error")


    def render(self):
        # Execute the transition
        api.content.transition(obj=self.context,
            transition='phase1-finish-comments'
        )
        conclusions = [c for c in self.context.values() if c.portal_type == 'Conclusion']
        conclusion = conclusions[0]
        url = '%s/edit' % conclusion.absolute_url()
        return self.request.response.redirect(url)

class EditConclusionP2AndCloseComments(grok.View):
    grok.name('edit-conclusions-and-close-comments-phase2')
    grok.context(IObservation)
    grok.require('zope2.View')

    def update(self):
        # Some checks:
        waction = self.request.get('workflow_action')
        if waction != 'phase2-finish-comments':
                status = IStatusMessage(self.request)
                msg = _(u'There was an error, try again please')
                status.addStatusMessage(msg, "error")


    def render(self):
        # Execute the transition
        api.content.transition(obj=self.context,
            transition='phase2-finish-comments'
        )
        conclusions = [c for c in self.context.values() if c.portal_type == 'ConclusionsPhase2']
        conclusion = conclusions[0]
        url = '%s/edit' % conclusion.absolute_url()
        return self.request.response.redirect(url)

class EditHighlightsForm(dexterity.EditForm):
    grok.name('edit-highlights')
    grok.context(IObservation)
    grok.require('cmf.ModifyPortalContent')

    def updateFields(self):
        super(EditHighlightsForm, self).updateFields()
        self.fields = field.Fields(IObservation).select('highlight')
        self.fields['highlight'].widgetFactory = CheckBoxFieldWidget
        self.groups = [g for g in self.groups if g.label == 'label_schema_default']
