"""@package data_structure.case_node

File that contains the class CaseNode to structure all data loaded in BladePy.

"""

from PyQt4 import QtCore
from occ_modules.shape_properties import shape_colorlist
import configparser


class CaseNode(object):
    """
    Class to structure all data loaded in Core.BladePyCore.addCase().

    This is the structure to be read by the model set to tree list view. The case node contains information of
    name of the case, handle of the ais_shape, the name of the sub-shapes in the loaded igs and the graphics
    generate by TecplotReader Widget.
    The model used to represent a CaseNode is an case_model.CaseModel object.

    """
    def __init__(self, name, shape=None, subshape_names=None, plot_lists=None, parent=None):
        """
        The constructor of the class.

        A CaseNode object is created in Core.BladePyCore.addCase()

        @param name [str] Name of the case
        @param shape [Handle_AIS_InteractiveObject] Handles of AIS_Shape for shape controlling
        @param plot_lists [list] List of lists of graphics generated by TecplotReader
        @param subshape_names [list] List of strings of sub-shapes names.
        @param parent [CaseNode] Is a CaseNode object itself. It is the parent of the node.

        """

        super(CaseNode, self).__init__()
        self.subshape = []
        self._name = name
        self._tecplot_lists = plot_lists
        self._h_aisshape = shape
        self._supshape_names = subshape_names
        self._parent = parent
        case_count = 0

        try:
            case_count = parent.childCount()
        except AttributeError:
            pass
        # loading the default preferences_modules.
        list_settings = []

        last_settings = QtCore.QSettings("BladePy", "BladePy\MainApp\LastMainOptions".format(number=1))
        user_settings = [QtCore.QSettings("BladePy", "BladePy\MainApp\Options{number}".format(number=1))]

        list_settings.append(last_settings)
        list_settings.extend(user_settings)

        list_settings[1].beginGroup("shapes_settings")

        self._transformation = list_settings[1].value("default_transformation")
        self._transparency = float(list_settings[1].value("default_shape_transparency"))
        self._color = (int(list_settings[1].value("default_shape_color")) + case_count) % len(shape_colorlist)
        self._quality = float(list_settings[1].value("default_shape_quality"))

        try:
            for _ in self._h_aisshape:
                self.subshape.append([(list_settings[1].value("default_transformation")),
                                      float(list_settings[1].value("default_shape_transparency")),
                                      (int(list_settings[1].value("default_shape_color")) + case_count) % len(
                                          shape_colorlist),
                                      float(list_settings[1].value("default_shape_quality"))])

        except TypeError:
            pass

        list_settings[1].endGroup()

        # if list not empty, tecplot mode is standard
        
        if self._tecplot_lists:  
            self._tecplot_mode = "standard"
        else:
            self._tecplot_mode = "None"

        self._tecplot_visibility = "visible"
        self._tecplot_meanline_visibility = "visible"
        self._tecplot_bladeprofile_visibility = "visible"

        self._tecplot_savedstyle_list = []
        self._tecplot_bladeprofile_savedstyle_list = []
        self._tecplot_meanline_savedstyle_list = []

        self._children = []

        if parent is not None:
            parent.addChild(self)

    # Below is defining the methods for getting and setting protected members of the class.
    def shapeHAIS(self):
        """
        Method for retrieving the handle of AIS_Shape for the node
        
        @return [Handle_AIS_InteractiveObject] The AIS_shape Handle.
        """
        return self._h_aisshape

    def shapeTransformation(self):
        """
        Method for getting the transformation for the shape of this case

        @return [list] The list of coordinates.
        """
        return self._transformation

    def setShapeTransformation(self, transformation, coord):
        """
        Method for setting the transformation for the shape of this case

        @param transformation [int] The value of transformation
        @param coord [int] Coordinate of transformation application. Indexes: [x:0, y:1, z:2, axis:3, theta:4]

        @return None
        """

        self._transformation[coord] = transformation

    def shapeTransparency(self):
        """
        Method for getting the transparency for the shape of this case

        @return [float] The transparency
        """
        return self._transparency

    def setShapeTransparency(self, transparency):
        """
        Method for setting the transparency for the shape of this case

        @param transparency [float] The value of transparency
        @return None
        """
        self._transparency = transparency

    def shapeColor(self):
        """
        Method for getting the color for the shape of this case

        @return [int] The index of the color list of occ_modules.shapeproperties.shape_colorlist
        """
        return self._color

    def setShapeColor(self, color):
        """
        Method for setting the color for the shape of this case

        @param color [int] The index of the color list of occ_modules.shapeproperties.shape_colorlist
        @return None
        """
        self._color = color

    def shapeQuality(self):
        """
        Method for getting the quality for the shape of this case

        @return [float] The quality of the shape of this case
        """
        return self._quality

    def setShapeQuality(self, quality):
        """
        Method for setting the quality for the shape of this case

        @param quality [float] The quality of the shape of this case
        @return None
        """
        self._quality = quality

    def tecplotLists(self):
        """
        Method for getting the tecplot graphics of this case

        @return [list] List of tecplots lines
        """
        return self._tecplot_lists

    def tecplotSavedStyleList(self):
        """
        Method for getting the saved tecplot graphics line-styles of this case.

        This is useful for retrieving the last line-style when manipulating with Core.tecplot_setNeutral(),
        Core.tecplot_setVisibility(). The program must remember it when toggling the options to save user's preference.

        @return [list] List of tecplots lines
        """
        return self._tecplot_savedstyle_list

    def tecplotMeanLinesSavedStyleList(self):
        """
        Method for getting the saved tecplot graphics line-styles of this case.

        This is useful for retrieving the last line-style when manipulating with Core.tecplot_setNeutral(),
        Core.tecplot_setVisibility(). The program must remember it when toggling the options to save user's preference.

        @return [list] List of tecplots lines
        """
        return self._tecplot_meanline_savedstyle_list

    def tecplotBladeProfilesSavedStyleList(self):
        """
        Method for getting the saved tecplot graphics line-styles of this case.

        This is useful for retrieving the last line-style when manipulating with Core.tecplot_setNeutral(),
        Core.tecplot_setVisibility(). The program must remember it when toggling the options to save user's preference.

        @return [list] List of tecplots lines
        """
        return self._tecplot_bladeprofile_savedstyle_list

    def setTecplotSavedStyleList(self, tecplot_list):
        """
        Method for setting the saved tecplot graphics line-styles of this case.

        See datastructure.case_node.tecplotSavedStyleList()

        @param tecplot_list [list] List of tecplot graphics line-styles to save
        @return None
        """
        self._tecplot_savedstyle_list = tecplot_list

    def setTecplotMeanLinesSavedStyleList(self, tecplot_list):
        """
        Method for setting the saved tecplot graphics line-styles of MeanLines of this case.

        See datastructure.case_node.tecplotSavedStyleList()

        @param tecplot_list [list] List of tecplot graphics line-styles to save
        @return None
        """
        self._tecplot_meanline_savedstyle_list = tecplot_list

    def setTecplotBladeProfilesSavedStyleList(self, tecplot_list):
        """
        Method for setting the saved tecplot graphics line-styles of MeanLines of this case.

        See datastructure.case_node.tecplotSavedStyleList()

        @param tecplot_list [list] List of tecplot graphics line-styles to save
        @return None
        """
        self._tecplot_bladeprofile_savedstyle_list = tecplot_list

    def tecplotVisibility(self):
        """
        Method for getting the current state of visibility the tecplot graphics for this case

        @return [str] The state of the tecplot. Can be "visible" or "invisible"
        """
        return self._tecplot_visibility

    def tecplotBladeProfilesVisibility(self):
        """
        Method for getting the current state of visibility the tecplot graphics for this case

        @return [str] The state of the tecplot. Can be "visible" or "invisible"
        """
        return self._tecplot_bladeprofile_visibility

    def tecplotMeanLinesVisibility(self):
        """
        Method for getting the current state of visibility the tecplot graphics for this case

        @return [str] The state of the tecplot. Can be "visible" or "invisible"
        """
        return self._tecplot_meanline_visibility

    def setTecplotVisibility(self, visibility):
        """
        Method for setting a state for the tecplot graphics for this case

        @param visibility [str] The visibility for this case. Can be "visible" or "invisible"
        @return None
        """
        self._tecplot_visibility = visibility

    def setTecplotBladeProfilesVisibility(self, visibility):
        """
        Method for setting a state for the tecplot graphics for this case

        @param visibility [str] The visibility for this case. Can be "visible" or "invisible"
        @return None
        """
        self._tecplot_bladeprofile_visibility = visibility

    def setTecplotMeanLinesVisibility(self, visibility):
        """
        Method for setting a state for the tecplot graphics for this case

        @param visibility [str] The visibility for this case. Can be "visible" or "invisible"
        @return None
        """
        self._tecplot_meanline_visibility = visibility

    def tecplotIsVisible(self):
        """
         Method for verifying if tecplot is visible

         @return [bool] True if it is "visible" False if it is "invisible"
         """
        if self.tecplotVisibility() == "visible":
            return True
        else:
            return False

    def tecplotMeanLinesIsVisible(self):
        """
         Method for verifying if tecplot MeanLines is visible

         @return [bool] True if it is "visible" False if it is "invisible"
         """
        if self.tecplotMeanLinesVisibility() == "visible":
            return True
        else:
            return False

    def tecplotBladeProfilesIsVisible(self):
        """
         Method for verifying if tecplot BladeProfiles is visible

         @return [bool] True if it is "visible" False if it is "invisible"
         """
        if self.tecplotBladeProfilesVisibility() == "visible":
            return True
        else:
            return False

    def tecplotMode(self):
        """
        Method for getting the current state of mode the tecplot graphics for this case.

        The mode is whether all the lines in tecplot graphics are neutral - black - and dashed.

        @return [str] The state of mode of the tecplot. Can be "neutral" or "standard"
        """
    
        return self._tecplot_mode

    def setTecplotMode(self, mode):
        """
        Method for setting the current state of mode the tecplot graphics for this case. 
        
        @param mode [str] The mode for this case. Can be "neutral" or "standard"
        @return None
        """
        self._tecplot_mode = mode

    def tecplotIsNeutral(self):
        """
         Method for verifying if tecplot is neutral

         @return [bool] True if it is "neutral" False if it is "standard"
         """
        if self.tecplotMode() == "neutral":
            return True
        else:
            return False
    
    def addChild(self, child):
        """
        Methods required by model tree view of PyQt. Not necessary to observe this method.
        """
        self._children.append(child)

    def insertChild(self, position, child):
        """
        Methods required by model tree view of PyQt. Not necessary to observe this method.
        """
        if position < 0 or position > len(self._children):
            return False

        self._children.insert(position, child)
        child._parent = self
        return True

    def removeChild(self, position):
        """
        Methods required by model tree view of PyQt. Not necessary to observe this method.
        """
        if position < 0 or position > len(self._children):
            return False

        child = self._children.pop(position)
        child._parent = None

        return True

    def name(self):
        """
        Methods required by model tree view of PyQt. Not necessary to observe this method.
        """
        return self._name

    def setName(self, name):
        """
        Methods required by model tree view of PyQt. Not necessary to observe this method.
        """
        self._name = name

    def child(self, row):
        """
        Methods required by model tree view of PyQt. Not necessary to observe this method.
        """
        return self._children[row]

    def childCount(self):
        """
        Methods required by model tree view of PyQt. Not necessary to observe this method.
        """
        return len(self._children)

    def parent(self):
        """
        Methods required by model tree view of PyQt. Not necessary to observe this method.
        """
        return self._parent

    def row(self):
        """
        Methods required by model tree view of PyQt. Not necessary to observe this method.
        """
        if self._parent is not None:
            return self._parent._children.index(self)
