from __future__ import division # Makes it so 1/4 = 0.25 instead of zero


from phidl import Device, quickplot
import numpy as np
import phidl.geometry as pg
#==============================================================================
# We'll start by assuming we have a function waveguide() which already exists
# and makes us a simple waveguide
#==============================================================================

def waveguide(width = 10, height = 1):
    wg = Device('waveguide')
    wg.add_polygon([(0, 0), (width, 0), (width, height), (0, height)])
    wg.add_port(name = 'wgport1', midpoint = [0,height/2], width = height, orientation = 180)
    wg.add_port(name = 'wgport2', midpoint = [width,height/2], width = height, orientation = 0)
    return wg
   

#==============================================================================
# Create a blank device
#==============================================================================
# Create a new device ``d`` which will act as a blank canvas,
# and add a few waveguides to it to start out.  add_device() returns
# the referenced object you added, allowing you to manipulate it later
d = Device('MultiWaveguide')
wg1 = d.add_device(waveguide(width=10, height = 1))
wg2 = d.add_device(waveguide(width=12, height = 2))
wg3 = d.add_device(waveguide(width=14, height = 3))

quickplot(d)


#==============================================================================
# Creating polygons
#==============================================================================
# Create and add a polygon from separate lists of x points and y points
# e.g. [(x1, x2, x3, ...), (y1, y2, y3, ...)]
poly1 = d.add_polygon([[8,6,7,9],[6,8,9,5]])
# Alternatively, create and add a polygon from a list of points
# e.g. [(x1,y1), (x2,y2), (x3,y3), ...] using the same function
poly2 = d.add_polygon([(0, 0), (1, 1), (1, 3), (-3, 3)])

quickplot(d)


#==============================================================================
# Manipulating geometry 1 - Basic movement and rotation
#==============================================================================
# There are several actions we can take to move and rotate the geometry.  These
# actions include movement, rotation, and reflection.

wg1.move([10,4]) # Shift the second waveguide we created over by dx = 10, dy = 4
wg2.move(origin = [1,1], destination = [2,2]) # Shift the second waveguide over by dx = 1, dy = 1
wg3.move([1,1], [5,5], axis = 'y') # Shift the third waveguide over by dx = 0, dy = 4 (motion only along y-axis)

wg1.rotate(45) # Rotate the first waveguie by 45 degrees around (0,0)
wg2.rotate(30, center = [1,1]) # Rotate the second waveguide by 30 degrees around (1,1)

wg3.reflect(p1 = [1,1], p2 = [3,1]) # Reflects wg3 across the line formed by p1 and p2


#==============================================================================
# Manipulating geometry 2 - Properties
#==============================================================================
# Each Device and SubDevice object has several properties which can be used to learn
# information about the object (for instance where it's center coordinate is).  Several
# of these properties can actually be used to move the geometry by assigning them
# new values

wg1.bbox # Will return the bounding box of wg1 in terms of [(xmin, ymin), (xmax, ymax)]
wg1.xsize # Will return the width of wg1 in the x dimension
wg1.ysize # Will return the height of wg1 in the y dimension

wg1.center # Gives you the center coordinate of its bounding box
wg1.center = [4,4] # Shift wg1 such that the center coordinate of its bounding box is at (4,4)

wg2.xmax # Gives you the rightmost (+x) edge of the wg2 bounding box
wg2.xmax = 25 # Moves wg2 such that it's rightmost edge is at x = 25

wg3.ymin # Gives you the bottommost (+y) edge of the wg3 bounding box
wg3.ymin = -14 # Moves wg3 such that it's bottommost edge is at y = -14


quickplot(d)


#==============================================================================
# Manipulating geometry 3 - Smarter movement with ports
#==============================================================================
# All the waveguides we made have two ports: 'wgport1' and 'wgport2'  We can 
# use these names in place of (x,y) pairs.  For instance, if we want to move
# wg1 such that its port 'wgport1' rests on the origin, we do:
wg1.move(origin = 'wgport1', destination = [0,0])
# Alternatively, we can use the Port object itself in the same manner.  We can
# access the Port objects for any Device (or SubDevice) by calling device.ports,
# --which returns a Python dictionary--and accessing its value with the key
wg3.move(origin = wg3.ports['wgport1'], destination = [0,0])
# We can even move one port to another 
wg2.move(origin = wg2.ports['wgport1'], destination = wg3.ports['wgport2'])
# Several functions beyond just move() can take Ports as inputs
wg1.rotate(angle = -60, center = wg1.ports['wgport2'])
wg3.reflect(p1 = wg3.ports['wgport1'].midpoint, p2 = wg3.ports['wgport1'].midpoint + np.array([1,0]))

quickplot(d)


#==============================================================================
# Manipulating geometry 4 - Chaining commands
#==============================================================================
# Many of the functions in Device return the object they manipulate.  We can use
# this to chain commands in a single line. For instance this:
wg1.rotate(angle = 15, center = [0,0])
wg1.move([10,20])
# ... is equivalent to this expression
wg1.rotate(angle = 15, center = [0,0]).move([10,20])



#==============================================================================
# Connecting devices with connect()
#==============================================================================
d.connect(port = wg1.ports['wgport1'], destination = wg2.ports['wgport2'])
d.connect(port = wg3.ports['wgport2'], destination = wg2.ports['wgport1'])

quickplot(d)



#==============================================================================
# Adding ports
#==============================================================================
# Although our waveguides have ports, ``d`` itself does not -- it only draws
# the subports (ports of wg1, wg2, wg3) as a convience.  We need to add ports
# that we specifically want in our new device ``d``
d.add_port(port = wg1.ports['wgport2'], name = 1)
d.add_port(port = wg3.ports['wgport1'], name = 2)

quickplot(d)



#==============================================================================
# Taking things a level higher
#==============================================================================
# Now that we have our device ``d`` which is a multi-waveguide device, we
# can add references to that device in a new blank canvas we'll call ``dsquared``.
# We'll add two copies of ``d`` to dsquared, and shift one so we can see them both
dsquared = Device('MultiMultiWaveguide')
mwg1 = dsquared.add_device(d)
mwg2 = dsquared.add_device(d)
mwg2.move(destination = [10,10])

quickplot(dsquared)

# Like before, let's connect mwg1 and mwg2 together then offset them slightly
dsquared.connect(port = mwg1.ports[1], destination = mwg2.ports[2])
mwg2.move(destination = [30,30])

quickplot(dsquared)


#==============================================================================
# Routing
#==============================================================================
# Routing allows us to connect two ports which face each other with a smooth
# polygon.  Since we connected our two 
dsquared.route(port1 = mwg1.ports[1], port2 = mwg2.ports[2], path_type = 'sine', width_type = 'straight')
quickplot(dsquared)


#==============================================================================
# Adding text
#==============================================================================
# The function text() creates a Device, just like waveguide.  Use it and 
# manipulate it like any other Device
t = dsquared.add_device( pg.text('Hello\nworld!', size = 10, justify = 'center'))
t.move([0,40]).rotate(45)
quickplot(dsquared)


#==============================================================================
# Labeling
#==============================================================================
# This label will display in a GDS viewer, but will not be rendered
# or printed like the polygons created by the text()
dsquared.add_label('First label', mwg1.center)
dsquared.add_label('Second label', mwg2.center)


#==============================================================================
# Saving the file as a .gds
#==============================================================================
dsquared.write_gds('MultiMultiWaveguideTutorial.gds')

