"""Plot the result of sampling clusters

{{header}}
"""

# ? include "plot_header.template"
# ? from "plot_macros.template" import xdg_open with context

import numpy as np
import seaborn as sns
from matplotlib import pyplot as plt

from msmbuilder.io import load_trajs, load_generic

sns.set_style('ticks')
colors = sns.color_palette()

## Load
meta, ttrajs = load_trajs('ttrajs')
txx = np.concatenate(list(ttrajs.values()))
kmeans = load_generic('kmeans.pickl')

inds = load_generic("cluster-sample-inds.pickl")
coordinates = [
    np.asarray([ttrajs[traj_i][frame_i, :] for traj_i, frame_i in state_inds])
    for state_inds in inds
    ]


## Overlay sampled states on histogram
def plot_sampled_states(ax):
    ax.hexbin(txx[:, 0], txx[:, 1],
              cmap='magma_r',
              mincnt=1,
              bins='log',
              alpha=0.8,
              )

    # Show sampled points as scatter
    # Annotate cluster index
    for i, coo in enumerate(coordinates):
        plt.scatter(coo[:, 0], coo[:, 1], c=colors[i % 6], s=40)
        ax.text(kmeans.cluster_centers_[i, 0],
                kmeans.cluster_centers_[i, 1],
                "{}".format(i),
                ha='center',
                va='center',
                size=16,
                bbox=dict(
                    boxstyle='round',
                    fc='w',
                    ec="0.5",
                    alpha=0.9,
                ),
                zorder=10,
                )

    ax.set_xlabel("tIC 1", fontsize=16)
    ax.set_ylabel("tIC 2", fontsize=16)


## Render a script for loading in vmd
def load_in_vmd(dirname='cluster_samples'):
    k = len(inds[0])
    templ = [
        '# autogenerated by msmbuilder',
        '# open with `vmd -e load-cluster-samples.tcl`',
        '',
        '# Defaults',
        'mol default material Transparent',
        'mol default representation NewCartoon',
        '',
    ]
    for i in range(len(inds)):
        templ += [
            '# State {}'.format(i),
            'mol new top.pdb',
            'mol addfile {}/{}.xtc waitfor all'.format(dirname, i),
            'animate delete beg 0 end 0 top',
            'mol rename top State-{}'.format(i),
            'mol modcolor 0 top ColorID {}'.format(i),
            'mol drawframes top 0 0:{k}'.format(k=k),
            '',
        ]
    return '\n'.join(templ)


## Plot
fig, ax = plt.subplots(figsize=(7, 5))
plot_sampled_states(ax)
fig.tight_layout()
fig.savefig('cluster-samples.pdf')
# {{xdg_open('cluster-samples.pdf')}}

## Render vmd
with open('load-cluster-samples.tcl', 'w') as f:
    f.write(load_in_vmd())
