import requests


class WhatPulse:
    # WhatPulse uses different terms for `username`, this list is used to convert them all
    # to `name`
    NAME_SYNONYMS = ['name', 'account_name', 'user_name', 'username']

    class API:
        BASE = 'http://api.whatpulse.org/'
        PATHS = {
            'user': 'user.php',
            'team': 'team.php',
            'pulses': 'pulses.php'
        }

        def __getitem__(self, item):
            return WhatPulse.API.BASE + WhatPulse.API.PATHS[item]

    def __init__(self):
        self.http = requests.Session()
        self.paths = self.API()

    def request(self, url, params, object, return_json=False):
        params.setdefault('format', 'json')
        response = self.http.get(url, params=params)

        if return_json:
            # Return raw JSON data
            return response

        if object is not None:
            dictionary = response.json()
            if 'error' in dictionary.keys():
                raise WhatPulseError(dictionary['error'])

            return self._make_object(object, dictionary)

    def _make_object(self, object, dict):
        def _make_snake_case(value):
            # Converts the camel case provided by WhatPulse to snake case
            if len(value) == 0:
                return value

            word = ''
            for char in value:
                if word == '':
                    word += char.lower()
                elif word != '' and char.isupper():
                    word += '_'
                    word += char.lower()
                else:
                    word += char
            return word

        def _convert_to_int_float(value):
            # Tries to convert values to integers or floats for convenience
            if not isinstance(value, str):
                return value

            if value.isdigit():
                return int(value)

            try:
                return float(value)
            except ValueError:
                return value

        instance = object()

        for key, item in dict.items():
            key = _make_snake_case(key)
            item = _convert_to_int_float(item)

            if key == 'ranks':
                rank_obj = self._make_object(Rank, item)

                setattr(instance, 'ranks', rank_obj)

            elif key == 'computers':
                computers = []

                for computer_key, computer_item in item.items():
                    computer_obj = self._make_object(Computer, computer_item)
                    computers.append(computer_obj)

                setattr(instance, 'computers', computers)

            elif key == 'team':
                team_obj = self._make_object(Team, item)
                setattr(instance, 'team', team_obj)

            elif key == 'members':
                # TODO Make object for each member
                pass

            elif key in self.NAME_SYNONYMS:
                key = 'name'

            if not hasattr(instance, key):
                setattr(instance, key, item)

        return instance

    def get_user(self, userid):
        params = {'user': userid}
        user = self.request(self.paths['user'], params=params, object=User)
        return user

    def get_team(self, teamid, members=False):
        params = {'team': teamid}
        if members:
            params['members'] = 'yes'
        team = self.request(self.paths['team'], params=params, object=Team)
        return team

    def get_pulses(self, userid):
        params = {'user': userid}
        pulses = self.request(self.paths['pulses'], params=params, object=Pulse)

        pulse_objs = []
        for pulse, data in pulses.items():
            obj = self._make_object(Pulse, data)
            obj.pulse_id = int(pulse.split('-')[1])
            pulse_objs.append(obj)
        return pulse_objs


class User:
    def __repr__(self):
        return '<WhatPulse User %s>' % self.name


class Team:
    def __repr__(self):
        return '<WhatPulse Team %s>' % self.name


class Pulse:
    pass


class Computer:
    def __repr__(self):
        return '<WhatPulse Team %s>' % self.name


class Rank:
    def __repr__(self):
        return '<WhatPulse Ranks>'


class WhatPulseError(Exception):
    pass
