from __future__ import print_function

# Copyright (c) 2013, Roger Lew
# All rights reserved.

import unittest
import numpy as np
from numpy.testing import assert_array_almost_equal
import matplotlib.pyplot as plt

from undaqTools.misc.cdf import CDF, percentile

testdata = './data/normaltestdata' # space delimited ASCII of 10,000 random digits
                                   # from normal distribution with mean of -270 and
                                   # sd of 270
                               
class Test_percentile(unittest.TestCase):
    def setUp(self):
        global testdata
        with open(testdata,'rb') as f:
            self.data = map(float, f.read().split())
            
    def test0(self):
        r_bins = \
            np.array([ -1050.231     , -1043.33599147, -1036.44098295, -1029.54597442,
                       -1022.6509659 , -1015.75595737, -1008.86094885, -1001.96594033,
                        -995.0709318 ,  -988.17592327,  -981.28091475,  -974.38590622,
                        -967.4908977 ,  -960.59588918,  -953.70088065,  -946.80587212,
                        -939.9108636 ,  -933.01585507,  -926.12084655,  -919.22583803,
                        -912.3308295 ,  -905.43582097,  -898.54081245,  -891.64580392,
                        -884.7507954 ,  -877.85578687,  -870.96077835,  -864.06576982,
                        -857.1707613 ,  -850.27575278,  -843.38074425,  -836.48573573,
                        -829.5907272 ,  -822.69571867,  -815.80071015,  -808.90570163,
                        -802.0106931 ,  -795.11568457,  -788.22067605,  -781.32566752,
                        -774.430659  ,  -767.53565048,  -760.64064195,  -753.74563343,
                        -746.8506249 ,  -739.95561637,  -733.06060785,  -726.16559932,
                        -719.2705908 ,  -712.37558227,  -705.48057375,  -698.58556522,
                        -691.6905567 ,  -684.79554818,  -677.90053965,  -671.00553112,
                        -664.1105226 ,  -657.21551407,  -650.32050555,  -643.42549702,
                        -636.5304885 ,  -629.63547997,  -622.74047145,  -615.84546292,
                        -608.9504544 ,  -602.05544588,  -595.16043735,  -588.26542882,
                        -581.3704203 ,  -574.47541177,  -567.58040325,  -560.68539472,
                        -553.7903862 ,  -546.89537767,  -540.00036915,  -533.10536063,
                        -526.2103521 ,  -519.31534357,  -512.42033505,  -505.52532652,
                        -498.630318  ,  -491.73530947,  -484.84030095,  -477.94529242,
                        -471.0502839 ,  -464.15527537,  -457.26026685,  -450.36525832,
                        -443.4702498 ,  -436.57524127,  -429.68023275,  -422.78522422,
                        -415.8902157 ,  -408.99520717,  -402.10019865,  -395.20519012,
                        -388.3101816 ,  -381.41517307,  -374.52016455,  -367.62515602,
                        -360.7301475 ,  -353.83513897,  -346.94013045,  -340.04512192,
                        -333.1501134 ,  -326.25510487,  -319.36009635,  -312.46508782,
                        -305.5700793 ,  -298.67507077,  -291.78006225,  -284.88505372,
                        -277.9900452 ,  -271.09503667,  -264.20002815,  -257.30501962,
                        -250.4100111 ,  -243.51500257,  -236.61999405,  -229.72498552,
                        -222.829977  ,  -215.93496847,  -209.03995995,  -202.14495142,
                        -195.2499429 ,  -188.35493437,  -181.45992585,  -174.56491732,
                        -167.6699088 ,  -160.77490027,  -153.87989175,  -146.98488322,
                        -140.0898747 ,  -133.19486617,  -126.29985765,  -119.40484912,
                        -112.5098406 ,  -105.61483207,   -98.71982355,   -91.82481502,
                         -84.9298065 ,   -78.03479797,   -71.13978945,   -64.24478092,
                         -57.3497724 ,   -50.45476387,   -43.55975535,   -36.66474682,
                         -29.7697383 ,   -22.87472977,   -15.97972125,    -9.08471272,
                          -2.1897042 ,     4.70530433,    11.60031285,    18.49532138,
                          25.3903299 ,    32.28533843,    39.18034695,    46.07535548,
                          52.970364  ,    59.86537253,    66.76038105,    73.65538958,
                          80.5503981 ,    87.44540663,    94.34041515,   101.23542368,
                         108.1304322 ,   115.02544073,   121.92044925,   128.81545778,
                         135.7104663 ,   142.60547483,   149.50048335,   156.39549188,
                         163.2905004 ,   170.18550893,   177.08051745,   183.97552598,
                         190.8705345 ,   197.76554303,   204.66055155,   211.55556008,
                         218.4505686 ,   225.34557713,   232.24058565,   239.13559418,
                         246.0306027 ,   252.92561123,   259.82061975,   266.71562828,
                         273.6106368 ,   280.50564533,   287.40065385,   294.29566238,
                         301.1906709 ,   308.08567943,   314.98068795,   321.87569648,
                         328.770705  ])

        r_percentiles = \
            np.array([  0.00788452,  0.00860721,  0.00932989,  0.01005257,  0.01077526,
                        0.01149794,  0.01222062,  0.01294331,  0.01366599,  0.01438868,
                        0.01511136,  0.01949853,  0.02447029,  0.02685264,  0.029235  ,
                        0.03816535,  0.03870205,  0.03923875,  0.03977545,  0.04031214,
                        0.04084884,  0.04138554,  0.04192224,  0.04245893,  0.04299563,
                        0.04353233,  0.04406903,  0.04460572,  0.04514242,  0.04587917,
                        0.04666457,  0.04744996,  0.04823536,  0.04902075,  0.04980615,
                        0.05059154,  0.05137694,  0.05216234,  0.05515276,  0.05817619,
                        0.06144373,  0.06480917,  0.07465794,  0.08016102,  0.0870895 ,
                        0.08841823,  0.08974696,  0.09107569,  0.09240443,  0.09606674,
                        0.10644608,  0.11426144,  0.11800995,  0.12511504,  0.13906042,
                        0.14134499,  0.14362955,  0.1500741 ,  0.15956579,  0.17179116,
                        0.18368956,  0.18592559,  0.18816161,  0.19211337,  0.19841317,
                        0.2029537 ,  0.20541953,  0.20788537,  0.21266664,  0.22193011,
                        0.22927148,  0.23452336,  0.23801757,  0.24402634,  0.24829478,
                        0.2593486 ,  0.26444222,  0.26710829,  0.2786814 ,  0.28588673,
                        0.29055915,  0.30362949,  0.30988025,  0.31149286,  0.31310547,
                        0.31486499,  0.31998583,  0.33378933,  0.34527317,  0.36329378,
                        0.36957257,  0.37964864,  0.38529406,  0.38825826,  0.39021331,
                        0.39216836,  0.41376866,  0.42376066,  0.43533998,  0.460219  ,
                        0.47645825,  0.49932141,  0.51131487,  0.5139218 ,  0.52057234,
                        0.52577312,  0.53170762,  0.54289616,  0.5514161 ,  0.55749038,
                        0.56657762,  0.57084495,  0.5756384 ,  0.58497631,  0.59406697,
                        0.60124967,  0.61191264,  0.61861008,  0.63526535,  0.64147876,
                        0.64424147,  0.64812518,  0.65628559,  0.6611236 ,  0.66513402,
                        0.67247715,  0.6865401 ,  0.69855775,  0.70405424,  0.70993554,
                        0.71712121,  0.74659026,  0.75933937,  0.76285267,  0.77698097,
                        0.77843327,  0.77988556,  0.78203854,  0.79001124,  0.79668959,
                        0.80160361,  0.81324827,  0.81889569,  0.83359027,  0.8376273 ,
                        0.84045596,  0.84676012,  0.85137567,  0.86174448,  0.86334239,
                        0.86940656,  0.88236389,  0.88443058,  0.88731063,  0.89183958,
                        0.89734229,  0.90254643,  0.90947058,  0.91505731,  0.92255361,
                        0.93161672,  0.93531119,  0.93928433,  0.94184878,  0.94358849,
                        0.9453282 ,  0.94657215,  0.94779349,  0.94905589,  0.95096511,
                        0.95565876,  0.9591155 ,  0.96179936,  0.96485932,  0.96873016,
                        0.97047112,  0.9715957 ,  0.97272028,  0.97393729,  0.97516837,
                        0.97661306,  0.97872525,  0.97977577,  0.98043513,  0.98109449,
                        0.98175384,  0.98267924,  0.9861103 ,  0.9879409 ,  0.99216572,
                        0.99465729,  0.99529709,  0.99593689,  0.99657669,  0.99721649,
                        0.99777873,  0.99833404,  0.99888936,  0.99944468,  1.        ])
        u = self.data
        cdf = percentile(u[:200])
        
        np.testing.assert_array_almost_equal(r_bins, cdf.bin_edges)
        np.testing.assert_array_almost_equal(r_percentiles, cdf.percentiles)

    def test1_spec_numbins(self):
        r_bins = \
            np.array([ -1265.052    , -1244.1328598, -1223.2137196, -1202.2945794,
                       -1181.3754392, -1160.456299 , -1139.5371588, -1118.6180186,
                       -1097.6988784, -1076.7797382, -1055.860598 , -1034.9414578,
                       -1014.0223176,  -993.1031774,  -972.1840372,  -951.264897 ,
                        -930.3457568,  -909.4266166,  -888.5074764,  -867.5883362,
                        -846.669196 ,  -825.7500558,  -804.8309156,  -783.9117754,
                        -762.9926352,  -742.073495 ,  -721.1543548,  -700.2352146,
                        -679.3160744,  -658.3969342,  -637.477794 ,  -616.5586538,
                        -595.6395136,  -574.7203734,  -553.8012332,  -532.882093 ,
                        -511.9629528,  -491.0438126,  -470.1246724,  -449.2055322,
                        -428.286392 ,  -407.3672518,  -386.4481116,  -365.5289714,
                        -344.6098312,  -323.690691 ,  -302.7715508,  -281.8524106,
                        -260.9332704,  -240.0141302,  -219.09499  ,  -198.1758498,
                        -177.2567096,  -156.3375694,  -135.4184292,  -114.499289 ,
                         -93.5801488,   -72.6610086,   -51.7418684,   -30.8227282,
                          -9.903588 ,    11.0155522,    31.9346924,    52.8538326,
                          73.7729728,    94.692113 ,   115.6112532,   136.5303934,
                         157.4495336,   178.3686738,   199.287814 ,   220.2069542,
                         241.1260944,   262.0452346,   282.9643748,   303.883515 ,
                         324.8026552,   345.7217954,   366.6409356,   387.5600758,
                         408.479216 ,   429.3983562,   450.3174964,   471.2366366,
                         492.1557768,   513.074917 ,   533.9940572,   554.9131974,
                         575.8323376,   596.7514778,   617.670618 ,   638.5897582,
                         659.5088984,   680.4280386,   701.3471788,   722.266319 ,
                         743.1854592,   764.1045994,   785.0237396,   805.9428798,
                         826.86202  ])

        r_percentiles = \
            np.array([   1.52682325e-04,   2.94558006e-04,   6.16047975e-04,
                         7.02918092e-04,   7.73474112e-04,   8.21793025e-04,
                         8.70111939e-04,   1.10401110e-03,   1.34481409e-03,
                         1.83295542e-03,   2.47881609e-03,   3.68607187e-03,
                         5.15865394e-03,   6.73055980e-03,   8.61060936e-03,
                         1.04788857e-02,   1.23147839e-02,   1.56306574e-02,
                         1.87118249e-02,   2.12461678e-02,   2.56479572e-02,
                         3.05641053e-02,   3.62309399e-02,   4.08826236e-02,
                         4.93068431e-02,   5.73786087e-02,   6.70094435e-02,
                         7.80975453e-02,   8.85867316e-02,   9.97965662e-02,
                         1.13123710e-01,   1.25876664e-01,   1.42376196e-01,
                         1.62244613e-01,   1.83208974e-01,   2.05366050e-01,
                         2.29556330e-01,   2.53282959e-01,   2.77540635e-01,
                         3.05427136e-01,   3.36100492e-01,   3.63831154e-01,
                         3.94043118e-01,   4.25583501e-01,   4.55163960e-01,
                         4.86837208e-01,   5.20599620e-01,   5.48129193e-01,
                         5.78151849e-01,   6.08547876e-01,   6.40669188e-01,
                         6.70746937e-01,   7.01026286e-01,   7.25971707e-01,
                         7.51021605e-01,   7.77398045e-01,   8.00059380e-01,
                         8.22450308e-01,   8.40847440e-01,   8.57498407e-01,
                         8.75729509e-01,   8.91310252e-01,   9.05933655e-01,
                         9.18565943e-01,   9.29274795e-01,   9.40657581e-01,
                         9.48881056e-01,   9.57538817e-01,   9.63601434e-01,
                         9.70164957e-01,   9.75166523e-01,   9.79733128e-01,
                         9.83692163e-01,   9.86929642e-01,   9.89432271e-01,
                         9.91544580e-01,   9.93344271e-01,   9.94115203e-01,
                         9.95269177e-01,   9.96206415e-01,   9.97047263e-01,
                         9.97575959e-01,   9.98237416e-01,   9.98678170e-01,
                         9.98903131e-01,   9.99062383e-01,   9.99389219e-01,
                         9.99710548e-01,   9.99739859e-01,   9.99802006e-01,
                         9.99820441e-01,   9.99861218e-01,   9.99912599e-01,
                         9.99946445e-01,   9.99963921e-01,   9.99975528e-01,
                         9.99981646e-01,   9.99987764e-01,   9.99993882e-01,
                         1.00000000e+00])
        
        u = self.data
        cdf = percentile(u, numbins=100)

        np.testing.assert_array_almost_equal(r_bins, cdf.bin_edges)
        np.testing.assert_array_almost_equal(r_percentiles, cdf.percentiles)

class Test_percentile_bounds(unittest.TestCase):
    def setUp(self):
        global testdata
        with open(testdata,'rb') as f:
            self.data = map(float, f.read().split())
            
    def test_find_at_lower_bound(self):
        u = self.data
        cdf = percentile(u)
        self.assertAlmostEqual(cdf.find(0.0), -1265.052)

    def test_find_at_upper_bound(self):
        u = self.data
        cdf = percentile(u)
        self.assertAlmostEqual(cdf.find(1.0), 806.14998224278111)
        
    def test_find_out_of_bounds(self):
        u = self.data
        cdf = percentile(u)

        with self.assertRaises(ValueError):
            cdf.find(-0.1)

    def test_find_out_of_bounds2(self):
        u = self.data
        cdf = percentile(u)
        
        with self.assertRaises(ValueError):
            cdf.find(1.1)
            
class Test_cdf_find(unittest.TestCase):
    def setUp(self):
        global testdata
        with open(testdata,'rb') as f:
            self.data = map(float, f.read().split())

    def test0(self):
        u = self.data
        cdf = percentile(u)
        self.assertAlmostEqual(cdf.find(.5), -315.27959365994514)

class Test_cdf_plot(unittest.TestCase):
    def setUp(self):
        global testdata
        with open(testdata,'rb') as f:
            self.data = map(float, f.read().split())

    def test0(self):
        u = self.data
        cdf = percentile(u)
        import matplotlib.pyplot as plt

        fig = cdf.plot()
        fig.savefig('./output/cdf.png')
        plt.close('all')

class Test_cdf_repr(unittest.TestCase):
    def setUp(self):
        global testdata
        with open(testdata,'rb') as f:
            self.data = map(float, f.read().split())

    def test0(self):
        u = self.data
        cdf = percentile(u, numbins=100)
        cdf2 = eval(repr(cdf))

        assert_array_almost_equal(cdf.percentiles, cdf2.percentiles)
        assert_array_almost_equal(cdf.bin_edges, cdf2.bin_edges)

class Test_cdf_str(unittest.TestCase):
    def setUp(self):
        global testdata
        with open(testdata,'rb') as f:
            self.data = map(float, f.read().split())

    def test0(self):
        u = self.data
        cdf = percentile(u, numbins=100)
        #open('./data/cdftest0','wb').write(str(cdf))
        r = open('./data/cdftest0').read()
        self.assertEqual(str(cdf), r)

    def test1(self):
        """truncated bins"""
        u = self.data
        cdf = percentile(u)
        #open('./data/cdftest1','wb').write(str(cdf))
        r = open('./data/cdftest1').read()
        self.assertEqual(str(cdf), r)
        
def suite():
    return unittest.TestSuite((
            unittest.makeSuite(Test_percentile),
            unittest.makeSuite(Test_percentile_bounds),
            unittest.makeSuite(Test_cdf_find),
            unittest.makeSuite(Test_cdf_plot),
            unittest.makeSuite(Test_cdf_repr),
            unittest.makeSuite(Test_cdf_str),
                              ))

if __name__ == "__main__":
##    # build test data
##    u  = np.random.normal(-270, 270, size=(3*60*60,))
##    with open('./normaltestdata','wb') as f:
##        f.write(' '.join(['%.3f'%v for v in u]))
    
    # run tests
    runner = unittest.TextTestRunner()
    runner.run(suite()) 
