"""
XML to list and XML to dict parsers.

The code of the XmlListConfig and XmlDictConfig classes came from:
 - http://code.activestate.com/recipes/410469-xml-as-dictionary/
 - http://code.activestate.com/recipes/410469-xml-as-dictionary/#c2, and
 - http://code.activestate.com/recipes/410469-xml-as-dictionary/#c3
"""

try:
    from xml.etree import ElementTree
except:
    from elementtree import ElementTree


def xml_string_to_dict(text):
    """Parse the XML in ``text`` and return a dict."""
    tree = ElementTree.XML(text)
    return XmlDictConfig(tree)


class XmlListConfig(list):
    def __init__(self, aList):
        for element in aList:
            if len(element) > 0:
                # treat like dict
                if len(element) == 1 or element[0].tag != element[1].tag:
                    self.append(XmlDictConfig(element))
                    # treat like list
                elif element[0].tag == element[1].tag:
                    self.append(XmlListConfig(element))
            elif element.text:
                text = element.text.strip()
                if text:
                    self.append(text)


class XmlDictConfig(dict):
    """
    Example usage:

    >>> tree = ElementTree.parse('your_file.xml')
    >>> root = tree.getroot()
    >>> xmldict = XmlDictConfig(root)

    Or, if you want to use an XML string:

    >>> root = ElementTree.XML(xml_string)
    >>> xmldict = XmlDictConfig(root)

    And then use xmldict for what it is... a dict.
    """

    def __init__(self, parent_element):
        childrenNames = [child.tag for child in parent_element.getchildren()]

        if parent_element.items():  # attributes
            self.update(dict(parent_element.items()))
        for element in parent_element:
            if len(element) > 0:
                # treat like dict - we assume that if the first two tags
                # in a series are different, then they are all different.
                #print len(element), element[0].tag, element[1].tag
                if len(element) == 1 or element[0].tag != element[1].tag:
                    aDict = XmlDictConfig(element)
                # treat like list - we assume that if the first two tags
                # in a series are the same, then the rest are the same.
                else:
                    # here, we put the list in dictionary; the key is the
                    # tag name the list elements all share in common, and
                    # the value is the list itself
                    aDict = {element[0].tag: XmlListConfig(element)}
                # if the tag has attributes, add those to the dict
                if element.items():
                    aDict.update(dict(element.items()))

                if childrenNames.count(element.tag) > 1:
                    try:
                        currentValue = self[element.tag]
                        currentValue.append(aDict)
                        self.update({element.tag: currentValue})
                    except:
                        # the first of its kind, an empty list must be created
                        # aDict is written in [], i.e. it will be a list
                        self.update({element.tag: [aDict]})

                else:
                    self.update({element.tag: aDict})
            # this assumes that if you've got an attribute in a tag,
            # you won't be having any text. This may or may not be a
            # good idea -- time will tell. It works for the way we are
            # currently doing XML configuration files...
            elif element.items():
                self.update({element.tag: dict(element.items())})
            # finally, if there are no child tags and no attributes, extract
            # the text
            else:
                self.update({element.tag: element.text})
