#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os
from os.path import expanduser, join, isfile, isdir, dirname
from sys import platform
from shutil import copy
import tempfile
from datetime import datetime
from subprocess import call as runcmd

import myterm.color
from myterm.parser import arg, verb, alias, VerbParser

from ablog_cli import __version__
from ablog_cli.client import Client, manage_error
from ablog_cli.util import Table, first_value
from ablog_cli.config import Config

#for support of py2
import sys
if sys.version_info[0] > 2:
    # py3k
    pass
else:
    # py2
    import codecs
    import warnings
    def open(file, mode='r', buffering=-1, encoding=None,
             errors=None, newline=None, closefd=True, opener=None):
        if newline is not None:
            warnings.warn('newline is not supported in py2')
        if not closefd:
            warnings.warn('closefd is not supported in py2')
        if opener is not None:
            warnings.warn('opener is not supported in py2')
        return codecs.open(filename=file, mode=mode, encoding=encoding,
                    errors=errors, buffering=buffering)


DESCRIPTION = """client for ablog"""
AUTHOR = "Frederic Aoustin"
PROG = "ablog_cli"
VERSION = __version__

GETBOOL = {'false': False, 'true': True}

CONFIG = {}

TEMPLATE_POST= """
Title
=====

.. post:: %s
   :tags:
   :category:

todo ...
"""

def vi(data="", suffix='rst'):
    """manage edit data"""
    if "win" in platform:
        last_editor = 'notepad'
    else:
        last_editor = 'vi'
    editor = first_value(CONFIG.get('ABLOG_CLI_EDITOR',None), os.environ.get('EDITOR',None),last_editor)
    tmpfile = tempfile.NamedTemporaryFile(mode='w',delete=False, suffix=suffix) 
    tmpfile.close
    with open(tmpfile.name, 'wb') as f:
        f.write(data)
    runcmd([editor, tmpfile.name])
    return tmpfile.name


def pposts(url=None, user=None, password=None,cols=[], orderby=[], fmttime='%b %d, %Y', limit=None, formulas=[]):
    """list of post"""
    cli = Client(url, user, password)
    posts = Table()
    posts.add_columns(*cols)
    posts.append(*cli.post('ls', login=True))
    for formula in formulas:    
        posts.filter(formula)
    posts.apply_formula('date',"datetime.strptime(value,current['post_date_format']).strftime('%Y%m%d%H%M%S')")
    posts.orderby(*orderby)
    posts.apply_formula('date',"datetime.strptime(value,'%Y%m%d%H%M%S').strftime('"+fmttime+"')")
    posts.remove_columns(*[ col for col in posts.headers if col not in cols])
    if limit:
        if limit < 0:
            del posts[:limit]
        else:
            del posts[limit:]
    return posts


def check_arg(verb, arg, value):
    if arg == 'url':
        return first_value(value, CONFIG.get('ABLOG_CLI_URL',None), 'http://127.0.0.1:5000/api')
    if arg == 'user':
        return first_value(value, CONFIG.get('ABLOG_CLI_USER',None), 'guest')
    if arg == 'password':
        return first_value(value, CONFIG.get('ABLOG_CLI_PASSWORD',None), 'guest')
    if arg == 'noheader':
        return GETBOOL[first_value(value, CONFIG.get('ABLOG_CLI_NOHEADER',None), 'false')] 
    if arg == 'deep':
        return first_value(value, CONFIG.get('ABLOG_CLI_DEEP',None), 'false') 
    if arg == 'fmttime':
        return first_value(value, CONFIG.get('ABLOG_CLI_FMTTIME',None), '%b %d, %Y') 
    if arg == 'all':
        return GETBOOL[first_value(value, CONFIG.get('ABLOG_CLI_ALL',None), 'false')] 
    if verb == 'version':
        if arg == 'cols':
            return first_value(value, CONFIG.get('ABLOG_CLI_VERSION_COLS',None), 'module,version').split(',') 
        if arg == 'orderby':
            return first_value(value, CONFIG.get('ABLOG_CLI_VERSION_ORDERBY',None), 'module').split(',') 
    if verb == 'ls':
        if arg == 'cols':
            return first_value(value, CONFIG.get('ABLOG_CLI_LS_COLS',None), 'docname,title').split(',') 
        if arg == 'orderby':
            return first_value(value, CONFIG.get('ABLOG_CLI_LS_ORDERBY',None), 'docname').split(',') 
    if verb == 'tags':
        if arg == 'cols':
            return first_value(value, CONFIG.get('ABLOG_CLI_TAGS_COLS',None), 'tag,count,docname,title').split(',') 
        if arg == 'orderby':
            return first_value(value, CONFIG.get('ABLOG_CLI_TAGS_ORDERBY',None), 'tag,docname').split(',') 
    if verb == 'categorys':
        if arg == 'cols':
            return first_value(value, CONFIG.get('ABLOG_CLI_CATEGORYS_COLS',None), 'cat,count,docname,title').split(',') 
        if arg == 'orderby':
            return first_value(value, CONFIG.get('ABLOG_CLI_CATEGORYS_ORDERBY',None), 'cat,docname').split(',') 
    if verb == 'authors':
        if arg == 'cols':
            return first_value(value, CONFIG.get('ABLOG_CLI_AUTHORS_COLS',None), 'auth,count,docname,title').split(',') 
        if arg == 'orderby':
            return first_value(value, CONFIG.get('ABLOG_CLI_AUTHORS_ORDERBY',None), 'auth,docname').split(',') 
 
def arg_connect(func):
    arg(func, '-H', '--url', dest='url', type=str, default=None, check=check_arg,
        help="url of ablog site")
    return func

def arg_login(func):
    arg(func, '-u', '--user', dest='user', type=str, default=None, check=check_arg,
        help="user of ablog site")
    arg(func, '-p', '--password', dest='password', type=str, default=None, check=check_arg,
        help="password of ablog site")
    return func

def arg_print_table(func):
    arg(func, '--noheader', dest='noheader', choices = ['true', 'false'], type='choice', default=None, check=check_arg,
        help="no print header of result")
    arg(func, '--cols', dest='cols', type=str, default=None, check=check_arg,
        help="list of cols in print")
    arg(func, '--order-by', dest='orderby', type=str, default=None, check=check_arg,
        help="order table before print")
    arg(func, '--fmttime', dest='fmttime', type=str, default=None, check=check_arg,
        help="format for time")
    arg(func, '--limit', dest='limit', type=int, default=None,
        help="limit the number of rows")
    return func

@verb('version')
@arg_connect
@arg_print_table
@manage_error
def version(url, noheader=False, cols=[], orderby=[], fmttime='%b %d, %Y', limit=None):
    """get version of ablog"""
    cli = Client(url)
    versions = Table()
    versions.add_columns(*cols)
    versions.append(*cli.get('version'))
    versions.append({'module': 'ablog_cli', 'version': VERSION})
    versions.append({'module' : 'python local', 'version': '.'.join([str(a) for a in sys.version_info[0:3]])})
    versions.orderby(*orderby)
    versions.remove_columns(*[ col for col in versions.headers if col not in cols])
    if limit:
        if limit < 0:
            del versions[:limit]
        else:
            del versions[limit:]
    versions.pprint(noheader)

@verb('ls')
@arg_connect
@arg_login
@arg_print_table
@arg('--tag', dest='tag', type=str, default=None, help="add filter tag")
@arg('--category', dest='category', type=str, default=None, help="add filter category")
@arg('--auth', dest='author', type=str, default=None, help="add filter author")
@manage_error
def ls(url=None, user=None, password=None, noheader=False, cols=[], orderby=[], fmttime='%b %d, %Y', limit=None, tag=None, category=None, author=None):
    """list of post"""
    formula = []
    if tag: formula.append("'%s' in current['tags']" % tag)
    if category: formula.append("'%s' in current['category']" % category)
    if author: formula.append("'%s' in current['author']" % author)
    pposts(url, user, password, cols, orderby, fmttime, limit, formula).pprint(noheader)

@verb('tags', usage="%s tags [options] [name of tag]" % PROG)
@arg_connect
@arg_login
@arg_print_table
@manage_error
def tags(tag=None, url=None, user=None, password=None, noheader=False, cols=[], orderby=[], fmttime='%b %d, %Y', limit=None):
    """post list of tag or list of tag"""
    cli = Client(url, user, password)
    if tag:
        cols = [col for col in cols if col not in ('tag','count')]
        formula = ["'%s' in current['tags']" % tag,]
        pposts(url, user, password, cols, orderby, fmttime, limit, formula).pprint(noheader)
    else:
        tags = {}
        for post in cli.post('ls', login=True):
            for tag in post['tags']:
                if tag not in tags.keys():
                    tags[tag]=0
                tags[tag] = tags[tag]+1    
        table = Table()
        table.add_columns('tag','count')
        table.append(*[ [tag, tags[tag]] for tag in tags])
        table.orderby(*orderby)
        table.remove_columns(*[ col for col in table.headers if col not in cols])
        if limit:
            if limit < 0:
                del table[:limit]
            else:
                del table[limit:]
        table.pprint(noheader)

@verb('categorys', usage='%s categorys [options] [name of category]' % PROG)
@arg_connect
@arg_login
@arg_print_table
@manage_error
def categorys(cat=None, url=None, user=None, password=None, noheader=False, cols=[], orderby=[], fmttime='%b %d, %Y', limit=None):
    """post list of category or list of category"""
    cli = Client(url, user, password)
    if cat:
        cols = [col for col in cols if col not in ('cat','count')]
        formula = ["'%s' in current['category']" % cat,]
        pposts(url, user, password, cols, orderby, fmttime, limit, formula).pprint(noheader)
    else:
        cats = {}
        for post in cli.post('ls', login=True):
            for cat in post['category']:
                if cat not in cats.keys():
                    cats[cat]=0
                cats[cat] = cats[cat]+1    
        table = Table()
        table.add_columns('cat','count')
        table.append(*[ [cat, cats[cat]] for cat in cats])
        table.orderby(*orderby)
        table.remove_columns(*[ col for col in table.headers if col not in cols])
        if limit:
            if limit < 0:
                del table[:limit]
            else:
                del table[limit:]
        table.pprint(noheader)

@verb('authors', usage="%s authors [options] [name of author]" % PROG)
@arg_connect
@arg_login
@arg_print_table
@manage_error
def authors(auth=None, url=None, user=None, password=None, noheader=False, cols=[], orderby=[], fmttime='%b %d, %Y', limit=None):
    """post list of author or list of author"""
    cli = Client(url, user, password)
    if auth:
        cols = [col for col in cols if col not in ('auth','count')]
        formula = ["'%s' in current['author']" % auth,]
        pposts(url, user, password, cols, orderby, fmttime, limit, formula).pprint(noheader)
    else:
        auths = {}
        for post in cli.post('ls', login=True):
            for auth in post['author']:
                if auth not in auths.keys():
                    auths[auth]=0
                auths[auth] = auths[auth]+1    
        table = Table()
        table.add_columns('auth','count')
        table.append(*[ [auth, auths[auth]] for auth in auths])
        table.orderby(*orderby)
        table.remove_columns(*[ col for col in table.headers if col not in cols])
        if limit:
            if limit < 0:
                del table[:limit]
            else:
                del table[limit:]
        table.pprint(noheader)

@verb('build')
@arg_connect
@arg_login
@arg('--all', dest='all', choices = ['true', 'false'], type='choice', default=None, check=check_arg,
        help="build all posts")
@manage_error
def build(url=None, user=None, password=None, all='false'):
    """load build of ablog"""
    cli = Client(url, user, password)
    if cli.post('build', login=True, all=all):
        print('build is ok')

@verb('clean')
@arg_connect
@arg_login
@arg('--deep', dest='deep', choices = ['true', 'false'], type='choice', default=None, check=check_arg,
        help="clean all posts")
@manage_error
def clean(url=None, user=None, password=None, deep='false'):
    """load clean of ablog"""
    cli = Client(url, user, password)
    if cli.post('clean', login=True, deep=deep):
        print('clean is ok')

@verb('cat', usage="%s cat [options] docname" % PROG)
@arg_connect
@manage_error
def cat(docname=None, url=None):
    """display post on the screen"""
    cli = Client(url)
    print(cli.post('get', docname=docname).content)

@verb('edit', usage="%s edit [options] docname" % PROG)
@alias('vi')
@arg_connect
@arg_login
@manage_error
def edit(docname=None, url=None, user=None, password=None):
    """edit post in with EDITOR variable"""
    editor = first_value(CONFIG.get('ABLOG_CLI_EDITOR',None), os.environ.get('EDITOR',None),'vi')
    cli = Client(url, user, password)
    res = cli.post('get',docname=docname)
    suffix = cli.post('conf', login=True, key='source_suffix')['source_suffix']
    file_name = vi(res.content, suffix)
    cli.post('edit/%s' % docname, login=True, data=open(file_name,'rb'))

@verb('new', usage="%s new [options] [docname]" % PROG)
@arg_connect
@arg_login
@manage_error
def new(docname=None, url=None, user=None, password=None):
    """edit new post in with EDITOR variable"""
    cli = Client(url, user, password)
    if not docname:
        docname = datetime.strftime(datetime.now(), '%Y%m%d%H%M%S')
    dte = datetime.strftime(datetime.now(), cli.post('conf',login=True, key='post_date_format')['post_date_format'])
    encoding = cli.post('conf', login=True, key='source_encoding')['source_encoding']
    data = CONFIG.get('ABLOG_CLI_TEMPLATE_POST', TEMPLATE_POST) % dte    
    suffix = cli.post('conf', login=True, key='source_suffix')['source_suffix']
    file_name = vi(str.encode(data), suffix)
    cli.post('edit/%s' % docname, login=True, data=open(file_name,'rb'))

@verb('rm', usage="%s rm [options] docname" % PROG)
@arg_connect
@arg_login
@manage_error
def rm(docname=None, url=None, user=None, password=None):
    """delete post for ablog site"""
    cli = Client(url, user, password)
    if cli.post('rm', login=True, docname=docname):
        print('delete is ok')

@verb('upload', usage="%s upload [options] SRC DEST" % PROG)
@arg_connect
@arg_login
@manage_error
def upload(src=None, dst=None, url=None, user=None, password=None):
    """upload file in ablog site"""
    cli = Client(url, user, password)
    if cli.post('upload'+dst, login=True, data=open(src,'rb')):
        print('upload is ok')

@verb('conf', usage="%s conf [options] parameter" % PROG)
@arg_connect
@arg_login
@arg('--noheader', dest='noheader', choices = ['true', 'false'], type='choice', default=None, check=check_arg,
        help="no print header of result")
@manage_error
def conf(param=None, url=None, user=None, password=None, noheader=False):
    """get conf value of ablog"""
    cli = Client(url, user, password)
    if param:
        confs= Table()
        confs.add_columns('param','value')
        confs.append([param, cli.post('conf', login=True, key=param)[param]])
        confs.pprint(noheader)

@verb('reload')
@arg_connect
@arg_login
@manage_error
def reload(url=None, user=None, password=None):
    """reload index of ablog"""
    cli = Client(url, user, password)
    cli.post('reload', login=True)
    print('reload is ok')

def main():
    global CONFIG
    try:
        if not os.path.isfile(join(expanduser("~"),'.ablog-cli','conf.py')):
            if not os.path.isdir(join(expanduser("~"),'.ablog-cli')):
                os.mkdir(join(expanduser("~"),'.ablog-cli'))
            copy(join(dirname(__file__),'conf.py'), join(expanduser("~"),'.ablog-cli','conf.py'))
    except:
        pass
    CONFIG = Config()
    CONFIG.from_pyfile(join(expanduser("~"),'.ablog-cli','conf.py'), silent=True)
    CONFIG.from_env('ABLOG_CLI_')
    parser = VerbParser()
    parser.prog = PROG
    parser.version = "%s %s" % (PROG, VERSION)
    parser.description= DESCRIPTION
    parser.epilog = AUTHOR
    parser.parse_args()

if __name__ == '__main__':
    main() 
