# -*- coding: utf-8 -*-

from urllib import urlencode
from datetime import datetime, timedelta

from li_common.comunicacao import requisicao

from pagador import configuracoes, servicos

GATEWAY = 'mercadopago'


class TipoToken(object):
    authorization_code = 'authorization_code'
    refresh_token = 'refresh_token'


class InstalaMeioDePagamento(servicos.InstalaMeioDePagamento):
    campos = ['usuario', 'token', 'token_expiracao', 'codigo_autorizacao']

    def __init__(self, loja_id, dados):
        super(InstalaMeioDePagamento, self).__init__(loja_id, dados)
        parametros = self.cria_entidade_pagador('ParametrosDeContrato', loja_id=loja_id).obter_para(GATEWAY)
        self.client_id = parametros['client_id']
        self.client_secret = parametros['client_secret']
        headers = {
            'Accept': 'application/json',
        }
        self.conexao = self.obter_conexao(formato_envio=requisicao.Formato.form_urlencode, formato_resposta=requisicao.Formato.json, headers=headers)

    def montar_url_autorizacao(self):
        try:
            parametros_redirect = urlencode({'next_url': self.dados['next_url'], 'fase_atual': '2'})
        except KeyError:
            raise self.InstalacaoNaoFinalizada(u'Você precisa informar a url de redirecionamento na volta do MercadoPago na chave next_url do parâmetro dados.')
        dados = {
            'response_type': 'code',
            'client_id': self.client_id,
            'redirect_uri': '{}?{}'.format(configuracoes.INSTALAR_REDIRECT_URL_SEGURA.format(self.loja_id, GATEWAY), parametros_redirect)
        }
        return 'http://auth.mercadolivre.com.br/authorization?{}'.format(urlencode(dados))

    def obter_user_id(self, access_token):
        url = 'https://api.mercadolibre.com/users/me?access_token={}'.format(access_token)
        resposta = self.conexao.get(url)
        if resposta.sucesso:
            return resposta.conteudo['id']
        return None

    @property
    def _dados_instalacao(self):
        try:
            parametros_redirect = urlencode({'next_url': self.dados['next_url'], 'fase_atual': '2'})
        except KeyError:
            raise self.InstalacaoNaoFinalizada(u'Você precisa informar a url de redirecionamento na volta do MercadoPago na chave next_url do parâmetro dados.')
        return {
            'code': self.dados['code'],
            'grant_type': TipoToken.authorization_code,
            'client_id': self.client_id,
            'client_secret': self.client_secret,
            'redirect_uri': '{}?{}'.format(configuracoes.INSTALAR_REDIRECT_URL_SEGURA.format(self.loja_id, GATEWAY), parametros_redirect)
        }

    @property
    def _dados_atualizacao(self):
        return {
            'refresh_token': self.dados['codigo_autorizacao'],
            'grant_type': TipoToken.refresh_token,
            'client_id': self.client_id,
            'client_secret': self.client_secret,
        }

    def obter_dados(self):
        url = 'https://api.mercadolibre.com/oauth/token'
        tipo = self.dados.get('tipo', 'instalar')
        if self.dados.get('error', '') == 'access-denied':
            raise self.InstalacaoNaoFinalizada(u'A autorização foi cancelada no MercadoPago.')
        if tipo == 'instalar':
            dados = self._dados_instalacao
        else:
            dados = self._dados_atualizacao
        resposta = self.conexao.post(url, dados)
        if resposta.nao_autorizado:
            raise self.InstalacaoNaoFinalizada(u'A autorização foi cancelada no MercadoPago.')
        if resposta.sucesso:
            resultado = {
                'token': resposta.conteudo['access_token'],
                'token_expiracao': datetime.utcnow() + timedelta(seconds=resposta.conteudo['expires_in']),
                'codigo_autorizacao': resposta.conteudo['refresh_token']
            }
            if tipo == 'instalar':
                user_id = self.obter_user_id(resposta.conteudo['access_token'])
                resultado['usuario'] = user_id
            return resultado
        raise self.InstalacaoNaoFinalizada(u'Erro ao entrar em contato com o MercadoPago. Código: {}, Resposta: {}'.format(resposta.status_code, resposta.conteudo))

    def desinstalar(self, dados):
        url_base = 'https://api.mercadolibre.com/users/{}/applications/{}?access_token={}'
        url = url_base.format(dados['usuario'], self.client_id, dados['token'])
        resposta = self.conexao.delete(url)
        if resposta.sucesso:
            return u'Aplicação removida com sucesso'
        return u'Desinstalação foi feita com sucesso, mas a aplicação não foi removida do MercadoPago. Você precisará <a href="https://www.mercadopago.com/mlb/account/security/applications/connections" _target="blank">remover manualmente</a>.'


class AtualizadorAccessToken(object):
    def __init__(self):
        self.configuracao = None
        self.conexao = None
        self.tentativa = 1
        self.tentativa_maxima = 2

    def define_configuracao_e_conexao(self, servico):
        self.configuracao = servico.configuracao
        self.conexao = servico.conexao

    def deve_reenviar(self, resposta):
        self.tentativa += 1
        deve = 1 < self.tentativa <= self.tentativa_maxima and (
            resposta.conteudo.get('message', '') in ['expired_token', 'invalid_token'] or
            resposta.conteudo.get('error', '') == 'invalid_access_token'
        )
        if deve:
            self.atualiza_credenciais()
        return deve

    def atualiza_credenciais(self):
        self.configuracao.instalar({'fase_atual': '2', 'tipo': 'atualizar', 'codigo_autorizacao': self.configuracao.codigo_autorizacao})
        self.conexao.credenciador.configuracao = self.configuracao
        self.conexao.credenciador.atualiza_credenciais()


class Credenciador(servicos.Credenciador):
    def __init__(self, tipo=None, configuracao=None):
        super(Credenciador, self).__init__(tipo, configuracao)
        self.tipo = self.TipoAutenticacao.query_string
        self.access_token = None
        self.refresh_token = None
        self.token_expiracao = None
        self.atualiza_credenciais()

    def atualiza_credenciais(self):
        self.access_token = getattr(self.configuracao, 'token', '')
        self.refresh_token = getattr(self.configuracao, 'codigo_autorizacao', '')
        self.token_expiracao = getattr(self.configuracao, 'token_expiracao', datetime.utcnow())

    def obter_credenciais(self):
        return self.access_token


class EntregaPagamento(servicos.EntregaPagamento):
    def __init__(self, loja_id, plano_indice=1, dados=None):
        super(EntregaPagamento, self).__init__(loja_id, plano_indice, dados=dados)
        self.tem_malote = True
        self.faz_http = True
        self.conexao = self.obter_conexao()
        self.resposta = None
        self.url = 'https://api.mercadolibre.com/checkout/preferences'
        self.tentativa = 1
        self.tentativa_maxima = 2

    def define_credenciais(self):
        self.conexao.credenciador = Credenciador(configuracao=self.configuracao)

    def atualiza_credenciais(self):
        self.configuracao.instalar({'fase_atual': '2', 'tipo': 'atualizar', 'codigo_autorizacao': self.configuracao.codigo_autorizacao})
        self.conexao.credenciador.configuracao = self.configuracao
        self.conexao.credenciador.atualiza_credenciais()

    def envia_pagamento(self, tentativa=1):
        self.tentativa = tentativa
        if self.tentativa > 1:
            self.atualiza_credenciais()
        self.dados_enviados = {'tentativa': tentativa}
        self.dados_enviados.update(self.malote.to_dict())
        self.resposta = self.conexao.post(self.url, self.malote.to_dict())
        if self.resposta.nao_autenticado or self.resposta.nao_autorizado:
            self.reenviar = self.tentativa <= self.tentativa_maxima and (
                self.resposta.conteudo.get('message', '') in ['expired_token', 'invalid_token'] or
                self.resposta.conteudo.get('error', '') == 'invalid_access_token'
            )
            if self.reenviar:
                self.resultado = {'erro': 'autenticacao', 'reenviar': self.reenviar, 'tentativa': self.tentativa, 'pago': False}
                raise self.EnvioNaoRealizado(u'Autenticação da loja com o MercadoPago falhou.', self.loja_id, self.pedido.numero)

    def processa_dados_pagamento(self):
        self.resultado = self._processa_resposta()

    def _processa_resposta(self):
        status_code = self.resposta.status_code
        if self.resposta.erro_servidor:
            return {'mensagem': u'O servidor do MercadoPago está indisponível nesse momento.', 'status_code': status_code, 'pago': False}
        if self.resposta.nao_autenticado or self.resposta.nao_autorizado:
            return {'mensagem': u'Autenticação da loja com o MercadoPago Falhou. Contate o SAC da loja.', 'status_code': self.resposta.status_code, 'pago': False}
        if self.resposta.timeout:
            return {'mensagem': u'O servidor do MercadoPago não respondeu em tempo útil.', 'status_code': status_code, 'pago': False}
        if self.resposta.requisicao_invalida:
            servicos.logger.error({
                'loja_id': self.loja_id,
                'pedido_numero': self.pedido.numero,
                'dados_envio': self.dados_enviados,
                'erros': [self.resposta.conteudo['error'], self.resposta.conteudo['message']]
            })
            raise self.EnvioNaoRealizado(
                u'Dados inválidos enviados ao MercadoPago',
                self.loja_id,
                self.pedido.numero
            )

        if self.resposta.sucesso:
            if 'error' in self.resposta.conteudo:
                erros = self.resposta.conteudo['error']
                raise self.EnvioNaoRealizado(u'Ocorreram erros no envio dos dados para o MercadoPago', self.loja_id, self.pedido.numero, dados_envio=self.malote.to_dict(), erros=erros)
            return {'url': self.resposta.conteudo['init_point'], 'pago': False}


class SituacoesDePagamento(servicos.SituacoesDePagamento):
    DE_PARA = {
        'in_process': servicos.SituacaoPedido.SITUACAO_AGUARDANDO_PAGTO,
        'rejected': servicos.SituacaoPedido.SITUACAO_AGUARDANDO_PAGTO,
        'pending': servicos.SituacaoPedido.SITUACAO_PAGTO_EM_ANALISE,
        'approved': servicos.SituacaoPedido.SITUACAO_PEDIDO_PAGO,
        'in_mediation': servicos.SituacaoPedido.SITUACAO_PAGTO_EM_DISPUTA,
        'refunded': servicos.SituacaoPedido.SITUACAO_PAGTO_DEVOLVIDO,
        'cancelled': servicos.SituacaoPedido.SITUACAO_PEDIDO_CANCELADO,
        'charged_back': servicos.SituacaoPedido.SITUACAO_PAGTO_CHARGEBACK
    }


class RegistraResultado(servicos.RegistraResultado):
    def __init__(self, loja_id, dados=None):
        super(RegistraResultado, self).__init__(loja_id, dados)
        self.redirect_para = dados.get('next_url', None)

    def monta_dados_pagamento(self):
        try:
            self.dados_pagamento['identificador_id'] = self.dados['preference_id']
            self.pedido_numero = self.dados['referencia']
            status_pagamento = self.dados.get('collection_status', 'pending')
            self.situacao_pedido = SituacoesDePagamento.do_tipo(status_pagamento)
            pedido = self.cria_entidade_pagador('Pedido', numero=self.pedido_numero, loja_id=self.configuracao.loja_id)
            if pedido and pedido.situacao_id not in [servicos.SituacaoPedido.SITUACAO_PEDIDO_EFETUADO]:
                self.situacao_pedido = None
                self.resultado = {'resultado': status_pagamento, 'fatal': pedido.situacao_id == servicos.SituacaoPedido.SITUACAO_PEDIDO_CANCELADO, 'pago': self.situacao_pedido in [servicos.SituacaoPedido.SITUACAO_PEDIDO_PAGO, servicos.SituacaoPedido.SITUACAO_PAGTO_EM_ANALISE]}
                return
            self.resultado = {'resultado': status_pagamento, 'pago': self.situacao_pedido in [servicos.SituacaoPedido.SITUACAO_PEDIDO_PAGO, servicos.SituacaoPedido.SITUACAO_PAGTO_EM_ANALISE]}
        except KeyError:
            self.resultado = {'resultado': 'cancelado', 'pago': False}


class Retorno(object):
    pagamento = 'payment'
    ordem_pagamento = 'merchant_order'

    def __init__(self, dados):
        self.dados_retorno = {}
        self.topico = dados.get('topic', None) 
        self.eh_pagamento = self.topico == self.pagamento
        self.eh_ordem_pagamento = self.topico == self.ordem_pagamento
        self.valido = 'id' in dados and 'topic' in dados and (self.eh_ordem_pagamento or self.eh_pagamento)
        self.chave = 'collection' if self.eh_pagamento else 'payments'

    def recebe_dados_de_retorno(self, dados):
        self.dados_retorno = dados

    @property
    def dados(self):
        try:
            if self.eh_pagamento:
                return self.dados_retorno[self.chave]
            if self.eh_ordem_pagamento:
                    return self.dados_retorno[self.chave][0]
        except (IndexError, KeyError):
            return {}


class RegistraNotificacao(servicos.RegistraResultado):
    def __init__(self, loja_id, dados=None):
        super(RegistraNotificacao, self).__init__(loja_id, dados)
        self.conexao = self.obter_conexao()
        self.retorno = Retorno(dados)
        self.faz_http = True
        self.tentativa = 1
        self.tentativa_maxima = 2

    def define_credenciais(self):
        self.conexao.credenciador = Credenciador(configuracao=self.configuracao)

    def monta_dados_pagamento(self):
        if self.resposta and self.resposta.sucesso:
            self.retorno.recebe_dados_de_retorno(self.resposta.conteudo)
            if not self.retorno.valido:
                self.resultado = {'resultado': 'erro', 'status_code': self.resposta.status_code, 'conteudo': self.resposta.conteudo}
                return
            self.pedido_numero = self.dados.get('referencia', None) or self.retorno.dados['external_reference']
            self.dados_pagamento = {
                'transacao_id': self.retorno.dados['id']
            }
            if 'total_paid_amount' in self.retorno.dados:
                self.dados_pagamento['valor_pago'] = self.retorno.dados['total_paid_amount']
            self.situacao_pedido = SituacoesDePagamento.do_tipo(self.retorno.dados.get('status', ''))
            self.resultado = {'resultado': 'OK'}
        elif self.resposta and (self.resposta.nao_autenticado or self.resposta.nao_autorizado):
            #TODO: isso está repetido na classe entrega... fazer um composition
            self.reenviar = self.tentativa <= self.tentativa_maxima and (
                self.resposta.conteudo.get('message', '') in ['expired_token', 'invalid_token'] or
                self.resposta.conteudo.get('error', '') == 'invalid_access_token'
            )
            #TODO: trocar isso para diparar uma exceção e tratar no Pagamento, como é na entrega
            if self.reenviar:
                self.reenviando()
            else:
                self.resultado = {'resultado': 'nao autorizado', 'conteudo': self.resposta.conteudo}
        elif self.resposta:
            self.resultado = {'resultado': 'erro', 'status_code': self.resposta.status_code, 'conteudo': self.resposta.conteudo}
        else:
            self.resultado = {'resultado': 'erro', 'status_code': 500, 'conteudo': {'mensagem': u'MercadoPago não retornou uma resposta válida'}}

    def reenviando(self):
        self.tentativa += 1
        self.obtem_informacoes_pagamento()
        self.monta_dados_pagamento()

    #TODO: isso está repetido na classe entrega... fazer um composition
    def atualiza_credenciais(self):
        self.configuracao.instalar({'fase_atual': '2', 'tipo': 'atualizar', 'codigo_autorizacao': self.configuracao.codigo_autorizacao})
        self.conexao.credenciador.configuracao = self.configuracao
        self.conexao.credenciador.atualiza_credenciais()

    def obtem_informacoes_pagamento(self):
        if not self.retorno.valido:
            return
        if self.tentativa > 1:
            self.atualiza_credenciais()
        self.resposta = self.conexao.get(self.url)

    @property
    def url(self):
        if not self.retorno.valido:
            return ''
        if self.retorno.topico == Retorno.pagamento:
            return 'https://api.mercadolibre.com/collections/notifications/{}'.format(self.dados['id'])
        if self.retorno.topico == Retorno.ordem_pagamento:
            return self.dados.get('resource', 'https://api.mercadolibre.com/merchant_orders/{}'.format(self.dados['id']))


class AtualizaTransacoes(servicos.AtualizaTransacoes):
    def __init__(self, loja_id, dados):
        super(AtualizaTransacoes, self).__init__(loja_id, dados)
        self.url = 'https://api.mercadopago.com/collections/search'
        self.conexao = self.obter_conexao(formato_envio=requisicao.Formato.querystring)
        self.atualizador_credenciais = AtualizadorAccessToken()

    def define_credenciais(self):
        self.conexao.credenciador = Credenciador(configuracao=self.configuracao)

    def _gera_dados_envio(self):
        initial_date = '{}T00:00:00Z'.format(self.dados['data_inicial'])
        final_date = '{}T23:59:59Z'.format(self.dados['data_final'])
        return {
            'criteria': 'desc',
            'sort': 'date_created',
            'range': 'date_created',
            'limit': 1000,
            'begin_date': initial_date,
            'end_date': final_date
        }

    def consulta_transacoes(self):
        self.atualizador_credenciais.define_configuracao_e_conexao(self)
        self._obtem_resposta()

    def _obtem_resposta(self):
        self.dados_enviados = self._gera_dados_envio()
        self.resposta = self.conexao.get(self.url, dados=self.dados_enviados)

    def analisa_resultado_transacoes(self):
        if self.resposta.sucesso:
            transacoes = self.resposta.conteudo['results']
            self.dados_pedido = []
            for transacao in transacoes:
                transacao = transacao['collection']
                if transacao['notification_url'] and GATEWAY in transacao['notification_url']:
                    self.dados_pedido.append({
                        'situacao_pedido': SituacoesDePagamento.do_tipo(transacao['status']),
                        'pedido_numero': transacao['external_reference']
                    })
        elif self.resposta.nao_autenticado or self.resposta.nao_autorizado:
            if self.atualizador_credenciais.deve_reenviar(self.resposta):
                self._obtem_resposta()
                self.analisa_resultado_transacoes()
        else:
            if 'error' in self.resposta.conteudo:
                self.erros = self.resposta.conteudo
