# -*- coding: utf-8 -*-
from pagador import configuracoes, entidades
from pagador_mercadopago import cadastro

CODIGO_GATEWAY = 10


class Item(entidades.BaseParaPropriedade):
    _atributos = ['id', 'title', 'currency_id', 'picture_url', 'category_id', 'quantity', 'unit_price']


class Payer(entidades.BaseParaPropriedade):
    _atributos = ['name', 'surname', 'email', 'date_created', 'address', 'phone', 'identification']


class Phone(entidades.BaseParaPropriedade):
    _atributos = ['area_code', 'number']


class Identification(entidades.BaseParaPropriedade):
    _atributos = ['type', 'number']


class PayerAddress(entidades.BaseParaPropriedade):
    _atributos = ['street_name', 'street_number', 'zip_code']


class BackUrls(entidades.BaseParaPropriedade):
    _atributos = ['success', 'pending', 'failure']


class Shipments(entidades.BaseParaPropriedade):
    _atributos = ['cost', 'receiver_address']


class ReceiverAddress(entidades.BaseParaPropriedade):
    _atributos = ['street_name', 'street_number', 'zip_code', 'floor', 'apartment']


class ShipmentsMercadoEnvios(entidades.BaseParaPropriedade):
    _atributos = ['mode', 'dimensions', 'receiver_address', 'default_shipping_method', 'free_methods']


class ZipCodeMercadoEnvios(entidades.BaseParaPropriedade):
    _atributos = ['zip_code']


class Malote(entidades.Malote):
    def __init__(self, configuracao):
        super(Malote, self).__init__(configuracao)
        self.notification_url = None
        self.external_reference = None
        self.auto_return = 'approved'
        self._pedido = None
        self.payer = None
        self.back_urls = None
        self.shipments = None
        self.items = None

    def monta_conteudo(self, pedido, parametros_contrato=None, dados=None):
        self._pedido = pedido
        notification_url = configuracoes.NOTIFICACAO_URL_SEGURA.format('mercadopago', self.configuracao.loja_id)
        self.notification_url = '{}/notificacao?referencia={}'.format(notification_url, self._pedido.numero)
        self.external_reference = self._pedido.numero
        self.payer = Payer(
            name=self.formatador.trata_unicode_com_limite(self._pedido.cliente_primeiro_nome),
            surname=self.formatador.trata_unicode_com_limite(self._pedido.cliente_ultimo_nome),
            email=self._pedido.cliente['email'],
            date_created=self.formatador.formata_data(self._pedido.cliente['data_criacao'], iso=True),
            phone=self._cliente_telefone,
            identification=self._cliente_documento,
            address=PayerAddress(
                street_name=self.formatador.trata_unicode_com_limite(self._pedido.endereco_cliente['endereco']),
                street_number=self._pedido.endereco_cliente['numero'],
                zip_code=self._pedido.endereco_cliente['cep'],
            )
        )
        back_url = '{}/resultado?referencia={}&next_url={}'.format(notification_url, self._pedido.numero, dados['next_url'])
        self.back_urls = BackUrls(
            failure='{}?sucesso=0'.format(back_url),
            success='{}?sucesso=1'.format(back_url),
            pending='{}?sucesso=2'.format(back_url)
        )
        if self._pedido.forma_envio_tipo == 'mercadoenvios_api':
            dimensoes = self._pedido.dimensoes()

            # Quebra de responsabilidade. A API do Pagador não tem que saber
            # qual é o código do envio.
            if self._pedido.forma_envio_codigo == 'mercadoenvios_normal':
                codigo_envio = 100009
            elif self._pedido.forma_envio_codigo == 'mercadoenvios_expresso':
                codigo_envio = 182
            else:
                codigo_envio = None

            self.shipments = ShipmentsMercadoEnvios(
                mode="me2",
                dimensions="{}x{}x{},{}".format(dimensoes[0], dimensoes[1], dimensoes[2], dimensoes[3]),
                receiver_address=ReceiverAddress(
                    street_name=self.formatador.trata_unicode_com_limite(self._pedido.endereco_entrega['endereco']),
                    street_number=self._pedido.endereco_entrega['numero'],
                    apartment=self.formatador.trata_unicode_com_limite(self._pedido.endereco_entrega['complemento']),
                    zip_code=self._pedido.endereco_entrega['cep'],
                ),
                default_shipping_method=codigo_envio
            )

            if self._pedido.valor_envio == 0:
                self.shipments.free_methods = [{"id": codigo_envio}]
        else:
            self.shipments = Shipments(
                cost=self.formatador.formata_decimal(self._pedido.valor_envio, como_float=True),
                receiver_address=ReceiverAddress(
                    street_name=self.formatador.trata_unicode_com_limite(self._pedido.endereco_entrega['endereco']),
                    street_number=self._pedido.endereco_entrega['numero'],
                    apartment=self.formatador.trata_unicode_com_limite(self._pedido.endereco_entrega['complemento']),
                    zip_code=self._pedido.endereco_entrega['cep'],
                )
            )
        self.items = self._monta_items()
        try:
            sponsor_id = int(parametros_contrato['sponsor_id'])
        except (KeyError, ValueError):
            sponsor_id = None
        if sponsor_id:
            setattr(self, 'sponsor_id', sponsor_id)

    @property
    def _cliente_telefone(self):
        telefone = self._pedido.cliente_telefone
        return Phone(area_code=telefone[0], number=telefone[1])

    @property
    def _cliente_documento(self):
        tipo = 'CPF' if self._pedido.endereco_cliente['tipo'] == "PF" else 'CNPJ'
        return Identification(type=tipo, number=self._pedido.cliente_documento)

    def _monta_items(self):
        items = [
            Item(
                id=self.formatador.trata_unicode_com_limite(item.sku),
                title=self.formatador.trata_unicode_com_limite(item.nome),
                currency_id='BRL',
                unit_price=self.formatador.formata_decimal(item.preco_venda, como_float=True),
                quantity=self.formatador.formata_decimal(item.quantidade, como_float=True),
                category_id='others',
                picture_url=''
            )
            for item in self._pedido.itens
        ]
        if self._pedido.valor_desconto:
            items.append(Item(id='desconto', title='Desconto', quantity=1, currency_id='BRL', category_id='', unit_price=self.formatador.formata_decimal((self._pedido.valor_desconto * -1), como_float=True)))
        return items


class ConfiguracaoMeioPagamento(entidades.ConfiguracaoMeioPagamento):
    modos_pagamento_aceitos = {
        'cartoes': ['visa', 'mastercard', 'amex', 'elo'],
        'outros': ['boleto']
    }

    def __init__(self, loja_id, codigo_pagamento=None, eh_listagem=False):
        self.campos = ['usuario', 'token', 'token_expiracao', 'codigo_autorizacao', 'ativo', 'valor_minimo_aceitado', 'valor_minimo_parcela', 'mostrar_parcelamento', 'maximo_parcelas', 'parcelas_sem_juros']
        self.codigo_gateway = CODIGO_GATEWAY
        self.eh_gateway = True
        super(ConfiguracaoMeioPagamento, self).__init__(loja_id, codigo_pagamento, eh_listagem=eh_listagem)
        _formulario = cadastro.FormularioMercadoPago()
        self._parcelas = _formulario.obter_parcelas_disponiveis()
        if not self.eh_listagem:
            self.formulario = _formulario
            self.eh_aplicacao = True
