import sys
import os
import yaml
import re
import logging


def import_by_name(name):
    """Import a module by name
    
        :param name: The name of the module.
        :type name: str.
        :returns: module -- the imported module.
    
    """
    __import__(name)
    return sys.modules[name]
    
def extract_metadata(text):
    """Extract the YAML front matter from a string, process it and return the content
       of the string
    
        :param text: The text containing the YAML front matter and body.
        :type text: str.
        :returns: dict -- A dictionary containing the metadata in the front-matter and the
                          body in the 'content' key
    
    """
    p = re.compile('^---.*---$', re.MULTILINE | re.DOTALL)
    yaml_text = p.findall(text)
    if len(yaml_text) == 0:
        return {'content': text}
    
    metadata = yaml.load(yaml_text[0].strip('-'))
    metadata['content'] = p.sub('', text)
    return metadata

def path_for_extra(extra_name):
    """Return the path for an extra resource given its name
    
        :param extra_name: The name of the resource.
        :type extra_name: str.
        :returns:  str -- the path of the resource directory.
    
    """
    return os.getcwd() + '/extras/' + extra_name

def clone_repo(url, path):
    """Clone a git repository
    
        :param url: The URL of the remote repository.
        :type url: str.
        :param path: The local path where to clone the repository
        :type path: str.
    
    """
    from git import Repo
    Repo.clone_from(url, path)
  
def fetch_extras(extras):
    """Retrieve the extra resources needed for the presentation. The method requires a
       list of tuples representing the extras to download as input parameter. Every
       tuple must contain two elements, a string with the name of the resource to
       download as first element and a string or a function to achieve the download as
       second element. If a string is passed as second element md2slides will assume
       it's the URL of a git repository and clone the repository in the default path.
       If a method is passed as second element instead, the method will be called with
       a string representing the path to download the resource to as input parameter.
    
        :param extras: A list of tuples with the info to download the extras.
        :type extras: list.
    
    """
    for extra in extras:
        extra_path = path_for_extra(extra[0])
        if not (os.path.exists(extra_path) or os.path.isdir(extra_path)):
            logging.info("Retrieving {0}".format(extra[0]))
            if isinstance(extra[1], str):
                clone_repo(extra[1],extra_path)
            else:
                extra[1](extra_path)