import markdown
from markdown.util import etree
import pkg_resources
from .. import extensions
from md2slides.util import clone_repo

class ImpressjsProcessor(markdown.treeprocessors.Treeprocessor):
    """ Tree processor to output html compatible with Impress.js presentations. """
    
    def create_slide(self, slide, i):
        """Modify a slide html element to conform to Impress.js.
        
            :param slide: The slide html element.
            :type slide: xml.etree.ElementTree.Element
            :param i: Index of the current slide.
            :type i: int
        
        """
        
        # In Impress.js slides are contained inside a div element with the class step
        slide.tag = 'div'
        
        # update the element attributes
        if 'class' not in slide.attrib:
            slide.set('class', 'step')
        else:
            slide.attrib['class'] += ' step'
        if 'id' not in slide.attrib:
            slide.set('id', str(i))

    def run(self, root):
        """Modify the html tree to conform to Impress.js.
        
            :param root: The root element of the tree.
            :type root: xml.etree.ElementTree.Element
            :returns:  xml.etree.ElementTree.Element -- the root element of the modified tree.
        
        """
        i = 1
        for c in root:
            if c.tag=='slide':
                self.create_slide(c,i)
                i += 1
        root.attrib['id'] = 'impress'
        new_root = etree.Element('div')
        new_root.append(root)
        return new_root

class Impressjs(extensions.SlideExtension):
    """ Subclass of SlideExtension that adds ImpressjsProcessor to the treeprocessor stack and
        register itself.
    """
    name = 'impressjs'
    
    def extendMarkdown(self, md, md_globals):
        super(Impressjs, self).extendMarkdown(md, md_globals)
        md.treeprocessors.add('impressjsprocessor', ImpressjsProcessor(), '>slidetreeprocessor')
        md.registerExtension(self)

    def reset(self):
        pass
        
    def template(self):
        return pkg_resources.resource_string(__name__,'impressjs.html')
        
    def download(self, path):
        clone_repo('https://github.com/bartaz/impress.js.git',path)