import markdown
from markdown.util import etree

class SlideBlockProcessor(markdown.blockprocessors.BlockProcessor):
    """ Process New Slides blocks. In md2slides a slide block is a line starting with 
        -*- and followed by a series of html attributes like "class", "id" or "data-x"
        defining the look and behaviour of this particular slide. The attributes will
        be attached to the slide element. 
    
    """

    def test(self, parent, block):
        # md2slides syntax requires a new slide to start with -*-
        if block.startswith('-*-'):
            return True
        return False

    def run(self, parent, blocks):
        """Process the slide block.
        
            :param parent: The parent element of the blocks.
            :type parent: xml.etree.ElementTree.Element        
        """
        # TODO: this whole method can be made faster probably
        
        # get the slide block, remove -*-, prepend <slide and append />
        block = blocks.pop(0)
        block = block[4:]
        block = '<slide ' + block
        block += ' />'
        
        # Create a slide element by letting etree parse it for us
        parent.append(etree.fromstring(block))
        
class SlideTreeProcessor(markdown.treeprocessors.Treeprocessor):
    """ Process the tree generated from a md2slides markdown file. """
    
    def run(self, root):
        """The SlideBlockProcessor replaced the slide blocks with self-closing slide 
           elements. We reconstruct the tree properly by looping through the root
           children and appending visual elements to their slide element.
        
            :param root: The root element of the tree.
            :type root: xml.etree.ElementTree.Element
            :returns:  xml.etree.ElementTree.Element -- the root element of the modified tree.
        
        """
        if len(root)==0:
            return root
        
        root = list(root)
        main_node = etree.Element('div')
        
        current_slide = etree.Element('slide')
        if root[0].tag=='slide':
            current_slide = root.pop(0)
            
        while len(root):
            if root[0].tag=='slide':
                main_node.append(current_slide)
                current_slide = root.pop(0)
                continue
            current_slide.append(root.pop(0))
        main_node.append(current_slide)
        return main_node

class SlideExtension(markdown.Extension):
    """ Markdown extension used to parse the extended markdown syntax of md2slides. 
        This class should be subclassed by other extensions that provide further processing
        required by the specific javascript presentation engine used.
    """
    
    # name is going to be used by md2slides to compose the path to download the extensions
    # files. The path will be "./extras/<name>" from the location of the markdown file.
    # the reference to the extension files in the html template must reflect this path.
    name = ''
    
    def extendMarkdown(self, md, md_globals):
        """ Subclasses should invoke this method before adding their specific extensions and 
            registering themselves.
        """
        md.parser.blockprocessors.add('slideblockprocessor',SlideBlockProcessor(md.parser),'<paragraph')
        md.treeprocessors.add('slidetreeprocessor', SlideTreeProcessor(), '>prettify')
        
    def template(self):
        """An extension for a particular Javascript presentation engine should provide its 
           html template through this method. Subclasses should override this to return the
           string with the html
        
           :returns:  str -- The string with the html template for the extension.
        
        """
        raise NotImplementedError
        
    def download(self, path):
        """Every extension should know how to download the file it needs for offline work.
           md2slides provides a clone_repo method in the util package to clone a git
           repository into path.
        
           :parama path: The path where to put the files.
           :type path: str.
        
        """
        raise NotImplementedError
