import markdown
from markdown.util import etree
import pkg_resources
from .. import extensions
from md2slides.util import clone_repo

class DeckjsProcessor(markdown.treeprocessors.Treeprocessor):
    """ Tree processor to output html compatible with Deck.js presentations. """
    
    def create_slide(self, slide, i):
        """Modify a slide html element to conform to Deck.js.
        
            :param slide: The slide html element.
            :type slide: xml.etree.ElementTree.Element
            :param i: Index of the current slide.
            :type i: int
        
        """
        
        # In Deck.js slides are contained inside section elements
        slide.tag = 'section'
        
        # update the element attributes
        if 'class' not in slide.attrib:
            slide.set('class', 'slide')
        else:
            slide.attrib['class'] += ' slide'
        if 'id' not in slide.attrib:
            slide.set('id', str(i))

    def create_effects(self, root):
        if root.tail and root.tail[-1] == '~':
            root.tail = root.tail[:-1]
            root.set('class', 'slide')
        elif root.text and root.text[-1] == '~':
            root.text = root.text[:-1]
            root.set('class', 'slide')
        for c in root:
            c = self.create_effects(c)
        return root

    def run(self, root):
        """Modify the html tree to conform to Deck.js.
        
            :param root: The root element of the tree.
            :type root: xml.etree.ElementTree.Element
            :returns:  xml.etree.ElementTree.Element -- the root element of the modified tree.
        
        """
        
        root = self.create_effects(root)
        i = 1
        for c in root:
            if c.tag=='slide':
                self.create_slide(c,i)
                i += 1
        return root

class Deckjs(extensions.SlideExtension):
    """ Subclass of SlideExtension that adds DeckjsProcessor to the treeprocessor stack and
        register itself.
    """
    name = 'deckjs'
    
    def extendMarkdown(self, md, md_globals):
        super(Deckjs, self).extendMarkdown(md, md_globals)
        md.treeprocessors.add('deckjsprocessor', DeckjsProcessor(), '>slidetreeprocessor')
        md.registerExtension(self)

    def reset(self):
        pass
    
    def template(self):
        return pkg_resources.resource_string(__name__,'deckjs.html')
        
    def download(self, path):
        clone_repo('https://github.com/imakewebthings/deck.js.git',path)
    