#!/usr/bin/env python

import inspect
import argparse
import os
import logging
import functools
import markdown
import jinja2
import pkg_resources
try:
    from tornado.log import enable_pretty_logging
except ImportError:
    from tornado.options import enable_pretty_logging
import md2slides
from reloader import start_livereload
from util import import_by_name, extract_metadata, fetch_extras

MATHJAX_CONF = ("<script type=\"text/x-mathjax-config\">\n"
                     "    MathJax.Hub.Config({tex2jax: {inlineMath: [['$','$'], ['\\(','\\)']]}});\n"
                     "</script>\n"
                     "<script type=\"text/javascript\" "
                     "src=\"extras/mathjax/MathJax.js?config=TeX-AMS-MML_HTMLorMML\">\n"
                     "</script>")
            
def create_engine(engine_name):
    """Create a markdown extension object to process the markdown into valid
       html for the presentation
    
        :param engine_name: The name of the javascript engine to be used for the presentation.
        :type engine_name: str.
        :returns: An instance of the markdown extension subclass
            
    """
    engine_module = import_by_name('md2slides.templates.{0}.{0}'.format(engine_name))
    for memeber_name, member in inspect.getmembers(engine_module, inspect.isclass):
        if issubclass(member, markdown.Extension):
            return member()
    return None
            
def compile_presentation(metadata):
    """Compile the presentation from metadata into the html file
        
        :param metadata: A dictionary containing the informations to compile the html from md.
        :type metadata: dict.
        :returns: the metadata dictionary with any addition or modification
    
    """
    logging.info('Compiling %s into %s' % (os.path.basename(metadata['input_path']), os.path.basename(metadata['output_path'])))
    
    with open(metadata['input_path'], 'r') as mdfile:
        content = mdfile.read()
    
    metadata.update(extract_metadata(content))    
    if metadata.get('latex', False):
        metadata['latex'] = MATHJAX_CONF
        metadata['extras'].append(('mathjax', 'https://github.com/mathjax/MathJax.git'))
    else:
        metadata['latex'] = ''
    
    # Generate the html body from the markdown input file
    metadata['content'] = markdown.markdown(metadata['content'], extensions=[metadata['engine']])
    
    # Open Template supplied as argument
    template = jinja2.Template(metadata['template'])
    # Render template with content and title
    text = template.render(metadata)
    
    with open(metadata['output_path'], 'w') as output_file:
        output_file.write(text.encode('utf-8'))
    
    return metadata

def parse_options():
    parser = argparse.ArgumentParser(description='Process some integers.')
    parser.add_argument('file', type=str, help='Path of the markdown file to process')
    parser.add_argument('-e', '--engine', type=str, choices=['deckjs','impressjs'], default='deckjs', help='javascript java engine to use')
    parser.add_argument('-a', '--autoupdate', action="store_true", default=False, help='automatic reload')
    return parser.parse_args()

def run(input_path = None, engine = None, autoupdate = False):
    logging.getLogger().setLevel(logging.INFO)
    enable_pretty_logging()
    
    if input_path is None:
        args = parse_options()
        input_path = os.path.abspath(args.file)
        engine = args.engine
        autoupdate = args.autoupdate
    
    # create the path of the input and output files from the input filename
    filename = os.path.basename(input_path)
    dirname = os.path.dirname(input_path)
    file_root, ext = os.path.splitext(filename)
    output_path = os.path.join(dirname, file_root + '.html')
        
    # Custom markdown extension to transform md into valid html
    if isinstance(engine, str):
        engine = md2slides.create_engine(engine)
    
    # metadata dictionary with default values
    metadata = {'title': 'untitled', 
                'content': '', 
                'engine': engine,
                'input_path': input_path,
                'output_path': output_path,
                'template': engine.template(),
                'css': file_root + '.css',
                'extras': [(engine.name, engine.download)]}
                
    md2slides.compile_presentation(metadata)
    
    # fetch all the needed extras 
    fetch_extras(metadata['extras'])
    
    if autoupdate:
        compile_func = functools.partial(compile_presentation, metadata)
        start_livereload(os.path.basename(metadata['output_path']), metadata['input_path'], func=compile_func)
        
if __name__ == '__main__':
    run()