#!/usr/bin/python
__author__ = 'Ronie Martinez'
import re
import inspect
import sys

_camel = re.compile('^[a-z0-9]+([A-Z][a-z0-9]+)+$')
_snake = re.compile('^[a-z0-9]+(_[a-z0-9]+)+$')
_camel_splitter = re.compile('([A-Z][a-z0-9]+)')


def _to_snake_case(string):
    return '_'.join([m.lower() for m in _camel_splitter.split(string) if m])


def _to_camel_case(string):
    words = string.split('_')
    for i in range(1, len(words)):
        words[i] = words[i].title()
    return ''.join(words)


class NoCase(object):
    """
    NoCase is a class that automatically converts method calls from camelCase to snake_case and vice versa.
    Sample usage:

    class MyClass(NoCase):

        def myMethod(self):
            return 'my method'

        def my_other_method(self):
            return 'my other method'

    my_class = MyClass()
    print(my_class.my_method())  # prints 'my method'
    print(my_class.myOtherMethod())  # prints 'my other method'
    """

    def __getattr__(self, name):
        if _camel.match(name):
            return getattr(self, _to_snake_case(name))
        elif _snake.match(name):
            return getattr(self, _to_camel_case(name))
        raise AttributeError(name)


def register_no_case(module=sys.modules['__main__']):
    """
    Finds all functions in the current module, 'module', and registers a function to it's equivalent CamelCase or snake_case name
    Sample usage:
    from nocase import register_no_case


    def myMethod():
        return 'my method'


    def my_other_method():
        return 'my other method'

    register_no_case()
    print(my_method())
    print(myOtherMethod())

    :param module: module name to inspect and to where new functions will be registered
    """
    functions = inspect.getmembers(module, inspect.isfunction)
    for name, function in functions:
        if _camel.match(name):
            setattr(module, _to_snake_case(name), function)
        elif _snake.match(name):
            setattr(module, _to_camel_case(name), function)

registerNoCase = register_no_case  # CamelCase call to register_no_case
