"""Optimizers."""

import autograd.numpy as np
import autograd


class NewtonOptimizer(object):
    """An optimizer based on Newton's method."""

    def __init__(self, f, tol=1.48e-08, maxiter=50):
        """Generate an optimizer from an objective function."""
        self.g = autograd.grad(f)
        self.h = autograd.hessian(f)
        self.tol = tol
        self.maxiter = maxiter

    def optimize(self, x0, target):
        """Calculate an optimum argument of an objective function."""
        x = x0
        for _ in range(self.maxiter):
            delta = np.linalg.solve(self.h(x, target), -self.g(x, target))
            x = x + delta
            if np.linalg.norm(delta) < self.tol:
                break
        return x


class SDOptimizer(object):
    """An optimizer based on steepest descent method."""

    def __init__(self, f, tol=1.48e-08, maxiter=50, alpha=1):
        """Generate an optimizer from an objective function."""
        self.g = autograd.grad(f)
        self.tol = tol
        self.maxiter = maxiter
        self.alpha = alpha

    def optimize(self, x0, target):
        """Calculate an optimum argument of an objective function."""
        x = x0
        for _ in range(self.maxiter):
            delta = self.alpha * self.g(x, target)
            x = x - delta
            if np.linalg.norm(delta) < self.tol:
                break
        return x
