# Rekall Memory Forensics
#
# Based on original code by:
# Copyright (C) 2007-2013 Volatility Foundation
#
# Authors:
# Michael Hale Ligh <michael.ligh@mnin.org>
#
# This code:
# Copyright 2014 Google Inc. All Rights Reserved.
#
# Authors:
# Michael Cohen <scudette@google.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or (at
# your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

import struct

from rekall import plugin
from rekall import testlib

from rekall.plugins.windows import common
from rekall.plugins.overlays.windows import pe_vtypes


class DecodingError(Exception):
    """Raised when unable to decode an instruction."""


class HookHeuristic(object):
    """A Hook heuristic detects possible hooks.

    This heuristic emulates some common CPU instructions to try and detect
    control flow jumps within the first few instructions of a function.

    These are essentially guesses based on the most common hook types. Be aware
    that these are pretty easy to defeat which will cause the hook to be missed.

    See rekall/src/hooks/amd64.asm and rekall/src/hooks/i386.asm For the test
    cases which illustrate the type of hooks that we will detect.
    """

    def __init__(self, session=None):
        self.session = session
        self.Reset()

    def Reset(self):
        # Keep track of registers, stack and main memory.
        self.regs = {}
        self.stack = []
        self.memory = {}

    def WriteToOperand(self, operand, value):
        if operand["type"] == "REG":
            self.regs[operand["reg"]] = value

        elif operand["type"] == "IMM":
            self.memory[operand["address"]] = value

        elif operand["type"] == "MEM":
            self.memory[self._get_mem_operand_target(operand)] = value

        else:
            raise DecodingError("Operand not supported")

    def ReadFromOperand(self, operand):
        """Read the operand.

        We support the following forms:

        - Immediate (IMM):  JMP 0x123456
        - Absolute Memory Address (MEM): JMP [0x123456]
        - Register (REG): JMP [EAX]
        """
        # Read from register.
        if operand["type"] == 'REG':
            return self.regs.get(operand["reg"], 0)

        # Immediate operand.
        elif operand["type"] == 'IMM':
            return operand["address"]

        # Read the content of memory.
        elif operand["type"] == "MEM":
            return self._GetMemoryAddress(
                self._get_mem_operand_target(operand), operand["size"])

        else:
            raise DecodingError("Operand not supported")

    def _get_mem_operand_target(self, operand):
        reg_base = operand["base"]
        if reg_base == "RIP":
            return operand["address"]
        else:
            # Register reference [base_reg + disp + index_reg * scale]
            return (self.regs.get(reg_base, 0) +
                    operand["disp"] +
                    self.regs.get(operand["index"], 0) * operand["scale"])

    def _GetMemoryAddress(self, offset, size):
        try:
            # First check our local cache for a previously written value.
            return self.memory[offset]
        except KeyError:
            data = self.address_space.read(offset, size)
            format_string = {1: "b", 2: "H", 4: "I", 8: "Q"}[size]

            return struct.unpack(format_string, data)[0]

    def process_lea(self, instruction):
        """Copies the address from the second operand to the first."""
        operand = instruction.operands[1]
        if operand["type"] == 'MEM':
            self.WriteToOperand(instruction.operands[0],
                                self._get_mem_operand_target(operand))
        else:
            raise DecodingError("Invalid LEA source.")

    def process_push(self, instruction):
        value = self.ReadFromOperand(instruction.operands[0])

        self.stack.append(value)

    def process_pop(self, instruction):
        try:
            value = self.stack.pop(-1)
        except IndexError:
            value = 0

        self.WriteToOperand(instruction.operands[0], value)

    def process_ret(self, _):
        if self.stack:
            return self.stack.pop(-1)

    def process_mov(self, instruction):
        value = self.ReadFromOperand(instruction.operands[1])

        self.WriteToOperand(instruction.operands[0], value)

    def process_inc(self, instruction):
        value = self.ReadFromOperand(instruction.operands[0])

        self.WriteToOperand(instruction.operands[0], value + 1)

    def process_dec(self, instruction):
        value = self.ReadFromOperand(instruction.operands[0])

        self.WriteToOperand(instruction.operands[0], value - 1)

    def process_cmp(self, instruction):
        """We dont do anything with the comparison since we dont test for it."""
        _ = instruction

    def process_test(self, instruction):
        """We dont do anything with the comparison since we dont test for it."""
        _ = instruction

    def _Operate(self, instruction, operator):
        value1 = self.ReadFromOperand(instruction.operands[0])
        value2 = self.ReadFromOperand(instruction.operands[1])

        self.WriteToOperand(
            instruction.operands[0], operator(value1, value2))

    def process_xor(self, instruction):
        return self._Operate(instruction, lambda x, y: x ^ y)

    def process_add(self, instruction):
        return self._Operate(instruction, lambda x, y: x + y)

    def process_sub(self, instruction):
        return self._Operate(instruction, lambda x, y: x - y)

    def process_and(self, instruction):
        return self._Operate(instruction, lambda x, y: x & y)

    def process_or(self, instruction):
        return self._Operate(instruction, lambda x, y: x | y)

    def process_shl(self, instruction):
        return self._Operate(instruction, lambda x, y: x << (y % 0xFF))

    def process_shr(self, instruction):
        return self._Operate(instruction, lambda x, y: x >> (y % 0xFF))

    def Inspect(self, function, instructions=10):
        """The main entry point to the Hook processor.

        We emulate the function instructions and try to determine the jump
        destination.

        Args:
           function: A basic.Function() instance.
        """
        self.Reset()
        self.address_space = function.obj_vm

        for instruction in function.disassemble(instructions=instructions):
            if instruction.is_return():
                # RET Instruction terminates processing.
                return self.process_ret(instruction)

            elif instruction.mnemonic == "call":
                return self.ReadFromOperand(instruction.operands[0])

            # A JMP instruction.
            elif instruction.is_branch():
                return self.ReadFromOperand(instruction.operands[0])

            else:
                try:
                    handler = getattr(self, "process_%s" % instruction.mnemonic)
                except AttributeError:
                    continue

                # Handle the instruction.
                handler(instruction)


class CheckPEHooks(plugin.VerbosityMixIn, common.WindowsCommandPlugin):
    """Checks a pe file mapped into memory for hooks."""

    name = "check_pehooks"

    @classmethod
    def args(cls, parser):
        super(CheckPEHooks, cls).args(parser)
        parser.add_argument("--image-base", default=0,
                            help="The base address of the pe image in memory.")

        parser.add_argument(
            "--type", default="all", choices=["all", "iat", "inline", "eat"],
            type="ChoiceArray", help="Type of hook to display.")

    def __init__(self, image_base=0, type="all", **kwargs):
        super(CheckPEHooks, self).__init__(**kwargs)
        self.image_base = self.session.address_resolver.get_address_by_name(
            image_base)
        self.heuristic = HookHeuristic(session=self.session)
        self.hook_type = type

    def detect_IAT_hooks(self):
        """Detect Import Address Table hooks.

        An IAT hook is where malware changes the IAT entry for a dll after its
        loaded so that when it is called from within the DLL, flow control is
        directed to the malware instead.

        We determine the IAT entry is hooked if the address is outside the dll
        which is imported.
        """
        pe = pe_vtypes.PE(image_base=self.image_base, session=self.session)

        # First try to find all the names of the imported functions.
        imports = [
            (dll, func_name) for dll, func_name, _ in pe.ImportDirectory()]

        resolver = self.session.address_resolver

        for idx, (dll, func_address, _) in enumerate(pe.IAT()):

            try:
                target_dll, target_func_name = imports[idx]
                target_dll = self.session.address_resolver.NormalizeModuleName(
                    target_dll)
            except IndexError:
                # We can not retrieve these function's name from the
                # OriginalFirstThunk array - possibly because it is not mapped
                # in.
                target_dll = dll
                target_func_name = ""

            self.session.report_progress(
                "Checking function %s!%s", target_dll, target_func_name)

            # We only want the containing module.
            module = resolver.GetContainingModule(func_address)
            if module and target_dll == module.name:
                continue

            function_name = "%s:%s" % (target_dll, target_func_name)

            yield function_name, func_address

    def render_iat_hooks(self, renderer):
        renderer.table_header([
            ("Import", "import", "60"),
            ("Dest", "destination", "[addrpad]"),
            ("Dest Name", "dest_name", "60"),
            ])

        for function_name, func_address in self.detect_IAT_hooks():
            destination = self.session.address_resolver.format_address(
                func_address, max_distance=2**64)
            if not destination:
                destination = "%#x" % func_address

            renderer.table_row(
                function_name, func_address, destination)

    def detect_EAT_hooks(self, size=0):
        """Detect Export Address Table hooks.

        An EAT hook is where malware changes the EAT entry for a dll after its
        loaded so that a new DLL wants to link against it, the new DLL will use
        the malware's function instead of the exporting DLL's function.

        We determine the EAT entry is hooked if the address lies outside the
        exporting dll.
        """
        address_space = self.session.GetParameter("default_address_space")
        pe = pe_vtypes.PE(image_base=self.image_base, session=self.session,
                          address_space=address_space)
        start = self.image_base
        end = self.image_base + size

        # If the dll size is not provided we parse it from the PE header.
        if not size:
            for _, _, virtual_address, section_size in pe.Sections():
                # Only count executable sections.
                section_end = self.image_base + virtual_address + section_size
                if section_end > end:
                    end = section_end

        resolver = self.session.address_resolver

        for dll, func, name, hint in pe.ExportDirectory():
            self.session.report_progress("Checking export %s!%s", dll, name)

            # Skip zero or invalid addresses.
            if address_space.read(func.v(), 10) == "\x00" * 10:
                continue

            if start < func.v() < end:
                continue

            function_name = "%s:%s (%s)" % (
                resolver.NormalizeModuleName(dll), name, hint)

            yield function_name, func

    def render_eat_hooks(self, renderer):
        renderer.table_header([
            ("Export", "Export", "[wrap:60]"),
            ("Dest", "destination", "[addrpad]"),
            ("Dest Name", "dest_name", "[wrap:60]"),
            ])

        for function_name, func_address in self.detect_IAT_hooks():
            destination = self.session.address_resolver.format_address(
                func_address, max_distance=2**64)
            if not destination:
                destination = "%#x" % func_address

            renderer.table_row(
                function_name, func_address, destination)

    def detect_inline_hooks(self):
        """A Generator of hooked exported functions from this PE file.

        Yields:
          A tuple of (function, name, jump_destination)
        """
        # Inspect the export directory for inline hooks.
        pe = pe_vtypes.PE(image_base=self.image_base, session=self.session)

        for _, function, name, _ in pe.ExportDirectory():
            self.session.report_progress(
                "Checking function %#x (%s)", function, name)

            # Try to detect an inline hook.
            destination = self.heuristic.Inspect(function, instructions=3) or ""

            # If we did not detect a hook we skip this function.
            if destination:
                yield function, name, destination

    def render_inline_hooks(self, renderer):
        renderer.table_header([("Name", "name", "20s"),
                               ("Hook", "hook", "30s"),
                               ("Disassembly", "location", "60s"),
                              ])
        for function, name, destination in self.detect_inline_hooks():
            hook_detected = False

            # Try to resolve the destination into a name.
            destination_name = self.session.address_resolver.format_address(
                destination, max_distance=2**64)

            # We know about it. We suppress the output for jumps that go into a
            # known module. These should be visible using the regular vad
            # module.
            if destination_name:
                destination = destination_name
            else:
                destination = "%#x" % destination
                hook_detected = True

            # Skip non hooked results if verbosity is too low.
            if self.verbosity < 10 and not hook_detected:
                continue

            # Only highlight results if verbosity is high.
            highlight = ""
            if hook_detected and self.verbosity > 1:
                highlight = "important"

            renderer.table_row(name, destination, function.deref(),
                               highlight=highlight)


    def render(self, renderer):
        if self.hook_type in ["all", "inline"]:
            self.render_inline_hooks(renderer)

        if self.hook_type in ["all", "iat"]:
            self.render_iat_hooks(renderer)

        if self.hook_type in ["all", "eat"]:
            self.render_eat_hooks(renderer)


class IATHooks(plugin.VerbosityMixIn, common.WinProcessFilter):
    """Detect IAT/EAT hooks in process and kernel memory"""

    name = "hooks_iat"

    def render_iat_hooks(self, task, dll, renderer):
        checker = self.session.plugins.check_pehooks(
            image_base=dll.base)

        for function_name, func_address in checker.detect_IAT_hooks():
            destination = self.session.address_resolver.format_address(
                func_address, max_distance=2**64)
            if not destination:
                destination = "%#x" % func_address

            renderer.table_row(
                task.pid, task.name, dll.name, function_name,
                func_address, destination)

    def render(self, renderer):
        cc = self.session.plugins.cc()
        renderer.table_header([
            ("Pid", "pid", "4"),
            ("Proc", "proc", "16"),
            ("Dll", "dll", "30"),
            ("Import", "import", "60"),
            ("Dest", "destination", "[addrpad]"),
            ("Dest Name", "dest_name", "60"),
            ])

        with cc:
            for task in self.filter_processes():
                cc.SwitchProcessContext(task)

                for dll in task.get_load_modules():
                    self.render_iat_hooks(task, dll, renderer)


class TestIATHooks(testlib.SimpleTestCase):
    PLUGIN = "hooks_iat"

    PARAMETERS = dict(
        commandline="hooks_iat --pid %(pid)s"
        )


class EATHooks(plugin.VerbosityMixIn, common.WinProcessFilter):
    """Detect EAT hooks in process and kernel memory"""

    name = "hooks_eat"

    def render_eat_hooks(self, task, dll, renderer):
        checker = self.session.plugins.check_pehooks(
            image_base=dll.base)

        for function_name, func_address in checker.detect_EAT_hooks():
            destination = self.session.address_resolver.format_address(
                func_address, max_distance=2**64)
            if not destination:
                destination = "%#x" % func_address

            renderer.table_row(
                task.pid, task.name, dll.name, function_name,
                func_address, destination)

    def render(self, renderer):
        cc = self.session.plugins.cc()
        renderer.table_header([
            ("Pid", "pid", "4"),
            ("Proc", "proc", "16"),
            ("Dll", "dll", "30"),
            ("Export", "export", "60"),
            ("Dest", "destination", "[addrpad]"),
            ("Dest Name", "dest_name", "60"),
            ])

        with cc:
            for task in self.filter_processes():
                cc.SwitchProcessContext(task)

                for dll in task.get_load_modules():
                    self.render_eat_hooks(task, dll, renderer)


class TestEATHooks(testlib.SimpleTestCase):
    PLUGIN = "hooks_eat"

    PARAMETERS = dict(
        commandline="hooks_eat --pid %(pid)s"
        )



class InlineHooks(plugin.VerbosityMixIn, common.WinProcessFilter):
    """Detect API hooks in process and kernel memory"""

    name = "hooks_inline"

    def render_inline_hooks(self, task, dll, renderer):
        checker = self.session.plugins.check_pehooks(
            image_base=dll.base)

        for function, name, destination in checker.detect_inline_hooks():
            hook_detected = False

            # Try to resolve the destination into a name.
            destination_name = self.session.address_resolver.format_address(
                destination, max_distance=2**64)

            # We know about it. We suppress the output for jumps that go into a
            # known module. These should be visible using the regular vad
            # module.
            if destination_name:
                destination = destination_name
            else:
                destination = "%#x" % destination
                hook_detected = True

            # Skip non hooked results if verbosity is too low.
            if self.verbosity < 10 and not hook_detected:
                continue

            # Only highlight results if verbosity is high.
            highlight = ""
            if hook_detected and self.verbosity > 1:
                highlight = "important"

            renderer.table_row(
                task.pid, task.name, dll.name, name, destination,
                function.deref(), highlight=highlight)


    def render(self, renderer):
        cc = self.session.plugins.cc()
        renderer.table_header([
            ("Pid", "pid", "4"),
            ("Proc", "proc", "16"),
            ("Dll", "dll", "16"),
            ("Name", "name", "[wrap:20]"),
            ("Hook", "hook", "20"),
            ("Disassembly", "location", "60"),
            ])

        with cc:
            for task in self.filter_processes():
                cc.SwitchProcessContext(task)

                for dll in task.get_load_modules():
                    self.render_inline_hooks(task, dll, renderer)


class TestInlineHooks(testlib.SimpleTestCase):
    PLUGIN = "hooks_inline"

    PARAMETERS = dict(
        commandline="hooks_inline --pid %(pid)s"
        )
