from xml.etree.ElementTree import ElementTree as Tree
import base64
import zlib
import pygame
import math


class MapError(Exception):
    """Exception for points outside the map.*
    """
    def __init__(self, x, y, max_x, max_y):
        self.x = x
        self.y = y
        self.max_x = max_x
        self.max_y = max_y

    def __str__(self):
        self.fin = ''
        if self.x > self.max_x:
            self.fin += 'x should be reduced by ' + str(self.x -
            self.max_x)
            if self.y > self.max_y:
                self.fin += ', '
        if self.y > self.max_y:
            self.fin += 'y should be reduced by ' + str(self.y -
            self.max_y)
        return self.fin


class MAP:
    """Bacis map class.
    """
    def __init__(self, width, height):
        self.width = width
        self.height = height
        self.objects = []

    def __repr__(self):
        if self.objects:
            self.fin = 'Map ' + str(self.width) + "x" + str(self.height) +\
            ":\n"
            self.count = 1
            for obj in self.objects:
                self.fin += str(self.count) + '. ' + str(obj) + '\n'
                self.count += 1
            return self.fin[:-1]
        else:
            return "Empty Map " + str(self.width) + "x" + str(self.height)\
            + ":"

    def __contains__(self, item):
        self.item = item
        return self.item in self.objects

    def __bool__(self):
        return bool(self.objects)

    def add_obj(self, obj):
        """Function that adds object(point, rect...) to map.
        """
        self.obj = obj
        if type(self.obj) == point:
            if self.obj.x > self.width or self.obj.y > self.height:
                raise MapError(obj.x, obj.y, self.width, self.height)
        self.objects.append(self.obj)

    def at(self, x, y):
        """Return generator of all items in map on x, y coordinates.
        """
        self.x = x
        self.y = y
        for obj in self.objects:
            if type(obj) == point:
                if obj.x == self.x and obj.y == self.y:
                    yield obj
            elif type(obj) == group_of_points:
                self.T = False
                for POINT in obj.at(self.x, self.y):
                    yield POINT
                    self.T = True
                if self.T:
                    yield obj
            elif type(obj) == rect:
                if obj.at(self.x, self.y):
                    yield obj


class point:
    """Basic point class.
    """
    blits = False

    def __init__(self, x, y, Map, description='Unknown', quiet=False):
        self.x = x
        self.y = y
        self.Map = Map
        self.description = description
        if not quiet:
            self.Map.add_obj(self)
        self.name = self.description

    def __tmx_init__(x, y, width, height, Map, name):
        if name:
            return point(x, y, Map, name)
        else:
            return point(x, y, Map)

    def __tmx_a_init__(x, y, width, height, Map, name, **prop):
        return ext_obj(point.__tmx_init__(x, y, width, height, Map.in_map, name),
        **prop)

    def __str__(self):
        return self.description + ' @ ' + str(self.x) + ', ' + str(self.y)

    __repr__ = __str__

    def distance(self, other):
        """Calculates distance between this and given point.
        """
        self.other = other
        return math.sqrt(abs(self.x - other.x) ** 2 + abs(self.y - other.y)
        ** 2)

    def get_xy(self):
        """Returns point's x and y.
        """
        return (self.x, self.y)


class line:
    """Basic line class.
    """
    blits = False

    def __init__(self, points, Map, description='Unknown', quiet=False,
        from_line_seg=False):
        self.points = points
        if from_line_seg:
            self.segment = from_line_seg
        else:
            self.segment = line_seg(self.points, Map, quiet=True,
            from_line=self)
        self.Map = Map
        self.description = description
        if not quiet:
            self.Map.add_obj(self)
        self.dif = self.x_dif, self.y_dif = self.segment.dif
        self.name = self.description
        self.cof = self.segment.cof

    def __str__(self):
        return self.description + ' line (' + str(self.points[0]) + ', '\
        + str(self.points[1]) + ')'

    __repr__ = __str__

    def __contains__(self, other):
        self.other = other
        if type(self.other) == point:
            if self.other in self.points:
                return True
            self.l = line((self.points[0], self.other), Map, quiet=True)
            if self.l.cof == self.cof:
                return True
        elif type(self.other) == group_of_points:
            for P in self.other.points:
                if not P in self:
                    return False
            return True
        elif type(self.other) == rect:
            if self.points[0] in self.other or self.points[1] in self.other:
                return True
            self.lines = []
            for p in self.other.points():
                self.lines.append(line((self.points[0], p), self.Map, quiet=True))
            self.angles = []
            for l in self.lines:
                self.angles.append(l.get_f_angle(0) + 360)
            if min(self.angles) <= self.get_f_angle(0) + 360 <= max(self.angles):
                return True
        return False

    def get_angle(self):
        """Returns line anlge.
        """
        return self.segment.get_angle()

    def get_f_angle(self, p_index):
        return self.segment.get_f_angle(p_index)

    def perpend(self, point, quiet=False):
        self.d = direction(point, (self.get_f_angle() + 90) % 360, self.Map, quiet=True)
        return self.d.line(quiet)

    def collide(self, L):
        self.L = L
        self.x1, self.y1 = self.points[0].get_xy()
        self.x2, self.y2 = self.points[1].get_xy()
        self.x3, self.y3 = self.L.points[0].get_xy()
        self.x4, self.y4 = self.L.points[1].get_xy()
        self.xs1 = self.x1 - self.x2
        self.xs2 = self.x3 - self.x4
        self.xs1 /= self.y1 - self.y2
        self.xs2 /= self.y3 - self.y4
        if self.xs1 == self.xs2:
            raise MapError("Lines are parallel")
        self.x3 += (self.y1 - self.y3) * self.xs2
        self.y3 = self.y1
        self.diff1 = self.x1 - self.x3
        self.diff2 = self.xs1 - self.xs2


class line_seg:
    """Bacis line segment class.
    """
    blits = False

    def __init__(self, points, Map, description='Unknown', quiet=False,
        from_line=False):
        self.points = points
        self.Map = Map
        self.description = description
        if not quiet:
            self.Map.add_obj(self)
        self.name = self.description
        self.dif = self.x_dif, self.y_dif = (abs(self.points[0].x -
        self.points[1].x), abs(self.points[0].y - self.points[1].y))
        if self.y_dif == 0:
            self.cof = "horizontal"
        else:
            self.cof = self.x_dif / self.y_dif
        if from_line:
            self.line = from_line
        else:
            self.line = line(self.points, self.Map, quiet=True,
            from_line_seg=self)

    def __len__(self):
        return int(self.points[0].distance(self.points[1]))

    def __str__(self):
        return self.description + ' line segment (' + str(self.points[0])
        + ', ' + str(self.points[1]) + ')'

    __repr__ = __str__

    def __contains__(self, other):
        self.other = other
        if type(self.other) == point:
            if self.other in self.points:
                return True
            self.l = line((self.points[0], self.other), Map, quiet=True)
            if self.l.cof == self.cof:
                if self.cof == "horizontal":
                    if self.points[0].y > self.points[1].y:
                        if self.points[0].y > self.other.y > self.points[1].y:
                            return True
                    else:
                        if self.points[0].y < self.other.y < self.points[1].y:
                            return True
                else:
                    if (self.points[0].y > self.other.y > self.points[1].y or
                    self.points[0].y < self.other.y < self.points[1].y) and\
                    (self.points[0].x > self.other.x > self.points[1].x or
                    self.points[0].x < self.other.x < self.points[1].x):
                        return True
        elif type(self.other) == group_of_points:
            for P in self.other.points:
                if not P in self:
                    return False
            return True
        elif type(self.other) == rect:
            if self.other in self.line:
                if ((self.points[0].x >= self.points[1].x and self.other.x <= self.points[0].x) or
                (self.points[1].x > self.points[0].x and self.other.x <= self.points[1].x)) and\
                ((self.points[0].y >= self.points[1].y and self.other.y <= self.points[0].y) or
                (self.points[1].y > self.points[0].y and self.other.y <= self.points[1].y)):
                    return True
        return False

    def get_angle(self):
        """
        Returns line rotation (0 vertical, 90 horizontal) in range 0 - 180.
        """
        if self.cof == "horizontal":
            return 90.0
        return math.degrees(math.atan(self.x_dif / self.y_dif))

    def get_f_angle(self, p_index):
        self.a = self.get_angle()
        self.x1, self.y1 = self.points[p_index].get_xy()
        self.x2, self.y2 = self.points[not p_index].get_xy()
        if self.x1 > self.x2:
            if self.y1 > self.y2:
                self.a = 180 - self.a
        else:
            if self.y1 > self.y2:
                self.a += 180
            else:
                self.a = 360 - self.a
        self.a %= 360
        return self.a


def q_points(x1, y1, x2, y2, Map):
    """Returns points for line and line_seg.
    """
    p1 = point(x1, y1, Map, quiet=True)
    p2 = point(x2, y2, Map, quiet=True)
    return (p1, p2)


class ray:
    """Basic ray class.
    """
    blits = False

    def __init__(self, start_p, some_p, Map, description='Unknown',
        quiet=False):
        self.start_p = start_p
        self.some_p = some_p
        self.line = line((self.start_p, self.some_p), Map, quiet=True,
        from_line_seg=True)
        self.line_seg = self.line.segment
        self.Map = Map
        if not quiet:
            self.Map.add_obj(self)
        self.description = description
        self.name = self.description
        self.dif = self.x_dif, self.y_dif = (abs(self.start_p.x -
        self.some_p.x), abs(self.start_p.y - self.some_p.y))
        if self.y_dif == 0:
            self.cof = "horizontal"
        else:
            self.cof = self.x_dif / self.y_dif

    def __contains__(self, other):
        self.other = other
        if type(self.other) == point:
            if self.other == self.start_p:
                return True
            self.l = line((self.start_p, self.other), Map, quiet=True)
            if self.l.cof == self.cof:
                if self.cof == "horizontal":
                    if self.start_p.y > self.some_p.y:
                        if self.start_p.y > self.other.y:
                            return True
                    else:
                        if self.start_p.y < self.other.y:
                            return True
                else:
                    if ((self.start_p.y > self.other.y and self.start_p >
                    self.some_p) or (self.start_p.y < self.other.y and
                    self.start_p.y < self.some_p.y)) and ((self.start_p.x >
                    self.other.x and self.start_p > self.some_p) or
                    (self.start_p.x < self.other.x and self.start_p >
                    self.some_p)):
                        return True
        elif type(self.other) == group_of_points:
            for P in self.other.points:
                if not P in self:
                    return False
            return True
        return False


class direction:
    """Bacis direction class.
    """
    blits = False

    def __init__(self, point, angle, Map, description='Unknown',
        quiet=False):
        self.angle = angle
        self.rd = math.radians(self.angle)
        self.point = point
        self.description = description
        self.Map = Map
        if not quiet:
            self.Map.add_obj(self)

    def __str__(self):
        return self.description + " direction @" + str(self.point.x) +\
        ", " + str(self.point.y) + "; angle: " + str(self.angle)

    __repr__ = __str__

    def get_pos(self, distance):
        """Gets point of direction with given distance.
        """
        self.distance = distance
        if self.angle == 0:
            return point(self.point.x, self.point.y - self.distance,
            self.Map, quiet=True)
        else:
            self.x = math.sin(self.rd) * self.distance
            self.y = math.cos(self.rd) * self.distance
            return point(self.point.x + self.x, self.point.y - self.y,
            self.Map, quiet=True)

    def move(self, distance):
        """'Moves' directions point.
        """
        self.point.x, self.point.y = self.get_pos(distance).get_xy()

    def set_angle(self, angle):
        """Sets new angle.
        """
        self.angle = angle
        self.rd = math.radians(self.angle)

    def get_angle(self):
        """Returns direction angle.
        """
        return self.angle

    def ch_angle(self, change):
        """Changes angle for given value.
        """
        self.angle += change
        self.rd = math.radians(self.angle)

    def line(self, quiet=False):
        return line((self.point, self.get_pos(10)), self.Map, self.description, quiet)


class group_of_points:
    """Class for group of points.
    """
    blits = False

    def __init__(self, Map, description='Unknown', *points, quiet=False):
        self.Map = Map
        self.description = description
        self.points = points
        self.counter = 0
        if not quiet:
            self.Map.add_obj(self)
        self.name = self.description

    def __str__(self):
        self.fin = self.description + ' group ['
        for Point in self.points:
            self.fin += str(Point) + '; '
        self.fin = self.fin[:-2] + ']'
        return self.fin

    __repr__ = __str__

    def at(self, x, y):
        """Return generator of all items in group on x, y coordinates.
        """
        self.x = x
        self.y = y
        for Point in self.points:
            if Point.x == self.x and Point.y == self.y:
                yield Point


class rect:
    """Bacis map rect class.
    """
    blits = False

    def __init__(self, x, y, width, height, Map, description='Unknown',
        quiet=False):
        self.x = x
        self.y = y
        self.width = width
        self.height = height
        self.Map = Map
        if not quiet:
            self.Map.add_obj(self)
        self.description = description
        self.name = self.description

    def __str__(self):
        return self.description + ' rect ' + str(self.width) + 'X' + \
        str(self.height) + ' @ ' + str(self.x) + ', ' + str(self.y)

    __repr__ = __str__

    def __contains__(self, item):
        self.item = item
        if type(self.item) == point:
            if self.at(self.item.x, self.item.y):
                return True
        elif type(self.item) == group_of_points:
            for p in self.item.points:
                if not p in self:
                    return False
            return True
        elif type(self.item) == rect:
            if self.x <= self.item.x and self.y <= self.item.y and self.x \
            + self.width >= self.item.x + self.item.width and self.y + \
            self.height >= self.item.y + self.item.height:
                return True
            return False
        else:
            raise TypeError("'in <rect>' doesn't support " +
            repr(self.item))

    def at(self, x, y):
        """Test if point is in rect.*
        """
        return self.x + self.width >= x >= self.x and self.y + \
        self.height >= y >= self.y

    def collide(self, other):
        """Tests colliding with given rect.
        """
        self.fin = [0, 0, 0, 0]
        if self.y + self.height > other.y and self.y < other.y +\
        other.height:
            if other.x + other.width > self.x + self.width > other.x:
                self.fin[0] = self.x + self.width - other.x
            if self.x + self.width > other.x + other.width > self.x:
                self.fin[2] = other.x + other.width - self.x
        if self.x + self.width > other.x and self.x < other.x +\
        other.width:
            if other.y + other.height > self.y + self.height > other.y:
                self.fin[1] = self.y + self.height - other.y
            if self.y + self.height > other.y + other.height > self.y:
                self.fin[3] = other.y + other.height - self.y
        return self.fin

    def touch(self, other):
        """Tests touching with other rect.
        """
        self.fin = [False, False, False, False]
        if self.y + self.height > other.y and self.y < other.y +\
        other.height:
            if self.x + self.width == other.x:
                self.fin[0] = True
            if other.x + other.width == self.x:
                self.fin[2] = True
        if self.x + self.width > other.x and self.x < other.x +\
        other.width:
            if self.y + self.height == other.y:
                self.fin[1] = True
            if other.y + other.height == self.y:
                self.fin[3] = True
        return self.fin

    def points(self, quiet=True):
        yield point(self.x, self.y, self.Map, quiet=quiet)
        yield point(self.x + self.width, self.y, self.Map, quiet=quiet)
        yield point(self.x, self.y + self.height, self.Map, quiet=quiet)
        yield point(self.x + self.width, self.y + self.height, self.Map, quiet=quiet)


class circle:
    def __init__(self, centre, radius, Map, description='Unknown', quiet=False):
        self.centre = centre
        self.x, self.y = self.centre.get_xy()
        self.radius = radius
        self.Map = Map
        if not quiet:
            self.Map.add_obj(self)
        self.description = description

    def __repr__(self):
        return 'circle ' + self.description + ' @' + str(self.x) + ', ' + str(self.y) + '; r=' + str(self.radius)

    __str__ = __repr__


class element:
    def __init__(self, name, props, Map):
        self.name = name

        self.Map = Map

        self.props = props
        self.opts = []
        for prop in dict(self.props):
            self.opts.append(prop)
            if not self.props[prop]:
                del self.props[prop]

    @classmethod
    def __tmx_x_init__(cls, obj, Map):
        name = ""
        if "name" in obj.attrib:
            name = obj.attrib["name"]
        properties = {}
        if obj.find("properties"):
            for prop in obj.find("properties"):
                properties[prop.attrib["name"]] = prop.attrib["value"]
        if cls.__tmx_x_init__.__func__ == element.__tmx_x_init__.__func__:
            return cls(name, properties, Map)
        else:
            cls.name = name
            cls.properties = properties


class layer(element):
    def __init__(self, name, props, mapping, Map):
        super().__init__(name, props, Map)
        self.mapping = mapping
        self.Map = Map
        self.screen = self.Map.screen
        self.tiles = self.Map.images
        self.tile_width, self.tile_height = self.Map.tile_width, self.Map.tile_width
        self.x = self.y = 0

    @classmethod
    def __tmx_x_init__(cls, obj, Map):
        super().__tmx_x_init__(obj, Map)
        BIN = zlib.decompress(base64.b64decode(obj.find("data").text[4:-3]))
        mapping = [[] for x in range(int(Map.t_height))]
        for p, x in enumerate(BIN):
            if not p % 4:
                mapping[int(p/4/Map.t_width)].append(int(x))
        if cls.__tmx_x_init__.__func__ == layer.__tmx_x_init__.__func__:
            return cls(cls.name, cls.properties, mapping, Map)
        else:
            cls.mapping = mapping

    def set_pos(self, x, y):
        self.x = x
        self.y = y

    def blit(self):
        for p_y, y in enumerate(self.mapping):
            for p_x, x in enumerate(y):
                for tile in self.tiles:
                    if tile.Index(x):
                        tile.blit((p_x * self.tile_width - self.x, p_y * self.tile_height - self.y), x - tile.Index(x))


class Object(element):
    def __init__(self, name, Type, props, Map, obj):
        super().__init__(name, props, Map)
        self.type = Type
        self.obj = obj
        self.Map.objects.append(self)
        self.Map.in_map.add_obj(self)

    @classmethod
    def __tmx_x_init__(cls, obj, Map):
        super().__tmx_x_init__(obj, Map)
        Type = ""
        x = int(obj.attrib["x"])
        y = int(obj.attrib["y"])
        width = height = 0
        if "type" in obj.attrib:
            Type = obj.attrib["type"]
        if "width" in obj.attrib:
            width = int(obj.attrib["width"])
        if "height" in obj.attrib:
            height = int(obj.attrib["height"])
        if (not width or height) and Map.gid_point:
            Obj = point(x, y, Map.in_map, cls.name, True)
        elif (not width and height) and Map.gid_line:
            Obj = line(q_points(x, y, x + width, y + height, Map.in_map), Map.in_map, cls.name, True)
        else:
            Obj = rect(x, y, width, height, Map.in_map, cls.name, True)
        if cls.__tmx_x_init__.__func__ == Object.__tmx_x_init__.__func__:
            return cls(cls.name, Type, cls.properties, Map, Obj)
        else:
            cls.type = Type
            cls.x, cls.y = x, y
            cls.width, cls.height = width, height
            cls.obj = Obj

    def __str__(self):
        self.fin = "object " + str(self.obj)
        if self.opts:
            self.fin += '; ['
            for opt in self.opts:
                self.fin += str(opt)
                self.fin += ', '
            self.fin = self.fin[:-2]
            self.fin += ']'
        if self.props:
            self.fin += '; {'
            for prop in self.props:
                self.fin += str(prop)
                self.fin += ': '
                self.fin += str(self.props[prop])
                self.fin += ', '
            self.fin = self.fin[:-2]
            self.fin += '}'
        return self.fin

    __repr__ = __str__


class map_obj(Object):
    def __init__(self, name, Type, props, picture, Map, obj):
        super().__init__(name, Type, props, Map, obj)
        self.picture = picture

    @classmethod
    def __tmx_x_init__(cls, obj, Map):
        super().__init__(obj, Map)
        if cls.__tmx_x_init__.__func__ == map_obj.__tmx_x_init__.__func__:
            return cls(cls.name, cls.type, cls.props, None, cls.Map, cls.obj)

    def blit(self):
        self.Map.screen.blit(self.picture, self.get_blit())

    def get_blit(self):
        return (self.x - self.Map.x + self.Map.off_x, self.y - self.Map.y + self.Map.off_y)

    def set_position(self, x, y):
        self.x = x
        self.y = y

    def move(self, x, y):
        self.x += x
        self.y += y


class objectgroup(element):
    def __init__(self, name, props, objects, Map):
        super().__init__(name, props, Map)
        self.objects = objects

    def __iter__(self):
        for o in self.objects:
            yield o

    @classmethod
    def __tmx_x_init__(cls, obj, Map):
        super().__tmx_x_init__(obj, Map)
        objects = []
        for P in obj:
            if P.tag == "object":
                objects.append(Object.__tmx_x_init__(P, Map))
        if cls.__tmx_x_init__.__func__ == objectgroup.__tmx_x_init__.__func__:
            return cls(cls.name, cls.properties, objects, Map)
        else:
            cls.objects = objects


class image:
    def __init__(self, name, tile_x, tile_y, first, screen):
        self.image = pygame.image.load(name)
        self.tile_size = self.tile_x, self.tile_y = (tile_x, tile_y)
        self.images = []
        self.size = self.x, self.y = self.image.get_size()
        self.xt = self.x // self.tile_x
        self.yt = self.y // self.tile_y
        for y in range(self.yt):
            for x in range(self.xt):
                self.images.append(pygame.Rect(x * tile_x, y * tile_y, tile_x, tile_y))
        self.size = self.xt * self.yt
        self.first = first
        self.screen = screen

    def Index(self, index):
        return (self.first <= index <= self.first + self.size) * self.first

    def blit(self, pos, index):
        self.screen.blit(self.image, pos, self.images[index])


class tiled_map:
    def __init__(self, name, gid_point=True, gid_line=True):
        self.name = name + '.tmx'
        self.xml = Tree(file=self.name)
        self.out_map = self.xml.getroot()

        self.screen = pygame.display.get_surface()
        self.screen_w, self.screen_h = self.screen.get_size()
        
        self.tiles = self.t_width, self.t_height = (int(self.out_map.attrib["width"]), int(self.out_map.attrib["height"]))
        self.tile_size = self.tile_width, self.tile_height = (int(self.out_map.attrib["tilewidth"]), int(self.out_map.attrib["tileheight"]))
        self.size = self.width, self.height = self.t_width * self.tile_width, self.t_height * self.tile_height
        
        self.images = []
        self.layers = []
        self.edge_x = self.edge_y = 0
        self.edge_width, self.edge_height = self.size
        self.x = self.y = 0
        
        self.objects = []
        self.gid_point = gid_point
        self.gid_line = gid_line
        self.objectgroups = []
        self.in_map = MAP(self.width, self.height)

        for part in self.out_map:
            if part.tag == "tileset":
                self.im = part.find("image")
                self.images.append(image(self.im.attrib["source"], int(part.attrib["tilewidth"]), int(part.attrib["tileheight"]), int(part.attrib["firstgid"]), self.screen))
            elif part.tag == "layer":
                self.layers.append(layer.__tmx_x_init__(part, self))
            elif part.tag == "objectgroup":
                self.objectgroups.append(objectgroup.__tmx_x_init__(part, self))

    def __str__(self):
        return "Tiled m" + str(self.in_map)[1:]

    __repr__ = __str__

    def set_camera_pos(self, x, y, pos=(50, 50), edge=True):
        self.x = x
        self.y = y
        self.off_x, self.off_y = self.screen_w * pos[0] / 100, self.screen_h * pos[1] / 100
        self.x -= self.off_x
        self.y -= self.off_y
        self.edge = edge
        if self.edge:
            self.x = max(self.edge_x, min(self.x, self.edge_width - self.screen_w))
            self.y = max(self.edge_y, min(self.y, self.edge_height - self.screen_h))
        for l in self.layers:
            l.set_pos(self.x, self.y)
        return (self.x + self.off_x, self.y + self.off_y)

    def get_camera_pos(self):
        return (self.x + self.off_x, self.y + self.off_y)

    def blit(self):
        for l in self.layers:
            l.blit()

    def clone_obj(self, key, key_type="name"):
        self.final = []
        if key_type in ("group", "objectgroup"):
            for group in self.objectgroups:
                if group.name == key:
                    for obj in group:
                        self.final.append(obj)
        else:
            for obj in self.objects:
                if (obj.name if key_type == "name" else obj.type) == key:
                    self.final.append(obj)
        if len(self.final) > 1:
            return self.final
        elif self.final:
            return self.final[0]
        else:
            return None

    def set_edge(self, width, height):
        self.edge_width = width
        self.edge_height = height

    def offset(self, x, y):
        self.edge_x = x
        self.edge_y = y


class moving_map(tiled_map):
    def __init__(self, name, x, y):
        super().__init__(name)
        self.X = x
        self.Y = y
        self.set_camera_pos(self.X, self.Y)

    def set_position(self, x, y):
        self.X = x
        self.Y = y
        self.set_camera_pos(self.X, self.Y)

    def get_position(self):
        return (self.X, self.Y)

    def move(self, hor, ver):
        self.X += hor
        self.Y += ver
        self.set_camera_pos(self.X, self.Y)
