# bloggerfs.py - blogger file system in userspace using FUSE
# Copyright (C) 2010 Marco Giusti
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the Free
# Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
# MA 02111-1307, USA


# rfc3339.py -- Implementation of the majority of RFC 3339 for python.
# Copyright (c) 2008, 2009, 2010 LShift Ltd. <query@lshift.net>


# gdata - Python client library for Google data APIs
# Copyright (C) 2009 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


import os
import sys
import stat
import errno
import time
import functools
import logging

import fuse
import gdata.blogger.client
import rfc3339

__author__ = "Marco Giusti"
__copyright__ = "Copyright 2007, Marco Giusti"
__license__ = "LGPL"
__version__ = "0.1"
__email__ = "marco.giusti@gmail.com"


class NullHandler(logging.Handler):

    def emit(self, record):
        pass


fuse.fuse_python_api = (0, 2)
logger = logging.getLogger('bloggerfs')
logger.setLevel(logging.WARNING)
debug = logger.debug
info = logger.info
warn = logger.warn
error = logger.error
critical = logger.critical


def debugfunc(func):
    @functools.wraps(func)
    def wrapper(*args, **kwds):
        debug(func.__name__)
        for i, arg in enumerate(args):
            debug('args[%s]: %s' % (i, arg))
        for k in kwds:
            debug('kwds[%s]: %s' % (k, kwds[k]))
        try:
            ret = func(*args, **kwds)
        except:
            logger.exception('')
            raise
        debug('return value: %s\n' % ret)
        return ret
    return wrapper


class Node(object):

    __mtime = None
    __children = None

    @property
    def mtime(self):
        if self.__mtime is None:
            self.__mtime = int(time.time())
        return self.__mtime

    ctime = mtime

    def _children(self):
        return {}

    @property
    def children(self):
        if self.__children is None:
            self.__children = self._children()
        return self.__children

    def locate(self, path):
        if path == []:
            return self
        name = path.pop(0)
        children = self.children
        if name in children:
            return children[name].locate(path)
        raise IOError(errno.ENOENT)


class FileNode(Node):

    name = None # Implement this in subclasses
    text = None # Implement this in subclasses

    def dirEntry(self):
        return fuse.Direntry(self.name, type=stat.S_IFREG)

    def getattr(self, path):
        assert path == [], 'Inaspected path'
        size = len(self.text)
        st = fuse.Stat(st_mode=stat.S_IFREG|0440, st_nlink=1,
                       st_uid=os.getuid(), st_gid=os.getgid(), st_size=size,
                       st_ctime=self.ctime, st_mtime=self.mtime)
        return st

    def read(self, path, size, offset):
        return self.text[offset:offset+size]


class DirNode(Node):

    def dirEntry(self):
        return fuse.Direntry(self.name, type=stat.S_IFDIR)

    def readdir(self, path, offset):
        assert offset == 0, 'readdir offset != 0'
        assert path == [], 'readdir, inaspected path %s' % path
        yield fuse.Direntry('.')
        yield fuse.Direntry('..')
        for e in self.children.values():
            yield e.dirEntry()

    def getattr(self, path):
        assert path == [], 'Inaspected path'
        st = fuse.Stat(st_mode=stat.S_IFDIR|0550, st_nlink=2,
                       st_uid=os.geteuid(), st_gid=os.getegid(), st_size=4096,
                       st_ctime=self.ctime, st_mtime=self.mtime)
        return st


class FeedNode(DirNode):

    _mtime = None
    # _ctime = None

    def __init__(self, element, client):
        self._element = element
        self._client = client

    @property
    def mtime(self):
        if self._mtime is None and self._element.updated.text:
            try:
                updated = rfc3339.parse_datetime(self._element.updated.text)
                self._mtime = int(time.mktime(updated.timetuple()))
            except:
                exception()
                self._mtime = int(time.time())
        return self._mtime

    # @property
    # def ctime(self):
    #     if self._ctime is None:
    #         if hasattr(self._element, 'published'):
                # published = rfc3339.parse_datetime(self._element.published.text)
                # self._ctime = int(time.mktime(published.timetuple()))
    #         else:
    #             self._ctime = self.mtime
    #     return self._ctime


class Comment(FileNode):

    def __init__(self, element, client):
        self._element = element
        self._client = client

    @property
    def name(self):
        return self._element.get_comment_id()

    @property
    def text(self):
        return self._element.title.text


class Post(FeedNode):

    def _children(self):
        feed = self._client.get_post_comments(self.blogId, self.name)
        return dict((e.get_comment_id(), Comment(e, self._client))
                    for e in feed.entry)

    @property
    def name(self):
        return self._element.get_post_id()

    @property
    def blogId(self):
        return self._element.get_blog_id()


class Blog(FeedNode):

    @debugfunc
    def _children(self):
        feed = self._client.get_posts(self._element.get_blog_id())
        return dict((e.get_post_id(), Post(e, self._client))
                 for e in feed.entry)

    @property
    def name(self):
        return self._element.get_blog_id()


class Account(FeedNode):

    __element = None

    def __init__(self, client):
        self._client = client

    @property
    def _element(self):
        if not self.__element:
            self.__element = self._client.get_blogs()
        return self.__element

    def _children(self):
        children = dict((e.get_blog_id(), Blog(e, self._client))
                        for e in self._element.entry)
        return children


class BloggerFS(fuse.Fuse):

    def __init__(self, *args, **kwds):
        fuse.Fuse.__init__(self, *args, **kwds)
        self._client = gdata.blogger.client.BloggerClient()
        self._account = Account(self._client)

    def main(self):
        if self.parser.fuse_args.mount_expected():
            options = self.cmdline[0]
            if options.debug:
                logger.setLevel(logging.DEBUG)
            if options.syslog:
                from logging import handlers
                handler = logging.handlers.SysLogHandler()
            elif options.logfile:
                from logging import handlers
                handler = logging.handlers.FileHandler(options.logfile)
            elif options.debug:
                handler = logging.StreamHandler()
            else:
                handler = NullHandler()
            formatter = logging.Formatter('[%(asctime)s] [%(levelname)s] '
                                          '[%(funcName)s]  %(message)s')
            handler.setFormatter(formatter)
            logger.addHandler(handler)

            if not options.email:
                from gdata.sample_util import authorize_client
                authorize_client(self._client, service='blogger',
                                 source='bloggerfs',
                                 scopes=['http://www.blogger.com/feeds/'])
            else:
                email = options.email
                if not options.password:
                    import getpass
                    password = getpass.getpass()
                else:
                    password = options.password
                self._client.client_login(email, password, source='bloggerfs',
                                          service='blogger')
        fuse.Fuse.main(self)

    def _splitpath(self, path):
        return (path.split('/')[1:], [])[path == '/']

    def getattr(self, path):
        parts = self._splitpath(path)
        return self._account.locate(parts).getattr(parts)

    def readdir(self, path, offset):
        parts = self._splitpath(path)
        return self._account.locate(parts).readdir(parts, offset)

    def read(self, path, size, offset):
        parts = self._splitpath(path)
        return self._account.locate(parts).read(parts, size, offset)

    def write(self, path, buf, offset):
    #     parts = self._splitpath(path)
    #     return self._account.locate(parts).write(parts, size, offset)
        return 0

    # @debugfunc
    # def open(self, path, flags):
    #     return 0


def main(argv=None):
    if argv is None:
        import sys
        argv = sys.argv[1:]

    server = BloggerFS()
    server.parser.add_option('-e', '--email', help='Google account email')
    server.parser.add_option('-p', '--password', help='Google account password')
    server.parser.add_option('-b', '--debug', help='Show debugging info',
                             action='store_true')
    server.parser.add_option('--syslog', action='store_true',
                             help='Log messages to syslog')
    server.parser.add_option('-l', '--logfile', help='Log messages to file')
    server.parse(args=argv, errex=1)
    server.main()


if __name__ == '__main__':
    main()

