# Copyright (c) 2014, The MITRE Corporation. All rights reserved.
# See LICENSE.txt for complete terms.

import stix
import stix.bindings.exploit_target as exploit_target_binding
from stix.common import StructuredText


class Weakness(stix.Entity):
    _binding = exploit_target_binding
    _binding_class = _binding.WeaknessType
    _namespace = "http://stix.mitre.org/ExploitTarget-1"

    def __init__(self, description=None, cwe_id=None):
        self.description = description
        self.cwe_id = cwe_id

    @property
    def cwe_id(self):
        return self._cwe_id

    @cwe_id.setter
    def cwe_id(self, value):
        self._cwe_id = value

    @property
    def description(self):
        return self._description

    @description.setter
    def description(self, value):
        if value:
            if isinstance(value, StructuredText):
                self._description = value
            else:
                self._description = StructuredText(value=value)
        else:
            self._description = None

    def to_obj(self, return_obj=None):
        if not return_obj:
            return_obj = self._binding_class()

        if self.description:
            return_obj.set_Description(self.description.to_obj())
        return_obj.set_CWE_ID(self.cwe_id)

        return return_obj

    @classmethod
    def from_obj(cls, obj, return_obj=None):
        if not obj:
            return None
        if not return_obj:
            return_obj = cls()

        return_obj.description = StructuredText.from_obj(obj.get_Description())
        return_obj.cwe_id = obj.get_CWE_ID()

        return return_obj

    def to_dict(self):
        d = {}

        if self.description:
            d['description'] = self.description.to_dict()
        if self.cwe_id:
            d['cwe_id'] = self.cwe_id

        return d

    @classmethod
    def from_dict(cls, dict_repr, return_obj=None):
        if not dict_repr:
            return None
        if not return_obj:
            return_obj = cls()

        return_obj.description = StructuredText.from_dict(dict_repr.get('description'))
        return_obj.cwe_id = dict_repr.get('cwe_id')

        return return_obj
