# Functions to convert HSV colors to RGB colors lovingly ported from FastLED
#
#  The basically fall into two groups: spectra, and rainbows.
#  Spectra and rainbows are not the same thing.  Wikipedia has a good
#  illustration here
#   http://upload.wikimedia.org/wikipedia/commons/f/f6/Prism_compare_rainbow_01.png
#  from this article
#   http://en.wikipedia.org/wiki/Rainbow#Number_of_colours_in_spectrum_or_rainbow
#  that shows a 'spectrum' and a 'rainbow' side by side.  Among other
#  differences, you'll see that a 'rainbow' has much more yellow than
#  a plain spectrum.  "Classic" LED color washes are spectrum based, and
#  usually show very little yellow.
#
#  Wikipedia's page on HSV color space, with pseudocode for conversion
#  to RGB color space
#   http://en.wikipedia.org/wiki/HSL_and_HSV
#  Note that their conversion algorithm, which is (naturally) very popular
#  is in the "maximum brightness at any given hue" style, vs the "uniform
#  brightness for all hues" style.
#
#  You can't have both; either purple is the same brightness as red, e.g
#    red = #FF0000 and purple = #800080 -> same "total light" output
#  OR purple is 'as bright as it can be', e.g.
#    red = #FF0000 and purple = #FF00FF -> purple is much brighter than red.
#  The colorspace conversions here try to keep the apparent brightness
#  constant even as the hue varies.
#
#  Adafruit's "Wheel" function, discussed here
#   http://forums.adafruit.com/viewtopic.php?f=47&t=22483
#  is also of the "constant apparent brightness" variety.
#
# More details here: https://github.com/FastLED/FastLED/wiki/FastLED-HSV-Colors


import colorsys
import types

def color_scale(color, level):
    """Scale RGB tuple by level, 0 - 255"""
    return tuple([(i * level) >> 8 for i in list(color)])

def color_blend(a, b):
    """Performs a Screen blend on RGB color tuples, a and b"""
    return (255 - (((255-a[0]) * (255-b[0])) >> 8), 255 - (((255-a[1]) * (255-b[1])) >> 8), 255 - (((255-a[2]) * (255-b[2])) >> 8))

def gamma_correct(color, gamma):
    """Applies a gamma correction to an RGB color tuple"""
    return (gamma[color[2]],gamma[color[0]],gamma[color[2]])

#pre-generated spectrums for the sake of speed
hue_raw = [(251, 0, 0), (247, 3, 0), (243, 7, 0), (239, 11, 0), (235, 15, 0), (231, 19, 0), (227, 23, 0), (223, 27, 0), (219, 31, 0), (215, 35, 0), (211, 39, 0), (207, 43, 0), (203, 47, 0), (199, 51, 0), (195, 55, 0), (191, 59, 0), (187, 63, 0), (183, 67, 0), (179, 71, 0), (175, 75, 0), (171, 79, 0), (167, 83, 0), (163, 87, 0), (159, 91, 0), (155, 95, 0), (151, 99, 0), (147, 103, 0), (143, 107, 0), (139, 111, 0), (135, 115, 0), (131, 119, 0), (127, 123, 0), (123, 127, 0), (119, 131, 0), (115, 135, 0), (111, 139, 0), (107, 143, 0), (103, 147, 0), (99, 151, 0), (95, 155, 0), (91, 159, 0), (87, 163, 0), (83, 167, 0), (79, 171, 0), (75, 175, 0), (71, 179, 0), (67, 183, 0), (63, 187, 0), (59, 191, 0), (55, 195, 0), (51, 199, 0), (47, 203, 0), (43, 207, 0), (39, 211, 0), (35, 215, 0), (31, 219, 0), (27, 223, 0), (23, 227, 0), (19, 231, 0), (15, 235, 0), (11, 239, 0), (7, 243, 0), (3, 247, 0), (0, 251, 0), (0, 251, 0), (0, 247, 3), (0, 243, 7), (0, 239, 11), (0, 235, 15), (0, 231, 19), (0, 227, 23), (0, 223, 27), (0, 219, 31), (0, 215, 35), (0, 211, 39), (0, 207, 43), (0, 203, 47), (0, 199, 51), (0, 195, 55), (0, 191, 59), (0, 187, 63), (0, 183, 67), (0, 179, 71), (0, 175, 75), (0, 171, 79), (0, 167, 83), (0, 163, 87), (0, 159, 91), (0, 155, 95), (0, 151, 99), (0, 147, 103), (0, 143, 107), (0, 139, 111), (0, 135, 115), (0, 131, 119), (0, 127, 123), (0, 123, 127), (0, 119, 131), (0, 115, 135), (0, 111, 139), (0, 107, 143), (0, 103, 147), (0, 99, 151), (0, 95, 155), (0, 91, 159), (0, 87, 163), (0, 83, 167), (0, 79, 171), (0, 75, 175), (0, 71, 179), (0, 67, 183), (0, 63, 187), (0, 59, 191), (0, 55, 195), (0, 51, 199), (0, 47, 203), (0, 43, 207), (0, 39, 211), (0, 35, 215), (0, 31, 219), (0, 27, 223), (0, 23, 227), (0, 19, 231), (0, 15, 235), (0, 11, 239), (0, 7, 243), (0, 3, 247), (0, 0, 251), (0, 0, 251), (3, 0, 247), (7, 0, 243), (11, 0, 239), (15, 0, 235), (19, 0, 231), (23, 0, 227), (27, 0, 223), (31, 0, 219), (35, 0, 215), (39, 0, 211), (43, 0, 207), (47, 0, 203), (51, 0, 199), (55, 0, 195), (59, 0, 191), (63, 0, 187), (67, 0, 183), (71, 0, 179), (75, 0, 175), (79, 0, 171), (83, 0, 167), (87, 0, 163), (91, 0, 159), (95, 0, 155), (99, 0, 151), (103, 0, 147), (107, 0, 143), (111, 0, 139), (115, 0, 135), (119, 0, 131), (123, 0, 127), (127, 0, 123), (131, 0, 119), (135, 0, 115), (139, 0, 111), (143, 0, 107), (147, 0, 103), (151, 0, 99), (155, 0, 95), (159, 0, 91), (163, 0, 87), (167, 0, 83), (171, 0, 79), (175, 0, 75), (179, 0, 71), (183, 0, 67), (187, 0, 63), (191, 0, 59), (195, 0, 55), (199, 0, 51), (203, 0, 47), (207, 0, 43), (211, 0, 39), (215, 0, 35), (219, 0, 31), (223, 0, 27), (227, 0, 23), (231, 0, 19), (235, 0, 15), (239, 0, 11), (243, 0, 7), (247, 0, 3), (251, 0, 0), (0, 0, 251), (3, 0, 247), (7, 0, 243), (11, 0, 239), (15, 0, 235), (19, 0, 231), (23, 0, 227), (27, 0, 223), (31, 0, 219), (35, 0, 215), (39, 0, 211), (43, 0, 207), (47, 0, 203), (51, 0, 199), (55, 0, 195), (59, 0, 191), (63, 0, 187), (67, 0, 183), (71, 0, 179), (75, 0, 175), (79, 0, 171), (83, 0, 167), (87, 0, 163), (91, 0, 159), (95, 0, 155), (99, 0, 151), (103, 0, 147), (107, 0, 143), (111, 0, 139), (115, 0, 135), (119, 0, 131), (123, 0, 127), (127, 0, 123), (131, 0, 119), (135, 0, 115), (139, 0, 111), (143, 0, 107), (147, 0, 103), (151, 0, 99), (155, 0, 95), (159, 0, 91), (163, 0, 87), (167, 0, 83), (171, 0, 79), (175, 0, 75), (179, 0, 71), (183, 0, 67), (187, 0, 63), (191, 0, 59), (195, 0, 55), (199, 0, 51), (203, 0, 47), (207, 0, 43), (211, 0, 39), (215, 0, 35), (219, 0, 31), (223, 0, 27), (227, 0, 23), (231, 0, 19), (235, 0, 15), (239, 0, 11), (243, 0, 7), (247, 0, 3), (251, 0, 0)]
hue_rainbow = [(255, 0, 0), (253, 2, 0), (250, 5, 0), (248, 7, 0), (245, 10, 0), (242, 13, 0), (240, 15, 0), (237, 18, 0), (234, 21, 0), (232, 23, 0), (229, 26, 0), (226, 29, 0), (224, 31, 0), (221, 34, 0), (218, 37, 0), (216, 39, 0), (213, 42, 0), (210, 45, 0), (208, 47, 0), (205, 50, 0), (202, 53, 0), (200, 55, 0), (197, 58, 0), (194, 61, 0), (192, 63, 0), (189, 66, 0), (186, 69, 0), (184, 71, 0), (181, 74, 0), (178, 77, 0), (176, 79, 0), (173, 82, 0), (171, 85, 0), (171, 87, 0), (171, 90, 0), (171, 92, 0), (171, 95, 0), (171, 98, 0), (171, 100, 0), (171, 103, 0), (171, 106, 0), (171, 108, 0), (171, 111, 0), (171, 114, 0), (171, 116, 0), (171, 119, 0), (171, 122, 0), (171, 124, 0), (171, 127, 0), (171, 130, 0), (171, 132, 0), (171, 135, 0), (171, 138, 0), (171, 140, 0), (171, 143, 0), (171, 146, 0), (171, 148, 0), (171, 151, 0), (171, 154, 0), (171, 156, 0), (171, 159, 0), (171, 162, 0), (171, 164, 0), (171, 167, 0), (171, 171, 0), (167, 173, 0), (161, 176, 0), (157, 178, 0), (151, 181, 0), (145, 184, 0), (141, 186, 0), (135, 189, 0), (129, 192, 0), (125, 194, 0), (119, 197, 0), (113, 200, 0), (109, 202, 0), (103, 205, 0), (97, 208, 0), (93, 210, 0), (87, 213, 0), (81, 216, 0), (77, 218, 0), (71, 221, 0), (65, 224, 0), (61, 226, 0), (55, 229, 0), (49, 232, 0), (45, 234, 0), (39, 237, 0), (33, 240, 0), (29, 242, 0), (23, 245, 0), (17, 248, 0), (13, 250, 0), (7, 253, 0), (0, 255, 0), (0, 253, 2), (0, 250, 5), (0, 248, 7), (0, 245, 10), (0, 242, 13), (0, 240, 15), (0, 237, 18), (0, 234, 21), (0, 232, 23), (0, 229, 26), (0, 226, 29), (0, 224, 31), (0, 221, 34), (0, 218, 37), (0, 216, 39), (0, 213, 42), (0, 210, 45), (0, 208, 47), (0, 205, 50), (0, 202, 53), (0, 200, 55), (0, 197, 58), (0, 194, 61), (0, 192, 63), (0, 189, 66), (0, 186, 69), (0, 184, 71), (0, 181, 74), (0, 178, 77), (0, 176, 79), (0, 173, 82), (0, 171, 85), (0, 167, 89), (0, 161, 95), (0, 157, 99), (0, 151, 105), (0, 145, 111), (0, 141, 115), (0, 135, 121), (0, 129, 127), (0, 125, 131), (0, 119, 137), (0, 113, 143), (0, 109, 147), (0, 103, 153), (0, 97, 159), (0, 93, 163), (0, 87, 169), (0, 81, 175), (0, 77, 179), (0, 71, 185), (0, 65, 191), (0, 61, 195), (0, 55, 201), (0, 49, 207), (0, 45, 211), (0, 39, 217), (0, 33, 223), (0, 29, 227), (0, 23, 233), (0, 17, 239), (0, 13, 243), (0, 7, 249), (0, 0, 255), (2, 0, 253), (5, 0, 250), (7, 0, 248), (10, 0, 245), (13, 0, 242), (15, 0, 240), (18, 0, 237), (21, 0, 234), (23, 0, 232), (26, 0, 229), (29, 0, 226), (31, 0, 224), (34, 0, 221), (37, 0, 218), (39, 0, 216), (42, 0, 213), (45, 0, 210), (47, 0, 208), (50, 0, 205), (53, 0, 202), (55, 0, 200), (58, 0, 197), (61, 0, 194), (63, 0, 192), (66, 0, 189), (69, 0, 186), (71, 0, 184), (74, 0, 181), (77, 0, 178), (79, 0, 176), (82, 0, 173), (85, 0, 171), (87, 0, 169), (90, 0, 166), (92, 0, 164), (95, 0, 161), (98, 0, 158), (100, 0, 156), (103, 0, 153), (106, 0, 150), (108, 0, 148), (111, 0, 145), (114, 0, 142), (116, 0, 140), (119, 0, 137), (122, 0, 134), (124, 0, 132), (127, 0, 129), (130, 0, 126), (132, 0, 124), (135, 0, 121), (138, 0, 118), (140, 0, 116), (143, 0, 113), (146, 0, 110), (148, 0, 108), (151, 0, 105), (154, 0, 102), (156, 0, 100), (159, 0, 97), (162, 0, 94), (164, 0, 92), (167, 0, 89), (171, 0, 85), (173, 0, 83), (176, 0, 80), (178, 0, 78), (181, 0, 75), (184, 0, 72), (186, 0, 70), (189, 0, 67), (192, 0, 64), (194, 0, 62), (197, 0, 59), (200, 0, 56), (202, 0, 54), (205, 0, 51), (208, 0, 48), (210, 0, 46), (213, 0, 43), (216, 0, 40), (218, 0, 38), (221, 0, 35), (224, 0, 32), (226, 0, 30), (229, 0, 27), (232, 0, 24), (234, 0, 22), (237, 0, 19), (240, 0, 16), (242, 0, 14), (245, 0, 11), (248, 0, 8), (250, 0, 6), (253, 0, 3)]
hue_spectrum = [(251, 0, 0), (251, 0, 0), (247, 3, 0), (243, 7, 0), (239, 11, 0), (239, 11, 0), (235, 15, 0), (231, 19, 0), (227, 23, 0), (227, 23, 0), (223, 27, 0), (219, 31, 0), (215, 35, 0), (215, 35, 0), (211, 39, 0), (207, 43, 0), (203, 47, 0), (203, 47, 0), (199, 51, 0), (195, 55, 0), (191, 59, 0), (191, 59, 0), (187, 63, 0), (183, 67, 0), (179, 71, 0), (179, 71, 0), (175, 75, 0), (171, 79, 0), (167, 83, 0), (167, 83, 0), (163, 87, 0), (159, 91, 0), (155, 95, 0), (155, 95, 0), (151, 99, 0), (147, 103, 0), (143, 107, 0), (143, 107, 0), (139, 111, 0), (135, 115, 0), (131, 119, 0), (131, 119, 0), (127, 123, 0), (123, 127, 0), (119, 131, 0), (119, 131, 0), (115, 135, 0), (111, 139, 0), (107, 143, 0), (107, 143, 0), (103, 147, 0), (99, 151, 0), (95, 155, 0), (95, 155, 0), (91, 159, 0), (87, 163, 0), (83, 167, 0), (83, 167, 0), (79, 171, 0), (75, 175, 0), (71, 179, 0), (71, 179, 0), (67, 183, 0), (63, 187, 0), (59, 191, 0), (59, 191, 0), (55, 195, 0), (51, 199, 0), (47, 203, 0), (47, 203, 0), (43, 207, 0), (39, 211, 0), (35, 215, 0), (35, 215, 0), (31, 219, 0), (27, 223, 0), (23, 227, 0), (23, 227, 0), (19, 231, 0), (15, 235, 0), (11, 239, 0), (11, 239, 0), (7, 243, 0), (3, 247, 0), (0, 251, 0), (0, 251, 0), (0, 251, 0), (0, 247, 3), (0, 243, 7), (0, 243, 7), (0, 239, 11), (0, 235, 15), (0, 231, 19), (0, 231, 19), (0, 227, 23), (0, 223, 27), (0, 219, 31), (0, 219, 31), (0, 215, 35), (0, 211, 39), (0, 207, 43), (0, 207, 43), (0, 203, 47), (0, 199, 51), (0, 195, 55), (0, 195, 55), (0, 191, 59), (0, 187, 63), (0, 183, 67), (0, 183, 67), (0, 179, 71), (0, 175, 75), (0, 171, 79), (0, 171, 79), (0, 167, 83), (0, 163, 87), (0, 159, 91), (0, 159, 91), (0, 155, 95), (0, 151, 99), (0, 147, 103), (0, 147, 103), (0, 143, 107), (0, 139, 111), (0, 135, 115), (0, 135, 115), (0, 131, 119), (0, 127, 123), (0, 123, 127), (0, 123, 127), (0, 119, 131), (0, 115, 135), (0, 111, 139), (0, 111, 139), (0, 107, 143), (0, 103, 147), (0, 99, 151), (0, 99, 151), (0, 95, 155), (0, 91, 159), (0, 87, 163), (0, 87, 163), (0, 83, 167), (0, 79, 171), (0, 75, 175), (0, 75, 175), (0, 71, 179), (0, 67, 183), (0, 63, 187), (0, 63, 187), (0, 59, 191), (0, 55, 195), (0, 51, 199), (0, 51, 199), (0, 47, 203), (0, 43, 207), (0, 39, 211), (0, 39, 211), (0, 35, 215), (0, 31, 219), (0, 27, 223), (0, 27, 223), (0, 23, 227), (0, 19, 231), (0, 15, 235), (0, 15, 235), (0, 11, 239), (0, 7, 243), (0, 3, 247), (0, 3, 247), (0, 0, 251), (0, 0, 251), (3, 0, 247), (3, 0, 247), (7, 0, 243), (11, 0, 239), (15, 0, 235), (15, 0, 235), (19, 0, 231), (23, 0, 227), (27, 0, 223), (27, 0, 223), (31, 0, 219), (35, 0, 215), (39, 0, 211), (39, 0, 211), (43, 0, 207), (47, 0, 203), (51, 0, 199), (51, 0, 199), (55, 0, 195), (59, 0, 191), (63, 0, 187), (63, 0, 187), (67, 0, 183), (71, 0, 179), (75, 0, 175), (75, 0, 175), (79, 0, 171), (83, 0, 167), (87, 0, 163), (87, 0, 163), (91, 0, 159), (95, 0, 155), (99, 0, 151), (99, 0, 151), (103, 0, 147), (107, 0, 143), (111, 0, 139), (111, 0, 139), (115, 0, 135), (119, 0, 131), (123, 0, 127), (123, 0, 127), (127, 0, 123), (131, 0, 119), (135, 0, 115), (135, 0, 115), (139, 0, 111), (143, 0, 107), (147, 0, 103), (147, 0, 103), (151, 0, 99), (155, 0, 95), (159, 0, 91), (159, 0, 91), (163, 0, 87), (167, 0, 83), (171, 0, 79), (171, 0, 79), (175, 0, 75), (179, 0, 71), (183, 0, 67), (183, 0, 67), (187, 0, 63), (191, 0, 59), (195, 0, 55), (195, 0, 55), (199, 0, 51), (203, 0, 47), (207, 0, 43), (207, 0, 43), (211, 0, 39), (215, 0, 35), (219, 0, 31), (219, 0, 31), (223, 0, 27), (227, 0, 23), (231, 0, 19), (231, 0, 19), (235, 0, 15), (239, 0, 11), (243, 0, 7), (243, 0, 7), (247, 0, 3), (251, 0, 0)]
hue_360 = [(255, 0, 0), (255, 4, 0), (255, 8, 0), (255, 12, 0), (255, 16, 0), (255, 21, 0), (255, 25, 0), (255, 29, 0), (255, 33, 0), (255, 38, 0), (255, 42, 0), (255, 46, 0), (255, 50, 0), (255, 55, 0), (255, 59, 0), (255, 63, 0), (255, 67, 0), (255, 72, 0), (255, 76, 0), (255, 80, 0), (255, 84, 0), (255, 89, 0), (255, 93, 0), (255, 97, 0), (255, 102, 0), (255, 106, 0), (255, 110, 0), (255, 114, 0), (255, 119, 0), (255, 123, 0), (255, 127, 0), (255, 131, 0), (255, 136, 0), (255, 140, 0), (255, 144, 0), (255, 148, 0), (255, 153, 0), (255, 157, 0), (255, 161, 0), (255, 165, 0), (255, 170, 0), (255, 174, 0), (255, 178, 0), (255, 182, 0), (255, 187, 0), (255, 191, 0), (255, 195, 0), (255, 199, 0), (255, 204, 0), (255, 208, 0), (255, 212, 0), (255, 216, 0), (255, 220, 0), (255, 225, 0), (255, 229, 0), (255, 233, 0), (255, 238, 0), (255, 242, 0), (255, 246, 0), (255, 250, 0), (255, 255, 0), (250, 255, 0), (246, 255, 0), (242, 255, 0), (238, 255, 0), (233, 255, 0), (229, 255, 0), (225, 255, 0), (221, 255, 0), (216, 255, 0), (212, 255, 0), (208, 255, 0), (203, 255, 0), (199, 255, 0), (195, 255, 0), (191, 255, 0), (187, 255, 0), (182, 255, 0), (178, 255, 0), (174, 255, 0), (170, 255, 0), (165, 255, 0), (161, 255, 0), (157, 255, 0), (153, 255, 0), (148, 255, 0), (144, 255, 0), (140, 255, 0), (136, 255, 0), (131, 255, 0), (127, 255, 0), (123, 255, 0), (119, 255, 0), (114, 255, 0), (110, 255, 0), (106, 255, 0), (101, 255, 0), (97, 255, 0), (93, 255, 0), (89, 255, 0), (84, 255, 0), (80, 255, 0), (76, 255, 0), (72, 255, 0), (68, 255, 0), (63, 255, 0), (59, 255, 0), (55, 255, 0), (51, 255, 0), (46, 255, 0), (42, 255, 0), (38, 255, 0), (33, 255, 0), (29, 255, 0), (25, 255, 0), (21, 255, 0), (16, 255, 0), (12, 255, 0), (8, 255, 0), (4, 255, 0), (0, 255, 0), (0, 255, 4), (0, 255, 8), (0, 255, 12), (0, 255, 16), (0, 255, 21), (0, 255, 25), (0, 255, 29), (0, 255, 33), (0, 255, 38), (0, 255, 42), (0, 255, 46), (0, 255, 50), (0, 255, 55), (0, 255, 59), (0, 255, 63), (0, 255, 67), (0, 255, 72), (0, 255, 76), (0, 255, 80), (0, 255, 85), (0, 255, 89), (0, 255, 93), (0, 255, 97), (0, 255, 102), (0, 255, 106), (0, 255, 110), (0, 255, 114), (0, 255, 119), (0, 255, 123), (0, 255, 127), (0, 255, 131), (0, 255, 135), (0, 255, 140), (0, 255, 144), (0, 255, 148), (0, 255, 153), (0, 255, 157), (0, 255, 161), (0, 255, 165), (0, 255, 169), (0, 255, 174), (0, 255, 178), (0, 255, 182), (0, 255, 187), (0, 255, 191), (0, 255, 195), (0, 255, 199), (0, 255, 203), (0, 255, 208), (0, 255, 212), (0, 255, 216), (0, 255, 221), (0, 255, 225), (0, 255, 229), (0, 255, 233), (0, 255, 237), (0, 255, 242), (0, 255, 246), (0, 255, 250), (0, 255, 255), (0, 250, 255), (0, 246, 255), (0, 242, 255), (0, 238, 255), (0, 233, 255), (0, 229, 255), (0, 225, 255), (0, 220, 255), (0, 216, 255), (0, 212, 255), (0, 208, 255), (0, 203, 255), (0, 199, 255), (0, 195, 255), (0, 191, 255), (0, 187, 255), (0, 182, 255), (0, 178, 255), (0, 174, 255), (0, 169, 255), (0, 165, 255), (0, 161, 255), (0, 157, 255), (0, 153, 255), (0, 148, 255), (0, 144, 255), (0, 140, 255), (0, 136, 255), (0, 131, 255), (0, 127, 255), (0, 123, 255), (0, 119, 255), (0, 114, 255), (0, 110, 255), (0, 106, 255), (0, 102, 255), (0, 97, 255), (0, 93, 255), (0, 89, 255), (0, 84, 255), (0, 80, 255), (0, 76, 255), (0, 72, 255), (0, 67, 255), (0, 63, 255), (0, 59, 255), (0, 55, 255), (0, 51, 255), (0, 46, 255), (0, 42, 255), (0, 38, 255), (0, 33, 255), (0, 29, 255), (0, 25, 255), (0, 21, 255), (0, 16, 255), (0, 12, 255), (0, 8, 255), (0, 4, 255), (0, 0, 255), (4, 0, 255), (8, 0, 255), (12, 0, 255), (16, 0, 255), (21, 0, 255), (25, 0, 255), (29, 0, 255), (33, 0, 255), (38, 0, 255), (42, 0, 255), (46, 0, 255), (50, 0, 255), (55, 0, 255), (59, 0, 255), (63, 0, 255), (67, 0, 255), (72, 0, 255), (76, 0, 255), (80, 0, 255), (84, 0, 255), (89, 0, 255), (93, 0, 255), (97, 0, 255), (101, 0, 255), (106, 0, 255), (110, 0, 255), (114, 0, 255), (119, 0, 255), (123, 0, 255), (127, 0, 255), (131, 0, 255), (135, 0, 255), (140, 0, 255), (144, 0, 255), (148, 0, 255), (153, 0, 255), (157, 0, 255), (161, 0, 255), (165, 0, 255), (170, 0, 255), (174, 0, 255), (178, 0, 255), (182, 0, 255), (187, 0, 255), (191, 0, 255), (195, 0, 255), (199, 0, 255), (204, 0, 255), (208, 0, 255), (212, 0, 255), (216, 0, 255), (221, 0, 255), (225, 0, 255), (229, 0, 255), (233, 0, 255), (238, 0, 255), (242, 0, 255), (246, 0, 255), (250, 0, 255), (255, 0, 255), (255, 0, 250), (255, 0, 246), (255, 0, 242), (255, 0, 238), (255, 0, 233), (255, 0, 229), (255, 0, 225), (255, 0, 221), (255, 0, 216), (255, 0, 212), (255, 0, 208), (255, 0, 203), (255, 0, 199), (255, 0, 195), (255, 0, 191), (255, 0, 187), (255, 0, 182), (255, 0, 178), (255, 0, 174), (255, 0, 170), (255, 0, 165), (255, 0, 161), (255, 0, 157), (255, 0, 152), (255, 0, 148), (255, 0, 144), (255, 0, 140), (255, 0, 135), (255, 0, 131), (255, 0, 127), (255, 0, 123), (255, 0, 119), (255, 0, 114), (255, 0, 110), (255, 0, 106), (255, 0, 102), (255, 0, 97), (255, 0, 93), (255, 0, 89), (255, 0, 85), (255, 0, 80), (255, 0, 76), (255, 0, 72), (255, 0, 67), (255, 0, 63), (255, 0, 59), (255, 0, 55), (255, 0, 51), (255, 0, 46), (255, 0, 42), (255, 0, 38), (255, 0, 34), (255, 0, 29), (255, 0, 25), (255, 0, 21), (255, 0, 16), (255, 0, 12), (255, 0, 8), (255, 0, 4)]


def hsv2rgb_raw(hsv):
    """
    Converts an HSV tuple to RGB. Intended for internal use.
    You should use hsv2rgb_spectrum or hsv2rgb_rainbow instead.
    """

    HSV_SECTION_6 = 0x20
    HSV_SECTION_3 = 0x40
    
    h, s, v = hsv

    #The brightness floor is minimum number that all of
    #R, G, and B will be set to.
    invsat = 255 - s
    brightness_floor = (v * invsat) / 256
    
    #The color amplitude is the maximum amount of R, G, and B
    #that will be added on top of the brightness_floor to
    #create the specific hue desired.
    color_amplitude = v - brightness_floor
    
    #figure out which section of the hue wheel we're in,
    #and how far offset we are within that section
    section = h / HSV_SECTION_3 #0..2
    offset = h % HSV_SECTION_3 #0..63
    
    rampup = offset
    rampdown = (HSV_SECTION_3 - 1) - offset
    
    # compute color-amplitude-scaled-down versions of rampup and rampdown
    rampup_amp_adj   = (rampup   * color_amplitude) / (256 / 4);
    rampdown_amp_adj = (rampdown * color_amplitude) / (256 / 4);
    
    # add brightness_floor offset to everything
    rampup_adj_with_floor   = rampup_amp_adj   + brightness_floor;
    rampdown_adj_with_floor = rampdown_amp_adj + brightness_floor;
    
    r, g, b = (0,0,0)
    
    if section:
        if section == 1:
            # section 1: 0x40..0x7F
            r = brightness_floor;
            g = rampdown_adj_with_floor;
            b = rampup_adj_with_floor;
        else:
            # section 2; 0x80..0xBF
            r = rampup_adj_with_floor;
            g = brightness_floor;
            b = rampdown_adj_with_floor;    
    else:
        # section 0: 0x00..0x3F
        r = rampdown_adj_with_floor;
        g = rampup_adj_with_floor;
        b = brightness_floor;
        
    return (r, g, b)

def hue2rgb_raw(hue):
    if hue >=0 or hue < 256:
        return hue_raw[hue]
    else:
        raise ValueError("hue must be between 0 and 255")
 
def hsv2rgb_spectrum(hsv):
    """Generates RGB values from  HSV values in line with a typical light spectrum"""
    h, s, v = hsv
    return hsv2rgb_raw(((h * 192) >> 8, s, v))

def hue2rgb_spectrum(hue):
    if hue >=0 or hue < 256:
        return hue_spectrum[hue]
    else:
        raise ValueError("hue must be between 0 and 255")
    
def _nscale8x3_video(r, g, b, scale):
    """Internal Use Only"""
    nonzeroscale = 0
    if scale != 0: nonzeroscale = 1
    if r != 0:
        r = ((r * scale) >> 8) + nonzeroscale
    if g != 0:
        g = ((g * scale) >> 8) + nonzeroscale
    if b != 0:
        b = ((b * scale) >> 8) + nonzeroscale
    return (r, g, b)
    
def _scale8_video_LEAVING_R1_DIRTY( i, scale):
    """Internal Use Only"""
    nonzeroscale = 0
    if scale != 0: nonzeroscale = 1
    if i != 0:
        i = ((i * scale) >> 8) + nonzeroscale
    return i;

def hsv2rgb_rainbow(hsv):
    """Generates RGB values from HSV that have an even visual distribution.
       Be careful as this method is only have as fast as hsv2rgb_spectrum.
    """

    h, s, v = hsv
    offset = h & 0x1F #0..31
    offset8 = offset * 8
    third = (offset8 * (256/3)) >> 8
    r, g, b = (0,0,0)
    
    if not (h & 0x80):
        if not (h & 0x40):
            if not (h & 0x20):
                r = 255 - third
                g = third
                b = 0
            else:
                r = 171
                g = 85 + third
                b = 0x00
        else:
            if not (h & 0x20):
                twothirds = (third << 1)
                r = 171 - twothirds
                g = 171 + third
                b = 0
            else:
                r = 0
                g = 255 - third
                b = third
    else:
        if not (h & 0x40):
            if not (h & 0x20):
                r = 0x00
                twothirds = (third << 1)
                g = 171 - twothirds
                b = 85 + twothirds
            else:
                r = third
                g = 0
                b = 255 - third
        else:
            if not (h & 0x20):
                r = 85 + third
                g = 0
                b = 171 - third
            else:
                r = 171 + third
                g = 0x00
                b = 85 - third
            
    if s != 255:
        r, g, b = _nscale8x3_video(r, g, b, s)
        desat = 255 - s
        desat = (desat * desat) >> 8
        brightness_floor = desat
        r = r + brightness_floor
        g = g + brightness_floor
        b = b + brightness_floor
        
    if v != 255:
        v = _scale8_video_LEAVING_R1_DIRTY(v, v)
        r, g, b = _nscale8x3_video( r, g, b, v)
        
    return (r, g, b)

def hue2rgb_rainbow(hue):
    if hue >=0 or hue < 256:
        return hue_rainbow[hue]
    else:
        raise ValueError("hue must be between 0 and 255")

hsv2rgb = hsv2rgb_rainbow
hue2rgb = hue2rgb_rainbow

def hsv2rgb_360(hsv):
    """
    Python default hsv to rgb conversion for when hue values 0-359 are preferred.
    Due to requiring float math, this method is slower than hsv2rgb_rainbow and hsv2rgb_spectrum.
    """

    h, s, v = hsv

    r, g, b = colorsys.hsv_to_rgb(h / 360.0, s, v)
    return (int(r * 255.0), int(g * 255.0), int(b * 255.0))

def hue2rgb_360(hue):
    if hue >=0 or hue < 360:
        return hue_360[hue]
    else:
        raise ValueError("hue must be between 0 and 359")
    
def hex2rgb(hex):
    """Helper for converting RGB and RGBA hex values to Color"""
    hex = hex.strip('#')
    if len(hex) == 6:
        split = (hex[0:2],hex[2:4],hex[4:6])
    else:
        raise ValueError('Must pass in a 6 character hex value!')

    r, g, b = [int(x, 16) for x in split]

    return (r, g, b)


WHEEL_MAX = 384
_wheel = [(127, 0, 0), (126, 1, 0), (125, 2, 0), (124, 3, 0), (123, 4, 0), (122, 5, 0), (121, 6, 0), (120, 7, 0), (119, 8, 0), (118, 9, 0), (117, 10, 0), (116, 11, 0), (115, 12, 0), (114, 13, 0), (113, 14, 0), (112, 15, 0), (111, 16, 0), (110, 17, 0), (109, 18, 0), (108, 19, 0), (107, 20, 0), (106, 21, 0), (105, 22, 0), (104, 23, 0), (103, 24, 0), (102, 25, 0), (101, 26, 0), (100, 27, 0), (99, 28, 0), (98, 29, 0), (97, 30, 0), (96, 31, 0), (95, 32, 0), (94, 33, 0), (93, 34, 0), (92, 35, 0), (91, 36, 0), (90, 37, 0), (89, 38, 0), (88, 39, 0), (87, 40, 0), (86, 41, 0), (85, 42, 0), (84, 43, 0), (83, 44, 0), (82, 45, 0), (81, 46, 0), (80, 47, 0), (79, 48, 0), (78, 49, 0), (77, 50, 0), (76, 51, 0), (75, 52, 0), (74, 53, 0), (73, 54, 0), (72, 55, 0), (71, 56, 0), (70, 57, 0), (69, 58, 0), (68, 59, 0), (67, 60, 0), (66, 61, 0), (65, 62, 0), (64, 63, 0), (63, 64, 0), (62, 65, 0), (61, 66, 0), (60, 67, 0), (59, 68, 0), (58, 69, 0), (57, 70, 0), (56, 71, 0), (55,72, 0), (54, 73, 0), (53, 74, 0), (52, 75, 0), (51, 76, 0), (50, 77, 0), (49, 78, 0), (48, 79, 0), (47, 80, 0), (46, 81, 0), (45, 82, 0), (44, 83, 0), (43, 84,0), (42, 85, 0), (41, 86, 0), (40, 87, 0), (39, 88, 0), (38, 89, 0), (37, 90, 0), (36, 91, 0), (35, 92, 0), (34, 93, 0), (33, 94, 0), (32, 95, 0), (31, 96, 0),(30, 97, 0), (29, 98, 0), (28, 99, 0), (27, 100, 0), (26, 101, 0), (25, 102, 0), (24, 103, 0), (23, 104, 0), (22, 105, 0), (21, 106, 0), (20, 107, 0), (19, 108, 0), (18, 109, 0), (17, 110, 0), (16, 111, 0), (15, 112, 0), (14, 113, 0), (13,114, 0), (12, 115, 0), (11, 116, 0), (10, 117, 0), (9, 118, 0), (8, 119, 0), (7, 120, 0), (6, 121, 0), (5, 122, 0), (4, 123, 0), (3, 124, 0), (2, 125, 0), (1, 126, 0), (0, 127, 0), (0, 127, 0), (0, 126, 1), (0, 125, 2), (0, 124, 3), (0, 123, 4), (0, 122, 5), (0, 121, 6), (0, 120, 7), (0, 119, 8), (0, 118, 9), (0, 117,10), (0, 116, 11), (0, 115, 12), (0, 114, 13), (0, 113, 14), (0, 112, 15), (0, 111, 16), (0, 110, 17), (0, 109, 18), (0, 108, 19), (0, 107, 20), (0, 106, 21), (0, 105, 22), (0, 104, 23), (0, 103, 24), (0, 102, 25), (0, 101, 26), (0, 100, 27), (0, 99, 28), (0, 98, 29), (0, 97, 30), (0, 96, 31), (0, 95, 32), (0, 94, 33), (0, 93, 34), (0, 92, 35), (0, 91, 36), (0, 90, 37), (0, 89, 38), (0, 88, 39), (0, 87, 40), (0, 86, 41), (0, 85, 42), (0, 84, 43), (0, 83, 44), (0, 82, 45), (0, 81, 46), (0, 80, 47), (0, 79, 48), (0, 78, 49), (0, 77, 50), (0, 76, 51), (0, 75, 52), (0, 74, 53), (0, 73, 54), (0, 72, 55), (0, 71, 56), (0, 70, 57), (0, 69, 58), (0, 68, 59), (0, 67, 60), (0, 66, 61), (0, 65, 62), (0, 64, 63), (0, 63, 64), (0, 62, 65), (0, 61, 66), (0, 60, 67), (0, 59, 68), (0, 58, 69), (0, 57, 70), (0, 56, 71), (0, 55, 72), (0, 54, 73), (0, 53, 74), (0, 52, 75), (0, 51, 76),(0, 50, 77), (0, 49, 78), (0, 48, 79), (0, 47, 80), (0, 46, 81), (0, 45, 82), (0, 44, 83), (0, 43, 84), (0, 42, 85), (0, 41, 86), (0, 40, 87), (0, 39, 88), (0,38, 89), (0, 37, 90), (0, 36, 91), (0, 35, 92), (0, 34, 93), (0, 33, 94), (0, 32, 95), (0, 31, 96), (0, 30, 97), (0, 29, 98), (0, 28, 99), (0, 27, 100), (0, 26, 101), (0, 25, 102), (0, 24, 103), (0, 23, 104), (0, 22, 105), (0, 21, 106), (0, 20, 107), (0, 19, 108), (0, 18, 109), (0, 17, 110), (0, 16, 111), (0, 15, 112), (0, 14, 113), (0, 13, 114), (0, 12, 115), (0, 11, 116), (0, 10, 117), (0, 9, 118), (0, 8, 119), (0, 7, 120), (0, 6, 121), (0, 5, 122), (0, 4, 123), (0, 3, 124), (0, 2, 125), (0, 1, 126), (0, 0, 127), (0, 0, 127), (1, 0, 126), (2, 0, 125),(3, 0, 124), (4, 0, 123), (5, 0, 122), (6, 0, 121), (7, 0, 120), (8, 0, 119), (9, 0, 118), (10, 0, 117), (11, 0, 116), (12, 0, 115), (13, 0, 114), (14, 0, 113), (15, 0, 112), (16, 0, 111), (17, 0, 110), (18, 0, 109), (19, 0, 108), (20, 0, 107), (21, 0, 106), (22, 0, 105), (23, 0, 104), (24, 0, 103), (25, 0, 102), (26,0, 101), (27, 0, 100), (28, 0, 99), (29, 0, 98), (30, 0, 97), (31, 0, 96), (32,0, 95), (33, 0, 94), (34, 0, 93), (35, 0, 92), (36, 0, 91), (37, 0, 90), (38, 0, 89), (39, 0, 88), (40, 0, 87), (41, 0, 86), (42, 0, 85), (43, 0, 84), (44, 0, 83), (45, 0, 82), (46, 0, 81), (47, 0, 80), (48, 0, 79), (49, 0, 78), (50, 0, 77), (51, 0, 76), (52, 0, 75), (53, 0, 74), (54, 0, 73), (55, 0, 72), (56, 0, 71),(57, 0, 70), (58, 0, 69), (59, 0, 68), (60, 0, 67), (61, 0, 66), (62, 0, 65), (63, 0, 64), (64, 0, 63), (65, 0, 62), (66, 0, 61), (67, 0, 60), (68, 0, 59), (69, 0, 58), (70, 0, 57), (71, 0, 56), (72, 0, 55), (73, 0, 54), (74, 0, 53), (75, 0, 52), (76, 0, 51), (77, 0, 50), (78, 0, 49), (79, 0, 48), (80, 0, 47), (81, 0,46), (82, 0, 45), (83, 0, 44), (84, 0, 43), (85, 0, 42), (86, 0, 41), (87, 0, 40), (88, 0, 39), (89, 0, 38), (90, 0, 37), (91, 0, 36), (92, 0, 35), (93, 0, 34), (94, 0, 33), (95, 0, 32), (96, 0, 31), (97, 0, 30), (98, 0, 29), (99, 0, 28), (100, 0, 27), (101, 0, 26), (102, 0, 25), (103, 0, 24), (104, 0, 23), (105, 0, 22), (106, 0, 21), (107, 0, 20), (108, 0, 19), (109, 0, 18), (110, 0, 17), (111, 0, 16), (112, 0, 15), (113, 0, 14), (114, 0, 13), (115, 0, 12), (116, 0, 11), (117, 0, 10), (118, 0, 9), (119, 0, 8), (120, 0, 7), (121, 0, 6), (122, 0, 5), (123, 0, 4), (124, 0, 3), (125, 0, 2), (126, 0, 1), (127, 0, 0)]

def wheel_color(position):
    """Get color from wheel value (0 - 384).
    Provided for those used to using it from Adafruit libraries
    """
    if position < 0:
        position = 0
    if position > 384:
        position = 384

    return _wheel[position]
    #Below is original implementation. Using a pre-generated list is faster
    #if position < 128:
    #    r = 127 - position % 128
    #    g = position % 128
    #    b = 0
    #elif position < 256:
    #    g = 127 - position % 128
    #    b = position % 128
    #    r = 0
    #else:
    #    b = 127 - position % 128
    #    r = position % 128
    #    g = 0

    #return (r, g, b)

def hue_helper(pos, length, cycle_step):
    return hue2rgb_rainbow(((pos * 255 / length) + cycle_step) % 255)

def wheel_helper(pos, length, cycle_step):
    """Helper for wheel_color that distributes colors over length and allows shifting position"""
    return wheel_color(((pos * WHEEL_MAX / length) + cycle_step) % WHEEL_MAX);

Off = (0, 0, 0)
Blue = (0, 0, 255)
Pink = (255, 192, 203)
Honeydew = (240, 255, 240)
Purple = (128, 0, 128)
Fuchsia = (255, 0, 255)
LawnGreen = (124, 252, 0)
AliceBlue = (240, 248, 255)
Crimson = (220, 20, 60)
White = (255, 255, 255)
NavajoWhite = (255, 222, 173)
Cornsilk = (255, 248, 220)
Bisque = (255, 228, 196)
PaleGreen = (152, 251, 152)
Brown = (165, 42, 42)
DarkTurquoise = (0, 206, 209)
DarkGreen = (0, 100, 0)
MediumOrchid = (186, 85, 211)
Chocolate = (210, 105, 30)
PapayaWhip = (255, 239, 213)
Olive = (128, 128, 0)
DarkSalmon = (233, 150, 122)
PeachPuff = (255, 218, 185)
Plum = (221, 160, 221)
DarkGoldenrod = (184, 134, 11)
MintCream = (245, 255, 250)
CornflowerBlue = (100, 149, 237)
HotPink = (255, 105, 180)
DarkBlue = (0, 0, 139)
LimeGreen = (50, 205, 50)
DeepSkyBlue = (0, 191, 255)
DarkKhaki = (189, 183, 107)
LightGrey = (211, 211, 211)
Yellow = (255, 255, 0)
LightSalmon = (255, 160, 122)
MistyRose = (255, 228, 225)
SandyBrown = (244, 164, 96)
DeepPink = (255, 20, 147)
Magenta = (255, 0, 255)
Amethyst = (153, 102, 204)
DarkCyan = (0, 139, 139)
GreenYellow = (173, 255, 47)
DarkOrchid = (153, 50, 204)
OliveDrab = (107, 142, 35)
Chartreuse = (127, 255, 0)
Peru = (205, 133, 63)
Orange = (255, 165, 0)
Red = (255, 0, 0)
Wheat = (245, 222, 179)
LightCyan = (224, 255, 255)
LightSeaGreen = (32, 178, 170)
BlueViolet = (138, 43, 226)
Cyan = (0, 255, 255)
MediumPurple = (147, 112, 219)
MidnightBlue = (25, 25, 112)
Gainsboro = (220, 220, 220)
PaleTurquoise = (175, 238, 238)
PaleGoldenrod = (238, 232, 170)
Gray = (128, 128, 128)
MediumSeaGreen = (60, 179, 113)
Moccasin = (255, 228, 181)
Ivory = (255, 255, 240)
SlateBlue = (106, 90, 205)
Green = (0, 255, 0)
Green_HTML = (0, 128, 0)
DarkSlateBlue = (72, 61, 139)
Teal = (0, 128, 128)
Azure = (240, 255, 255)
LightSteelBlue = (176, 196, 222)
Tan = (210, 180, 140)
AntiqueWhite = (250, 235, 215)
WhiteSmoke = (245, 245, 245)
GhostWhite = (248, 248, 255)
MediumTurquoise = (72, 209, 204)
FloralWhite = (255, 250, 240)
LavenderBlush = (255, 240, 245)
SeaGreen = (46, 139, 87)
Lavender = (230, 230, 250)
BlanchedAlmond = (255, 235, 205)
DarkOliveGreen = (85, 107, 47)
DarkSeaGreen = (143, 188, 143)
Violet = (238, 130, 238)
Navy = (0, 0, 128)
Beige = (245, 245, 220)
SaddleBrown = (139, 69, 19)
IndianRed = (205, 92, 92)
Snow = (255, 250, 250)
SteelBlue = (70, 130, 180)
MediumSlateBlue = (123, 104, 238)
Black = (0, 0, 0)
LightBlue = (173, 216, 230)
Turquoise = (64, 224, 208)
MediumVioletRed = (199, 21, 133)
DarkViolet = (148, 0, 211)
DarkGray = (169, 169, 169)
Salmon = (250, 128, 114)
DarkMagenta = (139, 0, 139)
Tomato = (255, 99, 71)
SkyBlue = (135, 206, 235)
Goldenrod = (218, 165, 32)
MediumSpringGreen = (0, 250, 154)
DodgerBlue = (30, 144, 255)
Aqua = (0, 255, 255)
ForestGreen = (34, 139, 34)
DarkRed = (139, 0, 0)
SlateGray = (112, 128, 144)
Indigo = (75, 0, 130)
CadetBlue = (95, 158, 160)
LightYellow = (255, 255, 224)
DarkOrange = (255, 140, 0)
PowderBlue = (176, 224, 230)
RoyalBlue = (65, 105, 225)
Sienna = (160, 82, 45)
Thistle = (216, 191, 216)
Lime = (0, 255, 0)
Seashell = (255, 245, 238)
LemonChiffon = (255, 250, 205)
LightSkyBlue = (135, 206, 250)
YellowGreen = (154, 205, 50)
Plaid = (204, 85, 51)
Aquamarine = (127, 255, 212)
LightCoral = (240, 128, 128)
DarkSlateGray = (47, 79, 79)
Coral = (255, 127, 80)
Khaki = (240, 230, 140)
BurlyWood = (222, 184, 135)
LightGoldenrodYellow = (250, 250, 210)
MediumBlue = (0, 0, 205)
LightSlateGray = (119, 136, 153)
RosyBrown = (188, 143, 143)
Silver = (192, 192, 192)
PaleVioletRed = (219, 112, 147)
FireBrick = (178, 34, 34)
SpringGreen = (0, 255, 127)
LightGreen = (144, 238, 144)
Linen = (250, 240, 230)
OrangeRed = (255, 69, 0)
DimGray = (105, 105, 105)
Maroon = (128, 0, 0)
LightPink = (255, 182, 193)
MediumAquamarine = (102, 205, 170)
Gold = (255, 215, 0)
Orchid = (218, 112, 214)
OldLace = (253, 245, 230)