import wx, json, requests, threading
from wx.lib.sized_controls import SizedFrame

class WorkerThread(threading.Thread):
 def __init__(self, *args, **kwargs):
  self._stop = threading.Event()
  return super(WorkerThread, self).__init__(*args, **kwargs)
 
 def stop(self):
  """Stop the update in it's tracks."""
  self._stop.set()
 
 def stopped(self):
  """Returns whether the thread is stopped."""
  return self._stop.isSet()

class UpdateCheckFrame(SizedFrame):
 """
 The update checker frame.
 
  Override this class to setup the fothods
  * self.displayError: The function which displays the errors.
  * self._update: The method which is called to retrieve the json from self.url.
  * self.updateCheck: The function which actually processes the received json, and either shows the frame and enables the update button, or else presents with some other form of update method..
  * self.onUpdate: The function which performs the update when the Update button is pressed.
  * self.onCancel: The method which is called by clicking the cancel button.
 
 """
 def __init__(self, name, version, url, title = 'Check for updates'):
  """
  Initialise the frame.
  
  Override this method to set properties:
  * self.errorMessage: The message which is printed when an error is encountered. {msg} is the str()'d exception.
  * self.panel: The parent for all controls on the frame.
  
  Arguments:
  * name is the name of the application.
  * version is the currently used version.
  * url is the URL where version.json is stored.
  * title is the title of the frame.
  
  """
  super(UpdateCheckFrame, self).__init__(None, title = title)
  self.panel = self.GetContentsPane()
  self.errorMessage = 'Error while checking for updates: {msg}'
  self.request = None
  self.appName = name
  self.appVersion = version
  self.appUrl = url
  wx.StaticText(self.panel, label = '&Status')
  self.status = wx.TextCtrl(self.panel, value = 'Idle')
  self.updateButton = wx.Button(self.panel, label = '&Update')
  self.updateButton.Disable()
  self.updateButton.Bind(wx.EVT_BUTTON, self.onUpdate)
  self.cancelButton = wx.Button(self.panel, label = '&Cancel')
  self.cancelButton.Bind(wx.EVT_BUTTON, self.onCancel)
  self._thread = threading.Thread(target = self._update)
  self._thread.start()
 
 def displayError(self, string):
  wx.MessageBox(string, 'Error')
 
 def _update(self):
  self.status.SetLabel('Working...')
  try:
   self.request = requests.get(self.appUrl)
  except requests.RequestException as e:
   self.displayError(self.errorMessage.format(msg = str(e)))
   return self.Close(True)
  if not getattr(self, '_thread', None):
   self.Close(True) # The cancel button was pressed.
  else:
   self._thread = None
   u = self.updateCheck()
   if u:
    self.status.SetLabel(u)
   else:
    self.Close(True)
 
 def updateCheck(self):
  """Checks the json for updates."""
  j = json.loads(self.request.content)
  self.updateButton.Enable()
  return str(j)
 
 def onUpdate(self, event):
  """Performs the actual update."""
  self.displayError('Not implemented.')
 
 def onCancel(self, event):
  """Cancel button was clicked."""
  if not getattr(self, '_thread', None):
   self.Close(True)
  else:
   self._thread = None
   self.status.SetLabel('Quitting, please wait...')
