/*
   Copyright 2014 Boris T. Darchiev (boris.darchiev@gmail.com)

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

#include "Python.h"
#include "datetime.h"
#include "nkit/types.h"
#include "nkit/tools.h"
#include "nkit/logger_brief.h"
#include "nkit/xml2var.h"
#include "nkit/var2xml.h"
#include <Python.h>
#include <string>

#if ((PY_MAJOR_VERSION == 2) && (PY_MINOR_VERSION <=4))
#define NKIT_PYTHON_OLDER_THEN_2_5
#endif


#if ((PY_MAJOR_VERSION == 2) && (PY_MINOR_VERSION <= 5))
#define NKIT_PYTHON_OLDER_THEN_2_6
#endif

#if NKIT_PYTHON_OLDER_THEN_2_6
#define NKIT_PYTHON_LONG_FROM_INT64(v) PyLong_FromLong(static_cast<long>(v))
#else
#define NKIT_PYTHON_LONG_FROM_INT64(v) PyLong_FromLongLong(v)
#endif

namespace nkit
{
  //----------------------------------------------------------------------------
  static PyObject * main_module_;
  static PyObject * dt_module_;
  static PyObject * dt_;
  static PyObject * fromtimestamp_;
  static PyObject * traceback_module_;
  static PyObject * traceback_dict_;
  static PyObject * traceback_format_exception_;
  static PyObject * traceback_format_exception_only_;
  static PyObject * json_module_;
  static PyObject * json_dumps_;
  static PyObject * string_module_;
  static PyObject * string_dict_;
  static PyObject * string_join_fields_;
  static PyObject * datetime_json_encoder_;

  //----------------------------------------------------------------------------
  bool py_to_string(PyObject * unicode, std::string * out,
      std::string * error)
  {
    if (PyUnicode_CheckExact(unicode))
    {
      PyObject * tmp = PyUnicode_AsUTF8String(unicode);
      out->assign(PyString_AsString(tmp));
      Py_DECREF(tmp);
    }
    else if (PyString_CheckExact(unicode))
    {
      out->assign(PyString_AsString(unicode));
    }
    else if (PyFloat_CheckExact(unicode))
      out->assign(string_cast(PyFloat_AsDouble(unicode)));
    else if (PyNumber_Check(unicode))
    {
      out->assign(string_cast(
              static_cast<int64_t>(PyNumber_AsSsize_t(unicode, NULL))));
    }
    else
    {
      PyObject * tmp = PyObject_Str(unicode);
      if (!tmp)
      {
        *error = "Could not represent variable to string";
        return false;
      }

      bool ret = py_to_string(tmp, out,error);
      Py_DECREF(tmp);
      return ret;
    }

    return true;
  }

  //----------------------------------------------------------------------------
  PyObject * py_fromtimestamp(time_t  timestamp)
  {
    return PyObject_CallFunction(fromtimestamp_, (char*)"i", timestamp);
  }

  //----------------------------------------------------------------------------
  std::string py_strftime(const PyObject * data, const std::string & format)
  {
    std::string ret, error;
    PyObject * unicode =
            PyObject_CallMethod(const_cast<PyObject *>(data),
                    const_cast<char *>("strftime"),
                    const_cast<char *>("s"), format.c_str());
    py_to_string(unicode, &ret, &error);
    Py_DECREF(unicode);
    return ret;
  }

  //----------------------------------------------------------------------------
  bool pyobj_to_json(PyObject * obj, std::string * out, std::string * error)
  {
    PyObject * args = PyTuple_New(1);
    Py_INCREF(obj);
    PyTuple_SetItem(args, 0, obj);

    PyObject * kw = PyDict_New();

    PyObject * indent = PyInt_FromSize_t(2);
    PyDict_SetItemString(kw, "indent", indent);
    Py_DECREF(indent);

    PyObject * ensure_ascii = PyBool_FromLong(0);
    PyDict_SetItemString(kw, "ensure_ascii", ensure_ascii);
    Py_DECREF(ensure_ascii);

    PyDict_SetItemString(kw, "cls", datetime_json_encoder_);

    PyObject * result = PyObject_Call(json_dumps_, args, kw);
    bool ret = py_to_string(result, out, error);
    Py_DECREF(result);
    Py_DECREF(args);
    Py_DECREF(kw);

    return ret;
  }

  //----------------------------------------------------------------------------
  class PythonBuilderPolicy: Uncopyable
  {
  private:
    friend class VarBuilder<PythonBuilderPolicy>;

    typedef PyObject* type;

    static const type & GetUndefined()
    {
      static PyObject * WarningWorkaround = Py_None;
      Py_INCREF(Py_None);
      return WarningWorkaround;
    }

    PythonBuilderPolicy(const detail::Options & options)
      : object_(NULL)
      , options_(options)
    {}

    ~PythonBuilderPolicy()
    {
      Py_CLEAR(object_);
    }

    void InitAsBoolean( std::string const & value )
    {
      int32_t i = nkit::bool_cast(value);

      Py_CLEAR(object_);
      object_ = PyBool_FromLong(i);
      assert(object_);
    }

    void InitAsInteger( std::string const & value )
    {
      int64_t i = !value.empty() ? NKIT_STRTOLL( value.c_str(), NULL, 10 ) : 0;

      Py_CLEAR(object_);
      object_ = NKIT_PYTHON_LONG_FROM_INT64(i);
      assert(object_);
    }

    void InitAsString( std::string const & value)
    {
      Py_CLEAR(object_);
      if (options_.unicode_)
        object_ = PyUnicode_FromStringAndSize(value.data(), value.size());
      else
        object_ = PyString_FromStringAndSize(value.data(), value.size());
      assert(object_);
    }

    void InitAsUndefined()
    {
      Py_CLEAR(object_);
      Py_INCREF(Py_None);
      object_ = Py_None;
    }

    void InitAsFloatFormat( std::string const & value, const char * format )
    {
      double d(0.0);
      if (!value.empty())
      {
        if (0 == NKIT_SSCANF(value.c_str(), format, &d))
          d = 0.0;
      }

      Py_CLEAR(object_);
      object_ = PyFloat_FromDouble(d);
      assert(object_);
    }

    void InitAsDatetimeFormat( std::string const & value,
        const char * format )
    {
#if defined(_WIN32) || defined(_WIN64)
      struct tm _tm =
        { 0, 0, 0, 0, 0, 0, 0, 0, 0};
#else
      struct tm _tm =
        { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
#endif
      if (value.empty() || NKIT_STRPTIME(value.c_str(), format, &_tm) == NULL)
      {
        InitAsUndefined();
        return;
      }

      time_t time = mktime(&_tm);
      Py_CLEAR(object_);
      object_ = py_fromtimestamp(time);
      assert(object_);
    }

    void InitAsList()
    {
      Py_CLEAR(object_);
      object_ = PyList_New(0);
      assert(object_);
    }

    void InitAsDict()
    {
      Py_CLEAR(object_);
      object_ = PyDict_New();
      assert(object_);
    }

    void ListCheck()
    {
      assert(PyList_CheckExact(object_));
    }

    void DictCheck()
    {
      assert(PyDict_CheckExact(object_));
    }

    void AppendToList( type const & obj )
    {
      int result = PyList_Append( object_, obj );
      assert(-1 != result);
      NKIT_FORCE_USED(result)
    }

    void SetDictKeyValue( std::string const & key, type const & var )
    {
      int result = PyDict_SetItemString( object_, key.c_str(), var );
      assert(-1 != result);
      NKIT_FORCE_USED(result)
    }

    type const & get() const
    {
      return object_;
    }

    std::string ToString() const;

  private:
    type object_;
    const detail::Options & options_;
  };

  typedef VarBuilder<PythonBuilderPolicy> PythonVarBuilder;
  typedef Xml2VarBuilder<PythonVarBuilder> Xml2PythonBuilder;

  ////--------------------------------------------------------------------------
  struct PythonReaderPolicy
  {
    typedef PyObject * type;

    //--------------------------------------------------------------------------
    struct DictConstIterator
    {
      DictConstIterator()
        : data_(NULL)
        , pos_(-1)
        , key_(NULL)
        , value_(NULL)
      {}

      DictConstIterator(PyObject * data)
        : data_(data)
        , pos_(0)
        , key_(NULL)
        , value_(NULL)
      {
        if (!PyDict_Next(data_, &pos_, &key_, &value_))
          pos_ = -1;
      }

      bool operator != (const DictConstIterator & another)
      {
        return pos_ != another.pos_;
      }

      DictConstIterator & operator++()
      {
        if (pos_ != -1 && !PyDict_Next(data_, &pos_, &key_, &value_))
          pos_ = -1;
        return *this;
      }

      std::string first() const
      {
        if (!key_)
          return S_EMPTY_;

        std::string ret, err;
        if (unlikely(!nkit::py_to_string(key_, &ret, &err)))
          return S_EMPTY_;
        return ret;
      }

      PyObject * second() const
      {
        if (!value_)
          return Py_None;
        return value_;
      }

      PyObject * data_;
      Py_ssize_t pos_;
      PyObject * key_;
      PyObject * value_;
    };

    //--------------------------------------------------------------------------
    struct ListConstIterator
    {
      ListConstIterator()
        : list_(NULL)
        , size_(0)
        , pos_(-1)
      {}

      ListConstIterator(PyObject * list)
        : list_(PySequence_Fast(list, "Not a sequence"))
        , size_(list_ != NULL ? PySequence_Size(list_) : 0)
        , pos_(size_ > 0 ? 0: -1)
      {}

      ListConstIterator(const ListConstIterator & copy)
        : list_(copy.list_)
        , size_(copy.size_)
        , pos_(copy.pos_)
      {
        Py_XINCREF(list_);
      }

      ListConstIterator & operator = (const ListConstIterator & copy)
      {
        list_ = copy.list_;
        size_ = copy.size_;
        pos_ = copy.pos_;
        Py_XINCREF(list_);
        return *this;
      }

      ~ListConstIterator()
      {
        Py_XDECREF(list_);
      }

      bool operator != (const ListConstIterator & another)
      {
        return pos_ != another.pos_;
      }

      ListConstIterator & operator++()
      {
        if (++pos_ >= size_)
          pos_ = -1;
        return *this;
      }

      PyObject * value() const
      {
        if (unlikely(pos_ >= size_ || pos_ < 0))
          return Py_None;
        return PySequence_Fast_GET_ITEM(list_, pos_);
      }

      PyObject * list_;
      Py_ssize_t size_;
      Py_ssize_t pos_;
    };

    //--------------------------------------------------------------------------
    static DictConstIterator begin_d(const PyObject * data)
    {
      return DictConstIterator(const_cast<PyObject *>(data));
    }

    static DictConstIterator end_d(const PyObject * data)
    {
      return DictConstIterator();
    }

    static ListConstIterator begin_l(const PyObject * data)
    {
      return ListConstIterator(const_cast<PyObject *>(data));
    }

    static ListConstIterator end_l(const PyObject * data)
    {
      return ListConstIterator();
    }

    static std::string First(const DictConstIterator & it)
    {
      return it.first();
    }

    static PyObject * Second(const DictConstIterator & it)
    {
      return it.second();
    }

    static PyObject * Value(const ListConstIterator & it)
    {
      return it.value();
    }

    static bool IsList(const PyObject * data)
    {
      bool ret = PyList_CheckExact(const_cast<PyObject *>(data)) ||
              PyTuple_CheckExact(const_cast<PyObject *>(data)) ||
              PySet_Check(const_cast<PyObject *>(data)) ;
      return ret;
    }

    static bool IsDict(const PyObject * data)
    {
      bool ret = PyDict_CheckExact(const_cast<PyObject *>(data));
      return ret;
    }

    static bool IsString(const PyObject * data)
    {
      return PyUnicode_CheckExact(data) || PyString_CheckExact(data);
    }

    static bool IsFloat(const PyObject * data)
    {
      bool ret = PyFloat_CheckExact(const_cast<PyObject *>(data));
      return ret;
    }

    static bool IsDateTime(const PyObject * data)
    {
      bool ret = PyDateTime_CheckExact(const_cast<PyObject *>(data));
      return ret;
    }

    static bool IsBool(const PyObject * data)
    {
      bool ret = PyBool_Check(const_cast<PyObject *>(data));
      return ret;
    }

    static std::string GetString(const PyObject * data)
    {
      std::string ret, err;
      py_to_string(const_cast<PyObject *>(data), &ret, &err);
      return ret;
    }

    static std::string GetStringAsDateTime(const PyObject * data,
            const std::string & format)
    {
      return py_strftime(data, format);
    }

    static std::string GetStringAsFloat(const PyObject * data,
            size_t precision)
    {
      return string_cast(PyFloat_AsDouble(const_cast<PyObject *>(data)),
              precision);
    }

    static const std::string & GetStringAsBool(const PyObject * data,
        const std::string & true_format, const std::string & false_format)
    {
      return data == Py_True ? true_format: false_format;
    }

    static PyObject * GetByKey(const PyObject * data, const std::string & key,
            bool * found)
    {
      PyObject * ret =
    		  PyDict_GetItemString(const_cast<PyObject *>(data), key.c_str());
      *found = ret != NULL;
      if (*found)
        return ret;
      else
        return Py_None;
    }
  };

  typedef Var2XmlConverter<PythonReaderPolicy> Python2XmlConverter;

} // namespace nkit

////----------------------------------------------------------------------------
/// exception
static PyObject * Nkit4PyError;

////----------------------------------------------------------------------------
bool parse_dict(PyObject * dict, std::string * out, std::string * error)
{
  if (PyDict_CheckExact(dict))
    return nkit::pyobj_to_json(dict, out, error);
  else
    return nkit::py_to_string(dict, out, error);
}

////----------------------------------------------------------------------------
template<typename T>
struct SharedPtrHolder
{
  SharedPtrHolder(NKIT_SHARED_PTR(T) & ptr) : ptr_(ptr) {}
  NKIT_SHARED_PTR(T) ptr_;
};

////----------------------------------------------------------------------------
struct Xml2PythonBuilderData
{
  PyObject_HEAD;
  SharedPtrHolder< nkit::Xml2PythonBuilder > * holder_;
};

////----------------------------------------------------------------------------
static PyObject* CreatePythonXml2VarBuilder(
    PyTypeObject * type, PyObject * args, PyObject *)
{
  PyObject * dict1 = NULL;
  PyObject * dict2 = NULL;
  int result = PyArg_ParseTuple(args, "O|O", &dict1, &dict2);
  if(!result)
  {
    PyErr_SetString(Nkit4PyError,
        "Expected one or two arguments:"
        " 1) mappings or 2) options and mappings");
    return NULL;
  }

  PyObject * options_dict = dict2 ? dict1 : NULL;
  PyObject * mapping_dict = dict2 ? dict2 : dict1;

  std::string options, error;
  if (!options_dict)
    options = "{}";
  else if (!parse_dict(options_dict, &options, &error))
  {
    PyErr_SetString( Nkit4PyError,
        ("Options parameter must be JSON-string or dictionary: " +
        error).c_str());
    return NULL;
  }

  if (options.empty())
  {
    PyErr_SetString(
        Nkit4PyError,
        "Options parameter must be dict or JSON object" );
    return NULL;
  }

  std::string mappings;
  if (!parse_dict(mapping_dict, &mappings, &error))
  {
    PyErr_SetString( Nkit4PyError,
        ("Mappings parameter must be JSON-string or dictionary: " +
        error).c_str());
    return NULL;
  }

  if(mappings.empty())
  {
    PyErr_SetString(
        Nkit4PyError,
        "Mappings parameter must be dict or JSON object" );
    return NULL;
  }

  Xml2PythonBuilderData * self =
      (Xml2PythonBuilderData *)type->tp_alloc( type, 0 );
  if (!self)
  {
    PyErr_SetString(Nkit4PyError, "Low memory");
    return NULL;
  }

  nkit::Xml2PythonBuilder::Ptr builder =
      nkit::Xml2PythonBuilder::Create(options, mappings, &error);
  if(!builder)
  {
    PyErr_SetString( Nkit4PyError, error.c_str() );
    return NULL;
  }
  self->holder_ =
      new SharedPtrHolder< nkit::Xml2PythonBuilder >(builder);

  return (PyObject *)self;
}

////----------------------------------------------------------------------------
static void DeletePythonXml2VarBuilder(PyObject * self)
{
  SharedPtrHolder< nkit::Xml2PythonBuilder > * ptr =
        ((Xml2PythonBuilderData *)self)->holder_;
  if (ptr)
    delete ptr;
  self->ob_type->tp_free(self);
}

////----------------------------------------------------------------------------
static PyObject * feed_method( PyObject * self, PyObject * args )
{
  const char* request = NULL;
  Py_ssize_t size = 0;
  int result = PyArg_ParseTuple( args, "s#", &request, &size );
  if(!result)
  {
    PyErr_SetString( Nkit4PyError, "Expected string arguments" );
    return NULL;
  }
  if( !request || !*request || !size )
  {
    PyErr_SetString(
        Nkit4PyError, "Parameter must not be empty string" );
    return NULL;
  }

  nkit::Xml2PythonBuilder::Ptr builder =
      ((Xml2PythonBuilderData *)self)->holder_->ptr_;

  std::string error("");
  if(!builder->Feed( request, size, false, &error ))
  {
    PyErr_SetString( Nkit4PyError, error.c_str() );
    return NULL;
  }

  Py_RETURN_NONE;
}

////----------------------------------------------------------------------------
static PyObject * get_method( PyObject * self, PyObject * args )
{
  const char* mapping_name = NULL;
  int result = PyArg_ParseTuple( args, "s", &mapping_name );
  if(!result)
  {
    PyErr_SetString( Nkit4PyError, "Expected string argument" );
    return NULL;
  }
  if( !mapping_name || !*mapping_name )
  {
    PyErr_SetString(
        Nkit4PyError, "Mapping name must not be empty" );
    return NULL;
  }

  nkit::Xml2PythonBuilder::Ptr builder =
      ((Xml2PythonBuilderData *)self)->holder_->ptr_;

  PyObject * item = builder->var(mapping_name);
  Py_INCREF(item);
  return item;
}

////------------------------------------------------------------------------------
static PyObject * end_method( PyObject * self, PyObject * /*args*/ )
{
  nkit::Xml2PythonBuilder::Ptr builder =
          ((Xml2PythonBuilderData *)self)->holder_->ptr_;

  std::string empty("");
  std::string error("");
  if(!builder->Feed( empty.c_str(), empty.size(), true, &error ))
  {
    PyErr_SetString( Nkit4PyError, error.c_str() );
    return NULL;
  }

  nkit::StringList mapping_names(builder->mapping_names());

  PyObject * result = PyDict_New();
  nkit::StringList::const_iterator mapping_name = mapping_names.begin(),
      end = mapping_names.end();
  for (; mapping_name != end; ++mapping_name)
  {
    PyObject * item = builder->var(*mapping_name);
    PyDict_SetItemString(result, mapping_name->c_str(), item);
  }
  return result;
}

////----------------------------------------------------------------------------
static PyMethodDef xml2var_methods[] =
{
  { "feed", feed_method, METH_VARARGS, "Usage: builder.feed()\n"
          "Invoke \"Feed\" function\n"
          "Returns None\n" },
  { "get", get_method, METH_VARARGS, "Usage: builder.get()\n"
          "Returns result by mapping name\n" },
  { "end", end_method, METH_VARARGS, "Usage: builder.end()\n"
          "Returns Dict: results for all mappings\n" },
  { NULL, NULL, 0, NULL } /* Sentinel */
};

////----------------------------------------------------------------------------
static PyTypeObject Xml2PythonBuilderType =
{
  PyVarObject_HEAD_INIT(NULL, 0)
  "nkit4py.Xml2VarBuilder", /*tp_name*/
  sizeof(Xml2PythonBuilderData), /*tp_basicsize*/
  0, /*tp_itemsize*/
  DeletePythonXml2VarBuilder, /*tp_dealloc*/
  0, /*tp_print*/
  0, /*tp_getattr*/
  0, /*tp_setattr*/
  0, /*tp_compare*/
  0, /*tp_repr*/
  0, /*tp_as_number*/
  0, /*tp_as_sequence*/
  0, /*tp_as_mapping*/
  0, /*tp_hash */
  0, /*tp_call*/
  0, /*tp_str*/
  0, /*tp_getattro*/
  0, /*tp_setattro*/
  0, /*tp_as_buffer*/
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
  "XML to object or list converter and filter", /* tp_doc */
  0,//tp_traverse
  0,//tp_clear,
  0,//tp_richcompare,
  0,//tp_weaklistoffset,
  0,//tp_iter,
  0,//tp_iternext,
  xml2var_methods,//tp_methods,
  0,//tp_members,
  0,//tp_getset,
  0,//tp_base,
  0,//tp_dict,
  0,//tp_descr_get,
  0,//tp_descr_set,
  0,//tp_dictoffset,
  0,//tp_init,
  0,//tp_alloc,
  CreatePythonXml2VarBuilder,//tp_new,
};

////----------------------------------------------------------------------------
static PyObject * var2xml_method( PyObject * self, PyObject * args )
{
  PyObject * data = NULL;
  PyObject * options_dict = NULL;
  int result = PyArg_ParseTuple( args, "O|O", &data, &options_dict);
  if(!result)
  {
    PyErr_SetString( Nkit4PyError,
            "Expected any object and optional 'options' Dict" );
    return NULL;
  }

  std::string options, error;
  nkit::Dynamic op = nkit::Dynamic::Dict();
  if (options_dict && !parse_dict(options_dict, &options, &error))
  {
    std::string tmp("Options parameter must be JSON-string or dictionary: " +
            error);
    PyErr_SetString( Nkit4PyError, tmp.c_str());
    return NULL;
  }
  else if (!options.empty())
  {
    op = nkit::DynamicFromJson(options, &error);
    if (!op && !error.empty())
    {
      std::string tmp("Options parameter must be JSON-string or dictionary: " +
              error);
      PyErr_SetString( Nkit4PyError, tmp.c_str());
      return NULL;
    }
  }

  std::string out;
  if(!nkit::Python2XmlConverter::Process(op, data, &out, &error))
  {
    PyErr_SetString( Nkit4PyError, error.c_str() );
    return NULL;
  }

  return PyString_FromStringAndSize(out.data(), out.size());
}

////----------------------------------------------------------------------------
static PyMethodDef ModuleMethods[] =
{
  { "var2xml", var2xml_method, METH_VARARGS,
          "Usage: nkit4py.var2xml(data, options)\n"
          "Converts python structure to xml string\n"
          "Returns None\n" },
  { NULL, NULL, 0, NULL } /* Sentinel */
};

#ifndef PyMODINIT_FUNC  /* declarations for DLL import/export */
#define PyMODINIT_FUNC void
#endif

////----------------------------------------------------------------------------
namespace nkit
{
  std::string get_python_error()
  {
    std::string error;

    PyObject *exc_obj = NULL;
    PyObject *exc_str = NULL;
    PyObject *exc_traceback = NULL;
    PyObject *lines = NULL;
    PyObject *exception_string = NULL;

    if (PyErr_Occurred() == NULL)
    return std::string("");

    PyErr_Fetch(&exc_obj, &exc_str, &exc_traceback);

    if (exc_obj == NULL)
    {
      error.append("Fatal error in 'GetPythonError': ExcObj == NULL");
    }
    else if (exc_str == NULL)
    {
      error.append("Fatal error in 'GetPythonError': ExcStr == NULL");
    }
    else
    {
      if (exc_traceback == NULL)
        lines = PyObject_CallFunction(
          traceback_format_exception_only_,
            const_cast<char*>("OO"), exc_obj, exc_str);
      else
        lines = PyObject_CallFunction(
          traceback_format_exception_,
          const_cast<char*>("OOO"), exc_obj, exc_str, exc_traceback);

      if (lines == NULL)
      {
        error.append("traceback.formatexception error");
      }
      else
      {
        exception_string = PyObject_CallFunction(
            string_join_fields_, const_cast<char*>("Os"), lines, "");

        if (exception_string == NULL)
          error.append("string.joinfields error");
        else
          error.append(PyString_AsString(exception_string));
      }
    }

    Py_XDECREF(lines);
    Py_XDECREF(exception_string);
    Py_XDECREF(exc_obj);
    Py_XDECREF(exc_str);
    Py_XDECREF(exc_traceback);

    PyErr_Clear();

    return std::string(error);
  }

  std::string PythonBuilderPolicy::ToString() const
  {
    std::string ret, error;
    try
    {
      nkit::pyobj_to_json(object_, &ret, &error);
      return ret;
    }
    catch (...)
    {
      return std::string("");
    }
  }

  const char DATETIME_JSON_ENCODER_CLASS[] =
        "class DatetimeJSONEncoder(nkit_tmp_json.JSONEncoder):\n"
        "    _datetime = nkit_tmp_datetime\n"
        "    _traceback = nkit_tmp_traceback\n"
        "    _json = nkit_tmp_json\n"
        "    def default( self, obj ):\n"
        "        try:\n"
        "            if isinstance(obj, DatetimeJSONEncoder._datetime.datetime):\n"
        "                return obj.strftime(\"%Y-%m-%d %H:%M:%S\")\n"
        "            if isinstance(obj, DatetimeJSONEncoder._datetime.date):\n"
        "                return obj.strftime(\"%Y-%m-%d\")\n"
        "            if isinstance(obj, DatetimeJSONEncoder._datetime.time):\n"
        "                return obj.strftime(\"%H:%M:%S\")\n"
        "        except Exception:\n"
        "            # text = DatetimeJSONEncoder._traceback.format_exc()\n"
        "            return ''\n"
        "        return DatetimeJSONEncoder._json.JSONEncoder.default( self, obj )\n"
        ;

} // namespace nkit

////----------------------------------------------------------------------------
PyMODINIT_FUNC initnkit4py(void)
{
  if( -1 == PyType_Ready(&Xml2PythonBuilderType) )
    return;

  PyObject * module = Py_InitModule("nkit4py", ModuleMethods);
  if( NULL == module )
    return;

  Nkit4PyError = PyErr_NewException( (char *)"nkit4py.Error", NULL, NULL );
  Py_INCREF(Nkit4PyError);
  PyModule_AddObject( module, "Error", Nkit4PyError );

  Py_INCREF(&Xml2PythonBuilderType);
  PyModule_AddObject( module,
          "Xml2VarBuilder", (PyObject *)&Xml2PythonBuilderType );

  nkit::traceback_module_ = PyImport_ImportModule("traceback");
  assert(nkit::traceback_module_);
  Py_INCREF(nkit::traceback_module_);

  nkit::traceback_dict_ = PyModule_GetDict(nkit::traceback_module_);
  Py_INCREF(nkit::traceback_dict_);

  nkit::traceback_format_exception_ =
      PyDict_GetItemString(nkit::traceback_dict_, "format_exception");
  Py_INCREF(nkit::traceback_format_exception_);

  nkit::traceback_format_exception_only_ =
      PyDict_GetItemString(nkit::traceback_dict_, "format_exception_only");
  Py_INCREF(nkit::traceback_format_exception_only_);

  nkit::string_module_ = PyImport_ImportModule("string");
  assert(nkit::string_module_);
  Py_INCREF(nkit::string_module_);

  nkit::string_dict_ = PyModule_GetDict(nkit::string_module_);
  Py_INCREF(nkit::string_dict_);

  nkit::string_join_fields_ =
      PyDict_GetItemString(nkit::string_dict_, "joinfields");
  Py_INCREF(nkit::string_join_fields_);

  nkit::json_module_ = PyImport_ImportModule("json");
  assert(nkit::json_module_);
  Py_INCREF(nkit::json_module_);

  nkit::json_dumps_ = PyObject_GetAttrString(nkit::json_module_, "dumps");
  assert(nkit::json_dumps_);
  Py_INCREF(nkit::json_dumps_);

  nkit::dt_module_ = PyImport_ImportModule("datetime");
  assert(nkit::dt_module_);
  Py_INCREF(nkit::dt_module_);

  PyDateTime_IMPORT;

  nkit::dt_ = PyObject_GetAttrString(nkit::dt_module_, "datetime");
  assert(nkit::dt_);
  Py_INCREF(nkit::dt_);

  nkit::fromtimestamp_ = PyObject_GetAttrString(nkit::dt_, "fromtimestamp");
  assert(nkit::fromtimestamp_);
  Py_INCREF(nkit::fromtimestamp_);

  // class DatetimeJSONEncoder
  nkit::main_module_ = PyImport_AddModule("__main__");
  Py_INCREF(nkit::main_module_);
  PyObject * globals = PyModule_GetDict(nkit::main_module_);
  PyObject * locals = PyDict_New();
  PyDict_SetItemString(globals, "nkit_tmp_datetime", nkit::dt_module_);
  PyDict_SetItemString(globals, "nkit_tmp_traceback", nkit::traceback_module_);
  PyDict_SetItemString(globals, "nkit_tmp_json", nkit::json_module_);
  PyObject * run = PyRun_String(nkit::DATETIME_JSON_ENCODER_CLASS,
      Py_file_input, globals, locals);
  if ( NULL == run)
  {
    Py_DECREF(locals);
    CERR("Could not run code statements");
    return;
  }

  nkit::datetime_json_encoder_ =
      PyDict_GetItemString(locals, "DatetimeJSONEncoder");
  assert(nkit::datetime_json_encoder_);
  Py_INCREF(nkit::datetime_json_encoder_);
  PyModule_AddObject(module, "DatetimeJSONEncoder",
      nkit::datetime_json_encoder_);

  PyDict_DelItemString(globals, "nkit_tmp_datetime");
  PyDict_DelItemString(globals, "nkit_tmp_traceback");
  PyDict_DelItemString(globals, "nkit_tmp_json");
  Py_DECREF(locals);
  Py_DECREF(run);
}
