#!/usr/bin/env python
# vim: set fileencoding=utf-8 :
# Elie Khoury < elie.khoury@idiap.ch>
# Thu Nov 14 22:17:56 CET 2013
#
# Copyright (C) 2011-2013 Idiap Research Institute, Martigny, Switzerland
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.


"""Tests audio reader and writer based on sox for bob and python
"""

import os, sys
import unittest
import scipy.io.wavfile
import numpy
import pkg_resources
import bob
from ._ext import reader

def F(name, f):
  """Returns the test file on the "data" subdirectory"""
  return pkg_resources.resource_filename(name, os.path.join('data', f))


class SoxTest(unittest.TestCase):
  """Performs various tests on audio reading and writing using sox for bob and python"""

  def run_for(self, sample, extension):

    # audio file name
    infile = F(__name__, '%s%s' % (sample, extension))

    # reading the file using xbob.sox
    sox_audio = reader(infile)
    (sox_rate, sox_data) = sox_audio.load()

    # reading the file using scipy.io.wavfile
    (scipy_rate, scipy_data)=scipy.io.wavfile.read(infile)
    
    # verify if the rates are the same
    self.assertTrue(numpy.equal(int(sox_rate), scipy_rate))

    # If one channel
    if sox_data.shape[0] == 1:
      # verify if the data are the same (x 2^15)
      sox_data_ =  sox_data[0] * pow(2,15)
      sox_data_.astype(int)
      self.assertTrue(numpy.array_equal(sox_data_, scipy_data))
    
    #If more than one channel
    elif sox_data.shape[0] > 1:
      # verify if the nchannels are the same
      self.assertTrue(numpy.equal(sox_data.shape[0], scipy_data.shape[1])) 
      
      # verify if the data are the same (x 2^15)
      for i in range(0, sox_data.shape[0]):
        sox_data_ =  sox_data[i] * pow(2,15)
        sox_data_.astype(int)
        scipy_data_ = scipy_data[:,i]
        self.assertTrue(numpy.array_equal(sox_data_, scipy_data_))
  
  
  
  # compare mono: ours vs scipy.io.wavfile  
  def test01_read_mono_wav(self):
    self.run_for('sample1', '.wav')
  

  # compare stereo: ours vs scipy.io.wavfile    
  def test02_read_stereo_wav(self):
    self.run_for('sample4', '.wav')



  # compare mono: ours vs matlab 
  def test03_read_mono_wav(self):
    
    # audio file name
    infile = F(__name__, 'sample3.wav')
    
    # reading the file using xbob.sox
    sox_audio = reader(infile)
    (sox_rate, sox_data) = sox_audio.load()
    
    # reading the .mat file that contains the data generated by the Matlab wavread function
    mat_data = bob.io.load(F(__name__, 'sample3.mat'))
    self.assertTrue(numpy.array_equal(sox_data, mat_data))

 # TODO: could add test for writing
 
