#MAEC Malware Action Class

#Copyright (c) 2014, The MITRE Corporation
#All rights reserved

#Compatible with MAEC v4.1
#Last updated 02/18/2014

import cybox
from cybox.core import Action

import maec
import maec.bindings.maec_bundle as bundle_binding

class APICall(maec.Entity):
    _namespace = maec.bundle._namespace

    def __init__(self):
        super(APICall, self).__init__()
        self.function_name = None
        self.normalized_function_name = None
        self.address = None
        self.return_value = None
        self.parameters = []

    def to_obj(self):
        api_call_obj = bundle_binding.APICallType()
        if self.function_name is not None: api_call_obj.set_function_name(self.function_name)
        if self.normalized_function_name is not None: api_call_obj.set_normalized_function_name(self.normalized_function_name)
        if self.address is not None: api_call_obj.set_Address(self.normalized_function_name)
        if self.return_value is not None: api_call_obj.set_Return_Value(self.return_value)
        if len(self.parameters) > 0: 
            parameter_list_obj = bundle_binding.ParameterListType()
            for parameter in self.parameters:
                parameter_list_obj.add_Parameter(parameter.to_obj())
            api_call_obj.set_Parameters(parameter_list_obj)
        return api_call_obj

    def to_dict(self):
        api_call_dict = {}
        if self.function_name is not None: api_call_dict['function_name'] = self.function_name
        if self.normalized_function_name is not None: api_call_dict['normalized_function_name'] = self.normalized_function_name
        if self.address is not None: api_call_dict['address'] = self.address
        if self.return_value is not None: api_call_dict['return_value'] = self.return_value
        if len(self.parameters) > 0: 
            parameter_list = []
            for parameter in self.parameters:
                parameter_list.append(parameter.to_dict())
            api_call_dict['parameters'] = parameter_list
        return api_call_dict

    @staticmethod
    def from_dict(api_call_dict):
        if not api_call_dict:
            return None
        api_call_ = APICall()
        api_call_.function_name = api_call_dict.get('function_name')
        api_call_.normalized_function_name = api_call_dict.get('normalized_function_name')
        api_call_.address = api_call_dict.get('address')
        api_call_.return_value = api_call_dict.get('return_value')
        api_call_.parameters = ParameterList.from_list(api_call_dict.get('parameters', []))
        return api_call_

    @staticmethod
    def from_obj(api_call_obj):
        if not api_call_obj:
            return None
        api_call_ = APICall()
        api_call_.function_name = api_call_obj.get_function_name()
        api_call_.normalized_function_name = api_call_obj.get_normalized_function_name()
        api_call_.address = api_call_obj.get_Address()
        api_call_.return_value = api_call_obj.get_Return_Value()
        if api_call_obj.get_Parameters() is not None : api_call_.parameters = ParameterList.from_obj(api_call_obj.get_Parameters())
        return api_call_

class Parameter(maec.Entity):
    _namespace = maec.bundle._namespace

    def __init__(self):
        super(Parameter, self).__init__()
        self.ordinal_position = None
        self.name = None
        self.value = None

    def to_obj(self):
        parameter_obj = bundle_binding.ParameterType()
        if self.ordinal_position is not None: parameter_obj.set_ordinal_position(self.ordinal_position)
        if self.name is not None: parameter_obj.set_name(self.name)
        if self.value is not None: parameter_obj.set_value(self.value)
        return parameter_obj

    def to_dict(self):
        parameter_dict = {}
        if self.ordinal_position is not None: parameter_dict['ordinal_position'] = self.ordinal_position
        if self.name is not None: parameter_dict['name'] = self.name
        if self.value is not None: parameter_dict['value'] = self.value
        return parameter_dict

    @staticmethod
    def from_dict(parameter_dict):
        if not parameter_dict:
            return None
        parameter_ = Parameter()
        parameter_.ordinal_position = parameter_dict.get('ordinal_position')
        parameter_.name = parameter_dict.get('name')
        parameter_.value = parameter_dict.get('value')
        return parameter_

    @staticmethod
    def from_obj(parameter_obj):
        if not parameter_obj:
            return None
        parameter_ = Parameter()
        parameter_.ordinal_position = parameter_obj.get_ordinal_position()
        parameter_.name = parameter_obj.get_name()
        parameter_.value = parameter_obj.get_value()
        return parameter_

class ParameterList(maec.EntityList):
    _contained_type = Parameter
    _binding_class = bundle_binding.ParameterListType
    _binding_var = "Parameter"
    _namespace = maec.bundle._namespace

class ActionImplementation(maec.Entity):
    _namespace = maec.bundle._namespace

    def __init__(self):
        super(ActionImplementation, self).__init__()
        self.id = None
        self.type = None
        self.compatible_platforms = []
        self.api_call = None
        self.code = []

    def to_obj(self):
        implementation_obj = bundle_binding.ActionImplementationType()
        if self.id is not None: implementation_obj.set_id(self.id)
        if self.type is not None: implementation_obj.set_type(self.type) 
        if self.compatible_platforms is not None: pass 
            #platform_list_obj = bundle_binding.PlatformListType()
            #for platform in self.compatible_platforms:
            #    platform_list_obj.add_Platform(platform.to_obj())
            #implementation_obj.set_Compatible_Platforms(platform_list_obj)
        if self.api_call is not None: implementation_obj.set_API_Call(self.api_call.to_obj())
        if self.code is not None: pass
        return implementation_obj

    def to_dict(self):
        implementation_dict = {}
        if self.id is not None: implementation_dict['id'] = self.id
        if self.type is not None: implementation_dict['type'] = self.type
        if self.compatible_platforms is not None: pass 
            #platform_list_obj = bundle_binding.PlatformListType()
            #for platform in self.compatible_platforms:
            #    platform_list_obj.add_Platform(platform.to_obj())
            #implementation_obj.set_Compatible_Platforms(platform_list_obj)
        if self.api_call is not None: implementation_dict['api_call'] = self.api_call.to_dict()
        if self.code is not None: pass
        return implementation_dict

    @staticmethod
    def from_dict(implementation_dict):
        if not implementation_dict:
            return None
        implementation_ = ActionImplementation()
        implementation_.id = implementation_dict.get('id')
        implementation_.type = implementation_dict.get('type')
        #implementation_.compatible_platforms = implementation_dict.get('compatible_platforms') #TODO: add support
        implementation_.api_call = APICall.from_dict(implementation_dict.get('api_call'))
        #implementation_.code = implementation_dict.get('code') #TODO: add support
        return implementation_

    @staticmethod
    def from_obj(implementation_obj):
        if not implementation_obj:
            return None
        implementation_ = ActionImplementation()
        implementation_.id = implementation_obj.get_id()
        implementation_.type = implementation_obj.get_type()
        #implementation_.compatible_platforms = implementation_dict.get('compatible_platforms') #TODO: add support
        implementation_.api_call = APICall.from_obj(implementation_obj.get_API_Call())
        #implementation_.code = implementation_dict.get('code') #TODO: add support
        return implementation_

class MalwareAction(Action):
    _binding = bundle_binding
    _binding_class = bundle_binding.MalwareActionType
    _namespace = 'http://maec.mitre.org/XMLSchema/maec-bundle-4'

    implementation = cybox.TypedField("Implementation", ActionImplementation)





