#This file is part of Higgins. The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license
#terms.

from itertools import groupby
import logging

from sqlalchemy import desc, func
from sqlalchemy.orm import joinedload, subqueryload

from . import db, config
from .db import Build, Run, Test, Repository, Project

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger('higgins')

try:
    import flask
except ImportError:
    logger.error('Unable to start server: Flask module not found.')
    exit()
from flask import Flask, render_template, request, Response

app = Flask("Higgins",
            template_folder=config.TEMPLATE_DIR,
            static_folder=config.STATIC_DIR,
        )
app.jinja_env.add_extension("jinja2.ext.loopcontrols")

@app.route("/")
def index():
    page = int(request.args.get('page', 0))
    project = request.args.get('project', None)
    page_len = 10

    with db.session() as session:
        stmt = session.query(
            func.substr(Build.timestamp, 0, 11).label('day')
        ).order_by(
            desc(Build.timestamp)
        ).group_by(
            'day'
        ).offset(page_len*page).limit(page_len).subquery()

        res = session.query(func.min(stmt.c.day), func.max(stmt.c.day)).first()
        min_day, max_day = res or (None, None)

        # Extract build info in this period
        query = session.query(
            Build
        ).join(
            Project
        ).options(
            joinedload('*')
        ).order_by(
            desc(Build.timestamp)
        )

        if project:
            query = query.filter(Project.key.ilike('%' + project + '%'))

        if min_day:
            query = query.filter(
                func.substr(Build.timestamp, 0, 11) >= min_day,
                func.substr(Build.timestamp, 0, 11) <= max_day,
            )
        else:
            query = query.filter(False)

        res = groupby(query, lambda x: x.timestamp.date())
        by_date = [(d, list(b)) for d, b in res]

        return render_template('index.html',
                               by_date=list(by_date),
                               page=page,
                               project=project,
                               page_len=page_len,
                               config=config)

@app.route("/build/<int:build_id>/<key>")
@app.route("/build/<int:build_id>")
def build(build_id, key=None):
    with db.session() as session:
        query = session.query(
            Build, Run, Repository, Test
        ).join(
            Run
        ).join(
            Repository
        ).join(
            Test
        )

        # build with id 0 mean latest
        if build_id == 0:
            stmt = session.query(
                func.max(Build.id)
            ).subquery()
            query = query.filter(Run.build_id==stmt)
        else:
            query = query.filter(Run.build_id==build_id)

        results = query.all()
        if results:
            build = results[0][0]
        else:
            return render_template('404.html')
        return render_template('build.html', build=build, active_key=key)

@app.route("/raw/<int:run_id>/<name>")
def raw(run_id, name):
    with db.session() as session:
        r = session.query(Run).filter_by(id=run_id).first()
        return Response(r.get_data(name), mimetype='text/plain')
