#This file is part of Higgins. The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license
#terms.

from threading import Timer
import errno
import logging
import os
import shutil
import signal

from . import config, db
from .db import Project, Repository, Build, Test
from .utils import call, mkdir

logger = logging.getLogger('higgins')
MONITOR_TIMER = None
EMPTY_CONFIG = '''
my-project:
  name: My Project
  path: .
  repository: {}
  triage: {}
  test: {}
  build: {}

'''


def do_init():
    mkdir(config.DOT_DIR)
    mkdir(config.BUILD_DIR)

    # Create config file
    if not config.CONFIG:
        config.save(EMPTY_CONFIG)

    # Init db
    db.create_all()

    # Copy templates
    code_dir = os.path.dirname(os.path.realpath(__file__))
    tpl_dir = os.path.join(code_dir, 'template')
    static_dir = os.path.join(code_dir, 'static')
    try:
        shutil.copytree(tpl_dir, config.TEMPLATE_DIR)
        shutil.copytree(static_dir, config.STATIC_DIR)
    except OSError as exc:
        if exc.errno != errno.EEXIST:
            raise

def do_trigger():
    with db.session() as session:
        db.check_config(Project, config.CONFIG)
        query = session.query(Project).filter_by(active=True)
        for project in query:
            # Sync config
            ctx = {'project_id': project.id}
            conf = config.CONFIG.get(project.key)
            repo_conf = conf.get('repository', {})
            db.check_config(Repository, repo_conf, **ctx)

            test_conf = conf.get('test', {})
            db.check_config(Test, test_conf, **ctx)

            build_conf = conf.get('build', {})
            if not build_conf:
                continue

            # Sync repos
            project.sync_repositories()

            # Find dirty ones
            dirty = project.dirty_repositories()
            if not dirty:
                continue

            # Create build and runs
            Build.new_build(build_conf, project, dirty)


def do_build():
    while True:
        with db.session() as session:
            next_build = Build.next_waiting_build()
            if not next_build:
                break
            # TODO try to launch in parallel for different project
            next_build.launch()


def handler(signum, frame):
    logging.info('SIGINT called, trying to stop monitoring')
    global MONITOR_TIMER
    MONITOR_TIMER.cancel()
    exit()


def monitor(first=False):
    global MONITOR_TIMER
    if not first:
        do_trigger()
        do_build()

    MONITOR_TIMER = Timer(config.cli('monitor'), monitor)
    MONITOR_TIMER.start()

def do_serve():

    if config.cli('monitor'):
        monitor(True)
        signal.signal(signal.SIGINT, handler)


    from .web import app
    host = config.cli('host')
    port = config.cli('port')
    app.run(host=host, port=port, debug=config.cli('debug'))
