# encoding: utf-8
# Copyright 2015 California Institute of Technology. ALL RIGHTS
# RESERVED. U.S. Government Sponsorship acknowledged.

u'''EDRN LabCAS User Interface'''

from .interfaces import IBackend, IVocabularies
from .resources import Root
from edrn.labcas.ui.resources import Datasets, Dataset, Upload
from pyramid.authentication import AuthTktAuthenticationPolicy
from pyramid.authorization import ACLAuthorizationPolicy
from pyramid.config import Configurator
from pyramid_ldap import groupfinder
from zope.component import provideUtility
from zope.interface import implements
import xmlrpclib, solr


PACKAGE_NAME = __name__


class _Backend(object):
    implements(IBackend)
    def __init__(self, fileMgrURL, workflowMgrURL, stagingDir, archiveDir, solrURL):
        self.fileMgr = xmlrpclib.ServerProxy(fileMgrURL)
        self.workflowMgr = xmlrpclib.ServerProxy(workflowMgrURL)
        self.stagingDir, self.archiveDir = stagingDir, archiveDir
        self.solr = solr.SolrConnection(solrURL)
    def getFileMgr(self):
        return self.fileMgr.filemgr
    def getWorkflowMgr(self):
        return self.workflowMgr.workflowmgr  # Note case
    def getStagingDirectory(self):
        return self.stagingDir
    def getArchiveDirectory(self):
        return self.archiveDir
    def getSearchEngine(self):
        return self.solr


_protocols = [
    u"A Methylation Panel for Bladder Cancer",
    u"Affinity-based strategies to fast track development of colon cancer biomarkers",
    u"Antibodies Against Tumor Antigens or Mucin Epitopes in Ovarian Cancer",
    u"Assessment of Clinical Potential of Digital SNP Analysis",
    u"Aurora Kinase A as a Biomarker for the Detection of Bladder Cancer - Supplemental Pre-Validation Protocol",
    u"Autoantibody Approach for Serum-Based Detection of Head and Neck Cancer",
    u"Barrett's Esophagus Methylation Profiles",
    u"BBD Reference Set Application: Jeffery Marks-Duke (2015)",
    u"Benign Breast Disease Team Project",
    u"Biomarker Detection Using NAPPA Tumor Antigen Arrays",
    u"Biomarker Detection Using NAPPA Tumor Antigen Arrays: EDRN Supplement",
    u"Biomarker Dev Lab At Moffitt Supplement",
    u"Biomarker Development Laboratory",
    u"Biomarker Discovery for Renal Cell Carcinoma Through Serum, Urine and Tissue Protein Profiles",
    u"Biomarker Reference Sets for Cancers in Women",
    u"Biomarkers for early detection of pancreatic cancer",
    u"Biomarkers for Prostate and Bladder Cancer",
    u"Biomarkers for the Early Detection of Pancreatic Cancer",
    u"Biomarkers in Familial Multiple Myeloma",
    u"Biomarkers of Risk for Colorectal Neoplasia (Team Project #2)",
    u"Biomarkers to Distinguish Aggressive Cancers from Non-aggressive or Non- progressing cancers",
    u"Biomarkers to Distinguish Aggressive Cancers from Non-aggressive or Non-progressing Cancer",
    u"Body Fluids as a Source of Diagnostic Biomarkers",
    u"Body Fluids as a Source of Diagnostic Biomarkers: Breast",
    u"Body Fluids as a Source of Diagnostic Biomarkers: Prostate",
    u"Breast Cancer: subgroups specific blood-biomarkers for early / predictive diagnosis and personalized treatment",
    u"Breast Ductal  Lavage Fluid Specimen Bank",
    u"Breast Reference Set Application:  Anu Mathew-Meso Scale (2012)",
    u"Breast Reference Set Application: Chris Li-FHCRC (2014)",
    u"Breast Reference Set Application: Chris Li-FHCRC (2015)",
    u"Breast Reference Set Application: GeorgeTuszynski-Temple (2012)",
    u"Breast Reference Set Application: Karen Abbott- University of Arkansas (2013)",
    u"Breast Reference Set Application: Karen Anderson-ASU (2014)",
    u"Breast Reference Set Application: Pinku Mukherjee-CanDiag (2014)",
    u"Breast Reference Set Application: Richard Zangar-PNNL (2012)",
    u"Breast tissue array",
    u"BRSCW Reference Set Application: Anna Lokshin - Univ of Pitts (2008)",
    u"BRSCW Reference Set Application: Anu Mathew - Meso Scale (2006)",
    u"BRSCW Reference Set Application: Eleftherios Diamandis - Mt Sinai (2006)",
    u"BRSCW Reference Set Application: GeorgeTuszynski - Temple University (2012)",
    u"BRSCW Reference Set Application: James Morre - NOX Technologies Inc. (2009)",
    u"BRSCW Reference Set Application: Joe Buechler - Biosite Inc (2009)",
    u"BRSCW Reference Set Application: Karen  Abbott -University of Arkansas (2014)",
    u"BRSCW Reference Set Application: Michael Birrer-Massachusetts General Hospital (2013)",
    u"BRSCW Reference Set Application: Michael Wilson - EIC Laboratories, Inc (2007)",
    u"Canary TMA",
    u"Chemokine Prostate Cancer Biomarkers",
    u"Clinical and Analytical Validation of Cancer Biomarkers: Developmental Studies",
    u"Clinical Utility of Urinary CD90 as a Biomarker for Prostate Cancer Detection",
    u"Clinical Validation of Molecular Signatures of Cervical Neoplasia",
    u"CLUE Studies: Evaluating Biomarkers of Carcinogenesis",
    u"Collaborative study on SELDI biomarker discovery and serum protein profiling of lung cancers",
    u"Colon  Reference Set Application: Robert Getzenberg - Johns Hopkins (2008)",
    u"Colon Reference Set Application: Mary Disis - University of Washington (2008)",
    u"Colon Reference Set Application: Moncef Jendoubi - Milagen (2006)",
    u"Colon Reference Set Application: Nam Kim - DiaDexus (2008)",
    u"DCIS Team Project",
    u"Detecting breast cancer protein signatures in body fluid",
    u"Detecting Breast Cancer Signatures in Body Fluids",
    u"Detection of Bladder CA by Microsatellite Analysis (MSA)",
    u"Detection of cancer DNA markers in urine",
    u"Detection of Serum Markers of Breast Cancer by SELDI",
    u"Detection of Serum Proteomic Patterns and SNPs in Breast Cancer: A Pilot Study",
    u"Development and Clinical Validation of Biomarkers for Prostate Cancer",
    u"Development and Evaluation of a Tumor Marker for Prostate Cancer",
    u"Development of a magneto-nanosensor (MNS) array for prostate cancer biomarkers",
    u"Development of FISH-based assay for early detection of high grade cervical dysplasia",
    u"Development of paraffin-based expression profiles to predict minimal vs. substantial risk of disease progression using retrospective datasets",
    u"Development of reference samples and interlaboratory testing to improve the confidence in the measurement of microRNAs",
    u"Dubinett - Targeted Sequencing 2012",
    u"Early Detection of Breast Cancer Using Autoantibody Markers",
    u"Early Detection of Cancer by Affinity Mass Spectrometry-Set Aside funds",
    u"Early detection of liver cancer and hepatitis",
    u"Early Detection of Urinary Bladder Cancer",
    u"Early Detection Research Network Registry for Hereditary Cancer",
    u"Early Diagnosis of Pancreatic Cancer-Network Collaborative Studies",
    u"Early Molecular Detection of Invasive Bladder Cancer by Urine Examination",
    u"Ecogenetic Study of Lung Cancer",
    u"EDRN Biomarker Reference Lab: Pacific Northwest National Laboratory",
    u"EDRN Breast and Ovary Cancer CVC, Study 1: Preliminary validation of breast cancer early detection biomarker candidates from our WHI discovery project",
    u"EDRN Breast and Ovary Cancer CVC, Study 2: Phase 3 retrospective validation of ovarian cancer early detection markers in serial preclinical samples from the PLCO trial",
    u"EDRN Breast and Ovary Cancer CVC, Study 3: Phase 3 Validation of screening decision rules in preclinical UKCTOCS serial samples",
    u"EDRN Breast and Ovary Cancer CVC, Study 4: Phase 3 Validation of Ovarian Cancer Serum Markers in Preclinical WHI Samples",
    u"EDRN Informatics Pilot Project",
    u"EDRN Longitudinal Serum Biorepository",
    u"EDRN Pre-Validation of Metabolic Biomarkers of Prostate Cancer: Follow up EDRN Challenge",
    u"EDRN Pre-Validation of Multiplex Biomarker in Urine",
    u"EDRN Resource Network Exchange",
    u"EDRN-WHI Pre-Clinical Colon Ca Specimens",
    u"Establishment of a serum standard for use in high-throughput proteomic",
    u"Expression based markers for breast cancer detection",
    u"GLNE 003: Preliminary Validation of Biomarkers Predictive of Barrett’s Esophagus Progression to Dysplasia and Adenocarcinoma",
    u"Glycoprotein Biomarkers for the Early Detection of Aggressive Prostate Cancer",
    u"GSTP1 Methylation",
    u"Hepatocellular carcinoma Early Detection Strategy study",
    u"Hereditary Cancer Institute family study specimen bank",
    u"High-Throughput Evaluation of Breast Cancer Markers",
    u"HNPCC colon cancer-associated SELDI mass spectrometry protein profiles in serum",
    u"Hollingsworth - Aggressive vs Indolent 2012",
    u"Identification of biomarkers for lung cancer in never smokers",
    u"Identification of New Markers in Normal and Abnormal Pap Smears",
    u"Identification of protein and genetic biomarkers of prostate cancer and risk factors for progression of disease",
    u"Inflammation as a Potential Marker of Risk ad Progression for Prostate Cancer",
    u"Integrated Development of Novel Molecular Markers",
    u"Intergrated Systems Biology Approach for Ovarian Cancer Biomarker Discovery",
    u"Light Scattering Spectroscopy for the Detection of Colorectal Neoplasia",
    u"Liver Full Reference Set Application :Timothy Block - Drexel Univ (2010)",
    u"Liver Full Reference Set Application: David Lubman - Univ of Michigan (2011)",
    u"Liver Full Reference Set Application: Hiro Yamada - Wako (2011)",
    u"Liver Rapid Reference Set Application ( #2): Lubman - Univ of Michigan (2010)",
    u"Liver Rapid Reference Set Application (#1): David Lubman - Univ of Michigan (2009)",
    u"Liver Rapid Reference Set Application (#1): Laura Beretta-FHCRC (2011)",
    u"Liver Rapid Reference Set Application (#2): Laura Beretta-FHCRC (2011)",
    u"Liver Rapid Reference Set Application: Gary Norman-INOVA (2012)",
    u"Liver Rapid Reference Set Application: Gianluigi Giannelli - Univ of Bari (2008)",
    u"Liver Rapid Reference Set Application: Hemken - Abbott (2015)",
    u"Liver Rapid Reference Set Application: Hiro Yamada - Wako (2011)",
    u"Liver Rapid Reference Set Application: Kevin Qu-Quest (2011)",
    u"Liver Rapid Reference Set Application: Timothy Block - Drexel Univ (2008)",
    u"Lung Reference Set A Application:  Edward Hirschowitz - University of Kentucky (2009)",
    u"Lung Reference Set A Application: Dawn Coverley- University of York (2011)",
    u"Lung Reference Set A Application: LaszloTakacs - Biosystems (2010)",
    u"Lung Reference Set A Application: Lee Goodglick/ Eleftherios Diamandis- UCLA/MSH  (2009)",
    u"Lung Reference Set C Application: Rachel Ostroff - SomaLogics (2010)",
    u"MALDI Dilution Data: Randolph",
    u"Methylation & Proteomics-Based Detection of Bladder Cancer",
    u"Methylation tumor markers in HNPCC",
    u"MMR protein expression in HNPCC",
    u"Molecular Basis of Barrett's -Associated Carcinogenesis",
    u"Molecular Genetics of Urinary Bladder Carcinoma",
    u"Molecular Progression of Methylation in Bladder Cancer",
    u"MSA Bladder Reference Set Application: Charles Rosser-Hawaii (2014)",
    u"Mutational load distribution analysis (MLDA) in pancreatic cancer",
    u"Nashville Early Diagnosis Lung Cancer Project",
    u"National Ovarian Cancer Early Detection Program Blood and Genetics",
    u"New Biomarkers & Dysplastic Respiratory Epithelium",
    u"Novel Surveillance Strategies For Hereditary Nonpolyposis Colon Cancer Family Members",
    u"NYU Lung Cancer Biomarker Center",
    u"Ornithine decarboxylase (ODC) Polymorphism and Breast and Prostate Cancer",
    u"Ovarian Cancer Screening Pilot Trial In High Risk Women",
    u"Pancreatic Cancer Early Detection Stem Cell Biomarkers (CA117452 Set-Aside)",
    u"Pancreatic Cystic Fluid Reference Set",
    u"Pancreatic Reference Set Application- Ann Killary-MD Anderson (2012)",
    u"Pancreatic Reference Set Application: Brian Haab-Van Andel  (2012)",
    u"Pancreatic Reference Set Application: Ivan Blasutig-University of Toronto (2014)",
    u"Pancreatic Reference Set Application: Kazufumi Honda-National Cancer Center (2014)",
    u"Pancreatic Reference Set Application: Michael Hollingsworth-University of Nebraska (2014)",
    u"Pancreatic Reference Set Application: Surinder Batra- University of Nebraska (2014)",
    u"PASS Reference Set Application: Carroll Univ. of California SF (2012) PC101769 Predicting Prostate Cancer Progression at Time of Diagnosis",
    u"PASS Reference Set Application: Lin Univ. of Washington/OncoChip (2013)",
    u"PASS Reference Set Application: Lin Univ. of Washington/OPKO (2015)",
    u"PASS Reference Set Application: Lin UW (2010) TMPRSS2-ERG-PCA-PASS",
    u"PASS Reference Set Application: Nelson Fred Hutchinson Cancer Research Center (2013) Telomere Length in men with Prostate Cancer under Active Surveillance",
    u"PASS Reference Set Application: Ruth Etzioni-William Beaumont Hospital (2014)",
    u"PASS Reference Set Application: Thompson UT Health Science Center San Antonio (2014) SNPs for PCa Outcome",
    u"PCA3 Reference Set Application: T2-Erg-Martin Sanda-Emory (2014)",
    u"PCA3 Validation Study and Urinary Reference Set",
    u"Phase II Validation of a New Panel of Biomarkers for Early Detection of Ovarian Cancer",
    u"PLCO Ovarian Phase III Validation Study",
    u"Polyp Prediction Study",
    u"Pre-operative plevic mass study",
    u"Pre-Validation of Metabolomic Biomarkers of Prostate Cancer",
    u"Preliminary Clinical Characterization of Serum Biomarkers for Colorectal Neoplasms",
    u"Preliminary Validation of Biomarkers for the Detection of Colorectal Adenomas (Team Project #1)",
    u"Presence of simian virus 40 DNA sequences in human",
    u"Prognostic Markers in Lynch Syndrome Colorectal Cancers",
    u"Prostate Active Surveillance Study",
    u"Prostate Rapid Pre-Validation Set",
    u"Prostate Rapid Reference Set Application:  Brian Haab - Van Andel (2006)",
    u"Prostate Rapid Reference Set Application: Daniel Chan/Patty Beckman- JHU/University of Minnesota (2007)",
    u"Prostate Rapid Reference Set Application: Eleftherios Diamandis - Mt Sinai (2006)",
    u"Prostate Rapid Reference Set Application: Robert Getzenberg - John Hopkins (2006)",
    u"Prostate Reference Set Application: proPSA-Dan Chan-JHU (2008)",
    u"Prostate Reference Set Application: William Isaacs-John Hopkins (2010)",
    u"Prostate Upgrading Team Project",
    u"Prostate-Specific Glycoproteins for Early Detection of Prostate Cancer",
    u"Protein profiling for detection of high risk breast cancer",
    u"Proteomic Approach for Diagnostic Applications in Head and Neck Cancer",
    u"Proteomic Data Resources for EDRN Ovary Cancer Researchers within the EDRN",
    u"Proteomic Genetic and Longitudinal Pathways to Ovarian Cancer Biomarkers",
    u"Proteomic Profiling of Serum and Other Body Fluids for the Early Detection of Colon Cancers",
    u"Proteomics Biomarker Development Laboratory",
    u"Protocol for Comparing Promoter Hypermethylation Methods in NSCLC",
    u"PTM Microarray: Request for Year 3 Set-Aside Funds",
    u"Release of Set Aside Funds - microRNA Biomarkers for Renal Cell Carcinoma",
    u"Research Depository of Normal and Neoplastic Human Tissues",
    u"San Antonio Center of Biomarkers of Risk for Prostate Cancer (SABOR)",
    u"Screen Detection of Mismatch Repair Deficient Colorectal Neoplasias Using a Stool Based DNA Analysis",
    u"SELDI and LC-MS/MS discovery and profiling study of hepatocarcinomas",
    u"SELDI Phase I:  Assay Validation-Prostate",
    u"SELDI Profiling of serum and urine from patients with urologic disease in support of bladder cancer studies",
    u"SELDI Validation Study Phase II",
    u"Soluble  Mesothelin Related Peptide (SMRP) and Osteopontin (OPN) as Early Detection Markers for Malignant Mesothelioma (MM)",
    u"Spectral Markers for Early Detection of Colon Neoplasia",
    u"SPORE/EDRN/PRE-PLCO Ovarian Phase II Validation Study",
    u"Standard Specimen Reference Set:  Breast",
    u"Standard Specimen Reference Set:  Liver",
    u"Standard Specimen Reference Set:  Lung",
    u"Standard Specimen Reference Set:  Pancreatic",
    u"Standard Specimen Reference Set:  Prostate",
    u"Standard Specimen Reference Set: Colon",
    u"Study 5:  Phase 2 and 3 Validations of Putative Breast Cancer Early Detection Candidates",
    u"Targeted Sequencing for Discovery and Validation of DNA Methylation Markers of Colon Cancer Metastasis",
    u"The Biomarker Knowledge System Informatics Pilot Project Supplement To The Biomarker Development Laboratory at Moffitt (Bedlam)",
    u"The Early Detection Research Network: Biomarker Reference Laboratories",
    u"The Prostate Tumor Microenvironment Exhibits differentially expressed Genes Useful for Diagnosis",
    u"Triple Negative Breast Cancer Team Project",
    u"Upgrading Reference Set",
    u"US-Turkey Cappadocia Mesothelioma Project",
    u"Using MALDI-IMS and MRM to stablish a pipeline for discovery and validation of tumor neovasculature biomarker candidates.",
    u"Using the methylome to identify aggressive Barrett’s esophagus",
    u"Utility of Biomarkers for Early Detection of Malignant Mesothelioma in a High-risk Population",
    u"Validation of Biomarkers for the Early Detection of Colorectal Adenocarcinoma (GLNE 010)",
    u"Validation of Early Detection Ovarian Cancer Biomarkers (Team Project)",
    u"Validation of Molecular Biomarkers for the Early Detection of Lung Cancer in the setting of Indeterminate Pulmonary Nodules (Lung Team Project #2)",
    u"Validation of Protein Markers for Lung Cancer Using CARET Sera and Proteomics Techniques",
    u"Validation of Serum Markers for the Early Detection of Hepatocellular Carcinoma",
    u"Validation of the Mutagen Sensitivity and Chromosomal Hotspot Assays",
    u"Year 2 Collaborative Project - UCSF and Battelle PNWD",
]

_names = [
    u"Abbott, Karen",
    u"Abola, Matthew",
    u"Acuna, Fernando",
    u"Adam, Bao-Ling",
    u"Addrizzo-Harris, Doreen",
    u"Aderca, Ileana Sundra",
    u"Agarwal, Rajeev",
    u"Agostino, Naomi Ruth",
    u"Ahmed Mohammed, Hager F.",
    u"Ainslie, Nancy",
    u"Akleh, Siham",
    u"Akumu, Atieno Jenipher",
    u"Alberts, David S.",
    u"Ali, Hamdi A",
    u"Allen, Peter J.",
    u"Alpaugh, Katherine",
    u"Alshebeeb, Kutaiba Taha Hamid",
    u"Altland, Alexandra",
    u"Ames, Jacquelyn",
    u"Amini, Neda",
    u"Amling, Christopher L",
    u"Anandasabapathy, Sharmila",
    u"Anderson, Karen",
    u"Anderson, Michelle",
    u"Anderson, Roosevelt C",
    u"Andrews, Barbara",
    u"Anglim, Paul",
    u"Anton, Kristen",
    u"Aparicio, Belen",
    u"Ard, Susan",
    u"Arnold, Allyn",
    u"Ashfaq, Sadaf",
    u"Atha, Donald",
    u"Auguste, Fanchette",
    u"Auman, Heidi J",
    u"Aung, Tony",
    u"Austin, Betty Jo",
    u"Austin, Curtis",
    u"Axvig, Claudia K",
    u"Baack, Vanessa",
    u"Backman, Vadim",
    u"Baichoo, Esha",
    u"Baksh, Bibi",
    u"Baldwin, Cindy",
    u"Balshem, Andrew",
    u"Banerjee, Jayati",
    u"Banerjee, Priyanka",
    u"Banks, Marian",
    u"Bantis, Katrina",
    u"Barakat, Rody",
    u"Barbour, Dana",
    u"Barker, Peter E.",
    u"Barkin, Jack",
    u"Barnett, Matt",
    u"Baron, John A.",
    u"Barrett, Carl J.",
    u"Barrett, Thomas W.",
    u"Basler, Joseph W.",
    u"Basset, Nancy",
    u"Bast, Robert C",
    u"Batra, Surinder",
    u"Beaber, Elisabeth",
    u"Beach, David",
    u"Beane, Jennifer",
    u"Bearden, James",
    u"Beaver, Allison Ahrens",
    u"Beckmann, Patty",
    u"Bedilion, Tod",
    u"Beer, David",
    u"Befeler, Alex",
    u"Begum, Shahnaz",
    u"Bekaii-Saab, Tanios",
    u"Belinsky, Steven",
    u"Beltran, Roxxana Valeria",
    u"Bem, Jiri",
    u"Bemis, Lynne",
    u"Benetz, Mary Ellen",
    u"Benoit, Nicole",
    u"Benya, Richard",
    u"Berenberg, Jeffrey",
    u"Beretta, Laura",
    u"Berg, Barbara",
    u"Berg, Christine",
    u"Berger, Alvin",
    u"Berger, Barry",
    u"Berger, Yitzhak",
    u"Berry, Laura Michelle",
    u"Bettelon (formerly Williams), Lora Lynn",
    u"Bey, Stephanie",
    u"Bhamre, Suvarna",
    u"Bhandari, Manuj",
    u"Bhutani, Manoop",
    u"Bianchi, Fabrizio",
    u"Bianchi, Laura K.",
    u"Bidair, Mohamed",
    u"Bigbee, William L.",
    u"Bigley, Joseph",
    u"Bilbee, Maryann",
    u"Birrer, Michael",
    u"Black, Brad",
    u"Black, Janet M",
    u"Blanco, Kendra J",
    u"Blasutig, Ivan",
    u"Blaylock, Taylor",
    u"Block, Timothy",
    u"Blomquist, Thomas",
    u"Blumberg, Alex",
    u"Blunier, Adam",
    u"Blythe, Nancy",
    u"Bobadilla, Sandra",
    u"Bochner, Bernard",
    u"Bodger, Willie",
    u"Boeve, Tracy",
    u"Bok, Laurie A",
    u"Bomba, John",
    u"Booker, Abigail",
    u"Boyer, Hilary D",
    u"Boyer5, Hilary",
    u"Boyle, Peter",
    u"Brand, Leigh Ann",
    u"Brand, Randall E.",
    u"Branski, Jessica",
    u"Brassil, Diane N.",
    u"Brayer, Samuel",
    u"Breakey, Caitlin",
    u"Brenke, Connie",
    u"Brenner, Dean",
    u"Bresalier, Bob",
    u"Bresalier, Robert",
    u"Brewer, Catherine",
    u"Brissaud, Catherine",
    u"Bristol, Rebecca L",
    u"Brock, Malcolm",
    u"Broderick, Gregory",
    u"Brooks, Christine",
    u"Brooks, Ersulan",
    u"Brooks, James D.",
    u"Brooks, Mai N.",
    u"Brooks, Susan",
    u"Brotzman, Michelle",
    u"Broussard, Angie",
    u"Brower, Paul",
    u"Brown, Elissa Catherine",
    u"Brown, Patrick Ray",
    u"Bruce, Margaret",
    u"Bryce, Angela Marie",
    u"Buchanan, Becky",
    u"Buechler, Joe",
    u"Bungum, Aaron",
    u"Bungum, Erin",
    u"Bunn, Paul A.",
    u"Burdette, Wanda M.",
    u"Burford, Julia",
    u"Burns, Carolyn Mary",
    u"Burton, Shelley",
    u"Busby, Erik",
    u"Bussell, Lindy",
    u"Busuego, Leonor N",
    u"Butler, Leslie A",
    u"Buttke, Karyn",
    u"Butuci, Artrit",
    u"Buxton, Richard",
    u"Buzdar, Aman",
    u"Byers, Tim",
    u"Cairns, Paul",
    u"Caldwell, Christina Sue",
    u"Califano, Joseph",
    u"Camino, Debra",
    u"Camp, David",
    u"Campbell, Carmella",
    u"Campos, David",
    u"Campos-Ruiz, David",
    u"Candia, Enzo",
    u"Canfield, Vikki",
    u"Canto, Marcia",
    u"Capers, Chaunda",
    u"Capriotti, Nicole",
    u"Caputo, Jean",
    u"Carbone, David P",
    u"Carey, Carol",
    u"Carey, Martin",
    u"Carey709, Carol Ann",
    u"Carlisle, Destin",
    u"Carr, Deborah",
    u"Carr, Melissa L.",
    u"Carr, Steven",
    u"Carroll, Peter R.",
    u"Carroll, Robert",
    u"Carter, Leslie A",
    u"Casey, Richard",
    u"Cassano, Jennifer",
    u"Castonguay, Alicia",
    u"Castro, Matt",
    u"Cazares, Lisa",
    u"Celestino, Joeseph",
    u"Cerkoski, Jacqueline Lynn",
    u"Chadwick, Samuel",
    u"Chalhoub, Walid",
    u"Chamberlin, Kim",
    u"Chambers, Amber",
    u"Chan, Alex",
    u"Chan, Daniel",
    u"Chan, Miriam",
    u"Chapman, Caroline J.",
    u"Chaturvedi, Apurva",
    u"Chen, David",
    u"Chen, Xi",
    u"Cheng, Shanna",
    u"Cheslock, Jolene A",
    u"Chhieng, Cheung F",
    u"Chia, David",
    u"Chiles, Paige",
    u"Chinnaiyan, Arul M.",
    u"Chitale, Dhananjay (Dan)",
    u"Christ, Kimberly R",
    u"Chung, Phyllis",
    u"Church, Tim R",
    u"Cirocco, Maria",
    u"Cirri, Holly O.",
    u"Clapper, Mary Ann Samonte",
    u"Clavero, José C",
    u"Cleary, Jaclyn",
    u"Clements, Mary Ann",
    u"Clipp, Sandra",
    u"Close, Kathleen",
    u"Coakley, Cindy",
    u"Colbert, Maureen A",
    u"Collins, Colin",
    u"Collins, Michael",
    u"Conaway, Caitlin",
    u"Cook, Bernard",
    u"Cook, Megan M",
    u"Cooper, Kara Lynn",
    u"Cooperberg, Matthew",
    u"Coopersmith, Allyson",
    u"Cordes, Jill",
    u"Cornelius, Gloria P",
    u"Correa, Jacqueline",
    u"Cortez, Norma",
    u"Costa, Jose",
    u"Cote, Donna",
    u"Cotter, Robert",
    u"Courter, Michael",
    u"Coverley, Dawn",
    u"Cowan, Kenneth H.",
    u"Cramer, Daniel",
    u"Crawley, Brigette",
    u"Crichley, Cheryl T",
    u"Crichton, Dan",
    u"Crigler, Elisabeth",
    u"Crissman, William",
    u"Critchlow, Cathy",
    u"Croce, Carlo",
    u"Crociani, Catrina",
    u"Crockett, Seth",
    u"Crofoot, Chelsea A",
    u"Crofoot, Chelsea Anne",
    u"Cross, Carlton L",
    u"Cross, Toni",
    u"Crowder, Kelly",
    u"Cunningham, Patricia",
    u"Cunningham, Windy",
    u"Curry, Victoriana K",
    u"Cutts, Alisa",
    u"Czerniak, Bogdan",
    u"Dahlgren, Jackie R.",
    u"Dai, Jianliang",
    u"Dakhil, Noma",
    u"Dakhil, Shaker",
    u"Dal Molin, Marco",
    u"Dalbagni, Guido",
    u"Daly, Patrick Michael",
    u"Daniels, Courtney",
    u"Daniels, Lisa",
    u"Dash, Atreya",
    u"Davila, Marta",
    u"Davis, Gerard",
    u"Davis, Harriet Stratton",
    u"Davis, Teleah",
    u"DeCotiis, Christopher",
    u"DeShazo, Michael",
    u"Deal, Mary Ellen",
    u"Dekovich, Alexander",
    u"Demarest, Chad",
    u"Demasco-Cody, Christine D.",
    u"Dempe, Amy",
    u"Derr, Kim",
    u"DesMarais, Donna",
    u"Desler, Michelle M",
    u"Devore, Nathan M",
    u"Dewey, Lauren",
    u"Dhaliwal, Rupi",
    u"Dhir, Rajiv",
    u"DiMaio, Christopher",
    u"Diamandis, Eleftherios P.",
    u"Dias, Hazel Marie",
    u"Diaz, Jose I.",
    u"Dibisceglie, Adrian",
    u"Digel, Jon",
    u"Dineen, Martin",
    u"Ding, Ivan",
    u"Dinh, Mai",
    u"Dinney, Colin P.",
    u"Disis, Mary L",
    u"Dixit, Darshana Ajit",
    u"Dizdar, Miral",
    u"Do, Kim Anh",
    u"Doan, Oanh Hoang",
    u"Dobi, Albert",
    u"Dobry, Craig",
    u"Donahue, Kathleen",
    u"Donato, Elizabeth",
    u"Donohue, Debra Gail",
    u"Donohue, Debra-Gail",
    u"Dotherow, Melinda",
    u"Douglas, Kristen",
    u"Douglas, Leslie",
    u"Dowell, Barry Lee",
    u"Downing, Sharon E",
    u"Drake, Richard",
    u"Drapkin, Ronny",
    u"Drew, Karon",
    u"Dubinett, Steve",
    u"Dudley, Beth",
    u"Dumas, Yves-Martine",
    u"Dumont, Charles",
    u"Dunn, Laura",
    u"Dunne, Robert",
    u"Duong, Trang",
    u"Duris, Robin Margaret",
    u"Dwyer, Danielle",
    u"Dwyer, Erin",
    u"Ebbel, Erin Elizabeth",
    u"Eckert, Emily A",
    u"Edelstein, Cim",
    u"Eeles, Rosalind",
    u"Ehret, Carolyn",
    u"Eisenberg, Joshua D",
    u"Eisenberg, Marcia",
    u"El-Hefnawy, Talal",
    u"El-Zein, Randa",
    u"Elashoff, David",
    u"Elborne, Vicki",
    u"Elias, Keren J",
    u"Ellinwood, Linda J",
    u"Elsaleh, Hany",
    u"Elton, Eric",
    u"Eltoum, Isam-Eldin",
    u"Emanuele, Vincent A",
    u"Ench, Yasmin Mistry",
    u"Engelhard, Deborah",
    u"Engels, Dale",
    u"Engstrom, Paul",
    u"Ernst, Amy B",
    u"Espino, Christian A",
    u"Esserman, Laura J.",
    u"Etzioni, Ruth",
    u"Euhus, David",
    u"Evans Long, Felicia",
    u"Eylers, Ellen",
    u"Faber, Angela",
    u"Fabian, Elizabeth Ashton",
    u"Fabian, Sarah",
    u"Fabrizio, Michael D.",
    u"Faino, Anna",
    u"Fan, Jing",
    u"Farley, Paul Joseph",
    u"Farlow, Erin C.",
    u"Farris, John David",
    u"Faulk, Marsha Lynn",
    u"Fazli, Ladan",
    u"Fazzari, Ruth",
    u"Fedarko, Neal S.",
    u"Feinberg, Emily",
    u"Fellenz, Lori L",
    u"Feng, Qinghua",
    u"Feng, Ziding",
    u"Fennimore, Todd",
    u"Fern-Wittek, Charlene",
    u"Ferrari, Michelle K",
    u"Ferris, Robert L.",
    u"Field, Nell A",
    u"Figueroa, Jonine",
    u"Finan, Michael",
    u"Finkelstein, Lisa",
    u"Finley, Katherine",
    u"Fishbein, Michael",
    u"Fisher (formerly Ruther), Nancy",
    u"Fisher, Natalie",
    u"Fishman, David",
    u"Flagg, Sarah",
    u"Fletcher, Sandy L.",
    u"Flores, Irma",
    u"Floyd, Stacy",
    u"Fore, Ian M",
    u"Foreman, Jessica T",
    u"Forry, Samuel P",
    u"Foster, Kevin",
    u"Francar, Lori",
    u"Frank, Angela",
    u"Franke, Judith",
    u"Franklin, Wilbur Alan",
    u"Frazier, Marsha",
    u"Frederick, NCI",
    u"Friedman, Natalie",
    u"Friend, Stephen",
    u"Frisby, Kathy",
    u"Fritz, Alan K",
    u"Frost, Andra",
    u"Frost, Stephen",
    u"Fu, Sherry",
    u"Fu196, Sherry",
    u"Fu197, Sherry",
    u"Gagliardi, Rosemarie",
    u"Gallagher, Stacey",
    u"Galvin, Nora K.",
    u"Gamage, Kristina",
    u"Gambhir, Sanjiv",
    u"Ganeshan, Srivenkatesh",
    u"Gann, Jordan",
    u"Gantman, Kathryn Ellen",
    u"Garcia, Marileila Varella",
    u"Gardner, Brian",
    u"Gardner, Kimberly",
    u"Garrett, Doris",
    u"Gaston, Sandra M.",
    u"Gazdar, Adi",
    u"Ge, Shaokui",
    u"Geisser, Mindy",
    u"Geradts, Joseph",
    u"Geraldizo, Dianne B",
    u"Gerlach, Scott Michael",
    u"Gerson, Stan",
    u"Getzenberg, Robert H.",
    u"Ghassemi, Faezeh",
    u"Ghosh, Mita",
    u"Ghosh-Janjigian, Sharmistha",
    u"Giama, Nasra",
    u"Giannelli, Gianluigi",
    u"Gibson, Donna Marlene",
    u"Gilchrist, Christy L",
    u"Giles, Karen",
    u"Gilkerson, Mona",
    u"Gillette, Michael",
    u"Gillies, Robert J.",
    u"Gimigliano, Sheri",
    u"Gite, Sadanand",
    u"Gladney, Jim",
    u"Glazier, Angie",
    u"Gleave, Martin",
    u"Glover, Nancy",
    u"Gnoleba, Annalisa",
    u"Godwin, Andrew K",
    u"Goeders, Christina Eleanor",
    u"Goff, Rachael",
    u"Goldberg, Judith D",
    u"Goldman, Radoslav",
    u"Goldsmith, Krista",
    u"Golubnitschaja, Prof. Dr. Olga",
    u"Golya, Kara",
    u"Gomez, Jorge",
    u"Gong, Lixin",
    u"Gonzales, Deanna R",
    u"Gonzales, Teresita",
    u"Gonzalez, Alex",
    u"Gonzalez, Kimberly",
    u"Goodall, Ginger",
    u"Goodglick, Lee",
    u"Goodman, Gary",
    u"Goodwin, Brianne",
    u"Gorelik, Dinah",
    u"Gormanns, Laurie",
    u"Gorzenski, Michael P",
    u"Gosbee, Wayne",
    u"Goudey, Kili",
    u"Gowland, Patricia Alice",
    u"Goyal, Parag",
    u"Graber, Susan",
    u"Grable, Michael",
    u"Grady UW, William",
    u"Grady, William",
    u"Granda, Kelly",
    u"Grant, Erica",
    u"Grantham, Rebecca",
    u"Grates, Harry E.",
    u"Graves, Judith",
    u"Green, Virginia A",
    u"Greenberg, Alissa",
    u"Greenberg, Alissa K.",
    u"Greenberg, Richard",
    u"Grizzle, William E.",
    u"Grogan, Thomas",
    u"Groover, Kathleen",
    u"Groskopf, Jack",
    u"Grossman, Herbert B",
    u"Group, BSA Working",
    u"Grove, Nancy",
    u"Gruidl, Mike",
    u"Gubb, Stephanie",
    u"Guerrero, Marta Elizabeth",
    u"Guerrero-Preston, Rafael",
    u"Guha, Sushovan",
    u"Guiles, Elizabeth",
    u"Gullstrand, Garth",
    u"Gurbaxani, Brian",
    u"Gurganus, Robin",
    u"Gutierrez, Alberto",
    u"HIrsch, Fred R.",
    u"Haab, Brian",
    u"Haczkiewicz, Erica",
    u"Hager, Linda",
    u"Haisfield-Wolfe, Mary Ellen",
    u"Halbherr, Theresa Catherine",
    u"Hall, James",
    u"Hall, Susannah",
    u"Halling, Kevin C",
    u"Hamilton, James P",
    u"Hamilton, Marjorie",
    u"Hammond, Mary",
    u"Han, Misop",
    u"Hanash, Samir",
    u"Hancock, William",
    u"Haney, Jerry",
    u"Hanlon, David",
    u"Hannon, Alice",
    u"Hanselman, Linda",
    u"Harbine, Becky",
    u"Hardigan, Signe",
    u"Harker, Matthew",
    u"Harnois, Denise",
    u"Harrelson, Stacey",
    u"Harrington, Ryan",
    u"Harris, Robert",
    u"Hart, Andrew Frederick",
    u"Harter, Amy",
    u"Harthun, Kenneth G.",
    u"Harvey, Lisa Manning",
    u"Hasalla, Besmira",
    u"Hashmi, Syed",
    u"Hasik, Karen",
    u"Hauge, Laurie L",
    u"Havard, Julia T",
    u"Havell, Thomas C",
    u"Haviland, Dana Lynne",
    u"Hawes, Stephen E.",
    u"Hawley, Sarah Joan",
    u"Haworth, Janet",
    u"Haydel, Brandy Michelle",
    u"Hayden, Alberto",
    u"Haywood, Cynthia",
    u"Hazelton, William Donald",
    u"Healy, Caroline",
    u"Heath, Darlene",
    u"Heath, Sara",
    u"Heimann-Nichols, Ellen Rose",
    u"Helber, Sarah",
    u"Heller, Dorothy",
    u"Helzlsouer, Kathy",
    u"Hemken, Philip M",
    u"Hench, Rita",
    u"Hendricks, Megan Anne",
    u"Hendrix, Thomas R",
    u"Herald, Barbara B",
    u"Herman, Kirk",
    u"Hernandez, Maria Herlinda",
    u"Hibberts, Janelle",
    u"Hickok, Melody",
    u"Hiebert, Janet",
    u"Higginbotham, Brooke",
    u"Highsmith, Edward",
    u"Hinton, David",
    u"Hirschberg, Sally",
    u"Hirschowitz, Edward",
    u"Hnosko, Donna",
    u"Hoang, Joseph",
    u"Hoerres, Mike",
    u"Hoffman-Bolton, Judith",
    u"Hoffmann, Joyce",
    u"Hogan, Michael",
    u"Holcomb, Chenee",
    u"Hollingsworth, Michael",
    u"Hollister, Becky",
    u"Hollister, Sally Marie",
    u"Holmes, Janet",
    u"Honda, Kazufumi",
    u"Honer, Kristin Marie",
    u"Honn, Susan",
    u"Hoover, Christine D",
    u"Hopkins, Judith",
    u"Horick, Nora",
    u"Horsley, Jonelle",
    u"Horvath, Steve",
    u"House, Joann",
    u"Hu, Jianhua",
    u"Hu, Ye (Tony)",
    u"Huang, Audrey",
    u"Huang, Han-Yao",
    u"Huang, Xin",
    u"Huang, Ying",
    u"Huflejt, Margaret E.",
    u"Hughes, Ruth",
    u"Hughes, Steven",
    u"Humphreys, Liz",
    u"Humrich, Jean",
    u"Hussain, Munira",
    u"Hutchinson, Robert",
    u"Ibaseta Fidalgo, Alvaro",
    u"Igel, Todd",
    u"Ignatov, Alex P",
    u"Infante, Elena",
    u"Isaacs, William",
    u"Isenberg, Matthew",
    u"Isozaki, Lisa",
    u"Ito, Tetsuo",
    u"Izbicka, Elzbieta",
    u"Jackson, Carla",
    u"Jackson, Donna E",
    u"Jackson, Renee",
    u"Jain, Surbhi",
    u"Jalayer, Annette",
    u"Jameson, JoAnn R",
    u"Jamil, Haris",
    u"Janakos, Maria",
    u"Jandorf, Lina",
    u"Janes, Holly Elizabeth",
    u"Jansz, Ken",
    u"Jaslowski, Anthony",
    u"Jatoi, Mashal",
    u"Jeames, Sanford",
    u"Jean, Tatyana",
    u"Jeldres, Claudio",
    u"Jendoubi, Moncef",
    u"Jenkins, Carol",
    u"Jeong, Sang Ho",
    u"Jett, James R",
    u"Jhala, Nirag",
    u"Jiang, Feng",
    u"Jin, Michelle",
    u"Jiwani, Ali Zul",
    u"Johnson, Candy",
    u"Johnson, Dianna",
    u"Johnston, Dennis",
    u"Jones, Elizabeth",
    u"Jones, Juanita L",
    u"Joon, Aron",
    u"Joost, Sarah",
    u"Jorgensen, Anna",
    u"Justusson, Carole",
    u"KANDUNOORI, PRAGATHI",
    u"Kagan, Jacob",
    u"Kalloger, Steve E.",
    u"Kalmykow, Brianna C.",
    u"Kamat, Ashish M.",
    u"Kaminski, Mary",
    u"Kandil, Tarek",
    u"Kang, Kwisa",
    u"Kastrinos, Fay",
    u"Kaul, Karen",
    u"Kayser, Michael",
    u"Kelleher, Timothy",
    u"Kelly, David",
    u"Kelly, Sean",
    u"Kelly, Suzanne M.",
    u"Kendrick, Kerri",
    u"Ketcham, Marsha A",
    u"Khan, Nadja",
    u"Khan, Seema A.",
    u"Kho, Jennilee",
    u"Kibel, Adam",
    u"Kiernan, Maureen",
    u"Killary, Ann M.",
    u"Kim, Bumjin",
    u"Kim, Jhingook",
    u"Kim, Nam",
    u"Kimes, Cedric L.",
    u"Kimes, Traci L.",
    u"Kimmie, Crystal Ann",
    u"Kincaid, Heather L.",
    u"King, Bonnie L.",
    u"King, Debra Lee",
    u"King, Meredith",
    u"Kingston, Susan",
    u"Kinsinger, Chris",
    u"Kinzler, Kenneth",
    u"Kiviat, Nancy B.",
    u"Klein, Alison",
    u"Klein, Eric",
    u"Kleiner, Brian",
    u"Klotz, Laurence",
    u"Knezevic, Vladimir",
    u"Knutson, Wilma",
    u"Ko, Tung-Chun",
    u"Koch, Jodi",
    u"Koch, Julie A",
    u"Kokubu, Joshua",
    u"Kokx, Patricia A",
    u"Kolb, Susan",
    u"Kolb, Suzanne",
    u"Kollath, Lori",
    u"Kontras, Dana",
    u"Kos, Mary (Beth) E",
    u"Koudele, Linda",
    u"Koulouris, Teresa",
    u"Krakar, April D",
    u"Kramer, Deneice",
    u"Kreps, Stig",
    u"Krueger, Karl",
    u"Krumpe, Clara",
    u"Ku, Winston",
    u"KuKuruga, Debra Lynn",
    u"Kucherlapati, Raju",
    u"Kuchinsky, Kyle",
    u"Kucmeroski, Diana Marie",
    u"Kueh, Alice",
    u"Kuhajda, Francis",
    u"Kulasingam, Vathany",
    u"Kulpa, Jan",
    u"Kumbongsi, Simi",
    u"Kurnit, David",
    u"Kuross, Steven A.",
    u"Kurowski, Karishma",
    u"Kusum, Sharma",
    u"Kutikov, Alexander",
    u"Kuwada, Scott",
    u"Kwon, Richard",
    u"Kwon, Sung Won",
    u"Kyle, Madison",
    u"Kyle, Tanya",
    u"LaBaer, Joshua",
    u"LaRock, Kylee",
    u"Lai, Hong",
    u"Laird, Julie Illi",
    u"Laird-Offringa, Ite",
    u"Lake, Douglas",
    u"Lam, Stephen",
    u"Lam, Wan",
    u"Lamb, Carla",
    u"Lamb, Janette",
    u"Lampe, Paul",
    u"Lance, Peter",
    u"Lance, Raymond S.",
    u"Landini, Cheryl",
    u"Landis, Patricia",
    u"Langseth, Laura",
    u"Larcom, William",
    u"Lau, Hiu-Lam",
    u"Laudone, Vincent P.",
    u"Law, Singwu",
    u"Law, Wendy",
    u"Lawter, Anna W.",
    u"Lawter, Heather Michelle",
    u"Lazaridis, Emmanuel",
    u"Le, Richard Hieu",
    u"LeVan, Rebecca M",
    u"Leach, Robin J",
    u"Lee, Amy J.",
    u"Lee, Cheryl T.",
    u"Lee, Chung",
    u"Lee, Daisy R",
    u"Lee, Gina",
    u"Lee, Yvonne",
    u"Leerapun, Apinya",
    u"Legge, Cindy",
    u"Legister, Miguel",
    u"Lehmann, Brenda",
    u"Leibert, Eric",
    u"Lemmen, Kelsey",
    u"Lenburg, Marc",
    u"Lennon, Anne Marie",
    u"Leone, Esther A",
    u"Leone, Victoria",
    u"Lerner, Seth P.",
    u"Leung, Hon-chiu Eastwood",
    u"Levchak, Branda",
    u"Levenson, Richard",
    u"Levin, Nancy",
    u"Levin, Stephen",
    u"Levine, Lyuba",
    u"Lewis, La Toya Amanda",
    u"Lewis, Shawnee",
    u"Li, Christopher",
    u"Li, Li",
    u"Li, Lin",
    u"Li, Ni",
    u"Li, Thomas",
    u"Lieban, Carol",
    u"Liebler, Daniel",
    u"Liles, Melinda",
    u"Lim, Mark",
    u"Lin UW, Daniel W.",
    u"Lin VA, Daniel W.",
    u"Lin, Cheryl",
    u"Lin, Peter",
    u"Lincoln, Samantha Jane",
    u"Lindsey, Lora",
    u"Linehan, Jennifer A",
    u"Lira, Ruth Rosalie",
    u"Listwin, Don",
    u"Litka, Jeanne A",
    u"Litschi, Erica",
    u"Littlejohn, Jessica",
    u"Liu, Alvin Y.",
    u"Liu, Robert",
    u"Liu, Tao",
    u"Liu, Yan",
    u"Livneh, Zvi",
    u"Lizcano, Carolina A",
    u"Locascio, Laurie",
    u"Lockhart, James B.",
    u"Logsdon, Craig D.",
    u"Lois, Auguso",
    u"Lok, Anna",
    u"Lokshin, Anna",
    u"Long, Joan L",
    u"Long, Joseph",
    u"Longhi, Kay",
    u"Longton, Gary",
    u"Lopez, Karen",
    u"Loprinzi, Charles",
    u"Lotan, Yair",
    u"Lotz, Cruz",
    u"Loughlin, Kevin R.",
    u"Lovato, Corinne",
    u"Lowry, Diana",
    u"Lozano, Marisa",
    u"Lu, Lili",
    u"Lubman, David",
    u"Luduena, Richard F",
    u"Lynch, Henry",
    u"Lynem, Ray",
    u"Lynn, Audrey H",
    u"Lyons, Shannon",
    u"Lyons-Weiler, James",
    u"M'Koma, Amosy E",
    u"Ma, Jonathan",
    u"MacDonnell, Marian A",
    u"Mace, Ashley Elaine",
    u"Mack, Erika",
    u"Macoska, Jill",
    u"Maes, Jessica",
    u"Mah, Vei",
    u"Mahoney, Edward",
    u"Main, Brian P.",
    u"Malcolm, Albert",
    u"Malehorn, David E.",
    u"Malick, John David",
    u"Malik, Gunjan",
    u"Malloy, Gloria",
    u"Malnik, Royceann",
    u"Maloney, Elizabeth M.",
    u"Maness, Deborah S",
    u"Mangold, Leslie",
    u"Manion, Martin",
    u"Manne, Upender",
    u"Mantor, Laurie",
    u"Manwaring, Carrie Louann",
    u"Manzano, Wilfred",
    u"Mao, Jenny",
    u"Maragh, Samantha",
    u"Marcon, Norman",
    u"Marcott, Valerie",
    u"Mardones, Sebastian",
    u"Maresh, Erin",
    u"Markowitz, Sanford",
    u"Marks, Jeffrey",
    u"Marks, Leonard Stephen",
    u"Marlow, Cameron",
    u"Marquis, Sara",
    u"Marrero, Jorge",
    u"Marsh, Tracey Lynn",
    u"Marshall, James",
    u"Marshall, Kathy Jo",
    u"Marshall-Booth, Heidi",
    u"Marticke, Simone",
    u"Martin, Debbie L.",
    u"Martin, Melissa ",
    u"Martinell, Connie",
    u"Martinez, Abraham",
    u"Massion, Pierre",
    u"Mateja, Gina",
    u"Mathew, Anu",
    u"Mathur, Arun",
    u"Mathur, Subbi P.",
    u"Matthai, Hoyt",
    u"Matthews, Deneice",
    u"Matthews, Denise",
    u"Mattmann, Chris",
    u"Matza, Martha",
    u"May, Gary",
    u"Mazumder, Raja",
    u"Mazurchuk, Richard",
    u"Mazurczak, Miroslaw",
    u"McAleer, Cecilia anne",
    u"McAnally, Kyle Christopher",
    u"McBeath, Katy",
    u"McCaskill, Shelley",
    u"McCaskill, Shelly",
    u"McCollum, Jennifer",
    u"McConnell, Daniel S.",
    u"McCreedy, Bruce",
    u"McCullough, Elizabeth T",
    u"McDermott, Kena A",
    u"McGinn, Travis",
    u"McGrath, Sandra",
    u"McIntosh, Martin",
    u"McKenney, Jesse",
    u"McLerran, Dale",
    u"McMellion, Sheila",
    u"McMillen, Jeannie",
    u"McNeill, Irene",
    u"McNie, Beverley",
    u"McWilliams, Kristin",
    u"McWilliams, Melissa A.",
    u"Medina, Maria Victoria",
    u"Mehta, Anand",
    u"Mehta, Neha",
    u"Meltzer, Stephen",
    u"Mena Canto, Francia Ayerim",
    u"Mendez-Meza, Rosario",
    u"Meng, Maxwell",
    u"Merchant, Nipun",
    u"Mercola, Dan",
    u"Mesfin, Asfaw",
    u"Messina, James Carmine",
    u"Messing, Edward",
    u"Messmer, Brandon",
    u"Mettler, Teresa",
    u"Meyer, Lori",
    u"Meyer, Tara",
    u"Meyers, Michael",
    u"Miguel, Juan",
    u"Mihindukulasuriya, Kusal Leonard",
    u"Milan-Robertson, Tanya Nicole",
    u"Mileti, Debbie",
    u"Millado, Kristen",
    u"Miller, Allison",
    u"Miller, Rebecca",
    u"Millspaugh, Lois Anne",
    u"Minihan, Anna",
    u"Miroballi, Yolanda",
    u"Misek, David",
    u"Mitcham, Josephine",
    u"Mitchell, Bree",
    u"Mitchell, Shamina",
    u"Miyamoto, Akemi",
    u"Mizell, Corey",
    u"Mizrahi, Isaac A",
    u"Modur, Vijay",
    u"Molina, Criselda",
    u"Molotsky, Sharon",
    u"Monje, Barbara",
    u"Monnig, William B.",
    u"Monroe, Brigitte Burlingame",
    u"Moon, Chulso",
    u"Moore, Dan",
    u"Moore, Victoria J",
    u"Mor, Gil",
    u"Morgan, Andrew R",
    u"Morgan, Linda",
    u"Morgan, Todd M.",
    u"Morgan, William Christopher",
    u"Mori, Yuriko",
    u"Morre, D. James",
    u"Mortensen, Diana",
    u"Morton, Thomas",
    u"Moser, Catherine",
    u"Moses, Harold",
    u"Mosquera, Juan Miguel",
    u"Mossauer, Heather D",
    u"Moyer, Suzanne",
    u"Mu, Yifei",
    u"Mueller, Gregory Paul",
    u"Mukherhjee, Pinku",
    u"Mukherjee, Pinku",
    u"Muller, Carolyn",
    u"Mullins, Casey",
    u"Mullis, Heather",
    u"Munoz-Antonia, Teresita",
    u"Murphy, Ruth G.",
    u"Murtha, Kit",
    u"Musovic, Sabina",
    u"Muterspaugh, Anel W",
    u"Muti, Paola",
    u"Mutter, George L.",
    u"Naidich, David",
    u"Nam, Robert",
    u"Nana-Sinkam, Serge Patrick",
    u"Nanez, Esther Elva",
    u"Naylor, Susan L.",
    u"Nelson (UW), Peter S",
    u"Nelson, Peter S",
    u"Nelson, William G.",
    u"Newcomb, Lisa",
    u"Newcomb5, Lisa",
    u"Newschaffer, Craig",
    u"Ng, Karen",
    u"Nghiem, Belinda Phan",
    u"Nghiem441, Belinda",
    u"Nguyen, Michael",
    u"Nguyen, Mindie",
    u"Nguyen, Minh-Tri",
    u"Nguyen, Pauline",
    u"Nguyen, Thu A",
    u"Nichols, Nancee",
    u"Nicks, Teresa",
    u"Nicks226, Teresa",
    u"Nielsen, Hans Jorgen",
    u"Nilsson, Olle",
    u"Ning, Jing",
    u"Nissenberg, Merel Grey",
    u"Nitta, Hiroaki",
    u"Noel, Jonathan",
    u"Nolan, Lorna Golla",
    u"Noll, Georgiann",
    u"Nomiyama, Erin",
    u"Norman, Gary L.",
    u"Norton, Larry",
    u"Noteboom, Jennifer",
    u"Novotny, Milos",
    u"Nyalwidhe, Julius O",
    u"O'Briant, Kathy",
    u"O'Brien, Trish",
    u"O'Leary, Judy",
    u"Oelschlager, Denise",
    u"Oettel, Kurt",
    u"Oh, Jane",
    u"Ojelabi, Michael",
    u"Oleson, Beth Dunn",
    u"Ontko, Mary",
    u"Osmack, Patricia",
    u"Ostovaneh, Mohammad Reza",
    u"Ostroff, Rachel",
    u"Ouellette, Michel",
    u"Owens, Janiqua",
    u"Owusu-Sarpong, Yvonne",
    u"Ozanne, Elissa",
    u"Padula, Gilbert",
    u"Page-Lester, Stephanie",
    u"Palko, Sharon R",
    u"Palmer, Maureen",
    u"Panicker, Gitika",
    u"Panik, Emily Theresa",
    u"Pannell, Lewis",
    u"Pape-Zambito, Dana Alyce",
    u"Parekh, Dipen J",
    u"Parikh, Neehar",
    u"Paris, Pamela",
    u"Park, John",
    u"Park, Jong",
    u"Park, Walter",
    u"Parker, Alexander",
    u"Parker, Beth",
    u"Parker, Suzanne Byan",
    u"Partin, Alan W",
    u"Partridge, Edward E",
    u"Pass, Harvey Ira",
    u"Patel, Anjli S",
    u"Patel, Divya",
    u"Patel, Nihir",
    u"Patenaude, Linda",
    u"Patriotis, Christos",
    u"Patz Jr., Edward F.",
    u"Patz, Edward",
    u"Pavri, Tanya Michelle",
    u"Peck, Vicky",
    u"Peltier-Siemers, Terry",
    u"Pena, Dalia J",
    u"Peng, Sophia",
    u"Pennel, Samantha",
    u"Pepe, Margaret Sullivan",
    u"Perkins, Doreen",
    u"Petricoin, Emanuel",
    u"Petrou, Steven",
    u"Petrovics, Gyorgy",
    u"Petty, Beth Ann",
    u"Pham, Kimberly",
    u"Pham, Xuan Thanh",
    u"Phan, Jeanine",
    u"Pichetsurnthorn, Pie",
    u"Pickens, Chrisley V.",
    u"Pierce, J. Michael",
    u"Pine, P Scott",
    u"Pittman, Dennis",
    u"Pittman, Jeffrey",
    u"Plant, Anne",
    u"Plate, Cindy",
    u"Plate607, Cynthia (Cindy)",
    u"Plonka, Caitlyn",
    u"Poe, Dolores",
    u"Polanco, Jacqueline",
    u"Ponder, Bruce",
    u"Poore, Baiba",
    u"Popel, Aleksander",
    u"Potter, John D",
    u"Preiss, Jordan Scott",
    u"Prentice, Ross",
    u"Presson, Angela",
    u"Presti, Joseph C.",
    u"Prichard, Jeffrey",
    u"Priest, Marie",
    u"Privatte, Samantha Joann",
    u"Prokrym, Kirill Davidson",
    u"Pruitt, Amy",
    u"Pugmire, Lindsey",
    u"Pui, Sandra",
    u"Pullin, Nancy",
    u"Pullin88, Nancy Alice",
    u"Punnen, Sanoj",
    u"Qu, Kevin",
    u"Quan, Susan",
    u"Raben, Marian Gillooly",
    u"Rahman, Jamshedur S.M.",
    u"Railey, Elda",
    u"Raju, Gottumukkala S",
    u"Rall, Christopher",
    u"Ramakrishnan, Anna",
    u"Randolph, Timothy",
    u"Ransohoff, David F.",
    u"Realiza, Ned",
    u"Reckers, Darcie",
    u"Recupero, Audrey L",
    u"Reczek, Peter",
    u"Reddy, Rajender",
    u"Reed, Ruth Carrie",
    u"Reid, Brian J.",
    u"Reid, Suzanna Lynne",
    u"Reiss, Jean",
    u"Rennert, Gad",
    u"Reploeg, Angela",
    u"Ressom, Habtom W.",
    u"Rettman196, Erin",
    u"Reynolds, Heather M",
    u"Reynolds, Thomas R",
    u"Rheaume, Robert Leo",
    u"Ribes, José C",
    u"Rice, Vicki",
    u"Richards, Kristy L.",
    u"Ricketts, Janette",
    u"Rickman, Otis",
    u"Ricks, Linda",
    u"Riffle, Stacy",
    u"Rinaudo, Jo Ann",
    u"Ring, Andrea",
    u"Rittmaster, Roger S.",
    u"Roayaie, Sasan",
    u"Roberson, Stacy",
    u"Roberts, Brian",
    u"Roberts, Lewis",
    u"Robertson, John ",
    u"Robinson, Bruce",
    u"Rocconi, Rodney",
    u"Rodgers, Kristin",
    u"Rodland, Karin",
    u"Rodriguez, Stephanie M",
    u"Roessig, Julie Marie",
    u"Roestenburg, Jessica",
    u"Rogers, Charlene",
    u"Rogut, Tatiana",
    u"Rom, William",
    u"Romany, Kellie",
    u"Romeo, Janae Ann",
    u"Rosen, Fran",
    u"Rosenblatt, Margery",
    u"Rosenzweig, Jason",
    u"Ross, Eric",
    u"Ross, William",
    u"Rosser, Charles J",
    u"Rothstein, RN, CCRC, Jamie Jay",
    u"Rowland, Alexander Parke",
    u"Rowland, Kendrith",
    u"Rowley, Jerry",
    u"Roy, Hemant K.",
    u"Rubin, Mark",
    u"Ruiz, Jose-Luis",
    u"Rushing, Kathy",
    u"Ruther, Nancy",
    u"Ryan, Nadine",
    u"Ryland, Betsy",
    u"Salit, Marc",
    u"Salnikov, Kathryn",
    u"Saltarelli, Kathleen",
    u"Salyer, Pamela",
    u"Sambamoorthi, Nethra",
    u"Sampliner, Richard",
    u"Samuels, Vanessa M",
    u"Sanda, Martin",
    u"Sasson, Aaron",
    u"Sawyers, Anne",
    u"Schaefer, Matthew T",
    u"Schellhammer, Paul",
    u"Schiestle, Jay",
    u"Schleissman, Katie",
    u"Schloss, Patrick",
    u"Schmidt, Bernd",
    u"Schmittgen, Thomas",
    u"Schoen, Robert E.",
    u"Schoenberg, Mark P.",
    u"Schooler, Phyllis",
    u"Schuster, Mary",
    u"Schwab, Richard",
    u"Schwartz, Myron",
    u"Selaru, Florin",
    u"Semmes, John",
    u"Sen, Subrata",
    u"Serbedzija, George N.",
    u"Sexton, Katherine C.",
    u"Shafi, Mehnaz A",
    u"Shaheen, Nicholas",
    u"Shahid, Ramza",
    u"Shak, Steve",
    u"Sharma, Kusum",
    u"Sheffield, Jackie",
    u"Shi, Xiaoyu",
    u"Shields, Deborah Carol",
    u"Shiggs, Brenda B.",
    u"Shinohara, Katsuto",
    u"Shipman, Donald Wayne",
    u"Shirriff, Karen",
    u"Shoe, Amy Beth",
    u"Shore, Neal",
    u"Shota, Kathleen Toyoko",
    u"Shue, Jing",
    u"Shulman, Keith",
    u"Siddiq, Fauzia",
    u"Siddiqui, Javed",
    u"Siddiqui, Rabia",
    u"Sidransky, David",
    u"Siegel, Mark",
    u"Siegner, Joan",
    u"Silverman, Greg Marc",
    u"Silverman, Joyce M",
    u"Silvestre, Josephine",
    u"Simcik, Amy E",
    u"Simeone, Diane M.",
    u"Singh, Harpreet",
    u"Singh, Pramiti",
    u"Singh, Shailender",
    u"Siropaides, Tim",
    u"Skates, Steven",
    u"Slebos, Robbert",
    u"Smart, Julie Ann",
    u"Smith, Dick",
    u"Smith, Elizabeth M.",
    u"Smith, Jill",
    u"Smith, Richard",
    u"Smith, Robin Scott",
    u"Smith, Shannon",
    u"Smith, Steven",
    u"Smith-McCune, Karen",
    u"Smoulder, Leah Noelle",
    u"Snyder, Debbie",
    u"So, Alan",
    u"Sokoll, Lori",
    u"Solomon, Ben",
    u"Solomon, Sheila",
    u"Sorbara, Lynn Rose",
    u"Soto, Ricardo",
    u"Spaulding, Amanda",
    u"Spaulding, Patricia E",
    u"Speicher, David W.",
    u"Spinarelli, Colette",
    u"Spira, Avrum",
    u"Spitz, Margaret R.",
    u"Squires, Michelle Dorrine",
    u"Srivastava, Shiv",
    u"Srivastava, Sudhir",
    u"St. Clare, Janeira",
    u"Stadheim, Linda Marie",
    u"Stanford, Janet L.",
    u"Stark, Azadeh",
    u"Stasi, Jason",
    u"Stass, Sanford",
    u"Steen SCCOC, Preston",
    u"Steen, Preston",
    u"Steffensen, Karina Dahl",
    u"Steiling, Katrina",
    u"Steinberg, Gary D.",
    u"Stelling, Deanna",
    u"Stengelin, Martin",
    u"Stevens, Victoria L.",
    u"Stewart, Susan",
    u"Stiller, Martha",
    u"Stimmel, Abby C",
    u"Stinchfield, Kathy",
    u"Stivers, David",
    u"Stocker, Susan Jane",
    u"Stone, Stacy M",
    u"Storms, Mary",
    u"Stout, Paul",
    u"Strahan, Francis X",
    u"Strawder, Terrence",
    u"Streed, Bonnie",
    u"Strickland, Paul",
    u"Stroehlein, John R.",
    u"Stromberg, Jennifer Ann",
    u"Su, Grace",
    u"Su, Ying Hsiu",
    u"Subryan, Shanni",
    u"Suddaby, Pamela",
    u"Sugita, Michio",
    u"Sullivan, Amy",
    u"Sullivan, Susan",
    u"Sullivan, Susan Vasterling",
    u"Sullivan, Travis",
    u"Sultan, Mohamed",
    u"Surette, Amanda",
    u"Surfus, Natalie Hope",
    u"Swaim, Michael Parks",
    u"Sweet, Crystal Autumn",
    u"Sweet, Sherrita",
    u"Sweney, Carol J",
    u"Syngal, Sapna",
    u"Taafaki, Munirh",
    u"Tabb, David",
    u"Tabb, David L.",
    u"Tainsky, Michael A.",
    u"Takano, Atsushi",
    u"Talerico, Veronica",
    u"Tamashiro, Sharon",
    u"Taneja, Samir",
    u"Tang, Hua",
    u"Tang, Jeffrey",
    u"Tang, Kuang S",
    u"Tavassoli, Peyman",
    u"Taylor, Jeremy",
    u"Taylor, Kimberly N",
    u"Taylor, Selima",
    u"Tchou-Wong, Kam-Meng",
    u"Tenggara, Imelda I.",
    u"Terrano, Robert John",
    u"Tewari, Ashutosh K.",
    u"Tewari, Muneesh",
    u"Thomas, Courtney M",
    u"Thomas, Louis",
    u"Thomas, Vicky",
    u"Thomas-Jones, Yolanda",
    u"Thompson, Anne R",
    u"Thompson, Ian M.",
    u"Thorlacius, Steinunn",
    u"Thornquist, Mark D",
    u"Thorsen, Jonean",
    u"Thunnissen, Erik",
    u"Thyagarajan, Veena",
    u"Thyagarajan196, Veena",
    u"Thyagarajan197, Veena",
    u"Tinley, Susan",
    u"Tipps, Tamara",
    u"Tockman, Melvyn S.",
    u"Tomasini, Maggie",
    u"Torlak, Mersiha",
    u"Toro, Charlene",
    u"Torres, Juan Pablo",
    u"Torres, Vanessa",
    u"Torretta, Susan",
    u"Townsend, R. Reid",
    u"Tran, Thuy",
    u"Tretiakova, Maria",
    u"Trevino, Sylvia",
    u"Trieu, Vanessa",
    u"Trinh, Sam",
    u"Tripon, Renee",
    u"Trock, Bruce",
    u"Troyer, Dean",
    u"True, Lawrence",
    u"Tsay, Junchieh",
    u"Tuck, Missy",
    u"Tucky, Barbara",
    u"Tucky, Barbara Helen",
    u"Turgeon, Kim",
    u"Tuszynski, George",
    u"Tuttle, Craig",
    u"Tyndall, Gail",
    u"UWAH, JUDE",
    u"Ujhazy, Peter",
    u"Ukaegbu, Chinedu",
    u"Ulrich, Neli",
    u"Umansky, Samuil",
    u"Unger, Elizabeth R.",
    u"Urban, Nicole",
    u"Urbano, Elane",
    u"Urzúa , Ilse",
    u"Usinowicz, Mary Beth",
    u"Utreja, Nipun",
    u"Uzzell, Norwood",
    u"Uzzo, Robert",
    u"Valcour, Andre",
    u"Van-Den-Ende, Alex",
    u"Varadarajan, Raj",
    u"Vargo, Janet",
    u"Varki, Ajit",
    u"Vedachalam, Srikanth",
    u"Veltri, Robert W.",
    u"Vernon, Cheryl B",
    u"Vessella, Robert L.",
    u"Vignesh, Shivakumar",
    u"Villagra, Cristina",
    u"Villena, Amelia",
    u"Vincent, Michael",
    u"Vinod, Manasi",
    u"Viterbo, Rosalia",
    u"Vitonis, Allison",
    u"Vo, Kevin P",
    u"Volk, Michael L.",
    u"VonMerveldt, Dendra",
    u"Vuskovic, Marko",
    u"Vydelingum, Nadarajen A.",
    u"Waclawski, Shelbie",
    u"Wadman, Sandi",
    u"Wagner, Andrew",
    u"Wagner, Paul",
    u"Wahr, Mary Catherine",
    u"Walker, Roger",
    u"Walker, Ronald Clark",
    u"Wallace, Deanna",
    u"Wallace, Deanna Orange City",
    u"Walter, Dawn Terese",
    u"Walters, Mindy",
    u"Wang, Denong",
    u"Wang, Dong",
    u"Wang, Gerald",
    u"Wang, Kenneth K",
    u"Wang, Pei",
    u"Wang, Vivien SW",
    u"Wang, WenFu",
    u"Wang, Wendy",
    u"Ward, David C",
    u"Ward, Maggie",
    u"Ward, Michael",
    u"Ward, Michelle",
    u"Ward, Patty",
    u"Warnick, Gregory S",
    u"Warrington, Sandra",
    u"Warrington, Sandy",
    u"Wasvary, Harry",
    u"Watson, Crystal",
    u"Watson, Jahme",
    u"Watt, Gavin",
    u"Weaver, Brandi",
    u"Weaver, Jo Ellen",
    u"Webb, Cynthia Salas",
    u"Wehle, Michael",
    u"Wehrmeyer, Kathryn",
    u"Wei, John",
    u"Weisenborn, Loretta Ann",
    u"Weisgerber, Adam J",
    u"Weiss, Matthias ",
    u"Weiss, Robert",
    u"Wells, Amy",
    u"Wertz, Joan",
    u"Wessells, Hunter",
    u"Westcott, Rick",
    u"White, Heather A",
    u"White, Rebekah",
    u"Whitten, Courtney",
    u"Wielma, Wilson",
    u"Wiener, Benjamin Russell",
    u"Willette, Bill",
    u"Williams, Melaine",
    u"Williams, Patrick",
    u"Williams, Tamara",
    u"Wilson, Kelly",
    u"Wilson, Michael S.",
    u"Winfrey, Carolyn Suzanne",
    u"Winter, Brendan Byron",
    u"Winter, Jordan",
    u"Winter, Lauren E.",
    u"Winters, Ryan Michael",
    u"Wirtala, Kelly",
    u"Wisdom, Ashlee Jeniel",
    u"Wise, Margaret",
    u"Witcher, Kiarra",
    u"Wlajnitz, Tina",
    u"Wolfert, Robert",
    u"Wolfsen, Herbert",
    u"Wooden, Carmen M.",
    u"Wright, Jonathan Lawrence",
    u"Wright, Liliana",
    u"Wrison, Kristina",
    u"Wu, Elizabeth",
    u"Wu, Lingling",
    u"Wynne-Jones, Cindy Lee",
    u"Xiao, Yan",
    u"Xie, Angela",
    u"Xu, Yanjuan",
    u"Yamada, Hiro",
    u"Yan, Aimin",
    u"Yancey, Corinne S",
    u"Yang, David",
    u"Yang, Hui-yu",
    u"Ye, Moncy",
    u"Yerokun, Olutola Alice",
    u"Yie, Ting-an",
    u"Yingst, Jamie",
    u"Yost, Kathleen",
    u"Young, Matthew Ryan",
    u"Young, Nancy",
    u"Young, Paul",
    u"Younus, Imamah",
    u"Zadra, Joseph",
    u"Zangar, Richard (Rick) C.",
    u"Zarate, Izelda Ann",
    u"Zeng, Gang",
    u"Zhang, Hui L.",
    u"Zhang, Jing",
    u"Zhang, Lan",
    u"Zhang, Ruo-Dan",
    u"Zhang, Xiaodan",
    u"Zhang, Yingxi",
    u"Zhang, Zhen",
    u"Zhao, Shanshan",
    u"Zhao, Yingming",
    u"Zheng, Minxue",
    u"Zheng, Yingye",
    u"Zhong, Li",
    u"Zhou, Boxin",
    u"Zhou, Jun",
    u"Zhu, Liu",
    u"Zhukov, Tatyana",
    u"Zielke, Brenda",
    u"Zimmerman, Abby M",
    u"Zriny, Sara Lynn",
    u"Zwick, Michael G.",
    u"de Frates, Rebecca A",
    u"de Luna, Patricia Whitney",
    u"jiang, xiao ying",
]


class _Vocabularies(object):
    implements(IVocabularies)
    def __init__(self):
        pass
    def getPeople(self):
        return _names
    def getProtocols(self):
        return _protocols


def main(global_config, **settings):
    """ This function returns a Pyramid WSGI application.
    """
    config = Configurator(settings=settings, root_factory=Root)
    config.include('pyramid_chameleon')
    config.include('pyramid_beaker')
    config.include('pyramid_ldap')
    config.ldap_setup(
        settings['ldap.url'], settings['ldap.manager'], settings['ldap.password'], timeout=int(settings['ldap.timeout'])
    )
    config.ldap_set_login_query(
        base_dn=settings['ldap.user.base'],
        filter_tmpl=settings['ldap.user.filter'].decode('utf-8').replace(u'\uff05', u'%'),
        scope=int(settings['ldap.user.scope'])
    )
    config.ldap_set_groups_query(
        base_dn=settings['ldap.group.base'],
        filter_tmpl=settings['ldap.group.filter'].decode('utf-8').replace(u'\uff05', u'%'),
        scope=int(settings['ldap.group.scope']),
        cache_period=int(settings['ldap.group.cache'])
    )
    config.set_authentication_policy(AuthTktAuthenticationPolicy(
        settings['authtkt.secret'], hashalg=settings['authtkt.hashalg'], callback=groupfinder
    ))
    config.set_authorization_policy(ACLAuthorizationPolicy())
    config.add_static_view('static', 'static', cache_max_age=3600)  # use ini?
    config.add_static_view('deform_static', 'deform:static')
    config.add_route('home', '/')
    config.add_route('datasets', '/datasets', factory=Datasets)
    config.add_route('dataset', '/datasets/{datasetID}', factory=Dataset)
    config.add_route('upload', '/upload', factory=Upload)
    config.add_route('metadata', '/upload/{workflowID}', factory=Upload)
    config.add_route('accept', '/upload/{workflowID}/accept', factory=Upload)
    config.add_route('login', '/login')
    config.add_route('logout', '/logout')
    config.scan()
    provideUtility(_Backend(
        settings['labcas.filemgr'],
        settings['labcas.workflow'],
        settings['labcas.staging'],
        settings['labcas.archive'],
        settings['labcas.solr']
    ))
    provideUtility(_Vocabularies())
    return config.make_wsgi_app()
