import pytest

from labgrid.resource import NetworkPowerPort
from labgrid.driver.powerdriver import ExternalPowerDriver, ManualPowerDriver, NetworkPowerDriver


class TestManualPowerDriver:
    def test_create(self, target):
        d = ManualPowerDriver(target, name='foo-board')
        assert (isinstance(d, ManualPowerDriver))

    def test_on(self, target, mocker):
        m = mocker.patch('builtins.input')

        d = ManualPowerDriver(target, name='foo-board')
        target.activate(d)
        d.on()

        m.assert_called_once_with(
            "Turn the target foo-board ON and press enter"
        )

    def test_off(self, target, mocker):
        m = mocker.patch('builtins.input')

        d = ManualPowerDriver(target, name='foo-board')
        target.activate(d)
        d.off()

        m.assert_called_once_with(
            "Turn the target foo-board OFF and press enter"
        )

    def test_cycle(self, target, mocker):
        m = mocker.patch('builtins.input')

        d = ManualPowerDriver(target, name='foo-board')
        target.activate(d)
        d.cycle()

        m.assert_called_once_with("CYCLE the target foo-board and press enter")


class TestExternalPowerDriver:
    def test_create(self, target):
        d = ExternalPowerDriver(
            target, cmd_on='set -1 foo-board', cmd_off='set -0 foo-board'
        )
        assert (isinstance(d, ExternalPowerDriver))

    def test_on(self, target, mocker):
        m = mocker.patch('subprocess.check_call')

        d = ExternalPowerDriver(
            target, cmd_on='set -1 foo-board', cmd_off='set -0 foo-board'
        )
        target.activate(d)
        d.on()

        m.assert_called_once_with(['set', '-1', 'foo-board'])

    def test_off(self, target, mocker):
        m = mocker.patch('subprocess.check_call')

        d = ExternalPowerDriver(
            target, cmd_on='set -1 foo-board', cmd_off='set -0 foo-board'
        )
        target.activate(d)
        d.off()

        m.assert_called_once_with(['set', '-0', 'foo-board'])

    def test_cycle(self, target, mocker):
        m_sleep = mocker.patch('time.sleep')
        m = mocker.patch('subprocess.check_call')

        d = ExternalPowerDriver(
            target, cmd_on='set -1 foo-board', cmd_off='set -0 foo-board'
        )
        target.activate(d)
        d.cycle()

        assert m.call_args_list == [
            mocker.call(['set', '-0', 'foo-board']),
            mocker.call(['set', '-1', 'foo-board']),
        ]
        m_sleep.assert_called_once_with(2.0)

    def test_cycle_explicit(self, target, mocker):
        m_sleep = mocker.patch('time.sleep')
        m = mocker.patch('subprocess.check_call')

        d = ExternalPowerDriver(
            target,
            cmd_on='set -1 foo-board',
            cmd_off='set -0 foo-board',
            cmd_cycle='set -c foo-board',
        )
        target.activate(d)
        d.cycle()

        m.assert_called_once_with(['set', '-c', 'foo-board'])
        m_sleep.assert_not_called()

class TestNetworkPowerDriver:
    def test_create(self, target):
        r = NetworkPowerPort(target, model='netio', host='dummy', index='1')
        d = NetworkPowerDriver(target)
        assert isinstance(d, NetworkPowerDriver)

    def test_import_backends(self):
        import labgrid.driver.power
        import labgrid.driver.power.apc
        import labgrid.driver.power.digipower
        import labgrid.driver.power.gude
        import labgrid.driver.power.netio

