# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
# STEPS - STochastic Engine for Pathway Simulation
# Copyright (C) 2007-2014 Okinawa Institute of Science and Technology, Japan.
# Copyright (C) 2003-2006 University of Antwerp, Belgium.
#
# See the file AUTHORS for details.
#
# This file is part of STEPS.
#
# STEPS is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# STEPS is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

#  Last Changed Rev:  $Rev: 352 $
#  Last Changed Date: $Date: 2010-08-09 10:49:54 +0900 (Mon, 09 Aug 2010) $
#  Last Changed By:   $Author: wchen $

try:
    import cubit
except ImportError:
    print("Unable to find CUBIT module.")

print "This is the obsolescent version of STEPS CUBIT supporting toolkit, try use the independent toolkit instead."

import random
import os
import sys
import steps.geom as sgeom

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def getSelectedNodes(node_proxy):
    """
    Return the STEPS indices and CUBIT indices of the selected vertex nodes.
        
    Parameters:
        * node_proxy   Vertex node element proxy generated by STEPS mesh importing function
        
    Return:
        (steps_ids, cubit_ids)
    """
    group_id = cubit.create_new_group()
    cubit.silent_cmd("group %i add selection" % (group_id))
    cubit_ids = cubit.get_group_nodes(group_id)
    steps_ids = [node_proxy.getSTEPSID(id) for id in cubit_ids]
    cubit.delete_group(group_id)
    return (steps_ids, cubit_ids)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def getSelectedTets(tet_proxy):
    """
    Return the STEPS indices and CUBIT indices of the selected tetrahedrons.
    
    Parameters:
        * tet_proxy   Tetrahedron element proxy generated by STEPS mesh importing function
        
    Return:
        (steps_ids, cubit_ids)
    """
    group_id = cubit.create_new_group()
    cubit.silent_cmd("group %i add selection" % (group_id))
    cubit_ids = cubit.get_group_tets(group_id)
    steps_ids = [tet_proxy.getSTEPSID(id) for id in cubit_ids]
    cubit.delete_group(group_id)
    return (steps_ids, cubit_ids)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def getSelectedTris(tri_proxy):
    """
    Return the STEPS indices and CUBIT indices of the selected triangles.
    
    Parameters:
        * tri_proxy   Triangle element proxy generated by STEPS mesh importing function
    
    Return:
        (steps_ids, cubit_ids)
    """
    group_id = cubit.create_new_group()
    cubit.silent_cmd("group %i add selection" % (group_id))
    cubit_ids = cubit.get_group_tris(group_id)
    steps_ids = [tri_proxy.getSTEPSID(id) for id in cubit_ids]
    cubit.delete_group(group_id)
    return (steps_ids, cubit_ids)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def SelectedTetsAsComp(mesh, tet_proxy, comp_name):
    """
    Create a compartment in the STEPS Tetmesh object using selected tetraedrons.
    
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * tet_proxy   Tetrahedron element proxy generated by STEPS mesh importing function
        * comp_name   Name of the compartment
    
    Return:
        steps.geom.TmComp object
    """
    
    steps_ids = getSelectedTets(tet_proxy)[0]
    comp = sgeom.TmComp(comp_name, mesh, steps_ids)
    return comp

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def SelectedTrisAsPatch(mesh, tri_proxy, patch_name, icomp = None, ocomp = None):
    """
    Create a patch in the STEPS Tetmesh object using selected triangles.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * tri_proxy   Triangle element proxy generated by STEPS mesh importing function
        * patch_name  Name of the patch
        * icomp       Inner TmComp compartment object (not name), None by default
        * ocomp       Outer TmComp compartment object (not name), None by default
        
    Return:
        steps.geom.TmPatch object
    """
    
    steps_ids = getSelectedTris(tri_proxy)[0]
    patch = sgeom.TmPatch(patch_name, mesh, steps_ids, icomp, ocomp)
    return patch

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def SelectedNodesAsROI(mesh, node_proxy, roi_name):
    """
    Store the selected nodes as Region of Interest in the STEPS Tetmesh object.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * node_proxy  Node element proxy generated by STEPS mesh importing function
        * roi_name    Name of the ROI
        
    Return:
        None
    """
    
    steps_ids = getSelectedNodes(node_proxy)[0]
    mesh.addROI(roi_name, sgeom.ELEM_VERTEX, steps_ids)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def SelectedTetsAsROI(mesh, tet_proxy, roi_name):
    """
    Store the selected tetrahedrons as Region of Interest in the STEPS Tetmesh object.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * tet_proxy   Tetrahedron element proxy generated by STEPS mesh importing function
        * roi_name    Name of the ROI
        
    Return:
        None
    """
    
    steps_ids = getSelectedTets(tet_proxy)[0]
    mesh.addROI(roi_name, sgeom.ELEM_TET, steps_ids)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def SelectedTrisAsROI(mesh, tri_proxy, roi_name):
    """
    Store the selected triangles as Region of Interest in the STEPS Tetmesh object.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * tri_proxy   Triangle element proxy generated by STEPS mesh importing function
        * roi_name    Name of the ROI
        
    Return:
        None
    """
    
    steps_ids = getSelectedTris(tri_proxy)[0]
    mesh.addROI(roi_name, sgeom.ELEM_TRI, steps_ids)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def getNodesBoundInSelectedVols(mesh, scale = 1e-6):
    """
    Return nodes bound in selected CUBIT volumes.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * scale       LENGTH scale from the CUBIT gemoetry to real geometry.
                    e.g. a radius of 10 in CUBIT to a radius of 1 micron in STEPS, scale is 1e-7.
                    By default the scale is 1e-6, i.e. 1 unit in CUBIT equals 1 micron in STEPS.
                    
    Return:
        list of STEPS indices of vertex nodes
    """
    
    group_id = cubit.create_new_group()
    cubit.silent_cmd("group %i add selection" % (group_id))
    volume_ids = cubit.get_group_volumes(group_id)
    nverts = mesh.nverts
    
    in_list = []
    
    for v in range(nverts):
        cords = mesh.getVertex(v)
        cords = [x / scale for x in cords]
        for vol_id in volume_ids:
            volume = cubit.volume(vol_id)
            body = volume.bodies()[0]
            status = body.point_containment(cords)
            if status == 1 or status == 2:
                in_list.append(v)
                break
    cubit.delete_group(group_id)
    return in_list

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def getTetsBoundInSelectedVols(mesh, scale = 1e-6):
    """
    Return tetraherons bound in selected CUBIT volumes.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * scale       LENGTH scale from the CUBIT gemoetry to real geometry.
                    e.g. a radius of 10 in CUBIT to a radius of 1 micron in STEPS, scale is 1e-7.
                    By default the scale is 1e-6, i.e. 1 unit in CUBIT equals 1 micron in STEPS.
                    
    Return:
        list of STEPS indices of tetrahedrons
    """
    
    group_id = cubit.create_new_group()
    cubit.silent_cmd("group %i add selection" % (group_id))
    volume_ids = cubit.get_group_volumes(group_id)
    ntets = mesh.ntets
    
    in_list = []
    
    for t in range(ntets):
        verts = mesh.getTet(t)
        with_in = True
        cords = []
        for v in verts:
            c = mesh.getVertex(v)
            c = [x / scale for x in c]
            cords.append(c)
        for vol_id in volume_ids:
            volume = cubit.volume(vol_id)
            body = volume.bodies()[0]
            within = True
            for cord in cords:
                status = body.point_containment(cord)
                if status == 0:
                    within = False
                    break
            if within:
                in_list.append(t)
                break
    cubit.delete_group(group_id)
    return in_list

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def getTrisBoundInSelectedVols(mesh, scale = 1e-6):
    """
    Return triangles bound in selected CUBIT volumes.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * scale       LENGTH scale from the CUBIT gemoetry to real geometry.
                    e.g. a radius of 10 in CUBIT to a radius of 1 micron in STEPS, scale is 1e-7.
                    By default the scale is 1e-6, i.e. 1 unit in CUBIT equals 1 micron in STEPS.
                    
    Return:
        list of STEPS indices of triangles
    """
    
    group_id = cubit.create_new_group()
    cubit.silent_cmd("group %i add selection" % (group_id))
    volume_ids = cubit.get_group_volumes(group_id)
    ntris = mesh.ntris
    
    in_list = []
    
    for t in range(ntris):
        verts = mesh.getTri(t)
        with_in = True
        cords = []
        for v in verts:
            c = mesh.getVertex(v)
            c = [x / scale for x in c]
            cords.append(c)
        for vol_id in volume_ids:
            volume = cubit.volume(vol_id)
            body = volume.bodies()[0]
            within = True
            for cord in cords:
                status = body.point_containment(cord)
                if status == 0:
                    within = False
                    break
            if within:
                in_list.append(t)
                break
    cubit.delete_group(group_id)
    return in_list

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def boundTetsAsComp(mesh, comp_name, scale = 1e-6):
    """
    Create a compartment in the STEPS Tetmesh object using tetraedrons bound in selected volumes.
        
    Parameters:
        mesh        Associated Tetmesh object created in STEPS
        comp_name   Name of the compartment
        scale       LENGTH scale from the CUBIT gemoetry to real geometry.
                    e.g. a radius of 10 in CUBIT to a radius of 1 micron in STEPS, scale is 1e-7.
                    By default the scale is 1e-6, i.e. 1 unit in CUBIT equals 1 micron in STEPS.
        
    Return:
        steps.geom.TmComp object
    """
    
    steps_ids = getTetsBoundInSelectedVols(mesh, scale)
    comp = sgeom.TmComp(comp_name, mesh, steps_ids)
    return comp

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def boundTrisAsPatch(mesh, patch_name, icomp = None, ocomp = None, scale = 1e-6):
    """
    Create a patch in the STEPS Tetmesh object using triangles bound in selected volumes.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * comp_name   Name of the compartment
        * scale       LENGTH scale from the CUBIT gemoetry to real geometry.
                    e.g. a radius of 10 in CUBIT to a radius of 1 micron in STEPS, scale is 1e-7.
                    By default the scale is 1e-6, i.e. 1 unit in CUBIT equals 1 micron in STEPS.
        
    Return:
        steps.geom.TmComp object
    """
    
    steps_ids = getTrisBoundInSelectedVols(mesh, scale)
    patch = sgeom.TmPatch(patch_name, mesh, steps_ids, icomp, ocomp)
    return patch

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def boundNodesAsROI(mesh, roi_name, scale = 1e-6):
    """
    Store nodes bound in selected volumes as Region of Interest in the STEPS Tetmesh object.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * roi_name    Name of the ROI
        * scale       LENGTH scale from the CUBIT gemoetry to real geometry.
                    e.g. a radius of 10 in CUBIT to a radius of 1 micron in STEPS, scale is 1e-7.
                    By default the scale is 1e-6, i.e. 1 unit in CUBIT equals 1 micron in STEPS.
        
    Return:
        None
    """
    
    steps_ids = getNodesBoundInSelectedVols(mesh, scale)
    mesh.addROI(roi_name, sgeom.ELEM_VERTEX, steps_ids)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def boundTetsAsROI(mesh, roi_name, scale = 1e-6):
    """
    Store tetraedrons bound in selected volumes as Region of Interest in the STEPS Tetmesh object.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * roi_name    Name of the ROI
        * scale       scale: LENGTH scale from the CUBIT gemoetry to real geometry.
                    e.g. a radius of 10 in CUBIT to a radius of 1 micron in STEPS, scale is 1e-7.
                    By default the scale is 1e-6, i.e. 1 unit in CUBIT equals 1 micron in STEPS.
        
    Return:
        None
    """
    
    steps_ids = getTetsBoundInSelectedVols(mesh, scale)
    mesh.addROI(roi_name, sgeom.ELEM_TET, steps_ids)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def boundTrisAsROI(mesh, roi_name, scale = 1e-6):
    """
    Store triangles bound in selected volumes as Region of Interest in the STEPS Tetmesh object.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * roi_name    Name of the ROI
        * scale       LENGTH scale from the CUBIT gemoetry to real geometry.
                    e.g. a radius of 10 in CUBIT to a radius of 1 micron in STEPS, scale is 1e-7.
                    By default the scale is 1e-6, i.e. 1 unit in CUBIT equals 1 micron in STEPS.
        
    Return:
        None
    """
    
    steps_ids = getTrisBoundInSelectedVols(mesh, scale)
    mesh.addROI(roi_name, sgeom.ELEM_TRI, steps_ids)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
################################################################################

def drawSTEPSNodes(steps_nodes, node_proxy):
    """
    Draw vertex nodes with given STEPS indices.
    
    Parameters:
        * steps_nodes STEPS indices of vertex nodes
        * node_proxy  Node element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    cmd_str = "draw node "
    for st in steps_nodes:
        cubit_id = node_proxy.getImportID(st)
        cmd_str += "%i," % (cubit_id)
    cubit.cmd(cmd_str)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
################################################################################

def drawSTEPSTris(steps_tris, tri_proxy):
    """
    Draw triangles with given STEPS indices.
    
    Parameters:
        * steps_tris  STEPS indices of triangles
        * tri_proxy   Triangle element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    cmd_str = "draw tri "
    for st in steps_tris:
        cubit_id = tri_proxy.getImportID(st)
        cmd_str += "%i," % (cubit_id)
    cubit.cmd(cmd_str)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
################################################################################

def drawSTEPSTets(steps_tets, tet_proxy):
    """
    Draw tetrahedrons with given STEPS indices.
    
    Parameters:
        * steps_tets  STEPS indices of tetrahedrons
        * tet_proxy   Tetrahedron element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    cmd_str = "draw tet "
    for st in steps_tets:
        cubit_id = tet_proxy.getImportID(st)
        cmd_str += "%i," % (cubit_id)
    cubit.cmd(cmd_str)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def drawROINodes(mesh, node_proxy, roi_name):
    """
    Draw node elements stored in ROI database with id roi_name.
    
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * node_proxy  Node element proxy generated by STEPS mesh importing function
        * roi_name    Name of the ROI
        
    Return:
        None
    """
    data = mesh.getROIData(roi_name)
    drawSTEPSNodes(data, node_proxy)


################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def drawROITets(mesh, tet_proxy, roi_name):
    """
    Draw tetrahedron elements stored in ROI database with id roi_name.
    
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * tet_proxy   Tetrahedron element proxy generated by STEPS mesh importing function
        * roi_name    Name of the ROI
        
    Return:
        None
    """
    data = mesh.getROIData(roi_name)
    drawSTEPSTets(data, tet_proxy)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def drawROITris(mesh, tri_proxy, roi_name):
    """
    Draw triangle elements stored in ROI database with id roi_name.
    
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * tri_proxy   Triangle element proxy generated by STEPS mesh importing function
        * roi_name    Name of the ROI
        
    Return:
        None
    """
    data = mesh.getROIData(roi_name)
    drawSTEPSTris(data, tri_proxy)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def drawComp(comp, tet_proxy):
    """
    Draw the STEPS compartment.
    
    Parameters:
        * comp        TmComp object
        * tet_proxy   Tetrahedron element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    data = comp.getAllTetIndices()
    drawSTEPSTets(data, tet_proxy)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def drawPatch(patch, tri_proxy):
    """
    Draw the STEPS compartment.
        
    Parameters:
        * patch       TmPatch object
        * tri_proxy   Triangle element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    data = patch.getAllTriIndices()
    drawSTEPSTris(data, tri_proxy)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def drawComp(mesh, comp_id, tet_proxy):
    """
    Draw the STEPS compartment.
    
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * comp_id     Name of the compartment
        * tet_proxy   Tetrahedron element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    comp = sgeom.castToTmComp(mesh.getComp(comp_id))
    data = comp.getAllTetIndices()
    drawSTEPSTets(data, tet_proxy)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def drawPatch(mesh, patch_id, tri_proxy):
    """
    Draw the STEPS compartment.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * patch_id    Name of the patch
        * tri_proxy   Triangle element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    patch = sgeom.castToTmPatch(mesh.getPatch(patch_id))
    data = patch.getAllTriIndices()
    drawSTEPSTris(data, tri_proxy)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
################################################################################

def highlightSTEPSNodes(steps_nodes, node_proxy):
    """
    Highlight vertex nodes with given STEPS indices.
    
    Parameters:
        * steps_nodes STEPS indices of vertex nodes
        * node_proxy  Node element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    cmd_str = "highlight node "
    for st in steps_nodes:
        cubit_id = node_proxy.getImportID(st)
        cmd_str += "%i," % (cubit_id)
    cubit.cmd(cmd_str)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
################################################################################

def highlightSTEPSTris(steps_tris, tri_proxy):
    """
    Highlight triangles with given STEPS indices.
    
    Parameters:
        * steps_tris  STEPS indices of triangles
        * tri_proxy   Triangle element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    cmd_str = "highlight tri "
    for st in steps_tris:
        cubit_id = tri_proxy.getImportID(st)
        cmd_str += "%i," % (cubit_id)
    cubit.cmd(cmd_str)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
################################################################################

def highlightSTEPSTets(steps_tets, tet_proxy):
    """
    Highlight tetrahedrons with given STEPS indices.
    
    Parameters:
        * steps_tets  STEPS indices of tetrahedrons
        * tet_proxy   Tetrahedron element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    cmd_str = "highlight tet "
    for st in steps_tets:
        cubit_id = tet_proxy.getImportID(st)
        cmd_str += "%i," % (cubit_id)
    cubit.cmd(cmd_str)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def highlightROINodes(mesh, node_proxy, roi_name):
    """
    Highlight node elements stored in ROI database with id roi_name.
    
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * node_proxy  Node element proxy generated by STEPS mesh importing function
        * roi_name    Name of the ROI
        
    Return:
        None
    """
    data = mesh.getROIData(roi_name)
    highlightSTEPSNodes(data, node_proxy)


################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def highlightROITets(mesh, tet_proxy, roi_name):
    """
    Highlight tetrahedron elements stored in ROI database with id roi_name.
    
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * tet_proxy   Tetrahedron element proxy generated by STEPS mesh importing function
        * roi_name    Name of the ROI
        
    Return:
        None
    """
    data = mesh.getROIData(roi_name)
    highlightSTEPSTets(data, tet_proxy)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def highlightROITris(mesh, tri_proxy, roi_name):
    """
    Highlight triangle elements stored in ROI database with id roi_name.
    
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * tri_proxy   Triangle element proxy generated by STEPS mesh importing function
        * roi_name    Name of the ROI
        
    Return:
        None
    """
    data = mesh.getROIData(roi_name)
    highlightSTEPSTris(data, tri_proxy)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def highlightComp(comp, tet_proxy):
    """
    Highlight the STEPS compartment.
    
    Parameters:
        * comp        TmComp object
        * tet_proxy   Tetrahedron element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    data = comp.getAllTetIndices()
    highlightSTEPSTets(data, tet_proxy)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def highlightPatch(patch, tri_proxy):
    """
    Highlight the STEPS compartment.
        
    Parameters:
        * patch       TmPatch object
        * tri_proxy   Triangle element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    data = patch.getAllTriIndices()
    highlightSTEPSTris(data, tri_proxy)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def highlightComp(mesh, comp_id, tet_proxy):
    """
    Highlight the STEPS compartment.
    
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * comp_id     Name of the compartment
        * tet_proxy   Tetrahedron element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    comp = sgeom.castToTmComp(mesh.getComp(comp_id))
    data = comp.getAllTetIndices()
    highlightSTEPSTets(data, tet_proxy)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################

def highlightPatch(mesh, patch_id, tri_proxy):
    """
    Highlight the STEPS compartment.
        
    Parameters:
        * mesh        Associated Tetmesh object created in STEPS
        * patch_id    Name of the patch
        * tri_proxy   Triangle element proxy generated by STEPS mesh importing function
        
    Return:
        None
    """
    patch = sgeom.castToTmPatch(mesh.getPatch(patch_id))
    data = patch.getAllTriIndices()
    highlightSTEPSTris(data, tri_proxy)

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
################################################################################

def getNodeImportIDs(steps_nodes, node_proxy):
    """
    Return the import indices of nodes giving their STEPS indices.
    
    Parameters:
        * steps_nodes STEPS indices of nodes
        * node_proxy  Node element proxy generated by STEPS mesh importing function
        
    Return:
        Import indices of the elements
    """
    ids = []
    for n in steps_nodes:
        ids.append(node_proxy.getImportID(n))
    return ids

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
################################################################################

def getTetImportIDs(steps_tets, tet_proxy):
    """
    Return the import indices of tetrahedrons giving their STEPS indices.
    
    Parameters:
        * steps_tets  STEPS indices of tetrahedrons
        * tet_proxy   Tetrahedron element proxy generated by STEPS mesh importing function
        
    Return:
        Import indices of the elements
    """
    ids = []
    for t in steps_tets:
        ids.append(tet_proxy.getImportID(t))
    return ids

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
################################################################################

def getTriImportIDs(steps_tris, tri_proxy):
    """
    Return the import indices of triangle giving their STEPS indices.
    
    Parameters:
        * steps_tris  STEPS indices of triangles
        * tri_proxy   Triangle element proxy generated by STEPS mesh importing function
        
    Return:
        Import indices of the elements
    """
    ids = []
    for t in steps_tris:
        ids.append(tri_proxy.getImportID(t))
    return ids

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
################################################################################

def toStr(e_list):
    return_str = ""
    for e in e_list:
        return_str += "%i," % (e)
    return return_str

################################################################################
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
################################################################################
