#!/usr/bin/env python3

import os
import binascii
import glob
from . import bbconstants

_capversion = bbconstants._capversion


def ghetto_convert(intsize):
    """
    Convert from decimal integer to little endian
    hexadecimal string, padded to 16 characters with zeros.
    :param intsize: Integer you wish to convert.
    :type intsize: integer
    """
    hexsize = format(intsize, '08x')  # '00AABBCC'
    newlist = [hexsize[i:i + 2]
               for i in range(0, len(hexsize), 2)]  # ['00', 'AA','BB','CC']
    while "00" in newlist:
        newlist.remove("00")  # extra padding
    newlist.reverse()
    ghetto_hex = "".join(newlist)  # 'CCBBAA'
    ghetto_hex = ghetto_hex.rjust(16, '0')
    return binascii.unhexlify(bytes(ghetto_hex.upper(), 'ascii'))


def make_offset(cap, firstfile, secondfile="", thirdfile="",
                fourthfile="", fifthfile="", sixthfile="", folder=os.getcwd()):
    """
    Create magic offset file for use in autoloader creation.
    Cap.exe MUST match separator version.
    Defined in _capversion.
    :param cap: Location of cap.exe file.
    :type cap: str
    :param firstfile: First signed file. Required.
    :type firstfile: str
    :param secondfile: Second signed file. Optional.
    :type secondfile: str
    :param thirdfile: Third signed file. Optional.
    :type thirdfile: str
    :param fourthfile: Fourth signed file. Optional.
    :type fourthfile: str
    :param fifthfile: Fifth signed file. Optional.
    :type fifthfile: str
    :param sixthfile: Sixth signed file. Optional.
    :type sixthfile: str
    :param folder: Working folder. Optional.
    :type folder: str
    """
    filecount = 0
    filelist = [
        firstfile,
        secondfile,
        thirdfile,
        fourthfile,
        fifthfile,
        sixthfile]
    for i in filelist:
        if i:
            filecount += 1
    # immutable things
    separator = binascii.unhexlify(
        """6ADF5D144E4B4D4E474F46464D4E532B170A0D1E0C14532B372A2D3E2C34522F3C534F514\
F514F514F534E464D514E4947514E51474F70709CD5C5979CD5C5979CD5C597""")
    password = binascii.unhexlify("0" * 160)
    singlepad = binascii.unhexlify("0" * 2)
    doublepad = binascii.unhexlify("0" * 4)
    signedpad = binascii.unhexlify("0" * 16)
    filepad = binascii.unhexlify(
        bytes(
            str(filecount).rjust(
                2,
                '0'),
            'ascii'))  # between 01 and 06
    trailermax = int(7 - int(filecount))
    trailermax = trailermax * 2
    trailer = "0" * trailermax  # 00 repeated between 1 and 6 times
    trailers = binascii.unhexlify(trailer)

    capfile = str(glob.glob(cap)[0])
    capsize = os.path.getsize(capfile)  # size of cap.exe, in bytes

    first = str(glob.glob(firstfile)[0])
    firstsize = os.path.getsize(first)  # required
    if (filecount >= 2):
        second = str(glob.glob(secondfile)[0])
        secondsize = os.path.getsize(second)
    if (filecount >= 3):
        third = str(glob.glob(thirdfile)[0])
        thirdsize = os.path.getsize(third)
    if (filecount >= 4):
        fourth = str(glob.glob(fourthfile)[0])
        fourthsize = os.path.getsize(fourth)
    if (filecount >= 5):
        fifth = str(glob.glob(fifthfile)[0])
        fifthsize = os.path.getsize(fifth)

    # start of first file; length of cap + length of offset
    firstoffset = len(separator) + len(password) + 64 + capsize
    firststart = ghetto_convert(firstoffset)
    if (filecount >= 2):
        secondoffset = firstoffset + firstsize  # start of second file
        secondstart = ghetto_convert(secondoffset)
    if (filecount >= 3):
        thirdoffset = secondstart + secondsize  # start of third file
        thirdstart = ghetto_convert(thirdoffset)
    if (filecount >= 4):
        fourthoffset = thirdoffset + thirdsize  # start of fourth file
        fourthstart = ghetto_convert(fourthoffset)
    if (filecount >= 5):
        fifthoffset = fourthstart + fourthsize  # start of fifth file
        fifthstart = ghetto_convert(fifthoffset)
    if (filecount == 6):
        sixthoffset = fifthoffset + fifthsize  # start of sixth file
        sixthstart = ghetto_convert(sixthoffset)

    with open(os.path.join(folder, "offset.hex"), "w+b") as file:
        file.write(separator)
        file.write(password)
        file.write(filepad)
        file.write(doublepad)
        file.write(firststart)
        file.write(singlepad)
        if (filecount >= 2):
            file.write(secondstart)
        else:
            file.write(signedpad)
        file.write(singlepad)
        if (filecount >= 3):
            file.write(thirdstart)
        else:
            file.write(signedpad)
        file.write(singlepad)
        if (filecount >= 4):
            file.write(fourthstart)
        else:
            file.write(signedpad)
        file.write(singlepad)
        if (filecount >= 5):
            file.write(fifthstart)
        else:
            file.write(signedpad)
        file.write(singlepad)
        if (filecount == 6):
            file.write(sixthstart)
        else:
            file.write(signedpad)
        file.write(singlepad)
        file.write(doublepad)
        file.write(trailers)


def make_autoloader(filename, cap, firstfile, secondfile="", thirdfile="",
                    fourthfile="", fifthfile="", sixthfile="",
                    folder=os.getcwd()):
    """
    Python implementation of cap.exe.
    Writes cap.exe, magic offset, signed files to a .exe file.
    Uses output of make_offset().
    :param filename: Name of autoloader.
    :type filename: str
    :param cap: Location of cap.exe file.
    :type cap: str
    :param firstfile: First signed file. Required.
    :type firstfile: str
    :param secondfile: Second signed file. Optional.
    :type secondfile: str
    :param thirdfile: Third signed file. Optional.
    :type thirdfile: str
    :param fourthfile: Fourth signed file. Optional.
    :type fourthfile: str
    :param fifthfile: Fifth signed file. Optional.
    :type fifthfile: str
    :param sixthfile: Sixth signed file. Optional.
    :type sixthfile: str
    :param folder: Working folder. Optional.
    :type folder: str
    """
    make_offset(
        cap,
        firstfile,
        secondfile,
        thirdfile,
        fourthfile,
        fifthfile,
        sixthfile,
        folder)

    filecount = 0
    filelist = [
        firstfile,
        secondfile,
        thirdfile,
        fourthfile,
        fifthfile,
        sixthfile]
    for i in filelist:
        if i:
            filecount += 1
    try:
        with open(os.path.join(os.path.abspath(folder),
                               filename), "wb") as autoloader:
            try:
                with open(os.path.normpath(cap), "rb") as capfile:
                    print("WRITING CAP VERSION", _capversion + "...")
                    while True:
                        chunk = capfile.read(4096)  # 4k chunks
                        if not chunk:
                            break
                        autoloader.write(chunk)
            except IOError as e:
                print("Operation failed:", e.strerror)
            try:
                with open(os.path.join(folder, "offset.hex"), "rb") as offset:
                    print("WRITING MAGIC OFFSET...")
                    autoloader.write(offset.read())
            except IOError as e:
                print("Operation failed:", e.strerror)
            try:
                with open(firstfile, "rb") as first:
                    print(
                        "WRITING SIGNED FILE #1...\n",
                        os.path.basename(firstfile))
                    while True:
                        chunk = first.read(4096)  # 4k chunks
                        if not chunk:
                            break
                        autoloader.write(chunk)
            except IOError as e:
                print("Operation failed:", e.strerror)
            if (filecount >= 2):
                try:
                    print(
                        "WRITING SIGNED FILE #2...\n",
                        os.path.basename(secondfile))
                    with open(secondfile, "rb") as second:
                        while True:
                            chunk = second.read(4096)  # 4k chunks
                            if not chunk:
                                break
                            autoloader.write(chunk)
                except IOError as e:
                    print("Operation failed:", e.strerror)
            if (filecount >= 3):
                try:
                    print(
                        "WRITING SIGNED FILE #3...\n",
                        os.path.basename(thirdfile))
                    with open(thirdfile, "rb") as third:
                        while True:
                            chunk = third.read(4096)  # 4k chunks
                            if not chunk:
                                break
                            autoloader.write(chunk)
                except IOError as e:
                    print("Operation failed:", e.strerror)
            if (filecount >= 4):
                try:
                    print(
                        "WRITING SIGNED FILE #5...\n",
                        os.path.basename(fourthfile))
                    with open(fourthfile, "rb") as fourth:
                        while True:
                            chunk = fourth.read(4096)  # 4k chunks
                            if not chunk:
                                break
                            autoloader.write(chunk)
                except IOError as e:
                    print("Operation failed:", e.strerror)
            if (filecount >= 5):
                try:
                    print(
                        "WRITING SIGNED FILE #5...\n",
                        os.path.basename(fifthfile))
                    with open(fifthfile, "rb") as fifth:
                        while True:
                            chunk = fifth.read(4096)  # 4k chunks
                            if not chunk:
                                break
                            autoloader.write(chunk)
                except IOError as e:
                    print("Operation failed:", e.strerror)
            if (filecount == 6):
                try:
                    print(
                        "WRITING SIGNED FILE #6...\n",
                        os.path.basename(sixthfile))
                    with open(sixthfile, "rb") as sixth:
                        while True:
                            chunk = sixth.read(4096)  # 4k chunks
                            if not chunk:
                                break
                            autoloader.write(chunk)
                except IOError as e:
                    print("Operation failed:", e.strerror)
    except IOError as e:
        print("Operation failed:", e.strerror)

    print(filename, "FINISHED!\n")
    os.remove(os.path.join(folder, "offset.hex"))


def generate_loaders(
        osversion, radioversion, radios=True,
        cap="cap.exe", localdir=os.getcwd()):
    """
    Create and properly label autoloaders.
    Leverages Python implementation of cap.exe.
    :param osversion: OS version, 10.x.y.zzzz.
    Autoconverted to 10.x.0y.zzzz if need be.
    :type osversion: str
    :param radioversion: Radio version, 10.x.y.zzzz.
    Autoconverted to 10.x.0y.zzzz if need be.
    :type radioversion: str
    :param radios: Whether to make radios or not. True by default.
    :type radios: bool
    :param cap: Path to cap.exe. Default is local dir\\cap.exe.
    :type cap: str
    :param localdir: Working path. Default is local dir.
    :type localdir: str
    """
    # #OS Images
    # 8960
    try:
        os_8960 = glob.glob(
            os.path.join(
                localdir,
                "*qc8960*_sfi.desktop*.signed"))[0]
    except IndexError:
        print("No 8960 image found")

    # 8x30 (10.3.1 MR+)
    try:
        os_8x30 = glob.glob(
            os.path.join(
                localdir,
                "*qc8x30*desktop*.signed"))[0]
    except IndexError:
        print("No 8x30 image found")

    # 8974
    try:
        os_8974 = glob.glob(
            os.path.join(
                localdir,
                "*qc8974*desktop*.signed"))[0]
    except IndexError:
        print("No 8974 image found")

    # OMAP (incl. 10.3.1)
    try:
        os_ti = glob.glob(os.path.join(localdir, "*winchester*.signed"))[0]
    except IndexError:
        print("No OMAP image found")

    # Radio files
    # STL100-1
    try:
        radio_z10_ti = glob.glob(
            os.path.join(
                localdir,
                "*radio.m5730*.signed"))[0]
    except IndexError:
        print("No OMAP radio found")

    # STL100-X
    try:
        radio_z10_qcm = glob.glob(
            os.path.join(
                localdir,
                "*radio.qc8960.BB*.signed"))[0]
    except IndexError:
        print("No 8960 radio found")

    # STL100-4
    try:
        radio_z10_vzw = glob.glob(
            os.path.join(
                localdir,
                "*radio.qc8960*omadm*.signed"))[0]
    except IndexError:
        print("No Verizon 8960 radio found")

    # Q10/Q5
    try:
        radio_q10 = glob.glob(os.path.join(localdir, "*8960*wtr.*.signed"))[0]
    except IndexError:
        print("No Q10/Q5 radio found")

    # Z30/Classic
    try:
        radio_z30 = glob.glob(os.path.join(localdir, "*8960*wtr5*.signed"))[0]
    except IndexError:
        print("No Z30/Classic radio found")

    # Z3
    try:
        radio_z3 = glob.glob(os.path.join(localdir, "*8930*wtr5*.signed"))[0]
    except IndexError:
        print("No Z3 radio found")

    # Passport
    try:
        radio_8974 = glob.glob(os.path.join(localdir, "*8974*wtr2*.signed"))[0]
    except IndexError:
        print("No Passport radio found")

    # Pretty format names
    splitos = osversion.split(".")
    if len(splitos[2]) == 1:
        splitos[2] = "0" + splitos[2]
    osversion = ".".join(splitos)
    splitrad = radioversion.split(".")
    if len(splitrad[2]) == 1:
        splitrad[2] = "0" + splitrad[2]
    radioversion = ".".join(splitrad)

    # Generate loaders
    # STL100-1
    try:
        print("Creating OMAP Z10 OS...")
        make_autoloader(  # @UndefinedVariable, since PyDev is dumb
            filename="Z10_" +
            osversion +
            "_STL100-1.exe",
            cap=cap,
            firstfile=os_ti,
            secondfile=radio_z10_ti,
            folder=localdir)
    except Exception:
        print("Could not create STL100-1 OS/radio loader")
    if radios:
        print("Creating OMAP Z10 radio...")
        try:
            make_autoloader(
                "Z10_" +
                radioversion +
                "_STL100-1.exe",
                cap=cap,
                firstfile=radio_z10_ti,
                folder=localdir)
        except Exception:
            print("Could not create STL100-1 radio loader")

    # STL100-X
    try:
        print("Creating Qualcomm Z10 OS...")
        make_autoloader(
            "Z10_" +
            osversion +
            "_STL100-2-3.exe",
            cap=cap,
            firstfile=os_8960,
            secondfile=radio_z10_qcm,
            folder=localdir)
    except Exception:
        print("Could not create Qualcomm Z10 OS/radio loader")
    if radios:
        print("Creating Qualcomm Z10 radio...")
        try:
            make_autoloader(
                "Z10_" +
                radioversion +
                "_STL100-2-3.exe",
                cap=cap,
                firstfile=radio_z10_qcm,
                folder=localdir)
        except Exception:
            print("Could not create Qualcomm Z10 radio loader")

    # STL100-4
    try:
        print("Creating Verizon Z10 OS...")
        make_autoloader(
            "Z10_" +
            osversion +
            "_STL100-4.exe",
            cap=cap,
            firstfile=os_8960,
            secondfile=radio_z10_vzw,
            folder=localdir)
    except Exception:
        print("Could not create Verizon Z10 OS/radio loader")
    if radios:
        print("Creating Verizon Z10 radio...")
        try:
            make_autoloader(
                "Z10_" +
                radioversion +
                "_STL100-4.exe",
                cap=cap,
                firstfile=radio_z10_vzw,
                folder=localdir)
        except Exception:
            print("Could not create Verizon Z10 radio loader")

    # Q10/Q5
    try:
        print("Creating Q10/Q5 OS...")
        make_autoloader(
            "Q10_" +
            osversion +
            "_SQN100-1-2-3-4-5.exe",
            cap=cap,
            firstfile=os_8960,
            secondfile=radio_q10,
            folder=localdir)
    except Exception:
        print("Could not create Q10/Q5 OS/radio loader")
    if radios:
        print("Creating Q10/Q5 radio...")
        try:
            make_autoloader(
                "Q10_" +
                radioversion +
                "_SQN100-1-2-3-4-5.exe",
                cap=cap,
                firstfile=radio_q10,
                folder=localdir)
        except Exception:
            print("Could not create Q10/Q5 radio loader")

    # Z30/Classic
    try:
        print("Creating Z30/Classic OS...")
        make_autoloader(
            "Z30_" +
            osversion +
            "_STA100-1-2-3-4-5-6.exe",
            cap=cap,
            firstfile=os_8960,
            secondfile=radio_z30,
            folder=localdir)
    except Exception:
        print("Could not create Z30/Classic OS/radio loader")
    if radios:
        print("Creating Z30/Classic radio...")
        try:
            make_autoloader(
                "Z30_" +
                radioversion +
                "_STA100-1-2-3-4-5-6.exe",
                cap=cap,
                firstfile=radio_z30,
                folder=localdir)
        except Exception:
            print("Could not create Z30/Classic radio loader")

    # Z3
    try:
        print("Creating Z3 OS...")
        make_autoloader(
            "Z3_" +
            osversion +
            "_STJ100-1-2.exe",
            cap=cap,
            firstfile=os_8x30,
            secondfile=radio_z3,
            folder=localdir)
    except Exception:
        print("Could not create Z3 OS/radio loader (8x30)")
    if radios:
        print("Creating Z3 radio...")
        try:
            make_autoloader(
                "Z3_" +
                radioversion +
                "_STJ100-1-2.exe",
                cap=cap,
                firstfile=radio_z3,
                folder=localdir)
        except Exception:
            print("Could not create Z3 radio loader")

    # Passport
    try:
        print("Creating Passport OS...")
        make_autoloader(
            "Passport_" +
            osversion +
            "_SQW100-1-2-3.exe",
            cap=cap,
            firstfile=os_8974,
            secondfile=radio_8974,
            folder=localdir)
    except Exception:
        print("Could not create Passport OS/radio loader")
    if radios:
        print("Creating Passport radio...")
        try:
            make_autoloader(
                "Passport_" +
                radioversion +
                "_SQW100-1-2-3.exe",
                cap=cap,
                firstfile=radio_8974,
                folder=localdir)
        except Exception:
            print("Could not create Passport radio loader")


def generate_lazy_loader(
        osversion, radioversion, device,
        cap="cap.exe", localdir=os.getcwd()):
    print("\nCREATING LOADER...")
    if device == 0:
        try:
            os_ti = str(glob.glob("*winchester*.signed")[0])
        except IndexError:
            print("No OMAP image found")
            return
        try:
            radio_z10_ti = str(glob.glob("*radio.m5730*.signed")[0])
        except IndexError:
            print("No OMAP radio found")
            return
        else:
            print("Creating OMAP Z10 OS...")
            try:
                make_autoloader(
                    filename="Z10_" +
                    osversion +
                    "_STL100-1.exe",
                    cap=cap,
                    firstfile=os_ti,
                    secondfile=radio_z10_ti,
                    thirdfile="",
                    fourthfile="",
                    fifthfile="",
                    sixthfile="",
                    folder=localdir)
            except Exception:
                print("Could not create STL100-1 OS/radio loader")
                return
    elif device == 1:
        try:
            os_8960 = str(glob.glob("*qc8960*_sfi.desktop*.signed")[0])
        except IndexError:
            print("No 8960 image found")
            return
        try:
            radio_z10_qcm = str(glob.glob("*radio.qc8960.BB*.signed")[0])
        except IndexError:
            print("No 8960 radio found")
            return
        else:
            print("Creating Qualcomm Z10 OS...")
            try:
                make_autoloader(
                    "Z10_" +
                    osversion +
                    "_STL100-2-3.exe",
                    cap,
                    os_8960,
                    radio_z10_qcm,
                    folder=localdir)
            except Exception:
                print("Could not create Qualcomm Z10 OS/radio loader")
                return
    elif device == 2:
        try:
            os_8960 = str(glob.glob("*qc8960*_sfi.desktop*.signed")[0])
        except IndexError:
            print("No 8960 image found")
            return
        try:
            radio_z10_vzw = str(glob.glob("*radio.qc8960*omadm*.signed")[0])
        except IndexError:
            print("No Verizon 8960 radio found")
            return
        else:
            print("Creating Verizon Z10 OS...")
            try:
                make_autoloader(
                    "Z10_" +
                    osversion +
                    "_STL100-4.exe",
                    cap,
                    os_8960,
                    radio_z10_vzw,
                    folder=localdir)
            except Exception:
                print("Could not create Verizon Z10 OS/radio loader")
                return
    elif device == 3:
        try:
            os_8960 = str(glob.glob("*qc8960*_sfi.desktop*.signed")[0])
        except IndexError:
            print("No 8960 image found")
            return
        try:
            radio_q10 = str(glob.glob("*8960*wtr.*.signed")[0])
        except IndexError:
            print("No Q10/Q5 radio found")
            return
        else:
            print("Creating Q10/Q5 OS...")
            try:
                make_autoloader(
                    "Q10_" +
                    osversion +
                    "_SQN100-1-2-3-4-5.exe",
                    cap,
                    os_8960,
                    radio_q10,
                    folder=localdir)
            except Exception:
                print("Could not create Q10/Q5 OS/radio loader")
                return
    elif device == 4:
        try:
            os_8960 = str(glob.glob("*qc8960*_sfi.desktop*.signed")[0])
        except IndexError:
            print("No 8960 image found")
            return
        try:
            radio_z30 = str(glob.glob("*8960*wtr5*.signed")[0])
        except IndexError:
            print("No Z30/Classic radio found")
            return
        else:
            print("Creating Z30/Classic OS...")
            try:
                make_autoloader(
                    "Z30_" +
                    osversion +
                    "_STA100-1-2-3-4-5-6.exe",
                    cap,
                    os_8960,
                    radio_z30,
                    folder=localdir)
            except Exception:
                print("Could not create Z30/Classic OS/radio loader")
                return
    elif device == 5:
        try:
            os_8x30 = str(glob.glob("*qc8x30*desktop*.signed")[0])
        except IndexError:
            print("No 8x30 image found")
            return
        try:
            radio_z3 = str(glob.glob("*8930*wtr5*.signed")[0])
        except IndexError:
            print("No Z3 radio found")
            return
        else:
            print("Creating Z3 OS...")
            try:
                make_autoloader(
                    "Z3_" +
                    osversion +
                    "_STJ100-1-2.exe",
                    cap,
                    os_8x30,
                    radio_z3,
                    folder=localdir)
            except Exception:
                print("Could not create Z3 OS/radio loader")
                return
    elif device == 6:
        try:
            os_8974 = str(glob.glob("*qc8974*desktop*.signed")[0])
        except IndexError:
            print("No 8974 image found")
            return
        try:
            radio_8974 = str(glob.glob("*8974*wtr2*.signed")[0])
        except IndexError:
            print("No Passport radio found")
            return
        else:
            print("Creating Passport OS...")
            try:
                make_autoloader(
                    "Passport_" +
                    osversion +
                    "_SQW100-1-2-3.exe",
                    cap,
                    os_8974,
                    radio_8974,
                    folder=localdir)
            except Exception:
                print("Could not create Passport OS/radio loader")
                return
    else:
        return
