from __future__ import absolute_import, division, print_function, with_statement, unicode_literals
import argparse
import sys
try:
    from . import chronqc_db
    from . import chronqc_plot
    from . import chronqc_annotation
    from . import chronqc_crongen
except (ValueError, SystemError, ImportError):
    import chronqc_db
    import chronqc_plot
    import chronqc_annotation
    import chronqc_crongen

def run_plot(args):
    """
    (args) -> plot
    runs main function from chronqc_plot.py
    """
    # print("running chronqc_plot")
    chronqc_plot.main(args)


def run_db(args):
    """
    (args) -> plot
    runs main function from chronqc_db.py
    """
    # print("running chronqc_db")
    chronqc_db.main(args)


def run_annotation(args):
    """
    runs main function from chronqc_annotation.py
    """
    #print("running chronqc_annotation")
    chronqc_annotation.main(args)


def run_chrongen(args):
    """
    runs main function from chronqc_crongen.py
    """
    #print("running chronqc_chrongen")
    chronqc_crongen.main(args)

class Highlander(argparse.Action):
    def __call__(self, parser, namespace, values, option_string=None):
        if getattr(namespace, self.dest, None) is not None:
            raise argparse.ArgumentError(self, 'This can be specified only once.')
        setattr(namespace, self.dest, values)
        
# @click.command()
def main():
     try:
        parser = argparse.ArgumentParser(prog='chronqc')
        subparsers = parser.add_subparsers()  # , annotation')
        # Parse database arguments
        db_parser = subparsers.add_parser('database',
                                          help='Generate ChronQC \
                                          database for ChronQC plots. \
                                          Type "chronqc database -h" \
                                          for help on generating/updating ChronQC \
                                          database.')
        db_requiredNamed = db_parser.add_argument_group('available arguments')
        group = db_requiredNamed.add_mutually_exclusive_group()
        # group.set_defaults(mode='create')
        db_requiredNamed.add_argument('--db',
                                      help='Name of ChronQC database to be created \
                                      (default name is chronqc.stats.sqlite) or \
                                      full path of ChronQC database to be updated.')
        group.add_argument('-u', '--update', action='store_const', dest='mode',
                           const='update', help="Update the ChronQC database \
                           provided by {--db} argument.")
        group.add_argument('-c', '--create', action='store_const', dest='mode',
                           const='create', help="Create the ChronQC database \
                           provided by {--db} argument.")
        db_requiredNamed.add_argument('--db-table', required=False, action=Highlander,
                               help='Name of ChronQC SQLlite database table \
                               to be created/updated. Default name is \
                               chronqc_stats_data. ')
        action = db_requiredNamed.add_mutually_exclusive_group(required=False)
        action.add_argument('--run-date-info', action=Highlander,
                            help='Run ID and sequencing date information for \
                            all samples in csv format. File should have three \
                            columns (Sample, Run, Date).If run-date-info is \
                            provided {multiqc-sources, module, run-dir-level} \
                            arguments are not required.')
        action.add_argument('--multiqc-sources', action=Highlander,
                            help='Input multiqc_sources.txt file generated by \
                            MultiQC. If run-date-info is not provided. \
                            Run ID and Sequencing date information is \
                            generated  using this file along with \
                            {module, run-dir-level} information.')
        db_requiredNamed.add_argument('--module', required=False,
                               default='FastQC', action=Highlander,
                               help='Default module is FastQC. \
                               Name of module from multiqc_sources.txt \
                               file. Time stamp from the output files of this \
                               module will be used to get the Sequencing \
                               date. If output files are not found then time \
                               stamp of multiqc-sources file is used instead. \
                               Path of output files of this module \
                               and run-dir-level (default = 3) is used to \
                               get Run ID.')
        db_requiredNamed.add_argument('--run-dir-level',
                               required=False, default=3, action=Highlander,
                               help='Upper directory level for path in \
                               multiqc_sources.txt file. This will be \
                               used generate Run ID. Default is 3.')
        db_parser.add_argument('-p', '--prefix', required=False, action=Highlander,
                               help='Prefix for ChronQC stats database.')
        db_parser.add_argument('-o', '--output', required=False, action=Highlander,
                               help='Directory path to save ChronQC \
                               stats database.')
        db_parser.add_argument('-f', '--force', required=False,
                               action='store_true', help='Overwrite the \
                               output in chronqc_db directory. ')
        db_requiredNamed.add_argument('mstats',  # required=True
                               action=Highlander,
                               help='Input multiqc_general_stats.txt file \
                               generated by MultiQC')
        db_requiredNamed.add_argument('panel',  action=Highlander,   #required=True,
                               help='Name of the panel (assay or project). \
                               This allows storing statistical data for \
                               multiple assays in same ChronQC database \
                               and comparison of data with historical \
                               data in the same assay. ')
        db_parser.set_defaults(func=run_db)
        # Parse plot arguments
        plot_parser = subparsers.add_parser('plot',
                                            help='Generate ChronQC plots. \
                                            Type "chronqc plot -h" for \
                                            help on generating ChronQC plots.')
        plot_parser.add_argument('-p', '--prefix', required=False, action=Highlander,
                                 help='Prefix for ChronQC output html file.')
        plot_parser.add_argument('-o', '--output',  required=False, action=Highlander,
                                 help='Directory path to save ChronQC output')
        plot_parser.add_argument('-f', '--force', required=False, 
                                 action='store_true', help='Overwrite the \
                                 output in chronqc_output directory.')
        plot_parser.add_argument('db',  action=Highlander,   #required=True,
                                 help='Input SQLite database name.')
        plot_parser.add_argument('panel', action=Highlander, # required=True,
                                 help='Name of the panel (assay). This enables \
                                 comparison of historical data \
                                 in the same assay.')
        plot_parser.add_argument('json', action=Highlander, #required=True,
                                 help='Input JSON filename with plot \
                                 properties.')
        plot_parser.set_defaults(func=run_plot)

        annotate_parser = subparsers.add_parser('annotation', help='Start connectivity for annotating plots. ' \
                                                'Type "chronqc annotation -h" \
                                                for help on starting ChronQC annotation server.')
        annotate_parser.add_argument('--port', required = False, action=Highlander, \
                        help='Four digit port number e.g. 8000. Default is 8000.')
        annotate_parser.set_defaults(func=run_annotation)
        # automation
        auto_parser = subparsers.add_parser('chrongen',
                                            help='Use this option for automating ChronQC plot generation. \
                                            Type "chronqc chrongen -h" for details on required arguments.')
        auto_parser.add_argument('config_file', help='Path of configuration file for chrongen.', type=str)
        auto_parser.set_defaults(func=run_chrongen)

        args = parser.parse_args()
        args.func(args)
     except AttributeError:
        parser.print_help()
        sys.exit(0)

if __name__ == '__main__':
    main()
