# ellipsesk.py
# KB 5/21/07

USEGL = False
import numpy as num
import time
from warnings import warn
import wx
from wx import xrc
from params import params
import imagesk
import matchidentities as m_id
import wx

#import pylab

if USEGL:
    import motmot.wxglvideo.simple_overlay as wxvideo # part of Motmot
else:
    import motmot.wxvideo.wxvideo as wxvideo

# scipy connected components labeling code
import scipy.ndimage as meas

from version import DEBUG

import pkg_resources # part of setuptools
ZOOM_RSRC_FILE = pkg_resources.resource_filename( __name__, "ellipses_zoom.xrc" )
SETTINGS_RSRC_FILE = pkg_resources.resource_filename( __name__, "ellipses_settings.xrc" )

class Point:
    def __init__( self, x, y ):
        self.x = x
        self.y = y

    def __eq__( self, other ):
        if other == params.empty_val:
            if self.x == params.empty_val and self.y == params.empty_val:
                return True
            else: return False
        elif type(other) != type(self):
            raise TypeError( "must compare points to points" )
        elif self.x == other.x and self.y == other.y: return True
        return False

    def __ne__( self, other ): return not self.__eq__( other )

    def __print__( self ): return "(%.1f, %.1f)"%(self.x, self.y)
    def __repr__( self ): return self.__print__()
    def __str__( self ): return self.__print__()

class Size:
    def __init__( self, width, height ):
        self.width = width
        self.height = height

    def __eq__( self, other ):
        if other == params.empty_val:
            if self.width == params.empty_val and self.height == params.empty_val:
                return True
            else: return False
        elif type(other) != type(self):
            raise TypeError( "must compare sizes to sizes" )
        elif self.width == other.width and self.height == other.height:
            return True
        return False

    def __ne__( self, other ): return not self.__eq__( other )

    def __print__( self ): return "(%.1f, %.1f)"%(self.width, self.height)
    def __repr__( self ): return self.__print__()
    def __str__( self ): return self.__print__()
    
class Ellipse:
    def __init__( self, centerX=params.empty_val, centerY=params.empty_val,
                  sizeW=params.empty_val, sizeH=params.empty_val,
                  angle=params.empty_val, area=params.empty_val, identity=-1 ):
        self.center = Point( centerX, centerY )
        self.size = Size( sizeW, sizeH )
        self.angle = angle
        self.area = area
        self.identity = identity

    def make_dummy( self ):
        """A dummy ellipse has its area (only) set to params.dummy_val ."""
        self.area = params.dummy_val

    def __eq__( self, other ):
        if other == params.empty_val:
            if not self.isDummy() and \
               self.center == params.empty_val and self.size == params.empty_val:
                return True
            else: return False
        elif other == params.dummy_val:
            if self.area == params.dummy_val:
                return True
            else: return False
        elif type(other) != type(self):
            raise TypeError( "must compare ellipses to ellipses" )
        elif self.center == other.center and self.size == other.size \
             and num.mod(self.angle-other.angle,TWOPI) == 0 \
             and self.identity == other.identity:
            return True
        return False

    def __ne__( self, other ): return not self.__eq__( other )

    def isEmpty( self ):
        return self.__eq__( params.empty_val )

    def isDummy( self ):
        return self.__eq__( params.dummy_val )

    def __nonzero__( self ):
        return not self.isEmpty()

    def __setattr__( self, name, value ):
        if name == 'major':
            self.size.height = value
        elif name == 'minor':
            self.size.width = value
        elif name == 'x':
            self.center.x = value
        elif name == 'y':
            self.center.y = value
        else:
            self.__dict__[name] = value

    def __getattr__( self, name ):
        if name == "width": return self.size.width
        elif name == "minor": return self.size.width
        elif name == "height": return self.size.height
        elif name == "major": return self.size.height
        elif name == "x": return self.center.x
        elif name == "y": return self.center.y
        elif name == "identity": return self.identity
        raise AttributeError( "Ellipse has no attribute %s"%name )

    def __print__( self, verbose=False ):
        if self.isEmpty():
            s = "[]"
        else:
            s = "[id=%d: center="%self.identity + self.center.__print__()
            s += ", axis lengths=" + self.size.__print__()
            s += ", angle=%.3f, area=%.1f]"%(self.angle,self.area)
        return s

    def __str__( self ): return self.__print__( False )
    def __repr__( self ): return self.__print__( True )

    def copy( self ):
        other = Ellipse( self.center.x, self.center.y,
                     self.size.width, self.size.height,
                     self.angle, self.area, self.identity )
        return other

    def Euc_dist( self, other ):
        """Euclidean distance between two ellipse centers."""
        return float((self.center.x - other.center.x)**2 + (self.center.y - other.center.y)**2)
    
    def dist( self, other ):
        """Calculates distance between ellipses, using some metric."""

        # compute angle distance, mod pi
        ang_dist = (( (self.angle-other.angle+num.pi/2.)%num.pi )-num.pi/2.)**2

        # compute euclidean distance between centers
        center_dist = self.Euc_dist(other)

        return (num.sqrt(center_dist + params.ang_dist_wt*ang_dist))

    def compute_area(self):
        self.area = self.size.width*self.size.height*num.pi*4.

class TargetList:
# container for targets
    def __init__( self ):
        self.list = {}
        #if params.use_colorblind_palette:
        #    self.colors = params.colorblind_palette
        #else:
        #    self.colors = params.normal_palette

    def __len__( self ): return len(self.list)

    def __setitem__(self, i, val):
        self.list[i] = val

    def __getitem__( self, i ):
        if self.hasItem(i):
            return self.list[i]
        else:
            return params.empty_val

    def __eq__( self, val ):
        """Test equality, either with another list of targets or with a single
        target. Returns a list of truth values."""
        if type(val) == type(params.empty_val):
            rtn = []
            for target in self.itervalues():
                if target == val:
                    rtn.append( True )
                else:
                    rtn.append( False )
        elif len(val) == len(self.list):
            rtn = []
            for i, target in self.iteritems():
                if val.hasItem(i) and target == val[i]:
                    rtn.append( True )
                else:
                    rtn.append( False )
        else:
            raise TypeError( "must compare with a list of equal length" )

        return rtn

    def __ne__( self, other ): return not self.__eq__( other )

    def hasItem(self, i):
        return self.list.has_key(i)

    def isEmpty( self ):
        return self.__eq__( params.empty_val )

    def __nonzero__( self ):
        return not self.isEmpty()

    def __print__( self, verbose=False ):
        s = "{"
        for target in self.itervalues():
            s += target.__print__( verbose ) + "; "
        s += "\b\b}\n"
        return s

    def __str__( self ): return self.__print__( False )
    def __repr__( self ): return self.__print__( True )

    def append( self, target ):
        self.list[target.identity] = target

    def pop( self, i ): return self.list.pop( i )

    def copy( self ):
        other = TargetList()
        for target in self.itervalues():
            other.append( target.copy() )
        return other

    def itervalues(self):
        return self.list.itervalues()

    def iterkeys(self):
        return self.list.iterkeys()

    def iteritems(self):
        return self.list.iteritems()

    def keys(self):
        return self.list.keys()

# code for estimating connected component observations
import estconncomps as est

def find_ellipses( dfore , bw, dofix=True ):
    """Fits ellipses to connected components in image.
    Returns an EllipseList, each member representing
    the x,y position and orientation of a single fly."""

    # check number of above-threshold pixels
    rows, cols = num.nonzero( bw )

    # store current time to find out how long computing ccs takes
    last_time = time.time()

    # find connected components
    (L,ncc) = meas.label(bw)

    # make sure there aren't too many connected components
    if ncc > params.max_n_clusters:
        warn( "too many objects found (>%d); truncating object search"%(params.max_n_clusters) )
        # for now, just throw out the last connected components.
        # in the future, we can sort based on area and keep those
        # with the largest area. hopefully, this never actually
        # happens.
        ncc = params.max_n_clusters
        L[L >= ncc] = 0

    #print 'time to compute connected components: %.2f'%(time.time()-last_time)

    # store current time to find out how long fitting ellipses takes
    last_time = time.time()
        
    # fit ellipses
    ellipses = est.weightedregionprops(L,ncc,dfore)

    #print 'initial list of ellipses:'
    #for i in range(len(ellipses)):
    #    print 'ellipse[%d] = '%i + str(ellipses[i])

    #print 'time to fit ellipses: %.2f'%(time.time() - last_time)

    if dofix:

        # store current time to find out how long fitting ellipses takes
        last_time = time.time()

        # check if any are small, and [try to] fix those
        est.fixsmall(ellipses,L,dfore)

        #print 'after fixing small, ellipses = '
        #for i in range(len(ellipses)):
        #    print 'ellipse[%d] = '%i + str(ellipses[i])

        #print 'time to fix small ellipses: %.2f'%(time.time() - last_time)

        last_time = time.time()

        # check if any are large, and [try to] fix those
        est.fixlarge(ellipses,L,dfore)

        #print 'after fixing large, ellipses = \n'
        #for i in range(len(ellipses)):
        #    print 'ellipse[%d] = '%i + str(ellipses[i])

        #print 'time to fix large ellipses: %.2f'%(time.time() - last_time)

        #est.deleteellipses(ellipses)

    return ellipses

def find_ellipses2( dfore , bw, dofix=True ):
    """Fits ellipses to connected components in image.
    Returns an EllipseList, each member representing
    the x,y position and orientation of a single fly."""

    # check number of above-threshold pixels
    rows, cols = num.nonzero( bw )

    # find connected components
    (L,ncc) = meas.label(bw)
    
    # make sure there aren't too many connected components
    if ncc > params.max_n_clusters:
        warn( "too many objects found (>%d); truncating object search"%(params.max_n_clusters) )
        # for now, just throw out the last connected components.
        # in the future, we can sort based on area and keep those
        # with the largest area. hopefully, this never actually
        # happens.
        ncc = params.max_n_clusters
        L[L >= ncc] = 0
        
    # fit ellipses
    ellipses = est.weightedregionprops(L,ncc,dfore)

    if dofix:

        # check if any are small, and [try to] fix those
        est.fixsmall(ellipses,L,dfore)

        # check if any are large, and [try to] fix those
        est.fixlarge(ellipses,L,dfore)

    return (ellipses,L)

def est_shape( bg, tracking_settings_frame=None ):
    """Estimate fly shape from a bunch of sample frames."""

    interactive = params.interactive and tracking_settings_frame is not None
    if interactive:
        progressbar = \
            wx.ProgressDialog('Computing Shape Model',
                              'Detecting observations in %d frames to estimate median and median absolute deviation of shape parameters'%params.n_frames_size,
                              params.n_frames_size,
                              tracking_settings_frame,
                              wx.PD_APP_MODAL|wx.PD_AUTO_HIDE|wx.PD_CAN_ABORT|wx.PD_REMAINING_TIME)

    # which frames will we estimate size from
    framelist = num.round( num.linspace( 0, params.n_frames-1,
                                         params.n_frames_size ) ).astype( num.int )

    ellipses = []

    i = 0
    for frame in framelist:
        # get background-subtracted image
        if interactive:
            (keepgoing,skip) = progressbar.Update(value=i,newmsg='Detecting observations in frame %d (%d / %d)'%(frame,i,params.n_frames_size))
            i+=1
            if not keepgoing:
                progressbar.Destroy()
                return False
        try:
            (dfore,bw) = bg.sub_bg( frame )
        except:
            continue
        (L,ncc) = meas.label(bw)
        ellipsescurr = est.weightedregionprops(L,ncc,dfore)
        ellipses += ellipsescurr

    n_ell = len(ellipses)

    if n_ell == 0: # probably threshold is too low
        return False

    # grab ellipse info
    major = num.zeros( (n_ell) )
    minor = num.zeros( (n_ell) )
    area = num.zeros( (n_ell) )
    for i in range(len(ellipses)):
        major[i] = ellipses[i].size.height
        minor[i] = ellipses[i].size.width
        area[i] = ellipses[i].area

    eccen = minor / major
    
    # compute the median
    iseven = num.mod(n_ell,2) == 0
    middle1 = num.floor(n_ell/2)
    middle2 = middle1 - 1
    major.sort()
    minor.sort()
    area.sort()
    eccen.sort()
    mu_maj = major[middle1]
    mu_min = minor[middle1]
    mu_area = area[middle1]
    mu_ecc = eccen[middle1]
    if iseven:
        mu_maj = (mu_maj + major[middle2])/2.
        mu_min = (mu_min + minor[middle2])/2.
        mu_area = (mu_area + area[middle2])/2.
        mu_ecc = (mu_ecc + eccen[middle2])/2.

    # compute absolute difference
    major = num.abs(major - mu_maj)
    minor = num.abs(minor - mu_min)
    area = num.abs(area - mu_area)
    eccen = num.abs(eccen - mu_ecc)

    # compute the median absolute difference
    major.sort()
    minor.sort()
    area.sort()
    eccen.sort()

    sigma_maj = major[middle1]
    sigma_min = minor[middle1]
    sigma_area = area[middle1]
    sigma_ecc = eccen[middle1]
    if iseven:
        sigma_maj = (sigma_maj + major[middle2])/2.
        sigma_min = (sigma_min + minor[middle2])/2.
        sigma_area = (sigma_area + area[middle2])/2.
        sigma_ecc = (sigma_ecc + eccen[middle2])/2.

    # estimate standard deviation assuming a Gaussian distribution
    # from the fact that half the data falls within mad
    # MADTOSTDFACTOR = 1./norminv(.75)
    MADTOSTDFACTOR = 1.482602
    sigma_maj *= MADTOSTDFACTOR
    sigma_min *= MADTOSTDFACTOR
    sigma_area *= MADTOSTDFACTOR
    sigma_ecc *= MADTOSTDFACTOR

    # fit Gaussians to the minor, major, eccentricity, and area
    #mu_maj = major.mean()
    #sigma_maj = major.std()
    #mu_min = minor.mean()
    #sigma_min = minor.std()
    #mu_ecc = eccen.mean()
    #sigma_ecc = eccen.std()
    #mu_area = area.mean()
    #sigma_area = area.std()

    # threshold at N standard deviations
    params.maxshape.major = mu_maj + params.n_std_thresh*sigma_maj
    params.minshape.major = mu_maj - params.n_std_thresh*sigma_maj
    if params.minshape.major < 0: params.minshape.major = 0
    params.maxshape.minor = mu_min + params.n_std_thresh*sigma_min
    params.minshape.minor = mu_min - params.n_std_thresh*sigma_min
    if params.minshape.minor < 0: params.minshape.minor = 0
    params.maxshape.ecc = mu_ecc + params.n_std_thresh*sigma_ecc
    params.minshape.ecc = mu_ecc - params.n_std_thresh*sigma_ecc
    if params.minshape.ecc < 0: params.minshape.ecc = 0
    if params.maxshape.ecc > 1: params.maxshape.ecc = 1
    params.maxshape.area = mu_area + params.n_std_thresh*sigma_area
    params.minshape.area = mu_area - params.n_std_thresh*sigma_area
    if params.minshape.area < 0: params.minshape.area = 0

    params.meanshape.major = mu_maj
    params.meanshape.minor = mu_min
    params.meanshape.ecc = mu_ecc
    params.meanshape.area = mu_area

    params.have_computed_shape = True

    if interactive:
        progressbar.Destroy()
    
    return True

class EllipseWindow:
    """Container class for each ellipse drawing window."""
    def __init__( self, img_panel, ell_id=0 ):
        # create drawing window
        self.box = wx.BoxSizer( wx.HORIZONTAL )
        self.window = wxvideo.DynamicImageCanvas( img_panel, -1 )
        self.window.set_resize(True)
        self.box.Add( self.window, 1, wx.EXPAND|wx.ALIGN_CENTER_VERTICAL )

        # create spin control and text label in a separate sizer
        inbox = wx.BoxSizer( wx.HORIZONTAL )
        self.stext = wx.StaticText( img_panel, -1, "ID" )
        inbox.Add( self.stext, 0, wx.ALIGN_CENTER_VERTICAL )
        self.spinner = wx.SpinCtrl( img_panel, -1, min=0, max=params.nids,
                                    size=(params.id_spinner_width,-1),
                                    style=wx.SP_ARROW_KEYS|wx.SP_WRAP )
        self.spinner.SetValue( ell_id )
        inbox.Add( self.spinner, 0, wx.ALIGN_CENTER_VERTICAL )
        img_panel.Bind(wx.EVT_SPINCTRL,self.OnSpinner,self.spinner)
        
        self.box.Add( inbox, 0, wx.ALIGN_TOP )

        # initialize display information
        self.data = num.ones( (params.zoom_window_pix,params.zoom_window_pix), num.uint8 )*127
        self.offset = (0,0)
        self.ellipse = Ellipse()
        self.color = [0,0,0]
        self.name = 'ellipse_%d'%ell_id

    def SetData(self,ellipses,img):
        self.ellipse = ellipses
        self.data = img
        self.spinner.SetRange(0,params.nids-1)

    def __del__( self ):
        try:
            self.window.Destroy() # could fail if EllipseFrame is already closed
            self.spinner.Destroy()
            self.stext.Destroy()
        except wx._core.PyDeadObjectError: pass
        # TODO: maybe try/catch isn't the right thing here...
        # what if this seg-faults on some other OS?

    def OnSpinner(self,evt):
        
        self.redraw()

    def redraw(self,eraseBackground=False):
        """Scale data and draw on window."""

        if (not hasattr(self,'ellipse')) or \
           (not hasattr(self.ellipse,'hasItem')):
            return

        ind = self.spinner.GetValue()
        if not self.ellipse.hasItem(ind):
            blank = num.ones( (params.zoom_window_pix,
                               params.zoom_window_pix), num.uint8 )*127
            self.window.update_image_and_drawings(self.name,blank,format='MONO8')
        else:
            # get box around ellipse
            valx = self.ellipse[ind].center.x - params.zoom_window_pix/2
            valx = max( valx, 0 )
            valx = min( valx, self.data.shape[1] - params.zoom_window_pix )
            valy = self.ellipse[ind].center.y - params.zoom_window_pix/2
            valy = max( valy, 0 )
            valy = min( valy, self.data.shape[0] - params.zoom_window_pix )
            self.offset = (valx,valy)
            zoomaxes = [self.offset[0],self.offset[0]+params.zoom_window_pix-1,
                        self.offset[1],self.offset[1]+params.zoom_window_pix-1]
            linesegs = draw_ellipses([self.ellipse[ind],],
                                     colors=[params.colors[ind%len(params.colors)],])
            im = imagesk.double2mono8(self.data,donormalize=False)
            linesegs,im = imagesk.zoom_linesegs_and_image(linesegs,im,zoomaxes)
            (linesegs,linecolors) = imagesk.separate_linesegs_colors(linesegs)
            self.window.update_image_and_drawings(self.name,im,format='MONO8',
                                                  linesegs=linesegs,lineseg_colors=linecolors)
            self.window.Refresh( eraseBackground=eraseBackground )
class EllipseFrame:
    """Window to show zoomed objects with their fit ellipses."""
    def __init__( self, parent):
        rsrc = xrc.XmlResource( ZOOM_RSRC_FILE )
        self.frame = rsrc.LoadFrame( parent, "frame_ellipses" )

        self.n_ell_spinner = xrc.XRCCTRL( self.frame, "spin_n_ellipses" )
        self.frame.Bind( wx.EVT_SPINCTRL, self.OnNEllSpinner, id=xrc.XRCID("spin_n_ellipses") )

        # set up image panel
        self.img_panel = xrc.XRCCTRL( self.frame, "panel_show" )
        self.img_box = wx.BoxSizer( wx.VERTICAL )
        self.img_panel.SetSizer( self.img_box )

        # make ellipse window
        self.ellipse_windows = [EllipseWindow( self.img_panel, 0 )]
        self.n_ell = 1

        # add ellipse window to img_panel
        self.img_box.Add( self.ellipse_windows[0].box, 1, wx.EXPAND )
        self.img_panel.SetAutoLayout( True )
        self.img_panel.Layout()

        self.frame.Show()

    def SetData(self,ellipses,img):
        self.ellipses = ellipses
        self.img = img
        for window in self.ellipse_windows:
            window.SetData(ellipses,img)

    def AddEllipseWindow( self, id ):
        id_list = [] # currently shown IDs
        for window in self.ellipse_windows:
            id_list.append( window.spinner.GetValue() )
        id_list.append(id)
        self.SetEllipseWindows(id_list)
        self.n_ell += 1
        self.n_ell_spinner.SetValue(self.n_ell)

    def OnNEllSpinner( self, evt ):
        """Remove or add a window."""

        id_list = [] # currently shown IDs
        for window in self.ellipse_windows:
            id_list.append( window.spinner.GetValue() )
                
        # determine whether to add or remove a window
        new_n_ell = self.n_ell_spinner.GetValue()
        if new_n_ell > self.n_ell:
            # add a window; try to be intelligent about which initial ID
            max_id = params.nids
            use_id = max_id-1
            while use_id in id_list and use_id >= 0:
                use_id -= 1
            id_list.append(use_id)
        elif new_n_ell < self.n_ell:
            id_list.pop()
        
        self.SetEllipseWindows(id_list)
        self.n_ell = new_n_ell
    
    def SetEllipseWindows(self,id_list):
        self.img_box.DeleteWindows()
        #for i in range(len(self.ellipse_windows)):
        #    self.ellipse_windows[i].Destroy()
        self.ellipse_windows = []
        for id in id_list:
            self.ellipse_windows.append( EllipseWindow( self.img_panel, id ) )
            self.ellipse_windows[-1].SetData(self.ellipses,self.img)
            self.img_box.Add( self.ellipse_windows[-1].box, 1, wx.EXPAND )
            # TODO: update a zoom window when its spinctrl spins
            #self.frame.Bind( wx.EVT_SPINCTRL, self.ellipse_windows[-1].

        # resize panel
        self.img_box.Layout()
        self.img_panel.Layout()
        self.Redraw(True)

    def Redraw( self,eraseBackground=False ):
        """Scale images and display."""
        for window in self.ellipse_windows:
            window.redraw(eraseBackground=eraseBackground)

def find_ellipses_display( dfore , bw ):
    """Fits ellipses to connected components in image.
    Returns an EllipseList, each member representing
    the x,y position and orientation of a single fly."""

    # check number of above-threshold pixels
    rows, cols = num.nonzero( bw )

    # find connected components
    (L,ncc) = meas.label(bw)
    
    # make sure there aren't too many connected components
    if ncc > params.max_n_clusters:
        warn( "too many objects found (>%d); truncating object search"%(params.max_n_clusters) )
        # for now, just throw out the last connected components.
        # in the future, we can sort based on area and keep those
        # with the largest area. hopefully, this never actually
        # happens.
        ncc = params.max_n_clusters
        L[L >= ncc] = 0
        
    # fit ellipses
    ellipses = est.weightedregionprops(L,ncc,dfore)

    # check if any are small, and [try to] fix those
    (issmall,didlowerthresh,didmerge,diddelete) = est.fixsmalldisplay(ellipses,L,dfore)

    # check if any are large, and [try to] fix those
    (islarge,didsplit) = est.fixlargedisplay(ellipses,L,dfore)
    
    return (ellipses,issmall,islarge,didlowerthresh,didmerge,diddelete,didsplit)

def draw_ellipses( targets, thickness=1, step=10*num.pi/180., colors=params.colors ):
    """Draw ellipses on a color image (MxNx3 numpy array).
    Refuses to draw empty ellipses (center==size==area==0)."""

    if hasattr(targets,'iteritems'):
        targetiterator = targets.iteritems()
    else:
        targetiterator = enumerate(targets)

    # create list of lines to draw
    lines = []
    for i, target in targetiterator:
        # don't draw empty ellipses
        if target.isEmpty(): continue

	# color of this ellipse
	color = colors[i%len(colors)]

        # create line	
        points = []
        alpha = num.sin( target.angle )
        beta = num.cos( target.angle )
	
	# do the first point
	aa = 0.
        ax = 2.*target.size.width * num.cos( aa )
        ay = 2.*target.size.height * num.sin( aa )
        xx = target.center.x + ax*alpha - ay*beta
        yy = target.center.y - ax*beta - ay*alpha

        for aa in num.arange( step, (2.*num.pi+step), step ):
	    
	    # save previous position
	    xprev = xx
	    yprev = yy

            ax = 2.*target.size.width * num.cos( aa )
            ay = 2.*target.size.height * num.sin( aa )
            xx = target.center.x + ax*alpha - ay*beta
            yy = target.center.y - ax*beta - ay*alpha
	    
            lines.append([xprev+1,yprev+1,xx+1,yy+1,color])

    return lines

def draw_ellipses_bmp( img, targets, thickness=1, step=10*num.pi/180., colors=params.colors, windowsize=None, zoomaxes=None ):
    """Draw ellipses on a color image (MxNx3 numpy array).
    Refuses to draw empty ellipses (center==size==area==0)."""

    if zoomaxes is None:
        zoomaxes = [0,img.shape[1]-1,0,img.shape[0]-1]
    
    if hasattr(targets,'iteritems'):
        targetiterator = targets.iteritems()
    else:
        targetiterator = enumerate(targets)

    # create list of line colors
    linecolors = []
    for i, target in targetiterator:
        # don't draw empty ellipses
        if target.isEmpty(): continue
        # set color
        linecolors.append(colors[i%len(colors)])

    if hasattr(targets,'iteritems'):
        targetiterator = targets.iteritems()
    else:
        targetiterator = enumerate(targets)

    # create list of lines to draw
    linelists = []
    for i, target in targetiterator:
        # don't draw empty ellipses
        if target.isEmpty(): continue
        # create line
        points = []
        alpha = num.sin( target.angle )
        beta = num.cos( target.angle )
        for aa in num.arange( 0., (2.*num.pi+step), step ):
            ax = 2.*target.size.width * num.cos( aa )
            ay = 2.*target.size.height * num.sin( aa )
            xx = target.center.x + ax*alpha - ay*beta
            yy = target.center.y - ax*beta - ay*alpha
            if xx < zoomaxes[0] or xx > zoomaxes[1] or yy < zoomaxes[2] or yy > zoomaxes[3]: continue
            points.append([xx+1,yy+1])
        linelists.append(points)

    bmp = imagesk.draw_annotated_image(img,linelists=linelists,windowsize=windowsize,zoomaxes=zoomaxes,
                                       linecolors=linecolors)

    return bmp

def find_flies( old0, old1, obs ):
    """All arguments are EllipseLists. Returns an EllipseList."""
    # possibly matchidentities should be smart enough to deal with this case
    # instead of handling it specially here

    if len(obs) == 0:
        flies = TargetList()
        #for e in old1:
        #    flies.append( Ellipse() ) # no obs for any target
        return flies

    # make predicted targets
    targ = m_id.cvpred( old0, old1 )

    if params.print_crap:
        print "targ (%d)"%len(targ), targ
        print "obs (%d)"%len(obs), obs

    # make a cost matrix containing the distance from each target to each obs
    ids = []
    for i in targ.iterkeys():
        ids.append(i)
    vals = []
    for i in targ.itervalues():
        vals.append(i)
    cost = num.zeros( (len(obs), len(targ)) )
    for i, observation in enumerate( obs ):
        for j, target in enumerate( vals ):
            if target.isDummy():
                cost[i,j] = params.max_jump + eps # will be ignored
            else:
                cost[i,j] = observation.dist( target )

    # find optimal matching between targ and observations
    obs_for_target, unass_obs = m_id.matchidentities( cost, params.max_jump )
    if params.print_crap: print "best matches:", obs_for_target

    # make a new list containing the best matches to each prediction
    flies = TargetList()
    for tt in range( len(targ) ):
        if obs_for_target[tt] >= 0:
            obs[obs_for_target[tt]].identity = ids[tt]
            flies.append( obs[obs_for_target[tt]] )
        #else:
        #    flies.append( Ellipse() ) # empty ellipse as a placeholder

    # append the targets that didn't match any observation
    for oo in range( len(obs) ):
        if unass_obs[oo]:
            obs[oo].identity = params.nids
            params.nids+=1
            flies.append( obs[oo] )
            
    if params.print_crap:
        print "returning", flies
    return flies

